/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.outposts;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.outposts.model.AccessDeniedException;
import software.amazon.awssdk.services.outposts.model.CancelOrderRequest;
import software.amazon.awssdk.services.outposts.model.CancelOrderResponse;
import software.amazon.awssdk.services.outposts.model.ConflictException;
import software.amazon.awssdk.services.outposts.model.CreateOrderRequest;
import software.amazon.awssdk.services.outposts.model.CreateOrderResponse;
import software.amazon.awssdk.services.outposts.model.CreateOutpostRequest;
import software.amazon.awssdk.services.outposts.model.CreateOutpostResponse;
import software.amazon.awssdk.services.outposts.model.CreateSiteRequest;
import software.amazon.awssdk.services.outposts.model.CreateSiteResponse;
import software.amazon.awssdk.services.outposts.model.DeleteOutpostRequest;
import software.amazon.awssdk.services.outposts.model.DeleteOutpostResponse;
import software.amazon.awssdk.services.outposts.model.DeleteSiteRequest;
import software.amazon.awssdk.services.outposts.model.DeleteSiteResponse;
import software.amazon.awssdk.services.outposts.model.GetCatalogItemRequest;
import software.amazon.awssdk.services.outposts.model.GetCatalogItemResponse;
import software.amazon.awssdk.services.outposts.model.GetConnectionRequest;
import software.amazon.awssdk.services.outposts.model.GetConnectionResponse;
import software.amazon.awssdk.services.outposts.model.GetOrderRequest;
import software.amazon.awssdk.services.outposts.model.GetOrderResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostResponse;
import software.amazon.awssdk.services.outposts.model.GetSiteAddressRequest;
import software.amazon.awssdk.services.outposts.model.GetSiteAddressResponse;
import software.amazon.awssdk.services.outposts.model.GetSiteRequest;
import software.amazon.awssdk.services.outposts.model.GetSiteResponse;
import software.amazon.awssdk.services.outposts.model.InternalServerException;
import software.amazon.awssdk.services.outposts.model.ListAssetsRequest;
import software.amazon.awssdk.services.outposts.model.ListAssetsResponse;
import software.amazon.awssdk.services.outposts.model.ListCatalogItemsRequest;
import software.amazon.awssdk.services.outposts.model.ListCatalogItemsResponse;
import software.amazon.awssdk.services.outposts.model.ListOrdersRequest;
import software.amazon.awssdk.services.outposts.model.ListOrdersResponse;
import software.amazon.awssdk.services.outposts.model.ListOutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ListOutpostsResponse;
import software.amazon.awssdk.services.outposts.model.ListSitesRequest;
import software.amazon.awssdk.services.outposts.model.ListSitesResponse;
import software.amazon.awssdk.services.outposts.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.outposts.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.outposts.model.NotFoundException;
import software.amazon.awssdk.services.outposts.model.OutpostsException;
import software.amazon.awssdk.services.outposts.model.OutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.outposts.model.StartConnectionRequest;
import software.amazon.awssdk.services.outposts.model.StartConnectionResponse;
import software.amazon.awssdk.services.outposts.model.TagResourceRequest;
import software.amazon.awssdk.services.outposts.model.TagResourceResponse;
import software.amazon.awssdk.services.outposts.model.UntagResourceRequest;
import software.amazon.awssdk.services.outposts.model.UntagResourceResponse;
import software.amazon.awssdk.services.outposts.model.UpdateOutpostRequest;
import software.amazon.awssdk.services.outposts.model.UpdateOutpostResponse;
import software.amazon.awssdk.services.outposts.model.UpdateSiteAddressRequest;
import software.amazon.awssdk.services.outposts.model.UpdateSiteAddressResponse;
import software.amazon.awssdk.services.outposts.model.UpdateSiteRackPhysicalPropertiesRequest;
import software.amazon.awssdk.services.outposts.model.UpdateSiteRackPhysicalPropertiesResponse;
import software.amazon.awssdk.services.outposts.model.UpdateSiteRequest;
import software.amazon.awssdk.services.outposts.model.UpdateSiteResponse;
import software.amazon.awssdk.services.outposts.model.ValidationException;
import software.amazon.awssdk.services.outposts.paginators.GetOutpostInstanceTypesPublisher;
import software.amazon.awssdk.services.outposts.paginators.ListAssetsPublisher;
import software.amazon.awssdk.services.outposts.paginators.ListCatalogItemsPublisher;
import software.amazon.awssdk.services.outposts.paginators.ListOrdersPublisher;
import software.amazon.awssdk.services.outposts.paginators.ListOutpostsPublisher;
import software.amazon.awssdk.services.outposts.paginators.ListSitesPublisher;
import software.amazon.awssdk.services.outposts.transform.CancelOrderRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.CreateOrderRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.CreateOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.CreateSiteRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.DeleteOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.DeleteSiteRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetCatalogItemRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetConnectionRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOrderRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostInstanceTypesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetSiteAddressRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetSiteRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListAssetsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListCatalogItemsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListOrdersRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListOutpostsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListSitesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.StartConnectionRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateSiteAddressRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateSiteRackPhysicalPropertiesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateSiteRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link OutpostsAsyncClient}.
 *
 * @see OutpostsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOutpostsAsyncClient implements OutpostsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultOutpostsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final OutpostsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultOutpostsAsyncClient(OutpostsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels the specified order for an Outpost.
     * </p>
     *
     * @param cancelOrderRequest
     * @return A Java Future containing the result of the CancelOrder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.CancelOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CancelOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelOrderResponse> cancelOrder(CancelOrderRequest cancelOrderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelOrder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelOrderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelOrderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelOrderRequest, CancelOrderResponse>()
                            .withOperationName("CancelOrder").withMarshaller(new CancelOrderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelOrderRequest));
            CompletableFuture<CancelOrderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an order for an Outpost.
     * </p>
     *
     * @param createOrderRequest
     * @return A Java Future containing the result of the CreateOrder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ServiceQuotaExceededException You have exceeded a service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.CreateOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateOrderResponse> createOrder(CreateOrderRequest createOrderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOrder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateOrderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateOrderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOrderRequest, CreateOrderResponse>()
                            .withOperationName("CreateOrder").withMarshaller(new CreateOrderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createOrderRequest));
            CompletableFuture<CreateOrderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Outpost.
     * </p>
     * <p>
     * You can specify either an Availability one or an AZ ID.
     * </p>
     *
     * @param createOutpostRequest
     * @return A Java Future containing the result of the CreateOutpost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ServiceQuotaExceededException You have exceeded a service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.CreateOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateOutpostResponse> createOutpost(CreateOutpostRequest createOutpostRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOutpost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateOutpostResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateOutpostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOutpostRequest, CreateOutpostResponse>()
                            .withOperationName("CreateOutpost")
                            .withMarshaller(new CreateOutpostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createOutpostRequest));
            CompletableFuture<CreateOutpostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a site for an Outpost.
     * </p>
     *
     * @param createSiteRequest
     * @return A Java Future containing the result of the CreateSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ServiceQuotaExceededException You have exceeded a service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.CreateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSiteResponse> createSite(CreateSiteRequest createSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSiteRequest, CreateSiteResponse>().withOperationName("CreateSite")
                            .withMarshaller(new CreateSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSiteRequest));
            CompletableFuture<CreateSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified Outpost.
     * </p>
     *
     * @param deleteOutpostRequest
     * @return A Java Future containing the result of the DeleteOutpost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.DeleteOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/DeleteOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteOutpostResponse> deleteOutpost(DeleteOutpostRequest deleteOutpostRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOutpost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteOutpostResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteOutpostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteOutpostRequest, DeleteOutpostResponse>()
                            .withOperationName("DeleteOutpost")
                            .withMarshaller(new DeleteOutpostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteOutpostRequest));
            CompletableFuture<DeleteOutpostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified site.
     * </p>
     *
     * @param deleteSiteRequest
     * @return A Java Future containing the result of the DeleteSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.DeleteSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/DeleteSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSiteResponse> deleteSite(DeleteSiteRequest deleteSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSiteRequest, DeleteSiteResponse>().withOperationName("DeleteSite")
                            .withMarshaller(new DeleteSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSiteRequest));
            CompletableFuture<DeleteSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified catalog item.
     * </p>
     *
     * @param getCatalogItemRequest
     * @return A Java Future containing the result of the GetCatalogItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetCatalogItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetCatalogItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCatalogItemResponse> getCatalogItem(GetCatalogItemRequest getCatalogItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCatalogItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCatalogItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCatalogItemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCatalogItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCatalogItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCatalogItemRequest, GetCatalogItemResponse>()
                            .withOperationName("GetCatalogItem")
                            .withMarshaller(new GetCatalogItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCatalogItemRequest));
            CompletableFuture<GetCatalogItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * Amazon Web Services uses this action to install Outpost servers.
     * </p>
     * </note>
     * <p>
     * Gets information about the specified connection.
     * </p>
     * <p>
     * Use CloudTrail to monitor this action or Amazon Web Services managed policy for Amazon Web Services Outposts to
     * secure it. For more information, see <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/security-iam-awsmanpol.html"> Amazon Web Services
     * managed policies for Amazon Web Services Outposts</a> and <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/logging-using-cloudtrail.html"> Logging Amazon Web
     * Services Outposts API calls with Amazon Web Services CloudTrail</a> in the <i>Amazon Web Services Outposts User
     * Guide</i>.
     * </p>
     *
     * @param getConnectionRequest
     * @return A Java Future containing the result of the GetConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetConnectionResponse> getConnection(GetConnectionRequest getConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConnectionRequest, GetConnectionResponse>()
                            .withOperationName("GetConnection")
                            .withMarshaller(new GetConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getConnectionRequest));
            CompletableFuture<GetConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified order.
     * </p>
     *
     * @param getOrderRequest
     * @return A Java Future containing the result of the GetOrder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetOrderResponse> getOrder(GetOrderRequest getOrderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOrder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOrderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOrderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOrderRequest, GetOrderResponse>().withOperationName("GetOrder")
                            .withMarshaller(new GetOrderRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getOrderRequest));
            CompletableFuture<GetOrderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified Outpost.
     * </p>
     *
     * @param getOutpostRequest
     * @return A Java Future containing the result of the GetOutpost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetOutpostResponse> getOutpost(GetOutpostRequest getOutpostRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutpost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOutpostResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOutpostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOutpostRequest, GetOutpostResponse>().withOperationName("GetOutpost")
                            .withMarshaller(new GetOutpostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getOutpostRequest));
            CompletableFuture<GetOutpostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the instance types for the specified Outpost.
     * </p>
     *
     * @param getOutpostInstanceTypesRequest
     * @return A Java Future containing the result of the GetOutpostInstanceTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetOutpostInstanceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpostInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOutpostInstanceTypesResponse> getOutpostInstanceTypes(
            GetOutpostInstanceTypesRequest getOutpostInstanceTypesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutpostInstanceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutpostInstanceTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOutpostInstanceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOutpostInstanceTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOutpostInstanceTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOutpostInstanceTypesRequest, GetOutpostInstanceTypesResponse>()
                            .withOperationName("GetOutpostInstanceTypes")
                            .withMarshaller(new GetOutpostInstanceTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getOutpostInstanceTypesRequest));
            CompletableFuture<GetOutpostInstanceTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the instance types for the specified Outpost.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getOutpostInstanceTypes(software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.GetOutpostInstanceTypesPublisher publisher = client.getOutpostInstanceTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.GetOutpostInstanceTypesPublisher publisher = client.getOutpostInstanceTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getOutpostInstanceTypes(software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param getOutpostInstanceTypesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetOutpostInstanceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpostInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    public GetOutpostInstanceTypesPublisher getOutpostInstanceTypesPaginator(
            GetOutpostInstanceTypesRequest getOutpostInstanceTypesRequest) {
        return new GetOutpostInstanceTypesPublisher(this, applyPaginatorUserAgent(getOutpostInstanceTypesRequest));
    }

    /**
     * <p>
     * Gets information about the specified Outpost site.
     * </p>
     *
     * @param getSiteRequest
     * @return A Java Future containing the result of the GetSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSiteResponse> getSite(GetSiteRequest getSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSiteRequest, GetSiteResponse>().withOperationName("GetSite")
                            .withMarshaller(new GetSiteRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSiteRequest));
            CompletableFuture<GetSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the site address of the specified site.
     * </p>
     *
     * @param getSiteAddressRequest
     * @return A Java Future containing the result of the GetSiteAddress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetSiteAddress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetSiteAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSiteAddressResponse> getSiteAddress(GetSiteAddressRequest getSiteAddressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSiteAddressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSiteAddress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSiteAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSiteAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSiteAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSiteAddressRequest, GetSiteAddressResponse>()
                            .withOperationName("GetSiteAddress")
                            .withMarshaller(new GetSiteAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSiteAddressRequest));
            CompletableFuture<GetSiteAddressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the hardware assets for the specified Outpost.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listAssetsRequest
     * @return A Java Future containing the result of the ListAssets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListAssets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListAssets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssetsResponse> listAssets(ListAssetsRequest listAssetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAssetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssetsRequest, ListAssetsResponse>().withOperationName("ListAssets")
                            .withMarshaller(new ListAssetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAssetsRequest));
            CompletableFuture<ListAssetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the hardware assets for the specified Outpost.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAssets(software.amazon.awssdk.services.outposts.model.ListAssetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListAssetsPublisher publisher = client.listAssetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListAssetsPublisher publisher = client.listAssetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.ListAssetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.ListAssetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssets(software.amazon.awssdk.services.outposts.model.ListAssetsRequest)} operation.</b>
     * </p>
     *
     * @param listAssetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListAssets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListAssets" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAssetsPublisher listAssetsPaginator(ListAssetsRequest listAssetsRequest) {
        return new ListAssetsPublisher(this, applyPaginatorUserAgent(listAssetsRequest));
    }

    /**
     * <p>
     * Lists the items in the catalog.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listCatalogItemsRequest
     * @return A Java Future containing the result of the ListCatalogItems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListCatalogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListCatalogItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListCatalogItemsResponse> listCatalogItems(ListCatalogItemsRequest listCatalogItemsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCatalogItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCatalogItems");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCatalogItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCatalogItemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCatalogItemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCatalogItemsRequest, ListCatalogItemsResponse>()
                            .withOperationName("ListCatalogItems")
                            .withMarshaller(new ListCatalogItemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCatalogItemsRequest));
            CompletableFuture<ListCatalogItemsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the items in the catalog.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCatalogItems(software.amazon.awssdk.services.outposts.model.ListCatalogItemsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListCatalogItemsPublisher publisher = client.listCatalogItemsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListCatalogItemsPublisher publisher = client.listCatalogItemsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.ListCatalogItemsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.ListCatalogItemsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCatalogItems(software.amazon.awssdk.services.outposts.model.ListCatalogItemsRequest)} operation.</b>
     * </p>
     *
     * @param listCatalogItemsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListCatalogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListCatalogItems" target="_top">AWS API
     *      Documentation</a>
     */
    public ListCatalogItemsPublisher listCatalogItemsPaginator(ListCatalogItemsRequest listCatalogItemsRequest) {
        return new ListCatalogItemsPublisher(this, applyPaginatorUserAgent(listCatalogItemsRequest));
    }

    /**
     * <p>
     * Lists the Outpost orders for your Amazon Web Services account.
     * </p>
     *
     * @param listOrdersRequest
     * @return A Java Future containing the result of the ListOrders operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOrders" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrdersResponse> listOrders(ListOrdersRequest listOrdersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOrdersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrdersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListOrdersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrdersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrdersRequest, ListOrdersResponse>().withOperationName("ListOrders")
                            .withMarshaller(new ListOrdersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOrdersRequest));
            CompletableFuture<ListOrdersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Outpost orders for your Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOrders(software.amazon.awssdk.services.outposts.model.ListOrdersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOrdersPublisher publisher = client.listOrdersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOrdersPublisher publisher = client.listOrdersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.ListOrdersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.ListOrdersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOrders(software.amazon.awssdk.services.outposts.model.ListOrdersRequest)} operation.</b>
     * </p>
     *
     * @param listOrdersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOrders" target="_top">AWS API
     *      Documentation</a>
     */
    public ListOrdersPublisher listOrdersPaginator(ListOrdersRequest listOrdersRequest) {
        return new ListOrdersPublisher(this, applyPaginatorUserAgent(listOrdersRequest));
    }

    /**
     * <p>
     * Lists the Outposts for your Amazon Web Services account.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listOutpostsRequest
     * @return A Java Future containing the result of the ListOutposts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListOutposts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListOutpostsResponse> listOutposts(ListOutpostsRequest listOutpostsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOutpostsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOutposts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOutpostsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListOutpostsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOutpostsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOutpostsRequest, ListOutpostsResponse>()
                            .withOperationName("ListOutposts").withMarshaller(new ListOutpostsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOutpostsRequest));
            CompletableFuture<ListOutpostsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Outposts for your Amazon Web Services account.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsPublisher publisher = client.listOutpostsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsPublisher publisher = client.listOutpostsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.ListOutpostsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.ListOutpostsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)} operation.</b>
     * </p>
     *
     * @param listOutpostsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListOutposts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListOutpostsPublisher listOutpostsPaginator(ListOutpostsRequest listOutpostsRequest) {
        return new ListOutpostsPublisher(this, applyPaginatorUserAgent(listOutpostsRequest));
    }

    /**
     * <p>
     * Lists the Outpost sites for your Amazon Web Services account. Use filters to return specific results.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listSitesRequest
     * @return A Java Future containing the result of the ListSites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSitesResponse> listSites(ListSitesRequest listSitesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSitesRequest, ListSitesResponse>().withOperationName("ListSites")
                            .withMarshaller(new ListSitesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSitesRequest));
            CompletableFuture<ListSitesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Outpost sites for your Amazon Web Services account. Use filters to return specific results.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesPublisher publisher = client.listSitesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesPublisher publisher = client.listSitesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.ListSitesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.ListSitesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)} operation.</b>
     * </p>
     *
     * @param listSitesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSitesPublisher listSitesPaginator(ListSitesRequest listSitesRequest) {
        return new ListSitesPublisher(this, applyPaginatorUserAgent(listSitesRequest));
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * Amazon Web Services uses this action to install Outpost servers.
     * </p>
     * </note>
     * <p>
     * Starts the connection required for Outpost server installation.
     * </p>
     * <p>
     * Use CloudTrail to monitor this action or Amazon Web Services managed policy for Amazon Web Services Outposts to
     * secure it. For more information, see <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/security-iam-awsmanpol.html"> Amazon Web Services
     * managed policies for Amazon Web Services Outposts</a> and <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/logging-using-cloudtrail.html"> Logging Amazon Web
     * Services Outposts API calls with Amazon Web Services CloudTrail</a> in the <i>Amazon Web Services Outposts User
     * Guide</i>.
     * </p>
     *
     * @param startConnectionRequest
     * @return A Java Future containing the result of the StartConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.StartConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/StartConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartConnectionResponse> startConnection(StartConnectionRequest startConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartConnectionRequest, StartConnectionResponse>()
                            .withOperationName("StartConnection")
                            .withMarshaller(new StartConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startConnectionRequest));
            CompletableFuture<StartConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Outpost.
     * </p>
     *
     * @param updateOutpostRequest
     * @return A Java Future containing the result of the UpdateOutpost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.UpdateOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateOutpostResponse> updateOutpost(UpdateOutpostRequest updateOutpostRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOutpost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateOutpostResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateOutpostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateOutpostRequest, UpdateOutpostResponse>()
                            .withOperationName("UpdateOutpost")
                            .withMarshaller(new UpdateOutpostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateOutpostRequest));
            CompletableFuture<UpdateOutpostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified site.
     * </p>
     *
     * @param updateSiteRequest
     * @return A Java Future containing the result of the UpdateSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.UpdateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSiteResponse> updateSite(UpdateSiteRequest updateSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSiteRequest, UpdateSiteResponse>().withOperationName("UpdateSite")
                            .withMarshaller(new UpdateSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSiteRequest));
            CompletableFuture<UpdateSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the address of the specified site.
     * </p>
     * <p>
     * You can't update a site address if there is an order in progress. You must wait for the order to complete or
     * cancel the order.
     * </p>
     * <p>
     * You can update the operating address before you place an order at the site, or after all Outposts that belong to
     * the site have been deactivated.
     * </p>
     *
     * @param updateSiteAddressRequest
     * @return A Java Future containing the result of the UpdateSiteAddress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.UpdateSiteAddress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateSiteAddress" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSiteAddressResponse> updateSiteAddress(UpdateSiteAddressRequest updateSiteAddressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSiteAddressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSiteAddress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSiteAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSiteAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSiteAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSiteAddressRequest, UpdateSiteAddressResponse>()
                            .withOperationName("UpdateSiteAddress")
                            .withMarshaller(new UpdateSiteAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSiteAddressRequest));
            CompletableFuture<UpdateSiteAddressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the physical and logistical details for a rack at a site. For more information about hardware requirements
     * for racks, see <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/outposts-requirements.html#checklist">Network
     * readiness checklist</a> in the Amazon Web Services Outposts User Guide.
     * </p>
     * <p>
     * To update a rack at a site with an order of <code>IN_PROGRESS</code>, you must wait for the order to complete or
     * cancel the order.
     * </p>
     *
     * @param updateSiteRackPhysicalPropertiesRequest
     * @return A Java Future containing the result of the UpdateSiteRackPhysicalProperties operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>ConflictException Updating or deleting this resource can cause an inconsistent state.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.UpdateSiteRackPhysicalProperties
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateSiteRackPhysicalProperties"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSiteRackPhysicalPropertiesResponse> updateSiteRackPhysicalProperties(
            UpdateSiteRackPhysicalPropertiesRequest updateSiteRackPhysicalPropertiesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateSiteRackPhysicalPropertiesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSiteRackPhysicalProperties");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSiteRackPhysicalPropertiesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateSiteRackPhysicalPropertiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSiteRackPhysicalPropertiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSiteRackPhysicalPropertiesRequest, UpdateSiteRackPhysicalPropertiesResponse>()
                            .withOperationName("UpdateSiteRackPhysicalProperties")
                            .withMarshaller(new UpdateSiteRackPhysicalPropertiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSiteRackPhysicalPropertiesRequest));
            CompletableFuture<UpdateSiteRackPhysicalPropertiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final OutpostsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OutpostsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends OutpostsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
