/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.outposts;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.outposts.model.AccessDeniedException;
import software.amazon.awssdk.services.outposts.model.CancelOrderRequest;
import software.amazon.awssdk.services.outposts.model.CancelOrderResponse;
import software.amazon.awssdk.services.outposts.model.ConflictException;
import software.amazon.awssdk.services.outposts.model.CreateOrderRequest;
import software.amazon.awssdk.services.outposts.model.CreateOrderResponse;
import software.amazon.awssdk.services.outposts.model.CreateOutpostRequest;
import software.amazon.awssdk.services.outposts.model.CreateOutpostResponse;
import software.amazon.awssdk.services.outposts.model.CreateSiteRequest;
import software.amazon.awssdk.services.outposts.model.CreateSiteResponse;
import software.amazon.awssdk.services.outposts.model.DeleteOutpostRequest;
import software.amazon.awssdk.services.outposts.model.DeleteOutpostResponse;
import software.amazon.awssdk.services.outposts.model.DeleteSiteRequest;
import software.amazon.awssdk.services.outposts.model.DeleteSiteResponse;
import software.amazon.awssdk.services.outposts.model.GetCatalogItemRequest;
import software.amazon.awssdk.services.outposts.model.GetCatalogItemResponse;
import software.amazon.awssdk.services.outposts.model.GetConnectionRequest;
import software.amazon.awssdk.services.outposts.model.GetConnectionResponse;
import software.amazon.awssdk.services.outposts.model.GetOrderRequest;
import software.amazon.awssdk.services.outposts.model.GetOrderResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostResponse;
import software.amazon.awssdk.services.outposts.model.GetSiteAddressRequest;
import software.amazon.awssdk.services.outposts.model.GetSiteAddressResponse;
import software.amazon.awssdk.services.outposts.model.GetSiteRequest;
import software.amazon.awssdk.services.outposts.model.GetSiteResponse;
import software.amazon.awssdk.services.outposts.model.InternalServerException;
import software.amazon.awssdk.services.outposts.model.ListAssetsRequest;
import software.amazon.awssdk.services.outposts.model.ListAssetsResponse;
import software.amazon.awssdk.services.outposts.model.ListCatalogItemsRequest;
import software.amazon.awssdk.services.outposts.model.ListCatalogItemsResponse;
import software.amazon.awssdk.services.outposts.model.ListOrdersRequest;
import software.amazon.awssdk.services.outposts.model.ListOrdersResponse;
import software.amazon.awssdk.services.outposts.model.ListOutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ListOutpostsResponse;
import software.amazon.awssdk.services.outposts.model.ListSitesRequest;
import software.amazon.awssdk.services.outposts.model.ListSitesResponse;
import software.amazon.awssdk.services.outposts.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.outposts.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.outposts.model.NotFoundException;
import software.amazon.awssdk.services.outposts.model.OutpostsException;
import software.amazon.awssdk.services.outposts.model.OutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.outposts.model.StartConnectionRequest;
import software.amazon.awssdk.services.outposts.model.StartConnectionResponse;
import software.amazon.awssdk.services.outposts.model.TagResourceRequest;
import software.amazon.awssdk.services.outposts.model.TagResourceResponse;
import software.amazon.awssdk.services.outposts.model.UntagResourceRequest;
import software.amazon.awssdk.services.outposts.model.UntagResourceResponse;
import software.amazon.awssdk.services.outposts.model.UpdateOutpostRequest;
import software.amazon.awssdk.services.outposts.model.UpdateOutpostResponse;
import software.amazon.awssdk.services.outposts.model.UpdateSiteAddressRequest;
import software.amazon.awssdk.services.outposts.model.UpdateSiteAddressResponse;
import software.amazon.awssdk.services.outposts.model.UpdateSiteRackPhysicalPropertiesRequest;
import software.amazon.awssdk.services.outposts.model.UpdateSiteRackPhysicalPropertiesResponse;
import software.amazon.awssdk.services.outposts.model.UpdateSiteRequest;
import software.amazon.awssdk.services.outposts.model.UpdateSiteResponse;
import software.amazon.awssdk.services.outposts.model.ValidationException;
import software.amazon.awssdk.services.outposts.paginators.GetOutpostInstanceTypesIterable;
import software.amazon.awssdk.services.outposts.paginators.ListAssetsIterable;
import software.amazon.awssdk.services.outposts.paginators.ListCatalogItemsIterable;
import software.amazon.awssdk.services.outposts.paginators.ListOrdersIterable;
import software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable;
import software.amazon.awssdk.services.outposts.paginators.ListSitesIterable;
import software.amazon.awssdk.services.outposts.transform.CancelOrderRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.CreateOrderRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.CreateOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.CreateSiteRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.DeleteOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.DeleteSiteRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetCatalogItemRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetConnectionRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOrderRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostInstanceTypesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetSiteAddressRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetSiteRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListAssetsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListCatalogItemsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListOrdersRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListOutpostsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListSitesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.StartConnectionRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateSiteAddressRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateSiteRackPhysicalPropertiesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.UpdateSiteRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link OutpostsClient}.
 *
 * @see OutpostsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOutpostsClient implements OutpostsClient {
    private static final Logger log = Logger.loggerFor(DefaultOutpostsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final OutpostsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultOutpostsClient(OutpostsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels the specified order for an Outpost.
     * </p>
     *
     * @param cancelOrderRequest
     * @return Result of the CancelOrder operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.CancelOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CancelOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelOrderResponse cancelOrder(CancelOrderRequest cancelOrderRequest) throws ValidationException, ConflictException,
            AccessDeniedException, NotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelOrderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelOrder");

            return clientHandler.execute(new ClientExecutionParams<CancelOrderRequest, CancelOrderResponse>()
                    .withOperationName("CancelOrder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelOrderRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelOrderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an order for an Outpost.
     * </p>
     *
     * @param createOrderRequest
     * @return Result of the CreateOrder operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws ServiceQuotaExceededException
     *         You have exceeded a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.CreateOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateOrderResponse createOrder(CreateOrderRequest createOrderRequest) throws ValidationException, ConflictException,
            AccessDeniedException, NotFoundException, InternalServerException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateOrderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOrder");

            return clientHandler.execute(new ClientExecutionParams<CreateOrderRequest, CreateOrderResponse>()
                    .withOperationName("CreateOrder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createOrderRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateOrderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Outpost.
     * </p>
     * <p>
     * You can specify either an Availability one or an AZ ID.
     * </p>
     *
     * @param createOutpostRequest
     * @return Result of the CreateOutpost operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws ServiceQuotaExceededException
     *         You have exceeded a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.CreateOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateOutpostResponse createOutpost(CreateOutpostRequest createOutpostRequest) throws ValidationException,
            ConflictException, NotFoundException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateOutpostResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOutpost");

            return clientHandler.execute(new ClientExecutionParams<CreateOutpostRequest, CreateOutpostResponse>()
                    .withOperationName("CreateOutpost").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createOutpostRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateOutpostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a site for an Outpost.
     * </p>
     *
     * @param createSiteRequest
     * @return Result of the CreateSite operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws ServiceQuotaExceededException
     *         You have exceeded a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.CreateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSiteResponse createSite(CreateSiteRequest createSiteRequest) throws ValidationException, ConflictException,
            AccessDeniedException, InternalServerException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSite");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSiteRequest, CreateSiteResponse>().withOperationName("CreateSite")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createSiteRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified Outpost.
     * </p>
     *
     * @param deleteOutpostRequest
     * @return Result of the DeleteOutpost operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.DeleteOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/DeleteOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteOutpostResponse deleteOutpost(DeleteOutpostRequest deleteOutpostRequest) throws ValidationException,
            ConflictException, NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteOutpostResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOutpost");

            return clientHandler.execute(new ClientExecutionParams<DeleteOutpostRequest, DeleteOutpostResponse>()
                    .withOperationName("DeleteOutpost").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteOutpostRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteOutpostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified site.
     * </p>
     *
     * @param deleteSiteRequest
     * @return Result of the DeleteSite operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.DeleteSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/DeleteSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSiteResponse deleteSite(DeleteSiteRequest deleteSiteRequest) throws ValidationException, ConflictException,
            NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSite");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSiteRequest, DeleteSiteResponse>().withOperationName("DeleteSite")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteSiteRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified catalog item.
     * </p>
     *
     * @param getCatalogItemRequest
     * @return Result of the GetCatalogItem operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetCatalogItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetCatalogItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCatalogItemResponse getCatalogItem(GetCatalogItemRequest getCatalogItemRequest) throws ValidationException,
            NotFoundException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCatalogItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCatalogItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCatalogItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCatalogItem");

            return clientHandler.execute(new ClientExecutionParams<GetCatalogItemRequest, GetCatalogItemResponse>()
                    .withOperationName("GetCatalogItem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getCatalogItemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCatalogItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * Amazon Web Services uses this action to install Outpost servers.
     * </p>
     * </note>
     * <p>
     * Gets information about the specified connection.
     * </p>
     * <p>
     * Use CloudTrail to monitor this action or Amazon Web Services managed policy for Amazon Web Services Outposts to
     * secure it. For more information, see <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/security-iam-awsmanpol.html"> Amazon Web Services
     * managed policies for Amazon Web Services Outposts</a> and <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/logging-using-cloudtrail.html"> Logging Amazon Web
     * Services Outposts API calls with Amazon Web Services CloudTrail</a> in the <i>Amazon Web Services Outposts User
     * Guide</i>.
     * </p>
     *
     * @param getConnectionRequest
     * @return Result of the GetConnection operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetConnectionResponse getConnection(GetConnectionRequest getConnectionRequest) throws AccessDeniedException,
            ValidationException, NotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConnectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnection");

            return clientHandler.execute(new ClientExecutionParams<GetConnectionRequest, GetConnectionResponse>()
                    .withOperationName("GetConnection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getConnectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified order.
     * </p>
     *
     * @param getOrderRequest
     * @return Result of the GetOrder operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetOrderResponse getOrder(GetOrderRequest getOrderRequest) throws ValidationException, NotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOrderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOrder");

            return clientHandler.execute(new ClientExecutionParams<GetOrderRequest, GetOrderResponse>()
                    .withOperationName("GetOrder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getOrderRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetOrderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified Outpost.
     * </p>
     *
     * @param getOutpostRequest
     * @return Result of the GetOutpost operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetOutpostResponse getOutpost(GetOutpostRequest getOutpostRequest) throws ValidationException, NotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOutpostResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutpost");

            return clientHandler
                    .execute(new ClientExecutionParams<GetOutpostRequest, GetOutpostResponse>().withOperationName("GetOutpost")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getOutpostRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetOutpostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the instance types for the specified Outpost.
     * </p>
     *
     * @param getOutpostInstanceTypesRequest
     * @return Result of the GetOutpostInstanceTypes operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetOutpostInstanceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpostInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOutpostInstanceTypesResponse getOutpostInstanceTypes(GetOutpostInstanceTypesRequest getOutpostInstanceTypesRequest)
            throws ValidationException, NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOutpostInstanceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetOutpostInstanceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutpostInstanceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutpostInstanceTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<GetOutpostInstanceTypesRequest, GetOutpostInstanceTypesResponse>()
                            .withOperationName("GetOutpostInstanceTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getOutpostInstanceTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetOutpostInstanceTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the instance types for the specified Outpost.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getOutpostInstanceTypes(software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.GetOutpostInstanceTypesIterable responses = client.getOutpostInstanceTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.GetOutpostInstanceTypesIterable responses = client
     *             .getOutpostInstanceTypesPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.GetOutpostInstanceTypesIterable responses = client.getOutpostInstanceTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getOutpostInstanceTypes(software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param getOutpostInstanceTypesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetOutpostInstanceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpostInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOutpostInstanceTypesIterable getOutpostInstanceTypesPaginator(
            GetOutpostInstanceTypesRequest getOutpostInstanceTypesRequest) throws ValidationException, NotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        return new GetOutpostInstanceTypesIterable(this, applyPaginatorUserAgent(getOutpostInstanceTypesRequest));
    }

    /**
     * <p>
     * Gets information about the specified Outpost site.
     * </p>
     *
     * @param getSiteRequest
     * @return Result of the GetSite operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSiteResponse getSite(GetSiteRequest getSiteRequest) throws ValidationException, NotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSite");

            return clientHandler.execute(new ClientExecutionParams<GetSiteRequest, GetSiteResponse>()
                    .withOperationName("GetSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSiteRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the site address of the specified site.
     * </p>
     *
     * @param getSiteAddressRequest
     * @return Result of the GetSiteAddress operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetSiteAddress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetSiteAddress" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSiteAddressResponse getSiteAddress(GetSiteAddressRequest getSiteAddressRequest) throws ValidationException,
            AccessDeniedException, NotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSiteAddressResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSiteAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSiteAddressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSiteAddress");

            return clientHandler.execute(new ClientExecutionParams<GetSiteAddressRequest, GetSiteAddressResponse>()
                    .withOperationName("GetSiteAddress").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSiteAddressRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSiteAddressRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the hardware assets for the specified Outpost.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listAssetsRequest
     * @return Result of the ListAssets operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListAssets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListAssets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAssetsResponse listAssets(ListAssetsRequest listAssetsRequest) throws ValidationException, AccessDeniedException,
            NotFoundException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAssetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssets");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAssetsRequest, ListAssetsResponse>().withOperationName("ListAssets")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listAssetsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAssetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the hardware assets for the specified Outpost.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAssets(software.amazon.awssdk.services.outposts.model.ListAssetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListAssetsIterable responses = client.listAssetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.ListAssetsIterable responses = client.listAssetsPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.ListAssetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListAssetsIterable responses = client.listAssetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssets(software.amazon.awssdk.services.outposts.model.ListAssetsRequest)} operation.</b>
     * </p>
     *
     * @param listAssetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListAssets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListAssets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAssetsIterable listAssetsPaginator(ListAssetsRequest listAssetsRequest) throws ValidationException,
            AccessDeniedException, NotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        return new ListAssetsIterable(this, applyPaginatorUserAgent(listAssetsRequest));
    }

    /**
     * <p>
     * Lists the items in the catalog.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listCatalogItemsRequest
     * @return Result of the ListCatalogItems operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListCatalogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListCatalogItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListCatalogItemsResponse listCatalogItems(ListCatalogItemsRequest listCatalogItemsRequest) throws ValidationException,
            NotFoundException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCatalogItemsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCatalogItemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCatalogItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCatalogItems");

            return clientHandler.execute(new ClientExecutionParams<ListCatalogItemsRequest, ListCatalogItemsResponse>()
                    .withOperationName("ListCatalogItems").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCatalogItemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCatalogItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the items in the catalog.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCatalogItems(software.amazon.awssdk.services.outposts.model.ListCatalogItemsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListCatalogItemsIterable responses = client.listCatalogItemsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.ListCatalogItemsIterable responses = client
     *             .listCatalogItemsPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.ListCatalogItemsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListCatalogItemsIterable responses = client.listCatalogItemsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCatalogItems(software.amazon.awssdk.services.outposts.model.ListCatalogItemsRequest)} operation.</b>
     * </p>
     *
     * @param listCatalogItemsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListCatalogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListCatalogItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListCatalogItemsIterable listCatalogItemsPaginator(ListCatalogItemsRequest listCatalogItemsRequest)
            throws ValidationException, NotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        return new ListCatalogItemsIterable(this, applyPaginatorUserAgent(listCatalogItemsRequest));
    }

    /**
     * <p>
     * Lists the Outpost orders for your Amazon Web Services account.
     * </p>
     *
     * @param listOrdersRequest
     * @return Result of the ListOrders operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOrders" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOrdersResponse listOrders(ListOrdersRequest listOrdersRequest) throws ValidationException, NotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOrdersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOrdersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOrdersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrders");

            return clientHandler
                    .execute(new ClientExecutionParams<ListOrdersRequest, ListOrdersResponse>().withOperationName("ListOrders")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listOrdersRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListOrdersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Outpost orders for your Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOrders(software.amazon.awssdk.services.outposts.model.ListOrdersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOrdersIterable responses = client.listOrdersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.ListOrdersIterable responses = client.listOrdersPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.ListOrdersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOrdersIterable responses = client.listOrdersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOrders(software.amazon.awssdk.services.outposts.model.ListOrdersRequest)} operation.</b>
     * </p>
     *
     * @param listOrdersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOrders" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOrdersIterable listOrdersPaginator(ListOrdersRequest listOrdersRequest) throws ValidationException,
            NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        return new ListOrdersIterable(this, applyPaginatorUserAgent(listOrdersRequest));
    }

    /**
     * <p>
     * Lists the Outposts for your Amazon Web Services account.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listOutpostsRequest
     * @return Result of the ListOutposts operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListOutposts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOutpostsResponse listOutposts(ListOutpostsRequest listOutpostsRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOutpostsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOutpostsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOutpostsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOutposts");

            return clientHandler.execute(new ClientExecutionParams<ListOutpostsRequest, ListOutpostsResponse>()
                    .withOperationName("ListOutposts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listOutpostsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListOutpostsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Outposts for your Amazon Web Services account.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable responses = client.listOutpostsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable responses = client.listOutpostsPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.ListOutpostsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable responses = client.listOutpostsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)} operation.</b>
     * </p>
     *
     * @param listOutpostsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListOutposts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOutpostsIterable listOutpostsPaginator(ListOutpostsRequest listOutpostsRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        return new ListOutpostsIterable(this, applyPaginatorUserAgent(listOutpostsRequest));
    }

    /**
     * <p>
     * Lists the Outpost sites for your Amazon Web Services account. Use filters to return specific results.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     *
     * @param listSitesRequest
     * @return Result of the ListSites operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSitesResponse listSites(ListSitesRequest listSitesRequest) throws ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSitesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSites");

            return clientHandler.execute(new ClientExecutionParams<ListSitesRequest, ListSitesResponse>()
                    .withOperationName("ListSites").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSitesRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListSitesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Outpost sites for your Amazon Web Services account. Use filters to return specific results.
     * </p>
     * <p>
     * Use filters to return specific results. If you specify multiple filters, the results include only the resources
     * that match all of the specified filters. For a filter where you can specify multiple values, the results include
     * items that match any of the values that you specify for the filter.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesIterable responses = client.listSitesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.ListSitesIterable responses = client.listSitesPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.ListSitesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesIterable responses = client.listSitesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)} operation.</b>
     * </p>
     *
     * @param listSitesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSitesIterable listSitesPaginator(ListSitesRequest listSitesRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        return new ListSitesIterable(this, applyPaginatorUserAgent(listSitesRequest));
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ValidationException, NotFoundException, AwsServiceException, SdkClientException,
            OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * Amazon Web Services uses this action to install Outpost servers.
     * </p>
     * </note>
     * <p>
     * Starts the connection required for Outpost server installation.
     * </p>
     * <p>
     * Use CloudTrail to monitor this action or Amazon Web Services managed policy for Amazon Web Services Outposts to
     * secure it. For more information, see <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/security-iam-awsmanpol.html"> Amazon Web Services
     * managed policies for Amazon Web Services Outposts</a> and <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/logging-using-cloudtrail.html"> Logging Amazon Web
     * Services Outposts API calls with Amazon Web Services CloudTrail</a> in the <i>Amazon Web Services Outposts User
     * Guide</i>.
     * </p>
     *
     * @param startConnectionRequest
     * @return Result of the StartConnection operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.StartConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/StartConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartConnectionResponse startConnection(StartConnectionRequest startConnectionRequest) throws AccessDeniedException,
            ValidationException, NotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartConnectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartConnection");

            return clientHandler.execute(new ClientExecutionParams<StartConnectionRequest, StartConnectionResponse>()
                    .withOperationName("StartConnection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startConnectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ValidationException, NotFoundException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ValidationException, NotFoundException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an Outpost.
     * </p>
     *
     * @param updateOutpostRequest
     * @return Result of the UpdateOutpost operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.UpdateOutpost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateOutpostResponse updateOutpost(UpdateOutpostRequest updateOutpostRequest) throws ValidationException,
            ConflictException, NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateOutpostResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateOutpostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOutpost");

            return clientHandler.execute(new ClientExecutionParams<UpdateOutpostRequest, UpdateOutpostResponse>()
                    .withOperationName("UpdateOutpost").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateOutpostRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateOutpostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified site.
     * </p>
     *
     * @param updateSiteRequest
     * @return Result of the UpdateSite operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.UpdateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSiteResponse updateSite(UpdateSiteRequest updateSiteRequest) throws ValidationException, ConflictException,
            NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSite");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSiteRequest, UpdateSiteResponse>().withOperationName("UpdateSite")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateSiteRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the address of the specified site.
     * </p>
     * <p>
     * You can't update a site address if there is an order in progress. You must wait for the order to complete or
     * cancel the order.
     * </p>
     * <p>
     * You can update the operating address before you place an order at the site, or after all Outposts that belong to
     * the site have been deactivated.
     * </p>
     *
     * @param updateSiteAddressRequest
     * @return Result of the UpdateSiteAddress operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.UpdateSiteAddress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateSiteAddress" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateSiteAddressResponse updateSiteAddress(UpdateSiteAddressRequest updateSiteAddressRequest)
            throws ValidationException, AccessDeniedException, NotFoundException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSiteAddressResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSiteAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSiteAddressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSiteAddress");

            return clientHandler.execute(new ClientExecutionParams<UpdateSiteAddressRequest, UpdateSiteAddressResponse>()
                    .withOperationName("UpdateSiteAddress").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSiteAddressRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSiteAddressRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update the physical and logistical details for a rack at a site. For more information about hardware requirements
     * for racks, see <a
     * href="https://docs.aws.amazon.com/outposts/latest/userguide/outposts-requirements.html#checklist">Network
     * readiness checklist</a> in the Amazon Web Services Outposts User Guide.
     * </p>
     * <p>
     * To update a rack at a site with an order of <code>IN_PROGRESS</code>, you must wait for the order to complete or
     * cancel the order.
     * </p>
     *
     * @param updateSiteRackPhysicalPropertiesRequest
     * @return Result of the UpdateSiteRackPhysicalProperties operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.UpdateSiteRackPhysicalProperties
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/UpdateSiteRackPhysicalProperties"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSiteRackPhysicalPropertiesResponse updateSiteRackPhysicalProperties(
            UpdateSiteRackPhysicalPropertiesRequest updateSiteRackPhysicalPropertiesRequest) throws ValidationException,
            ConflictException, NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSiteRackPhysicalPropertiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSiteRackPhysicalPropertiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateSiteRackPhysicalPropertiesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSiteRackPhysicalProperties");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSiteRackPhysicalPropertiesRequest, UpdateSiteRackPhysicalPropertiesResponse>()
                            .withOperationName("UpdateSiteRackPhysicalProperties").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateSiteRackPhysicalPropertiesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSiteRackPhysicalPropertiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends OutpostsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OutpostsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final OutpostsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
