/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.outposts;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.outposts.model.AccessDeniedException;
import software.amazon.awssdk.services.outposts.model.CreateOutpostRequest;
import software.amazon.awssdk.services.outposts.model.CreateOutpostResponse;
import software.amazon.awssdk.services.outposts.model.DeleteOutpostRequest;
import software.amazon.awssdk.services.outposts.model.DeleteOutpostResponse;
import software.amazon.awssdk.services.outposts.model.DeleteSiteRequest;
import software.amazon.awssdk.services.outposts.model.DeleteSiteResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostResponse;
import software.amazon.awssdk.services.outposts.model.InternalServerException;
import software.amazon.awssdk.services.outposts.model.ListOutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ListOutpostsResponse;
import software.amazon.awssdk.services.outposts.model.ListSitesRequest;
import software.amazon.awssdk.services.outposts.model.ListSitesResponse;
import software.amazon.awssdk.services.outposts.model.NotFoundException;
import software.amazon.awssdk.services.outposts.model.OutpostsException;
import software.amazon.awssdk.services.outposts.model.OutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.outposts.model.ValidationException;
import software.amazon.awssdk.services.outposts.paginators.ListOutpostsPublisher;
import software.amazon.awssdk.services.outposts.paginators.ListSitesPublisher;
import software.amazon.awssdk.services.outposts.transform.CreateOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.DeleteOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.DeleteSiteRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostInstanceTypesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListOutpostsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListSitesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link OutpostsAsyncClient}.
 *
 * @see OutpostsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOutpostsAsyncClient implements OutpostsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultOutpostsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultOutpostsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Outpost.
     * </p>
     *
     * @param createOutpostRequest
     * @return A Java Future containing the result of the CreateOutpost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>ServiceQuotaExceededException You have exceeded a service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.CreateOutpost
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateOutpostResponse> createOutpost(CreateOutpostRequest createOutpostRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOutpost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateOutpostResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateOutpostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOutpostRequest, CreateOutpostResponse>()
                            .withOperationName("CreateOutpost")
                            .withMarshaller(new CreateOutpostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createOutpostRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createOutpostRequest.overrideConfiguration().orElse(null);
            executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOutpostRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Outpost.
     * </p>
     *
     * @param deleteOutpostRequest
     * @return A Java Future containing the result of the DeleteOutpost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.DeleteOutpost
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/DeleteOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteOutpostResponse> deleteOutpost(DeleteOutpostRequest deleteOutpostRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOutpost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteOutpostResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteOutpostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteOutpostRequest, DeleteOutpostResponse>()
                            .withOperationName("DeleteOutpost")
                            .withMarshaller(new DeleteOutpostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteOutpostRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteOutpostRequest.overrideConfiguration().orElse(null);
            executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOutpostRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the site.
     * </p>
     *
     * @param deleteSiteRequest
     * @return A Java Future containing the result of the DeleteSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.DeleteSite
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/DeleteSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSiteResponse> deleteSite(DeleteSiteRequest deleteSiteRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSiteRequest, DeleteSiteResponse>().withOperationName("DeleteSite")
                            .withMarshaller(new DeleteSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSiteRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteSiteRequest.overrideConfiguration().orElse(null);
            executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSiteRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified Outpost.
     * </p>
     *
     * @param getOutpostRequest
     * @return A Java Future containing the result of the GetOutpost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetOutpost
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetOutpostResponse> getOutpost(GetOutpostRequest getOutpostRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutpost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOutpostResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOutpostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOutpostRequest, GetOutpostResponse>().withOperationName("GetOutpost")
                            .withMarshaller(new GetOutpostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getOutpostRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getOutpostRequest.overrideConfiguration().orElse(null);
            executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutpostRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the instance types for the specified Outpost.
     * </p>
     *
     * @param getOutpostInstanceTypesRequest
     * @return A Java Future containing the result of the GetOutpostInstanceTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>NotFoundException The specified request is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.GetOutpostInstanceTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpostInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOutpostInstanceTypesResponse> getOutpostInstanceTypes(
            GetOutpostInstanceTypesRequest getOutpostInstanceTypesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutpostInstanceTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOutpostInstanceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOutpostInstanceTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOutpostInstanceTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOutpostInstanceTypesRequest, GetOutpostInstanceTypesResponse>()
                            .withOperationName("GetOutpostInstanceTypes")
                            .withMarshaller(new GetOutpostInstanceTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getOutpostInstanceTypesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getOutpostInstanceTypesRequest.overrideConfiguration()
                    .orElse(null);
            executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutpostInstanceTypesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the Outposts for your AWS account.
     * </p>
     *
     * @param listOutpostsRequest
     * @return A Java Future containing the result of the ListOutposts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListOutposts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListOutpostsResponse> listOutposts(ListOutpostsRequest listOutpostsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOutposts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOutpostsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListOutpostsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOutpostsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOutpostsRequest, ListOutpostsResponse>()
                            .withOperationName("ListOutposts").withMarshaller(new ListOutpostsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOutpostsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listOutpostsRequest.overrideConfiguration().orElse(null);
            executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOutpostsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the Outposts for your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsPublisher publisher = client.listOutpostsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsPublisher publisher = client.listOutpostsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.ListOutpostsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.ListOutpostsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)} operation.</b>
     * </p>
     *
     * @param listOutpostsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListOutposts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListOutpostsPublisher listOutpostsPaginator(ListOutpostsRequest listOutpostsRequest) {
        return new ListOutpostsPublisher(this, applyPaginatorUserAgent(listOutpostsRequest));
    }

    /**
     * <p>
     * Lists the sites for the specified AWS account.
     * </p>
     *
     * @param listSitesRequest
     * @return A Java Future containing the result of the ListSites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListSites
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSitesResponse> listSites(ListSitesRequest listSitesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSitesRequest, ListSitesResponse>().withOperationName("ListSites")
                            .withMarshaller(new ListSitesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSitesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listSitesRequest.overrideConfiguration().orElse(null);
            executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSitesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the sites for the specified AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesPublisher publisher = client.listSitesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesPublisher publisher = client.listSitesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.outposts.model.ListSitesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.outposts.model.ListSitesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)} operation.</b>
     * </p>
     *
     * @param listSitesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException A parameter is not valid.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>InternalServerException An internal error has occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample OutpostsAsyncClient.ListSites
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSitesPublisher listSitesPaginator(ListSitesRequest listSitesRequest) {
        return new ListSitesPublisher(this, applyPaginatorUserAgent(listSitesRequest));
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OutpostsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends OutpostsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
