/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.outposts;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.outposts.model.AccessDeniedException;
import software.amazon.awssdk.services.outposts.model.CreateOutpostRequest;
import software.amazon.awssdk.services.outposts.model.CreateOutpostResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostInstanceTypesResponse;
import software.amazon.awssdk.services.outposts.model.GetOutpostRequest;
import software.amazon.awssdk.services.outposts.model.GetOutpostResponse;
import software.amazon.awssdk.services.outposts.model.InternalServerException;
import software.amazon.awssdk.services.outposts.model.ListOutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ListOutpostsResponse;
import software.amazon.awssdk.services.outposts.model.ListSitesRequest;
import software.amazon.awssdk.services.outposts.model.ListSitesResponse;
import software.amazon.awssdk.services.outposts.model.NotFoundException;
import software.amazon.awssdk.services.outposts.model.OutpostsException;
import software.amazon.awssdk.services.outposts.model.OutpostsRequest;
import software.amazon.awssdk.services.outposts.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.outposts.model.ValidationException;
import software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable;
import software.amazon.awssdk.services.outposts.paginators.ListSitesIterable;
import software.amazon.awssdk.services.outposts.transform.CreateOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostInstanceTypesRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.GetOutpostRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListOutpostsRequestMarshaller;
import software.amazon.awssdk.services.outposts.transform.ListSitesRequestMarshaller;

/**
 * Internal implementation of {@link OutpostsClient}.
 *
 * @see OutpostsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOutpostsClient implements OutpostsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultOutpostsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Outpost.
     * </p>
     *
     * @param createOutpostRequest
     * @return Result of the CreateOutpost operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws ServiceQuotaExceededException
     *         You have exceeded a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.CreateOutpost
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/CreateOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateOutpostResponse createOutpost(CreateOutpostRequest createOutpostRequest) throws ValidationException,
            NotFoundException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateOutpostResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateOutpostRequest, CreateOutpostResponse>()
                .withOperationName("CreateOutpost").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createOutpostRequest)
                .withMarshaller(new CreateOutpostRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about the specified Outpost.
     * </p>
     *
     * @param getOutpostRequest
     * @return Result of the GetOutpost operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetOutpost
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetOutpostResponse getOutpost(GetOutpostRequest getOutpostRequest) throws ValidationException, NotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOutpostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOutpostResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetOutpostRequest, GetOutpostResponse>()
                .withOperationName("GetOutpost").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getOutpostRequest)
                .withMarshaller(new GetOutpostRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the instance types for the specified Outpost.
     * </p>
     *
     * @param getOutpostInstanceTypesRequest
     * @return Result of the GetOutpostInstanceTypes operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws NotFoundException
     *         The specified request is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.GetOutpostInstanceTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/GetOutpostInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOutpostInstanceTypesResponse getOutpostInstanceTypes(GetOutpostInstanceTypesRequest getOutpostInstanceTypesRequest)
            throws ValidationException, NotFoundException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOutpostInstanceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetOutpostInstanceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetOutpostInstanceTypesRequest, GetOutpostInstanceTypesResponse>()
                .withOperationName("GetOutpostInstanceTypes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getOutpostInstanceTypesRequest)
                .withMarshaller(new GetOutpostInstanceTypesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the Outposts for your AWS account.
     * </p>
     *
     * @param listOutpostsRequest
     * @return Result of the ListOutposts operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListOutposts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOutpostsResponse listOutposts(ListOutpostsRequest listOutpostsRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOutpostsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOutpostsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListOutpostsRequest, ListOutpostsResponse>()
                .withOperationName("ListOutposts").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listOutpostsRequest)
                .withMarshaller(new ListOutpostsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the Outposts for your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable responses = client.listOutpostsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable responses = client.listOutpostsPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.ListOutpostsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListOutpostsIterable responses = client.listOutpostsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOutposts(software.amazon.awssdk.services.outposts.model.ListOutpostsRequest)} operation.</b>
     * </p>
     *
     * @param listOutpostsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListOutposts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListOutposts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOutpostsIterable listOutpostsPaginator(ListOutpostsRequest listOutpostsRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        return new ListOutpostsIterable(this, applyPaginatorUserAgent(listOutpostsRequest));
    }

    /**
     * <p>
     * Lists the sites for the specified AWS account.
     * </p>
     *
     * @param listSitesRequest
     * @return Result of the ListSites operation returned by the service.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListSites
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSitesResponse listSites(ListSitesRequest listSitesRequest) throws ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSitesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListSitesRequest, ListSitesResponse>()
                .withOperationName("ListSites").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listSitesRequest)
                .withMarshaller(new ListSitesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the sites for the specified AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesIterable responses = client.listSitesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.outposts.paginators.ListSitesIterable responses = client.listSitesPaginator(request);
     *     for (software.amazon.awssdk.services.outposts.model.ListSitesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.outposts.paginators.ListSitesIterable responses = client.listSitesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSites(software.amazon.awssdk.services.outposts.model.ListSitesRequest)} operation.</b>
     * </p>
     *
     * @param listSitesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         A parameter is not valid.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample OutpostsClient.ListSites
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/outposts-2019-12-03/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSitesIterable listSitesPaginator(ListSitesRequest listSitesRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, OutpostsException {
        return new ListSitesIterable(this, applyPaginatorUserAgent(listSitesRequest));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OutpostsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends OutpostsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
