/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opsworks.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.services.opsworks.transform.StackMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a stack.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Stack implements StructuredPojo, ToCopyableBuilder<Stack.Builder, Stack> {
    private final String stackId;

    private final String name;

    private final String arn;

    private final String region;

    private final String vpcId;

    private final Map<String, String> attributes;

    private final String serviceRoleArn;

    private final String defaultInstanceProfileArn;

    private final String defaultOs;

    private final String hostnameTheme;

    private final String defaultAvailabilityZone;

    private final String defaultSubnetId;

    private final String customJson;

    private final StackConfigurationManager configurationManager;

    private final ChefConfiguration chefConfiguration;

    private final Boolean useCustomCookbooks;

    private final Boolean useOpsworksSecurityGroups;

    private final Source customCookbooksSource;

    private final String defaultSshKeyName;

    private final String createdAt;

    private final String defaultRootDeviceType;

    private final String agentVersion;

    private Stack(BuilderImpl builder) {
        this.stackId = builder.stackId;
        this.name = builder.name;
        this.arn = builder.arn;
        this.region = builder.region;
        this.vpcId = builder.vpcId;
        this.attributes = builder.attributes;
        this.serviceRoleArn = builder.serviceRoleArn;
        this.defaultInstanceProfileArn = builder.defaultInstanceProfileArn;
        this.defaultOs = builder.defaultOs;
        this.hostnameTheme = builder.hostnameTheme;
        this.defaultAvailabilityZone = builder.defaultAvailabilityZone;
        this.defaultSubnetId = builder.defaultSubnetId;
        this.customJson = builder.customJson;
        this.configurationManager = builder.configurationManager;
        this.chefConfiguration = builder.chefConfiguration;
        this.useCustomCookbooks = builder.useCustomCookbooks;
        this.useOpsworksSecurityGroups = builder.useOpsworksSecurityGroups;
        this.customCookbooksSource = builder.customCookbooksSource;
        this.defaultSshKeyName = builder.defaultSshKeyName;
        this.createdAt = builder.createdAt;
        this.defaultRootDeviceType = builder.defaultRootDeviceType;
        this.agentVersion = builder.agentVersion;
    }

    /**
     * <p>
     * The stack ID.
     * </p>
     * 
     * @return The stack ID.
     */
    public String stackId() {
        return stackId;
    }

    /**
     * <p>
     * The stack name.
     * </p>
     * 
     * @return The stack name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The stack's ARN.
     * </p>
     * 
     * @return The stack's ARN.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The stack AWS region, such as "ap-northeast-2". For more information about AWS regions, see <a
     * href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
     * </p>
     * 
     * @return The stack AWS region, such as "ap-northeast-2". For more information about AWS regions, see <a
     *         href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
     */
    public String region() {
        return region;
    }

    /**
     * <p>
     * The VPC ID; applicable only if the stack is running in a VPC.
     * </p>
     * 
     * @return The VPC ID; applicable only if the stack is running in a VPC.
     */
    public String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The stack's attributes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The stack's attributes.
     */
    public Map<StackAttributesKeys, String> attributes() {
        return TypeConverter.convert(attributes, StackAttributesKeys::fromValue, Function.identity(),
                (k, v) -> !Objects.equals(k, StackAttributesKeys.UNKNOWN_TO_SDK_VERSION));
    }

    /**
     * <p>
     * The stack's attributes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The stack's attributes.
     */
    public Map<String, String> attributesStrings() {
        return attributes;
    }

    /**
     * <p>
     * The stack AWS Identity and Access Management (IAM) role.
     * </p>
     * 
     * @return The stack AWS Identity and Access Management (IAM) role.
     */
    public String serviceRoleArn() {
        return serviceRoleArn;
    }

    /**
     * <p>
     * The ARN of an IAM profile that is the default profile for all of the stack's EC2 instances. For more information
     * about IAM ARNs, see <a href="http://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">Using
     * Identifiers</a>.
     * </p>
     * 
     * @return The ARN of an IAM profile that is the default profile for all of the stack's EC2 instances. For more
     *         information about IAM ARNs, see <a
     *         href="http://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">Using Identifiers</a>.
     */
    public String defaultInstanceProfileArn() {
        return defaultInstanceProfileArn;
    }

    /**
     * <p>
     * The stack's default operating system.
     * </p>
     * 
     * @return The stack's default operating system.
     */
    public String defaultOs() {
        return defaultOs;
    }

    /**
     * <p>
     * The stack host name theme, with spaces replaced by underscores.
     * </p>
     * 
     * @return The stack host name theme, with spaces replaced by underscores.
     */
    public String hostnameTheme() {
        return hostnameTheme;
    }

    /**
     * <p>
     * The stack's default Availability Zone. For more information, see <a
     * href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
     * </p>
     * 
     * @return The stack's default Availability Zone. For more information, see <a
     *         href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
     */
    public String defaultAvailabilityZone() {
        return defaultAvailabilityZone;
    }

    /**
     * <p>
     * The default subnet ID; applicable only if the stack is running in a VPC.
     * </p>
     * 
     * @return The default subnet ID; applicable only if the stack is running in a VPC.
     */
    public String defaultSubnetId() {
        return defaultSubnetId;
    }

    /**
     * <p>
     * A JSON object that contains user-defined attributes to be added to the stack configuration and deployment
     * attributes. You can use custom JSON to override the corresponding default stack configuration attribute values or
     * to pass data to recipes. The string should be in the following format:
     * </p>
     * <p>
     * <code>"{\"key1\": \"value1\", \"key2\": \"value2\",...}"</code>
     * </p>
     * <p>
     * For more information on custom JSON, see <a
     * href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-json.html">Use Custom JSON to Modify the
     * Stack Configuration Attributes</a>.
     * </p>
     * 
     * @return A JSON object that contains user-defined attributes to be added to the stack configuration and deployment
     *         attributes. You can use custom JSON to override the corresponding default stack configuration attribute
     *         values or to pass data to recipes. The string should be in the following format:</p>
     *         <p>
     *         <code>"{\"key1\": \"value1\", \"key2\": \"value2\",...}"</code>
     *         </p>
     *         <p>
     *         For more information on custom JSON, see <a
     *         href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-json.html">Use Custom JSON to
     *         Modify the Stack Configuration Attributes</a>.
     */
    public String customJson() {
        return customJson;
    }

    /**
     * <p>
     * The configuration manager.
     * </p>
     * 
     * @return The configuration manager.
     */
    public StackConfigurationManager configurationManager() {
        return configurationManager;
    }

    /**
     * <p>
     * A <code>ChefConfiguration</code> object that specifies whether to enable Berkshelf and the Berkshelf version. For
     * more information, see <a
     * href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-creating.html">Create a New Stack</a>.
     * </p>
     * 
     * @return A <code>ChefConfiguration</code> object that specifies whether to enable Berkshelf and the Berkshelf
     *         version. For more information, see <a
     *         href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-creating.html">Create a New
     *         Stack</a>.
     */
    public ChefConfiguration chefConfiguration() {
        return chefConfiguration;
    }

    /**
     * <p>
     * Whether the stack uses custom cookbooks.
     * </p>
     * 
     * @return Whether the stack uses custom cookbooks.
     */
    public Boolean useCustomCookbooks() {
        return useCustomCookbooks;
    }

    /**
     * <p>
     * Whether the stack automatically associates the AWS OpsWorks Stacks built-in security groups with the stack's
     * layers.
     * </p>
     * 
     * @return Whether the stack automatically associates the AWS OpsWorks Stacks built-in security groups with the
     *         stack's layers.
     */
    public Boolean useOpsworksSecurityGroups() {
        return useOpsworksSecurityGroups;
    }

    /**
     * Returns the value of the CustomCookbooksSource property for this object.
     * 
     * @return The value of the CustomCookbooksSource property for this object.
     */
    public Source customCookbooksSource() {
        return customCookbooksSource;
    }

    /**
     * <p>
     * A default Amazon EC2 key pair for the stack's instances. You can override this value when you create or update an
     * instance.
     * </p>
     * 
     * @return A default Amazon EC2 key pair for the stack's instances. You can override this value when you create or
     *         update an instance.
     */
    public String defaultSshKeyName() {
        return defaultSshKeyName;
    }

    /**
     * <p>
     * The date when the stack was created.
     * </p>
     * 
     * @return The date when the stack was created.
     */
    public String createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The default root device type. This value is used by default for all instances in the stack, but you can override
     * it when you create an instance. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device">Storage
     * for the Root Device</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #defaultRootDeviceType} will return {@link RootDeviceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #defaultRootDeviceTypeString}.
     * </p>
     * 
     * @return The default root device type. This value is used by default for all instances in the stack, but you can
     *         override it when you create an instance. For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device"
     *         >Storage for the Root Device</a>.
     * @see RootDeviceType
     */
    public RootDeviceType defaultRootDeviceType() {
        return RootDeviceType.fromValue(defaultRootDeviceType);
    }

    /**
     * <p>
     * The default root device type. This value is used by default for all instances in the stack, but you can override
     * it when you create an instance. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device">Storage
     * for the Root Device</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #defaultRootDeviceType} will return {@link RootDeviceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #defaultRootDeviceTypeString}.
     * </p>
     * 
     * @return The default root device type. This value is used by default for all instances in the stack, but you can
     *         override it when you create an instance. For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device"
     *         >Storage for the Root Device</a>.
     * @see RootDeviceType
     */
    public String defaultRootDeviceTypeString() {
        return defaultRootDeviceType;
    }

    /**
     * <p>
     * The agent version. This parameter is set to <code>LATEST</code> for auto-update. or a version number for a fixed
     * agent version.
     * </p>
     * 
     * @return The agent version. This parameter is set to <code>LATEST</code> for auto-update. or a version number for
     *         a fixed agent version.
     */
    public String agentVersion() {
        return agentVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(attributesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(defaultInstanceProfileArn());
        hashCode = 31 * hashCode + Objects.hashCode(defaultOs());
        hashCode = 31 * hashCode + Objects.hashCode(hostnameTheme());
        hashCode = 31 * hashCode + Objects.hashCode(defaultAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(defaultSubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(customJson());
        hashCode = 31 * hashCode + Objects.hashCode(configurationManager());
        hashCode = 31 * hashCode + Objects.hashCode(chefConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(useCustomCookbooks());
        hashCode = 31 * hashCode + Objects.hashCode(useOpsworksSecurityGroups());
        hashCode = 31 * hashCode + Objects.hashCode(customCookbooksSource());
        hashCode = 31 * hashCode + Objects.hashCode(defaultSshKeyName());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(defaultRootDeviceTypeString());
        hashCode = 31 * hashCode + Objects.hashCode(agentVersion());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Stack)) {
            return false;
        }
        Stack other = (Stack) obj;
        return Objects.equals(stackId(), other.stackId()) && Objects.equals(name(), other.name())
                && Objects.equals(arn(), other.arn()) && Objects.equals(region(), other.region())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(attributesStrings(), other.attributesStrings())
                && Objects.equals(serviceRoleArn(), other.serviceRoleArn())
                && Objects.equals(defaultInstanceProfileArn(), other.defaultInstanceProfileArn())
                && Objects.equals(defaultOs(), other.defaultOs()) && Objects.equals(hostnameTheme(), other.hostnameTheme())
                && Objects.equals(defaultAvailabilityZone(), other.defaultAvailabilityZone())
                && Objects.equals(defaultSubnetId(), other.defaultSubnetId()) && Objects.equals(customJson(), other.customJson())
                && Objects.equals(configurationManager(), other.configurationManager())
                && Objects.equals(chefConfiguration(), other.chefConfiguration())
                && Objects.equals(useCustomCookbooks(), other.useCustomCookbooks())
                && Objects.equals(useOpsworksSecurityGroups(), other.useOpsworksSecurityGroups())
                && Objects.equals(customCookbooksSource(), other.customCookbooksSource())
                && Objects.equals(defaultSshKeyName(), other.defaultSshKeyName())
                && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(defaultRootDeviceTypeString(), other.defaultRootDeviceTypeString())
                && Objects.equals(agentVersion(), other.agentVersion());
    }

    @Override
    public String toString() {
        return ToString.builder("Stack").add("StackId", stackId()).add("Name", name()).add("Arn", arn()).add("Region", region())
                .add("VpcId", vpcId()).add("Attributes", attributesStrings()).add("ServiceRoleArn", serviceRoleArn())
                .add("DefaultInstanceProfileArn", defaultInstanceProfileArn()).add("DefaultOs", defaultOs())
                .add("HostnameTheme", hostnameTheme()).add("DefaultAvailabilityZone", defaultAvailabilityZone())
                .add("DefaultSubnetId", defaultSubnetId()).add("CustomJson", customJson())
                .add("ConfigurationManager", configurationManager()).add("ChefConfiguration", chefConfiguration())
                .add("UseCustomCookbooks", useCustomCookbooks()).add("UseOpsworksSecurityGroups", useOpsworksSecurityGroups())
                .add("CustomCookbooksSource", customCookbooksSource()).add("DefaultSshKeyName", defaultSshKeyName())
                .add("CreatedAt", createdAt()).add("DefaultRootDeviceType", defaultRootDeviceTypeString())
                .add("AgentVersion", agentVersion()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackId":
            return Optional.of(clazz.cast(stackId()));
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Arn":
            return Optional.of(clazz.cast(arn()));
        case "Region":
            return Optional.of(clazz.cast(region()));
        case "VpcId":
            return Optional.of(clazz.cast(vpcId()));
        case "Attributes":
            return Optional.of(clazz.cast(attributesStrings()));
        case "ServiceRoleArn":
            return Optional.of(clazz.cast(serviceRoleArn()));
        case "DefaultInstanceProfileArn":
            return Optional.of(clazz.cast(defaultInstanceProfileArn()));
        case "DefaultOs":
            return Optional.of(clazz.cast(defaultOs()));
        case "HostnameTheme":
            return Optional.of(clazz.cast(hostnameTheme()));
        case "DefaultAvailabilityZone":
            return Optional.of(clazz.cast(defaultAvailabilityZone()));
        case "DefaultSubnetId":
            return Optional.of(clazz.cast(defaultSubnetId()));
        case "CustomJson":
            return Optional.of(clazz.cast(customJson()));
        case "ConfigurationManager":
            return Optional.of(clazz.cast(configurationManager()));
        case "ChefConfiguration":
            return Optional.of(clazz.cast(chefConfiguration()));
        case "UseCustomCookbooks":
            return Optional.of(clazz.cast(useCustomCookbooks()));
        case "UseOpsworksSecurityGroups":
            return Optional.of(clazz.cast(useOpsworksSecurityGroups()));
        case "CustomCookbooksSource":
            return Optional.of(clazz.cast(customCookbooksSource()));
        case "DefaultSshKeyName":
            return Optional.of(clazz.cast(defaultSshKeyName()));
        case "CreatedAt":
            return Optional.of(clazz.cast(createdAt()));
        case "DefaultRootDeviceType":
            return Optional.of(clazz.cast(defaultRootDeviceTypeString()));
        case "AgentVersion":
            return Optional.of(clazz.cast(agentVersion()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        StackMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Stack> {
        /**
         * <p>
         * The stack ID.
         * </p>
         * 
         * @param stackId
         *        The stack ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackId(String stackId);

        /**
         * <p>
         * The stack name.
         * </p>
         * 
         * @param name
         *        The stack name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The stack's ARN.
         * </p>
         * 
         * @param arn
         *        The stack's ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The stack AWS region, such as "ap-northeast-2". For more information about AWS regions, see <a
         * href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
         * </p>
         * 
         * @param region
         *        The stack AWS region, such as "ap-northeast-2". For more information about AWS regions, see <a
         *        href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The VPC ID; applicable only if the stack is running in a VPC.
         * </p>
         * 
         * @param vpcId
         *        The VPC ID; applicable only if the stack is running in a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The stack's attributes.
         * </p>
         * 
         * @param attributes
         *        The stack's attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        /**
         * <p>
         * The stack AWS Identity and Access Management (IAM) role.
         * </p>
         * 
         * @param serviceRoleArn
         *        The stack AWS Identity and Access Management (IAM) role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRoleArn(String serviceRoleArn);

        /**
         * <p>
         * The ARN of an IAM profile that is the default profile for all of the stack's EC2 instances. For more
         * information about IAM ARNs, see <a
         * href="http://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">Using Identifiers</a>.
         * </p>
         * 
         * @param defaultInstanceProfileArn
         *        The ARN of an IAM profile that is the default profile for all of the stack's EC2 instances. For more
         *        information about IAM ARNs, see <a
         *        href="http://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">Using Identifiers</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultInstanceProfileArn(String defaultInstanceProfileArn);

        /**
         * <p>
         * The stack's default operating system.
         * </p>
         * 
         * @param defaultOs
         *        The stack's default operating system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultOs(String defaultOs);

        /**
         * <p>
         * The stack host name theme, with spaces replaced by underscores.
         * </p>
         * 
         * @param hostnameTheme
         *        The stack host name theme, with spaces replaced by underscores.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostnameTheme(String hostnameTheme);

        /**
         * <p>
         * The stack's default Availability Zone. For more information, see <a
         * href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
         * </p>
         * 
         * @param defaultAvailabilityZone
         *        The stack's default Availability Zone. For more information, see <a
         *        href="http://docs.aws.amazon.com/general/latest/gr/rande.html">Regions and Endpoints</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultAvailabilityZone(String defaultAvailabilityZone);

        /**
         * <p>
         * The default subnet ID; applicable only if the stack is running in a VPC.
         * </p>
         * 
         * @param defaultSubnetId
         *        The default subnet ID; applicable only if the stack is running in a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultSubnetId(String defaultSubnetId);

        /**
         * <p>
         * A JSON object that contains user-defined attributes to be added to the stack configuration and deployment
         * attributes. You can use custom JSON to override the corresponding default stack configuration attribute
         * values or to pass data to recipes. The string should be in the following format:
         * </p>
         * <p>
         * <code>"{\"key1\": \"value1\", \"key2\": \"value2\",...}"</code>
         * </p>
         * <p>
         * For more information on custom JSON, see <a
         * href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-json.html">Use Custom JSON to Modify
         * the Stack Configuration Attributes</a>.
         * </p>
         * 
         * @param customJson
         *        A JSON object that contains user-defined attributes to be added to the stack configuration and
         *        deployment attributes. You can use custom JSON to override the corresponding default stack
         *        configuration attribute values or to pass data to recipes. The string should be in the following
         *        format:</p>
         *        <p>
         *        <code>"{\"key1\": \"value1\", \"key2\": \"value2\",...}"</code>
         *        </p>
         *        <p>
         *        For more information on custom JSON, see <a
         *        href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-json.html">Use Custom JSON to
         *        Modify the Stack Configuration Attributes</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customJson(String customJson);

        /**
         * <p>
         * The configuration manager.
         * </p>
         * 
         * @param configurationManager
         *        The configuration manager.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationManager(StackConfigurationManager configurationManager);

        /**
         * <p>
         * The configuration manager.
         * </p>
         * This is a convenience that creates an instance of the {@link StackConfigurationManager.Builder} avoiding the
         * need to create one manually via {@link StackConfigurationManager#builder()}.
         *
         * When the {@link Consumer} completes, {@link StackConfigurationManager.Builder#build()} is called immediately
         * and its result is passed to {@link #configurationManager(StackConfigurationManager)}.
         * 
         * @param configurationManager
         *        a consumer that will call methods on {@link StackConfigurationManager.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurationManager(StackConfigurationManager)
         */
        default Builder configurationManager(Consumer<StackConfigurationManager.Builder> configurationManager) {
            return configurationManager(StackConfigurationManager.builder().apply(configurationManager).build());
        }

        /**
         * <p>
         * A <code>ChefConfiguration</code> object that specifies whether to enable Berkshelf and the Berkshelf version.
         * For more information, see <a
         * href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-creating.html">Create a New
         * Stack</a>.
         * </p>
         * 
         * @param chefConfiguration
         *        A <code>ChefConfiguration</code> object that specifies whether to enable Berkshelf and the Berkshelf
         *        version. For more information, see <a
         *        href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-creating.html">Create a New
         *        Stack</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder chefConfiguration(ChefConfiguration chefConfiguration);

        /**
         * <p>
         * A <code>ChefConfiguration</code> object that specifies whether to enable Berkshelf and the Berkshelf version.
         * For more information, see <a
         * href="http://docs.aws.amazon.com/opsworks/latest/userguide/workingstacks-creating.html">Create a New
         * Stack</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link ChefConfiguration.Builder} avoiding the need to
         * create one manually via {@link ChefConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ChefConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #chefConfiguration(ChefConfiguration)}.
         * 
         * @param chefConfiguration
         *        a consumer that will call methods on {@link ChefConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #chefConfiguration(ChefConfiguration)
         */
        default Builder chefConfiguration(Consumer<ChefConfiguration.Builder> chefConfiguration) {
            return chefConfiguration(ChefConfiguration.builder().apply(chefConfiguration).build());
        }

        /**
         * <p>
         * Whether the stack uses custom cookbooks.
         * </p>
         * 
         * @param useCustomCookbooks
         *        Whether the stack uses custom cookbooks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder useCustomCookbooks(Boolean useCustomCookbooks);

        /**
         * <p>
         * Whether the stack automatically associates the AWS OpsWorks Stacks built-in security groups with the stack's
         * layers.
         * </p>
         * 
         * @param useOpsworksSecurityGroups
         *        Whether the stack automatically associates the AWS OpsWorks Stacks built-in security groups with the
         *        stack's layers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder useOpsworksSecurityGroups(Boolean useOpsworksSecurityGroups);

        /**
         * Sets the value of the CustomCookbooksSource property for this object.
         *
         * @param customCookbooksSource
         *        The new value for the CustomCookbooksSource property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customCookbooksSource(Source customCookbooksSource);

        /**
         * Sets the value of the CustomCookbooksSource property for this object.
         *
         * This is a convenience that creates an instance of the {@link Source.Builder} avoiding the need to create one
         * manually via {@link Source#builder()}.
         *
         * When the {@link Consumer} completes, {@link Source.Builder#build()} is called immediately and its result is
         * passed to {@link #customCookbooksSource(Source)}.
         * 
         * @param customCookbooksSource
         *        a consumer that will call methods on {@link Source.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customCookbooksSource(Source)
         */
        default Builder customCookbooksSource(Consumer<Source.Builder> customCookbooksSource) {
            return customCookbooksSource(Source.builder().apply(customCookbooksSource).build());
        }

        /**
         * <p>
         * A default Amazon EC2 key pair for the stack's instances. You can override this value when you create or
         * update an instance.
         * </p>
         * 
         * @param defaultSshKeyName
         *        A default Amazon EC2 key pair for the stack's instances. You can override this value when you create
         *        or update an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultSshKeyName(String defaultSshKeyName);

        /**
         * <p>
         * The date when the stack was created.
         * </p>
         * 
         * @param createdAt
         *        The date when the stack was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(String createdAt);

        /**
         * <p>
         * The default root device type. This value is used by default for all instances in the stack, but you can
         * override it when you create an instance. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device"
         * >Storage for the Root Device</a>.
         * </p>
         * 
         * @param defaultRootDeviceType
         *        The default root device type. This value is used by default for all instances in the stack, but you
         *        can override it when you create an instance. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device"
         *        >Storage for the Root Device</a>.
         * @see RootDeviceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RootDeviceType
         */
        Builder defaultRootDeviceType(String defaultRootDeviceType);

        /**
         * <p>
         * The default root device type. This value is used by default for all instances in the stack, but you can
         * override it when you create an instance. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device"
         * >Storage for the Root Device</a>.
         * </p>
         * 
         * @param defaultRootDeviceType
         *        The default root device type. This value is used by default for all instances in the stack, but you
         *        can override it when you create an instance. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ComponentsAMIs.html#storage-for-the-root-device"
         *        >Storage for the Root Device</a>.
         * @see RootDeviceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RootDeviceType
         */
        Builder defaultRootDeviceType(RootDeviceType defaultRootDeviceType);

        /**
         * <p>
         * The agent version. This parameter is set to <code>LATEST</code> for auto-update. or a version number for a
         * fixed agent version.
         * </p>
         * 
         * @param agentVersion
         *        The agent version. This parameter is set to <code>LATEST</code> for auto-update. or a version number
         *        for a fixed agent version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentVersion(String agentVersion);
    }

    static final class BuilderImpl implements Builder {
        private String stackId;

        private String name;

        private String arn;

        private String region;

        private String vpcId;

        private Map<String, String> attributes;

        private String serviceRoleArn;

        private String defaultInstanceProfileArn;

        private String defaultOs;

        private String hostnameTheme;

        private String defaultAvailabilityZone;

        private String defaultSubnetId;

        private String customJson;

        private StackConfigurationManager configurationManager;

        private ChefConfiguration chefConfiguration;

        private Boolean useCustomCookbooks;

        private Boolean useOpsworksSecurityGroups;

        private Source customCookbooksSource;

        private String defaultSshKeyName;

        private String createdAt;

        private String defaultRootDeviceType;

        private String agentVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(Stack model) {
            stackId(model.stackId);
            name(model.name);
            arn(model.arn);
            region(model.region);
            vpcId(model.vpcId);
            attributes(model.attributes);
            serviceRoleArn(model.serviceRoleArn);
            defaultInstanceProfileArn(model.defaultInstanceProfileArn);
            defaultOs(model.defaultOs);
            hostnameTheme(model.hostnameTheme);
            defaultAvailabilityZone(model.defaultAvailabilityZone);
            defaultSubnetId(model.defaultSubnetId);
            customJson(model.customJson);
            configurationManager(model.configurationManager);
            chefConfiguration(model.chefConfiguration);
            useCustomCookbooks(model.useCustomCookbooks);
            useOpsworksSecurityGroups(model.useOpsworksSecurityGroups);
            customCookbooksSource(model.customCookbooksSource);
            defaultSshKeyName(model.defaultSshKeyName);
            createdAt(model.createdAt);
            defaultRootDeviceType(model.defaultRootDeviceType);
            agentVersion(model.agentVersion);
        }

        public final String getStackId() {
            return stackId;
        }

        @Override
        public final Builder stackId(String stackId) {
            this.stackId = stackId;
            return this;
        }

        public final void setStackId(String stackId) {
            this.stackId = stackId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final Map<String, String> getAttributes() {
            return attributes;
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = StackAttributesCopier.copy(attributes);
            return this;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = StackAttributesCopier.copy(attributes);
        }

        public final String getServiceRoleArn() {
            return serviceRoleArn;
        }

        @Override
        public final Builder serviceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
            return this;
        }

        public final void setServiceRoleArn(String serviceRoleArn) {
            this.serviceRoleArn = serviceRoleArn;
        }

        public final String getDefaultInstanceProfileArn() {
            return defaultInstanceProfileArn;
        }

        @Override
        public final Builder defaultInstanceProfileArn(String defaultInstanceProfileArn) {
            this.defaultInstanceProfileArn = defaultInstanceProfileArn;
            return this;
        }

        public final void setDefaultInstanceProfileArn(String defaultInstanceProfileArn) {
            this.defaultInstanceProfileArn = defaultInstanceProfileArn;
        }

        public final String getDefaultOs() {
            return defaultOs;
        }

        @Override
        public final Builder defaultOs(String defaultOs) {
            this.defaultOs = defaultOs;
            return this;
        }

        public final void setDefaultOs(String defaultOs) {
            this.defaultOs = defaultOs;
        }

        public final String getHostnameTheme() {
            return hostnameTheme;
        }

        @Override
        public final Builder hostnameTheme(String hostnameTheme) {
            this.hostnameTheme = hostnameTheme;
            return this;
        }

        public final void setHostnameTheme(String hostnameTheme) {
            this.hostnameTheme = hostnameTheme;
        }

        public final String getDefaultAvailabilityZone() {
            return defaultAvailabilityZone;
        }

        @Override
        public final Builder defaultAvailabilityZone(String defaultAvailabilityZone) {
            this.defaultAvailabilityZone = defaultAvailabilityZone;
            return this;
        }

        public final void setDefaultAvailabilityZone(String defaultAvailabilityZone) {
            this.defaultAvailabilityZone = defaultAvailabilityZone;
        }

        public final String getDefaultSubnetId() {
            return defaultSubnetId;
        }

        @Override
        public final Builder defaultSubnetId(String defaultSubnetId) {
            this.defaultSubnetId = defaultSubnetId;
            return this;
        }

        public final void setDefaultSubnetId(String defaultSubnetId) {
            this.defaultSubnetId = defaultSubnetId;
        }

        public final String getCustomJson() {
            return customJson;
        }

        @Override
        public final Builder customJson(String customJson) {
            this.customJson = customJson;
            return this;
        }

        public final void setCustomJson(String customJson) {
            this.customJson = customJson;
        }

        public final StackConfigurationManager.Builder getConfigurationManager() {
            return configurationManager != null ? configurationManager.toBuilder() : null;
        }

        @Override
        public final Builder configurationManager(StackConfigurationManager configurationManager) {
            this.configurationManager = configurationManager;
            return this;
        }

        public final void setConfigurationManager(StackConfigurationManager.BuilderImpl configurationManager) {
            this.configurationManager = configurationManager != null ? configurationManager.build() : null;
        }

        public final ChefConfiguration.Builder getChefConfiguration() {
            return chefConfiguration != null ? chefConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder chefConfiguration(ChefConfiguration chefConfiguration) {
            this.chefConfiguration = chefConfiguration;
            return this;
        }

        public final void setChefConfiguration(ChefConfiguration.BuilderImpl chefConfiguration) {
            this.chefConfiguration = chefConfiguration != null ? chefConfiguration.build() : null;
        }

        public final Boolean getUseCustomCookbooks() {
            return useCustomCookbooks;
        }

        @Override
        public final Builder useCustomCookbooks(Boolean useCustomCookbooks) {
            this.useCustomCookbooks = useCustomCookbooks;
            return this;
        }

        public final void setUseCustomCookbooks(Boolean useCustomCookbooks) {
            this.useCustomCookbooks = useCustomCookbooks;
        }

        public final Boolean getUseOpsworksSecurityGroups() {
            return useOpsworksSecurityGroups;
        }

        @Override
        public final Builder useOpsworksSecurityGroups(Boolean useOpsworksSecurityGroups) {
            this.useOpsworksSecurityGroups = useOpsworksSecurityGroups;
            return this;
        }

        public final void setUseOpsworksSecurityGroups(Boolean useOpsworksSecurityGroups) {
            this.useOpsworksSecurityGroups = useOpsworksSecurityGroups;
        }

        public final Source.Builder getCustomCookbooksSource() {
            return customCookbooksSource != null ? customCookbooksSource.toBuilder() : null;
        }

        @Override
        public final Builder customCookbooksSource(Source customCookbooksSource) {
            this.customCookbooksSource = customCookbooksSource;
            return this;
        }

        public final void setCustomCookbooksSource(Source.BuilderImpl customCookbooksSource) {
            this.customCookbooksSource = customCookbooksSource != null ? customCookbooksSource.build() : null;
        }

        public final String getDefaultSshKeyName() {
            return defaultSshKeyName;
        }

        @Override
        public final Builder defaultSshKeyName(String defaultSshKeyName) {
            this.defaultSshKeyName = defaultSshKeyName;
            return this;
        }

        public final void setDefaultSshKeyName(String defaultSshKeyName) {
            this.defaultSshKeyName = defaultSshKeyName;
        }

        public final String getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(String createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(String createdAt) {
            this.createdAt = createdAt;
        }

        public final String getDefaultRootDeviceType() {
            return defaultRootDeviceType;
        }

        @Override
        public final Builder defaultRootDeviceType(String defaultRootDeviceType) {
            this.defaultRootDeviceType = defaultRootDeviceType;
            return this;
        }

        @Override
        public final Builder defaultRootDeviceType(RootDeviceType defaultRootDeviceType) {
            this.defaultRootDeviceType(defaultRootDeviceType.toString());
            return this;
        }

        public final void setDefaultRootDeviceType(String defaultRootDeviceType) {
            this.defaultRootDeviceType = defaultRootDeviceType;
        }

        public final String getAgentVersion() {
            return agentVersion;
        }

        @Override
        public final Builder agentVersion(String agentVersion) {
            this.agentVersion = agentVersion;
            return this;
        }

        public final void setAgentVersion(String agentVersion) {
            this.agentVersion = agentVersion;
        }

        @Override
        public Stack build() {
            return new Stack(this);
        }
    }
}
