/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Container for the cluster configuration of an OpenSearch Service domain. For more information, see <a
 * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html">Creating and
 * managing Amazon OpenSearch Service domains</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClusterConfig implements SdkPojo, Serializable, ToCopyableBuilder<ClusterConfig.Builder, ClusterConfig> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(ClusterConfig::instanceTypeAsString)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceCount").getter(getter(ClusterConfig::instanceCount)).setter(setter(Builder::instanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCount").build()).build();

    private static final SdkField<Boolean> DEDICATED_MASTER_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DedicatedMasterEnabled").getter(getter(ClusterConfig::dedicatedMasterEnabled))
            .setter(setter(Builder::dedicatedMasterEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DedicatedMasterEnabled").build())
            .build();

    private static final SdkField<Boolean> ZONE_AWARENESS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ZoneAwarenessEnabled").getter(getter(ClusterConfig::zoneAwarenessEnabled))
            .setter(setter(Builder::zoneAwarenessEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZoneAwarenessEnabled").build())
            .build();

    private static final SdkField<ZoneAwarenessConfig> ZONE_AWARENESS_CONFIG_FIELD = SdkField
            .<ZoneAwarenessConfig> builder(MarshallingType.SDK_POJO).memberName("ZoneAwarenessConfig")
            .getter(getter(ClusterConfig::zoneAwarenessConfig)).setter(setter(Builder::zoneAwarenessConfig))
            .constructor(ZoneAwarenessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZoneAwarenessConfig").build())
            .build();

    private static final SdkField<String> DEDICATED_MASTER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DedicatedMasterType").getter(getter(ClusterConfig::dedicatedMasterTypeAsString))
            .setter(setter(Builder::dedicatedMasterType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DedicatedMasterType").build())
            .build();

    private static final SdkField<Integer> DEDICATED_MASTER_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("DedicatedMasterCount").getter(getter(ClusterConfig::dedicatedMasterCount))
            .setter(setter(Builder::dedicatedMasterCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DedicatedMasterCount").build())
            .build();

    private static final SdkField<Boolean> WARM_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("WarmEnabled").getter(getter(ClusterConfig::warmEnabled)).setter(setter(Builder::warmEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WarmEnabled").build()).build();

    private static final SdkField<String> WARM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WarmType").getter(getter(ClusterConfig::warmTypeAsString)).setter(setter(Builder::warmType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WarmType").build()).build();

    private static final SdkField<Integer> WARM_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("WarmCount").getter(getter(ClusterConfig::warmCount)).setter(setter(Builder::warmCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WarmCount").build()).build();

    private static final SdkField<ColdStorageOptions> COLD_STORAGE_OPTIONS_FIELD = SdkField
            .<ColdStorageOptions> builder(MarshallingType.SDK_POJO).memberName("ColdStorageOptions")
            .getter(getter(ClusterConfig::coldStorageOptions)).setter(setter(Builder::coldStorageOptions))
            .constructor(ColdStorageOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColdStorageOptions").build())
            .build();

    private static final SdkField<Boolean> MULTI_AZ_WITH_STANDBY_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("MultiAZWithStandbyEnabled")
            .getter(getter(ClusterConfig::multiAZWithStandbyEnabled)).setter(setter(Builder::multiAZWithStandbyEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MultiAZWithStandbyEnabled").build())
            .build();

    private static final SdkField<List<NodeOption>> NODE_OPTIONS_FIELD = SdkField
            .<List<NodeOption>> builder(MarshallingType.LIST)
            .memberName("NodeOptions")
            .getter(getter(ClusterConfig::nodeOptions))
            .setter(setter(Builder::nodeOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NodeOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(NodeOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            INSTANCE_COUNT_FIELD, DEDICATED_MASTER_ENABLED_FIELD, ZONE_AWARENESS_ENABLED_FIELD, ZONE_AWARENESS_CONFIG_FIELD,
            DEDICATED_MASTER_TYPE_FIELD, DEDICATED_MASTER_COUNT_FIELD, WARM_ENABLED_FIELD, WARM_TYPE_FIELD, WARM_COUNT_FIELD,
            COLD_STORAGE_OPTIONS_FIELD, MULTI_AZ_WITH_STANDBY_ENABLED_FIELD, NODE_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final Integer instanceCount;

    private final Boolean dedicatedMasterEnabled;

    private final Boolean zoneAwarenessEnabled;

    private final ZoneAwarenessConfig zoneAwarenessConfig;

    private final String dedicatedMasterType;

    private final Integer dedicatedMasterCount;

    private final Boolean warmEnabled;

    private final String warmType;

    private final Integer warmCount;

    private final ColdStorageOptions coldStorageOptions;

    private final Boolean multiAZWithStandbyEnabled;

    private final List<NodeOption> nodeOptions;

    private ClusterConfig(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.instanceCount = builder.instanceCount;
        this.dedicatedMasterEnabled = builder.dedicatedMasterEnabled;
        this.zoneAwarenessEnabled = builder.zoneAwarenessEnabled;
        this.zoneAwarenessConfig = builder.zoneAwarenessConfig;
        this.dedicatedMasterType = builder.dedicatedMasterType;
        this.dedicatedMasterCount = builder.dedicatedMasterCount;
        this.warmEnabled = builder.warmEnabled;
        this.warmType = builder.warmType;
        this.warmCount = builder.warmCount;
        this.coldStorageOptions = builder.coldStorageOptions;
        this.multiAZWithStandbyEnabled = builder.multiAZWithStandbyEnabled;
        this.nodeOptions = builder.nodeOptions;
    }

    /**
     * <p>
     * Instance type of data nodes in the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link OpenSearchPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return Instance type of data nodes in the cluster.
     * @see OpenSearchPartitionInstanceType
     */
    public final OpenSearchPartitionInstanceType instanceType() {
        return OpenSearchPartitionInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * Instance type of data nodes in the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link OpenSearchPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return Instance type of data nodes in the cluster.
     * @see OpenSearchPartitionInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * Number of data nodes in the cluster. This number must be greater than 1, otherwise you receive a validation
     * exception.
     * </p>
     * 
     * @return Number of data nodes in the cluster. This number must be greater than 1, otherwise you receive a
     *         validation exception.
     */
    public final Integer instanceCount() {
        return instanceCount;
    }

    /**
     * <p>
     * Indicates whether dedicated master nodes are enabled for the cluster.<code>True</code> if the cluster will use a
     * dedicated master node.<code>False</code> if the cluster will not.
     * </p>
     * 
     * @return Indicates whether dedicated master nodes are enabled for the cluster.<code>True</code> if the cluster
     *         will use a dedicated master node.<code>False</code> if the cluster will not.
     */
    public final Boolean dedicatedMasterEnabled() {
        return dedicatedMasterEnabled;
    }

    /**
     * <p>
     * Indicates whether multiple Availability Zones are enabled. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html"
     * >Configuring a multi-AZ domain in Amazon OpenSearch Service</a>.
     * </p>
     * 
     * @return Indicates whether multiple Availability Zones are enabled. For more information, see <a
     *         href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html"
     *         >Configuring a multi-AZ domain in Amazon OpenSearch Service</a>.
     */
    public final Boolean zoneAwarenessEnabled() {
        return zoneAwarenessEnabled;
    }

    /**
     * <p>
     * Container for zone awareness configuration options. Only required if <code>ZoneAwarenessEnabled</code> is
     * <code>true</code>.
     * </p>
     * 
     * @return Container for zone awareness configuration options. Only required if <code>ZoneAwarenessEnabled</code> is
     *         <code>true</code>.
     */
    public final ZoneAwarenessConfig zoneAwarenessConfig() {
        return zoneAwarenessConfig;
    }

    /**
     * <p>
     * OpenSearch Service instance type of the dedicated master nodes in the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dedicatedMasterType} will return {@link OpenSearchPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #dedicatedMasterTypeAsString}.
     * </p>
     * 
     * @return OpenSearch Service instance type of the dedicated master nodes in the cluster.
     * @see OpenSearchPartitionInstanceType
     */
    public final OpenSearchPartitionInstanceType dedicatedMasterType() {
        return OpenSearchPartitionInstanceType.fromValue(dedicatedMasterType);
    }

    /**
     * <p>
     * OpenSearch Service instance type of the dedicated master nodes in the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dedicatedMasterType} will return {@link OpenSearchPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #dedicatedMasterTypeAsString}.
     * </p>
     * 
     * @return OpenSearch Service instance type of the dedicated master nodes in the cluster.
     * @see OpenSearchPartitionInstanceType
     */
    public final String dedicatedMasterTypeAsString() {
        return dedicatedMasterType;
    }

    /**
     * <p>
     * Number of dedicated master nodes in the cluster. This number must be greater than 2 and not 4, otherwise you
     * receive a validation exception.
     * </p>
     * 
     * @return Number of dedicated master nodes in the cluster. This number must be greater than 2 and not 4, otherwise
     *         you receive a validation exception.
     */
    public final Integer dedicatedMasterCount() {
        return dedicatedMasterCount;
    }

    /**
     * <p>
     * Whether to enable warm storage for the cluster.
     * </p>
     * 
     * @return Whether to enable warm storage for the cluster.
     */
    public final Boolean warmEnabled() {
        return warmEnabled;
    }

    /**
     * <p>
     * The instance type for the cluster's warm nodes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #warmType} will
     * return {@link OpenSearchWarmPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #warmTypeAsString}.
     * </p>
     * 
     * @return The instance type for the cluster's warm nodes.
     * @see OpenSearchWarmPartitionInstanceType
     */
    public final OpenSearchWarmPartitionInstanceType warmType() {
        return OpenSearchWarmPartitionInstanceType.fromValue(warmType);
    }

    /**
     * <p>
     * The instance type for the cluster's warm nodes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #warmType} will
     * return {@link OpenSearchWarmPartitionInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #warmTypeAsString}.
     * </p>
     * 
     * @return The instance type for the cluster's warm nodes.
     * @see OpenSearchWarmPartitionInstanceType
     */
    public final String warmTypeAsString() {
        return warmType;
    }

    /**
     * <p>
     * The number of warm nodes in the cluster.
     * </p>
     * 
     * @return The number of warm nodes in the cluster.
     */
    public final Integer warmCount() {
        return warmCount;
    }

    /**
     * <p>
     * Container for cold storage configuration options.
     * </p>
     * 
     * @return Container for cold storage configuration options.
     */
    public final ColdStorageOptions coldStorageOptions() {
        return coldStorageOptions;
    }

    /**
     * <p>
     * A boolean that indicates whether a multi-AZ domain is turned on with a standby AZ. For more information, see <a
     * href
     * ="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html">Configuring a
     * multi-AZ domain in Amazon OpenSearch Service</a>.
     * </p>
     * 
     * @return A boolean that indicates whether a multi-AZ domain is turned on with a standby AZ. For more information,
     *         see <a
     *         href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html"
     *         >Configuring a multi-AZ domain in Amazon OpenSearch Service</a>.
     */
    public final Boolean multiAZWithStandbyEnabled() {
        return multiAZWithStandbyEnabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodeOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNodeOptions() {
        return nodeOptions != null && !(nodeOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of node options for the domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodeOptions} method.
     * </p>
     * 
     * @return List of node options for the domain.
     */
    public final List<NodeOption> nodeOptions() {
        return nodeOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(dedicatedMasterEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(zoneAwarenessEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(zoneAwarenessConfig());
        hashCode = 31 * hashCode + Objects.hashCode(dedicatedMasterTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dedicatedMasterCount());
        hashCode = 31 * hashCode + Objects.hashCode(warmEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(warmTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(warmCount());
        hashCode = 31 * hashCode + Objects.hashCode(coldStorageOptions());
        hashCode = 31 * hashCode + Objects.hashCode(multiAZWithStandbyEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodeOptions() ? nodeOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClusterConfig)) {
            return false;
        }
        ClusterConfig other = (ClusterConfig) obj;
        return Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(instanceCount(), other.instanceCount())
                && Objects.equals(dedicatedMasterEnabled(), other.dedicatedMasterEnabled())
                && Objects.equals(zoneAwarenessEnabled(), other.zoneAwarenessEnabled())
                && Objects.equals(zoneAwarenessConfig(), other.zoneAwarenessConfig())
                && Objects.equals(dedicatedMasterTypeAsString(), other.dedicatedMasterTypeAsString())
                && Objects.equals(dedicatedMasterCount(), other.dedicatedMasterCount())
                && Objects.equals(warmEnabled(), other.warmEnabled())
                && Objects.equals(warmTypeAsString(), other.warmTypeAsString()) && Objects.equals(warmCount(), other.warmCount())
                && Objects.equals(coldStorageOptions(), other.coldStorageOptions())
                && Objects.equals(multiAZWithStandbyEnabled(), other.multiAZWithStandbyEnabled())
                && hasNodeOptions() == other.hasNodeOptions() && Objects.equals(nodeOptions(), other.nodeOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClusterConfig").add("InstanceType", instanceTypeAsString())
                .add("InstanceCount", instanceCount()).add("DedicatedMasterEnabled", dedicatedMasterEnabled())
                .add("ZoneAwarenessEnabled", zoneAwarenessEnabled()).add("ZoneAwarenessConfig", zoneAwarenessConfig())
                .add("DedicatedMasterType", dedicatedMasterTypeAsString()).add("DedicatedMasterCount", dedicatedMasterCount())
                .add("WarmEnabled", warmEnabled()).add("WarmType", warmTypeAsString()).add("WarmCount", warmCount())
                .add("ColdStorageOptions", coldStorageOptions()).add("MultiAZWithStandbyEnabled", multiAZWithStandbyEnabled())
                .add("NodeOptions", hasNodeOptions() ? nodeOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "InstanceCount":
            return Optional.ofNullable(clazz.cast(instanceCount()));
        case "DedicatedMasterEnabled":
            return Optional.ofNullable(clazz.cast(dedicatedMasterEnabled()));
        case "ZoneAwarenessEnabled":
            return Optional.ofNullable(clazz.cast(zoneAwarenessEnabled()));
        case "ZoneAwarenessConfig":
            return Optional.ofNullable(clazz.cast(zoneAwarenessConfig()));
        case "DedicatedMasterType":
            return Optional.ofNullable(clazz.cast(dedicatedMasterTypeAsString()));
        case "DedicatedMasterCount":
            return Optional.ofNullable(clazz.cast(dedicatedMasterCount()));
        case "WarmEnabled":
            return Optional.ofNullable(clazz.cast(warmEnabled()));
        case "WarmType":
            return Optional.ofNullable(clazz.cast(warmTypeAsString()));
        case "WarmCount":
            return Optional.ofNullable(clazz.cast(warmCount()));
        case "ColdStorageOptions":
            return Optional.ofNullable(clazz.cast(coldStorageOptions()));
        case "MultiAZWithStandbyEnabled":
            return Optional.ofNullable(clazz.cast(multiAZWithStandbyEnabled()));
        case "NodeOptions":
            return Optional.ofNullable(clazz.cast(nodeOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("InstanceCount", INSTANCE_COUNT_FIELD);
        map.put("DedicatedMasterEnabled", DEDICATED_MASTER_ENABLED_FIELD);
        map.put("ZoneAwarenessEnabled", ZONE_AWARENESS_ENABLED_FIELD);
        map.put("ZoneAwarenessConfig", ZONE_AWARENESS_CONFIG_FIELD);
        map.put("DedicatedMasterType", DEDICATED_MASTER_TYPE_FIELD);
        map.put("DedicatedMasterCount", DEDICATED_MASTER_COUNT_FIELD);
        map.put("WarmEnabled", WARM_ENABLED_FIELD);
        map.put("WarmType", WARM_TYPE_FIELD);
        map.put("WarmCount", WARM_COUNT_FIELD);
        map.put("ColdStorageOptions", COLD_STORAGE_OPTIONS_FIELD);
        map.put("MultiAZWithStandbyEnabled", MULTI_AZ_WITH_STANDBY_ENABLED_FIELD);
        map.put("NodeOptions", NODE_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ClusterConfig, T> g) {
        return obj -> g.apply((ClusterConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClusterConfig> {
        /**
         * <p>
         * Instance type of data nodes in the cluster.
         * </p>
         * 
         * @param instanceType
         *        Instance type of data nodes in the cluster.
         * @see OpenSearchPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchPartitionInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * Instance type of data nodes in the cluster.
         * </p>
         * 
         * @param instanceType
         *        Instance type of data nodes in the cluster.
         * @see OpenSearchPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchPartitionInstanceType
         */
        Builder instanceType(OpenSearchPartitionInstanceType instanceType);

        /**
         * <p>
         * Number of data nodes in the cluster. This number must be greater than 1, otherwise you receive a validation
         * exception.
         * </p>
         * 
         * @param instanceCount
         *        Number of data nodes in the cluster. This number must be greater than 1, otherwise you receive a
         *        validation exception.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <p>
         * Indicates whether dedicated master nodes are enabled for the cluster.<code>True</code> if the cluster will
         * use a dedicated master node.<code>False</code> if the cluster will not.
         * </p>
         * 
         * @param dedicatedMasterEnabled
         *        Indicates whether dedicated master nodes are enabled for the cluster.<code>True</code> if the cluster
         *        will use a dedicated master node.<code>False</code> if the cluster will not.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dedicatedMasterEnabled(Boolean dedicatedMasterEnabled);

        /**
         * <p>
         * Indicates whether multiple Availability Zones are enabled. For more information, see <a
         * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html"
         * >Configuring a multi-AZ domain in Amazon OpenSearch Service</a>.
         * </p>
         * 
         * @param zoneAwarenessEnabled
         *        Indicates whether multiple Availability Zones are enabled. For more information, see <a href=
         *        "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html"
         *        >Configuring a multi-AZ domain in Amazon OpenSearch Service</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zoneAwarenessEnabled(Boolean zoneAwarenessEnabled);

        /**
         * <p>
         * Container for zone awareness configuration options. Only required if <code>ZoneAwarenessEnabled</code> is
         * <code>true</code>.
         * </p>
         * 
         * @param zoneAwarenessConfig
         *        Container for zone awareness configuration options. Only required if <code>ZoneAwarenessEnabled</code>
         *        is <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zoneAwarenessConfig(ZoneAwarenessConfig zoneAwarenessConfig);

        /**
         * <p>
         * Container for zone awareness configuration options. Only required if <code>ZoneAwarenessEnabled</code> is
         * <code>true</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ZoneAwarenessConfig.Builder} avoiding the
         * need to create one manually via {@link ZoneAwarenessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ZoneAwarenessConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #zoneAwarenessConfig(ZoneAwarenessConfig)}.
         * 
         * @param zoneAwarenessConfig
         *        a consumer that will call methods on {@link ZoneAwarenessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #zoneAwarenessConfig(ZoneAwarenessConfig)
         */
        default Builder zoneAwarenessConfig(Consumer<ZoneAwarenessConfig.Builder> zoneAwarenessConfig) {
            return zoneAwarenessConfig(ZoneAwarenessConfig.builder().applyMutation(zoneAwarenessConfig).build());
        }

        /**
         * <p>
         * OpenSearch Service instance type of the dedicated master nodes in the cluster.
         * </p>
         * 
         * @param dedicatedMasterType
         *        OpenSearch Service instance type of the dedicated master nodes in the cluster.
         * @see OpenSearchPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchPartitionInstanceType
         */
        Builder dedicatedMasterType(String dedicatedMasterType);

        /**
         * <p>
         * OpenSearch Service instance type of the dedicated master nodes in the cluster.
         * </p>
         * 
         * @param dedicatedMasterType
         *        OpenSearch Service instance type of the dedicated master nodes in the cluster.
         * @see OpenSearchPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchPartitionInstanceType
         */
        Builder dedicatedMasterType(OpenSearchPartitionInstanceType dedicatedMasterType);

        /**
         * <p>
         * Number of dedicated master nodes in the cluster. This number must be greater than 2 and not 4, otherwise you
         * receive a validation exception.
         * </p>
         * 
         * @param dedicatedMasterCount
         *        Number of dedicated master nodes in the cluster. This number must be greater than 2 and not 4,
         *        otherwise you receive a validation exception.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dedicatedMasterCount(Integer dedicatedMasterCount);

        /**
         * <p>
         * Whether to enable warm storage for the cluster.
         * </p>
         * 
         * @param warmEnabled
         *        Whether to enable warm storage for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warmEnabled(Boolean warmEnabled);

        /**
         * <p>
         * The instance type for the cluster's warm nodes.
         * </p>
         * 
         * @param warmType
         *        The instance type for the cluster's warm nodes.
         * @see OpenSearchWarmPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchWarmPartitionInstanceType
         */
        Builder warmType(String warmType);

        /**
         * <p>
         * The instance type for the cluster's warm nodes.
         * </p>
         * 
         * @param warmType
         *        The instance type for the cluster's warm nodes.
         * @see OpenSearchWarmPartitionInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenSearchWarmPartitionInstanceType
         */
        Builder warmType(OpenSearchWarmPartitionInstanceType warmType);

        /**
         * <p>
         * The number of warm nodes in the cluster.
         * </p>
         * 
         * @param warmCount
         *        The number of warm nodes in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warmCount(Integer warmCount);

        /**
         * <p>
         * Container for cold storage configuration options.
         * </p>
         * 
         * @param coldStorageOptions
         *        Container for cold storage configuration options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder coldStorageOptions(ColdStorageOptions coldStorageOptions);

        /**
         * <p>
         * Container for cold storage configuration options.
         * </p>
         * This is a convenience method that creates an instance of the {@link ColdStorageOptions.Builder} avoiding the
         * need to create one manually via {@link ColdStorageOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ColdStorageOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #coldStorageOptions(ColdStorageOptions)}.
         * 
         * @param coldStorageOptions
         *        a consumer that will call methods on {@link ColdStorageOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #coldStorageOptions(ColdStorageOptions)
         */
        default Builder coldStorageOptions(Consumer<ColdStorageOptions.Builder> coldStorageOptions) {
            return coldStorageOptions(ColdStorageOptions.builder().applyMutation(coldStorageOptions).build());
        }

        /**
         * <p>
         * A boolean that indicates whether a multi-AZ domain is turned on with a standby AZ. For more information, see
         * <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html">
         * Configuring a multi-AZ domain in Amazon OpenSearch Service</a>.
         * </p>
         * 
         * @param multiAZWithStandbyEnabled
         *        A boolean that indicates whether a multi-AZ domain is turned on with a standby AZ. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/managedomains-multiaz.html"
         *        >Configuring a multi-AZ domain in Amazon OpenSearch Service</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder multiAZWithStandbyEnabled(Boolean multiAZWithStandbyEnabled);

        /**
         * <p>
         * List of node options for the domain.
         * </p>
         * 
         * @param nodeOptions
         *        List of node options for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeOptions(Collection<NodeOption> nodeOptions);

        /**
         * <p>
         * List of node options for the domain.
         * </p>
         * 
         * @param nodeOptions
         *        List of node options for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeOptions(NodeOption... nodeOptions);

        /**
         * <p>
         * List of node options for the domain.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.opensearch.model.NodeOption.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.opensearch.model.NodeOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.opensearch.model.NodeOption.Builder#build()} is called immediately and
         * its result is passed to {@link #nodeOptions(List<NodeOption>)}.
         * 
         * @param nodeOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.opensearch.model.NodeOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeOptions(java.util.Collection<NodeOption>)
         */
        Builder nodeOptions(Consumer<NodeOption.Builder>... nodeOptions);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private Integer instanceCount;

        private Boolean dedicatedMasterEnabled;

        private Boolean zoneAwarenessEnabled;

        private ZoneAwarenessConfig zoneAwarenessConfig;

        private String dedicatedMasterType;

        private Integer dedicatedMasterCount;

        private Boolean warmEnabled;

        private String warmType;

        private Integer warmCount;

        private ColdStorageOptions coldStorageOptions;

        private Boolean multiAZWithStandbyEnabled;

        private List<NodeOption> nodeOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ClusterConfig model) {
            instanceType(model.instanceType);
            instanceCount(model.instanceCount);
            dedicatedMasterEnabled(model.dedicatedMasterEnabled);
            zoneAwarenessEnabled(model.zoneAwarenessEnabled);
            zoneAwarenessConfig(model.zoneAwarenessConfig);
            dedicatedMasterType(model.dedicatedMasterType);
            dedicatedMasterCount(model.dedicatedMasterCount);
            warmEnabled(model.warmEnabled);
            warmType(model.warmType);
            warmCount(model.warmCount);
            coldStorageOptions(model.coldStorageOptions);
            multiAZWithStandbyEnabled(model.multiAZWithStandbyEnabled);
            nodeOptions(model.nodeOptions);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(OpenSearchPartitionInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final Boolean getDedicatedMasterEnabled() {
            return dedicatedMasterEnabled;
        }

        public final void setDedicatedMasterEnabled(Boolean dedicatedMasterEnabled) {
            this.dedicatedMasterEnabled = dedicatedMasterEnabled;
        }

        @Override
        public final Builder dedicatedMasterEnabled(Boolean dedicatedMasterEnabled) {
            this.dedicatedMasterEnabled = dedicatedMasterEnabled;
            return this;
        }

        public final Boolean getZoneAwarenessEnabled() {
            return zoneAwarenessEnabled;
        }

        public final void setZoneAwarenessEnabled(Boolean zoneAwarenessEnabled) {
            this.zoneAwarenessEnabled = zoneAwarenessEnabled;
        }

        @Override
        public final Builder zoneAwarenessEnabled(Boolean zoneAwarenessEnabled) {
            this.zoneAwarenessEnabled = zoneAwarenessEnabled;
            return this;
        }

        public final ZoneAwarenessConfig.Builder getZoneAwarenessConfig() {
            return zoneAwarenessConfig != null ? zoneAwarenessConfig.toBuilder() : null;
        }

        public final void setZoneAwarenessConfig(ZoneAwarenessConfig.BuilderImpl zoneAwarenessConfig) {
            this.zoneAwarenessConfig = zoneAwarenessConfig != null ? zoneAwarenessConfig.build() : null;
        }

        @Override
        public final Builder zoneAwarenessConfig(ZoneAwarenessConfig zoneAwarenessConfig) {
            this.zoneAwarenessConfig = zoneAwarenessConfig;
            return this;
        }

        public final String getDedicatedMasterType() {
            return dedicatedMasterType;
        }

        public final void setDedicatedMasterType(String dedicatedMasterType) {
            this.dedicatedMasterType = dedicatedMasterType;
        }

        @Override
        public final Builder dedicatedMasterType(String dedicatedMasterType) {
            this.dedicatedMasterType = dedicatedMasterType;
            return this;
        }

        @Override
        public final Builder dedicatedMasterType(OpenSearchPartitionInstanceType dedicatedMasterType) {
            this.dedicatedMasterType(dedicatedMasterType == null ? null : dedicatedMasterType.toString());
            return this;
        }

        public final Integer getDedicatedMasterCount() {
            return dedicatedMasterCount;
        }

        public final void setDedicatedMasterCount(Integer dedicatedMasterCount) {
            this.dedicatedMasterCount = dedicatedMasterCount;
        }

        @Override
        public final Builder dedicatedMasterCount(Integer dedicatedMasterCount) {
            this.dedicatedMasterCount = dedicatedMasterCount;
            return this;
        }

        public final Boolean getWarmEnabled() {
            return warmEnabled;
        }

        public final void setWarmEnabled(Boolean warmEnabled) {
            this.warmEnabled = warmEnabled;
        }

        @Override
        public final Builder warmEnabled(Boolean warmEnabled) {
            this.warmEnabled = warmEnabled;
            return this;
        }

        public final String getWarmType() {
            return warmType;
        }

        public final void setWarmType(String warmType) {
            this.warmType = warmType;
        }

        @Override
        public final Builder warmType(String warmType) {
            this.warmType = warmType;
            return this;
        }

        @Override
        public final Builder warmType(OpenSearchWarmPartitionInstanceType warmType) {
            this.warmType(warmType == null ? null : warmType.toString());
            return this;
        }

        public final Integer getWarmCount() {
            return warmCount;
        }

        public final void setWarmCount(Integer warmCount) {
            this.warmCount = warmCount;
        }

        @Override
        public final Builder warmCount(Integer warmCount) {
            this.warmCount = warmCount;
            return this;
        }

        public final ColdStorageOptions.Builder getColdStorageOptions() {
            return coldStorageOptions != null ? coldStorageOptions.toBuilder() : null;
        }

        public final void setColdStorageOptions(ColdStorageOptions.BuilderImpl coldStorageOptions) {
            this.coldStorageOptions = coldStorageOptions != null ? coldStorageOptions.build() : null;
        }

        @Override
        public final Builder coldStorageOptions(ColdStorageOptions coldStorageOptions) {
            this.coldStorageOptions = coldStorageOptions;
            return this;
        }

        public final Boolean getMultiAZWithStandbyEnabled() {
            return multiAZWithStandbyEnabled;
        }

        public final void setMultiAZWithStandbyEnabled(Boolean multiAZWithStandbyEnabled) {
            this.multiAZWithStandbyEnabled = multiAZWithStandbyEnabled;
        }

        @Override
        public final Builder multiAZWithStandbyEnabled(Boolean multiAZWithStandbyEnabled) {
            this.multiAZWithStandbyEnabled = multiAZWithStandbyEnabled;
            return this;
        }

        public final List<NodeOption.Builder> getNodeOptions() {
            List<NodeOption.Builder> result = NodeOptionsListCopier.copyToBuilder(this.nodeOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNodeOptions(Collection<NodeOption.BuilderImpl> nodeOptions) {
            this.nodeOptions = NodeOptionsListCopier.copyFromBuilder(nodeOptions);
        }

        @Override
        public final Builder nodeOptions(Collection<NodeOption> nodeOptions) {
            this.nodeOptions = NodeOptionsListCopier.copy(nodeOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeOptions(NodeOption... nodeOptions) {
            nodeOptions(Arrays.asList(nodeOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeOptions(Consumer<NodeOption.Builder>... nodeOptions) {
            nodeOptions(Stream.of(nodeOptions).map(c -> NodeOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ClusterConfig build() {
            return new ClusterConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
