/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mgn;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mgn.model.AccessDeniedException;
import software.amazon.awssdk.services.mgn.model.ChangeServerLifeCycleStateRequest;
import software.amazon.awssdk.services.mgn.model.ChangeServerLifeCycleStateResponse;
import software.amazon.awssdk.services.mgn.model.ConflictException;
import software.amazon.awssdk.services.mgn.model.CreateReplicationConfigurationTemplateRequest;
import software.amazon.awssdk.services.mgn.model.CreateReplicationConfigurationTemplateResponse;
import software.amazon.awssdk.services.mgn.model.DeleteJobRequest;
import software.amazon.awssdk.services.mgn.model.DeleteJobResponse;
import software.amazon.awssdk.services.mgn.model.DeleteReplicationConfigurationTemplateRequest;
import software.amazon.awssdk.services.mgn.model.DeleteReplicationConfigurationTemplateResponse;
import software.amazon.awssdk.services.mgn.model.DeleteSourceServerRequest;
import software.amazon.awssdk.services.mgn.model.DeleteSourceServerResponse;
import software.amazon.awssdk.services.mgn.model.DeleteVcenterClientRequest;
import software.amazon.awssdk.services.mgn.model.DeleteVcenterClientResponse;
import software.amazon.awssdk.services.mgn.model.DescribeJobLogItemsRequest;
import software.amazon.awssdk.services.mgn.model.DescribeJobLogItemsResponse;
import software.amazon.awssdk.services.mgn.model.DescribeJobsRequest;
import software.amazon.awssdk.services.mgn.model.DescribeJobsResponse;
import software.amazon.awssdk.services.mgn.model.DescribeReplicationConfigurationTemplatesRequest;
import software.amazon.awssdk.services.mgn.model.DescribeReplicationConfigurationTemplatesResponse;
import software.amazon.awssdk.services.mgn.model.DescribeSourceServersRequest;
import software.amazon.awssdk.services.mgn.model.DescribeSourceServersResponse;
import software.amazon.awssdk.services.mgn.model.DescribeVcenterClientsRequest;
import software.amazon.awssdk.services.mgn.model.DescribeVcenterClientsResponse;
import software.amazon.awssdk.services.mgn.model.DisconnectFromServiceRequest;
import software.amazon.awssdk.services.mgn.model.DisconnectFromServiceResponse;
import software.amazon.awssdk.services.mgn.model.FinalizeCutoverRequest;
import software.amazon.awssdk.services.mgn.model.FinalizeCutoverResponse;
import software.amazon.awssdk.services.mgn.model.GetLaunchConfigurationRequest;
import software.amazon.awssdk.services.mgn.model.GetLaunchConfigurationResponse;
import software.amazon.awssdk.services.mgn.model.GetReplicationConfigurationRequest;
import software.amazon.awssdk.services.mgn.model.GetReplicationConfigurationResponse;
import software.amazon.awssdk.services.mgn.model.InitializeServiceRequest;
import software.amazon.awssdk.services.mgn.model.InitializeServiceResponse;
import software.amazon.awssdk.services.mgn.model.InternalServerException;
import software.amazon.awssdk.services.mgn.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mgn.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mgn.model.MarkAsArchivedRequest;
import software.amazon.awssdk.services.mgn.model.MarkAsArchivedResponse;
import software.amazon.awssdk.services.mgn.model.MgnException;
import software.amazon.awssdk.services.mgn.model.MgnRequest;
import software.amazon.awssdk.services.mgn.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mgn.model.RetryDataReplicationRequest;
import software.amazon.awssdk.services.mgn.model.RetryDataReplicationResponse;
import software.amazon.awssdk.services.mgn.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.mgn.model.StartCutoverRequest;
import software.amazon.awssdk.services.mgn.model.StartCutoverResponse;
import software.amazon.awssdk.services.mgn.model.StartReplicationRequest;
import software.amazon.awssdk.services.mgn.model.StartReplicationResponse;
import software.amazon.awssdk.services.mgn.model.StartTestRequest;
import software.amazon.awssdk.services.mgn.model.StartTestResponse;
import software.amazon.awssdk.services.mgn.model.TagResourceRequest;
import software.amazon.awssdk.services.mgn.model.TagResourceResponse;
import software.amazon.awssdk.services.mgn.model.TerminateTargetInstancesRequest;
import software.amazon.awssdk.services.mgn.model.TerminateTargetInstancesResponse;
import software.amazon.awssdk.services.mgn.model.ThrottlingException;
import software.amazon.awssdk.services.mgn.model.UninitializedAccountException;
import software.amazon.awssdk.services.mgn.model.UntagResourceRequest;
import software.amazon.awssdk.services.mgn.model.UntagResourceResponse;
import software.amazon.awssdk.services.mgn.model.UpdateLaunchConfigurationRequest;
import software.amazon.awssdk.services.mgn.model.UpdateLaunchConfigurationResponse;
import software.amazon.awssdk.services.mgn.model.UpdateReplicationConfigurationRequest;
import software.amazon.awssdk.services.mgn.model.UpdateReplicationConfigurationResponse;
import software.amazon.awssdk.services.mgn.model.UpdateReplicationConfigurationTemplateRequest;
import software.amazon.awssdk.services.mgn.model.UpdateReplicationConfigurationTemplateResponse;
import software.amazon.awssdk.services.mgn.model.UpdateSourceServerReplicationTypeRequest;
import software.amazon.awssdk.services.mgn.model.UpdateSourceServerReplicationTypeResponse;
import software.amazon.awssdk.services.mgn.model.ValidationException;
import software.amazon.awssdk.services.mgn.paginators.DescribeJobLogItemsPublisher;
import software.amazon.awssdk.services.mgn.paginators.DescribeJobsPublisher;
import software.amazon.awssdk.services.mgn.paginators.DescribeReplicationConfigurationTemplatesPublisher;
import software.amazon.awssdk.services.mgn.paginators.DescribeSourceServersPublisher;
import software.amazon.awssdk.services.mgn.paginators.DescribeVcenterClientsPublisher;
import software.amazon.awssdk.services.mgn.transform.ChangeServerLifeCycleStateRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.CreateReplicationConfigurationTemplateRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DeleteJobRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DeleteReplicationConfigurationTemplateRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DeleteSourceServerRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DeleteVcenterClientRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DescribeJobLogItemsRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DescribeJobsRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DescribeReplicationConfigurationTemplatesRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DescribeSourceServersRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DescribeVcenterClientsRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.DisconnectFromServiceRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.FinalizeCutoverRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.GetLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.GetReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.InitializeServiceRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.MarkAsArchivedRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.RetryDataReplicationRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.StartCutoverRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.StartReplicationRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.StartTestRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.TerminateTargetInstancesRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.UpdateLaunchConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.UpdateReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.UpdateReplicationConfigurationTemplateRequestMarshaller;
import software.amazon.awssdk.services.mgn.transform.UpdateSourceServerReplicationTypeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MgnAsyncClient}.
 *
 * @see MgnAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMgnAsyncClient implements MgnAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMgnAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMgnAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Allows the user to set the SourceServer.LifeCycle.state property for specific Source Server IDs to one of the
     * following: READY_FOR_TEST or READY_FOR_CUTOVER. This command only works if the Source Server is already
     * launchable (dataReplicationInfo.lagDuration is not null.)
     * </p>
     *
     * @param changeServerLifeCycleStateRequest
     * @return A Java Future containing the result of the ChangeServerLifeCycleState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.ChangeServerLifeCycleState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/ChangeServerLifeCycleState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ChangeServerLifeCycleStateResponse> changeServerLifeCycleState(
            ChangeServerLifeCycleStateRequest changeServerLifeCycleStateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, changeServerLifeCycleStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ChangeServerLifeCycleState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ChangeServerLifeCycleStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ChangeServerLifeCycleStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ChangeServerLifeCycleStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ChangeServerLifeCycleStateRequest, ChangeServerLifeCycleStateResponse>()
                            .withOperationName("ChangeServerLifeCycleState")
                            .withMarshaller(new ChangeServerLifeCycleStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(changeServerLifeCycleStateRequest));
            CompletableFuture<ChangeServerLifeCycleStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new ReplicationConfigurationTemplate.
     * </p>
     *
     * @param createReplicationConfigurationTemplateRequest
     * @return A Java Future containing the result of the CreateReplicationConfigurationTemplate operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>AccessDeniedException Operating denied due to a file permission or access check error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.CreateReplicationConfigurationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/CreateReplicationConfigurationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateReplicationConfigurationTemplateResponse> createReplicationConfigurationTemplate(
            CreateReplicationConfigurationTemplateRequest createReplicationConfigurationTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createReplicationConfigurationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReplicationConfigurationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateReplicationConfigurationTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateReplicationConfigurationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateReplicationConfigurationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateReplicationConfigurationTemplateRequest, CreateReplicationConfigurationTemplateResponse>()
                            .withOperationName("CreateReplicationConfigurationTemplate")
                            .withMarshaller(new CreateReplicationConfigurationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createReplicationConfigurationTemplateRequest));
            CompletableFuture<CreateReplicationConfigurationTemplateResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a single Job by ID.
     * </p>
     *
     * @param deleteJobRequest
     * @return A Java Future containing the result of the DeleteJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DeleteJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DeleteJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobResponse> deleteJob(DeleteJobRequest deleteJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobRequest, DeleteJobResponse>().withOperationName("DeleteJob")
                            .withMarshaller(new DeleteJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteJobRequest));
            CompletableFuture<DeleteJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a single Replication Configuration Template by ID
     * </p>
     *
     * @param deleteReplicationConfigurationTemplateRequest
     * @return A Java Future containing the result of the DeleteReplicationConfigurationTemplate operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DeleteReplicationConfigurationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DeleteReplicationConfigurationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReplicationConfigurationTemplateResponse> deleteReplicationConfigurationTemplate(
            DeleteReplicationConfigurationTemplateRequest deleteReplicationConfigurationTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteReplicationConfigurationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReplicationConfigurationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReplicationConfigurationTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteReplicationConfigurationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReplicationConfigurationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReplicationConfigurationTemplateRequest, DeleteReplicationConfigurationTemplateResponse>()
                            .withOperationName("DeleteReplicationConfigurationTemplate")
                            .withMarshaller(new DeleteReplicationConfigurationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteReplicationConfigurationTemplateRequest));
            CompletableFuture<DeleteReplicationConfigurationTemplateResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a single source server by ID.
     * </p>
     *
     * @param deleteSourceServerRequest
     * @return A Java Future containing the result of the DeleteSourceServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DeleteSourceServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DeleteSourceServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSourceServerResponse> deleteSourceServer(DeleteSourceServerRequest deleteSourceServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSourceServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSourceServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSourceServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSourceServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSourceServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSourceServerRequest, DeleteSourceServerResponse>()
                            .withOperationName("DeleteSourceServer")
                            .withMarshaller(new DeleteSourceServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSourceServerRequest));
            CompletableFuture<DeleteSourceServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a given vCenter client by ID.
     * </p>
     *
     * @param deleteVcenterClientRequest
     * @return A Java Future containing the result of the DeleteVcenterClient operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DeleteVcenterClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DeleteVcenterClient" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVcenterClientResponse> deleteVcenterClient(
            DeleteVcenterClientRequest deleteVcenterClientRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVcenterClientRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVcenterClient");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVcenterClientResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVcenterClientResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVcenterClientResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVcenterClientRequest, DeleteVcenterClientResponse>()
                            .withOperationName("DeleteVcenterClient")
                            .withMarshaller(new DeleteVcenterClientRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVcenterClientRequest));
            CompletableFuture<DeleteVcenterClientResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed job log items with paging.
     * </p>
     *
     * @param describeJobLogItemsRequest
     * @return A Java Future containing the result of the DescribeJobLogItems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeJobLogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeJobLogItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobLogItemsResponse> describeJobLogItems(
            DescribeJobLogItemsRequest describeJobLogItemsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobLogItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobLogItems");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobLogItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobLogItemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobLogItemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobLogItemsRequest, DescribeJobLogItemsResponse>()
                            .withOperationName("DescribeJobLogItems")
                            .withMarshaller(new DescribeJobLogItemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobLogItemsRequest));
            CompletableFuture<DescribeJobLogItemsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed job log items with paging.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobLogItems(software.amazon.awssdk.services.mgn.model.DescribeJobLogItemsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeJobLogItemsPublisher publisher = client.describeJobLogItemsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeJobLogItemsPublisher publisher = client.describeJobLogItemsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mgn.model.DescribeJobLogItemsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mgn.model.DescribeJobLogItemsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobLogItems(software.amazon.awssdk.services.mgn.model.DescribeJobLogItemsRequest)} operation.</b>
     * </p>
     *
     * @param describeJobLogItemsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeJobLogItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeJobLogItems" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeJobLogItemsPublisher describeJobLogItemsPaginator(DescribeJobLogItemsRequest describeJobLogItemsRequest) {
        return new DescribeJobLogItemsPublisher(this, applyPaginatorUserAgent(describeJobLogItemsRequest));
    }

    /**
     * <p>
     * Returns a list of Jobs. Use the JobsID and fromDate and toData filters to limit which jobs are returned. The
     * response is sorted by creationDataTime - latest date first. Jobs are normally created by the StartTest,
     * StartCutover, and TerminateTargetInstances APIs. Jobs are also created by DiagnosticLaunch and
     * TerminateDiagnosticInstances, which are APIs available only to *Support* and only used in response to relevant
     * support tickets.
     * </p>
     *
     * @param describeJobsRequest
     * @return A Java Future containing the result of the DescribeJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobsResponse> describeJobs(DescribeJobsRequest describeJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobsRequest, DescribeJobsResponse>()
                            .withOperationName("DescribeJobs").withMarshaller(new DescribeJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobsRequest));
            CompletableFuture<DescribeJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Jobs. Use the JobsID and fromDate and toData filters to limit which jobs are returned. The
     * response is sorted by creationDataTime - latest date first. Jobs are normally created by the StartTest,
     * StartCutover, and TerminateTargetInstances APIs. Jobs are also created by DiagnosticLaunch and
     * TerminateDiagnosticInstances, which are APIs available only to *Support* and only used in response to relevant
     * support tickets.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeJobs(software.amazon.awssdk.services.mgn.model.DescribeJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeJobsPublisher publisher = client.describeJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeJobsPublisher publisher = client.describeJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mgn.model.DescribeJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mgn.model.DescribeJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobs(software.amazon.awssdk.services.mgn.model.DescribeJobsRequest)} operation.</b>
     * </p>
     *
     * @param describeJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeJobsPublisher describeJobsPaginator(DescribeJobsRequest describeJobsRequest) {
        return new DescribeJobsPublisher(this, applyPaginatorUserAgent(describeJobsRequest));
    }

    /**
     * <p>
     * Lists all ReplicationConfigurationTemplates, filtered by Source Server IDs.
     * </p>
     *
     * @param describeReplicationConfigurationTemplatesRequest
     * @return A Java Future containing the result of the DescribeReplicationConfigurationTemplates operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeReplicationConfigurationTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeReplicationConfigurationTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeReplicationConfigurationTemplatesResponse> describeReplicationConfigurationTemplates(
            DescribeReplicationConfigurationTemplatesRequest describeReplicationConfigurationTemplatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeReplicationConfigurationTemplatesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReplicationConfigurationTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReplicationConfigurationTemplatesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeReplicationConfigurationTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReplicationConfigurationTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReplicationConfigurationTemplatesRequest, DescribeReplicationConfigurationTemplatesResponse>()
                            .withOperationName("DescribeReplicationConfigurationTemplates")
                            .withMarshaller(new DescribeReplicationConfigurationTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(describeReplicationConfigurationTemplatesRequest));
            CompletableFuture<DescribeReplicationConfigurationTemplatesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all ReplicationConfigurationTemplates, filtered by Source Server IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeReplicationConfigurationTemplates(software.amazon.awssdk.services.mgn.model.DescribeReplicationConfigurationTemplatesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeReplicationConfigurationTemplatesPublisher publisher = client.describeReplicationConfigurationTemplatesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeReplicationConfigurationTemplatesPublisher publisher = client.describeReplicationConfigurationTemplatesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mgn.model.DescribeReplicationConfigurationTemplatesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mgn.model.DescribeReplicationConfigurationTemplatesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReplicationConfigurationTemplates(software.amazon.awssdk.services.mgn.model.DescribeReplicationConfigurationTemplatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReplicationConfigurationTemplatesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeReplicationConfigurationTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeReplicationConfigurationTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeReplicationConfigurationTemplatesPublisher describeReplicationConfigurationTemplatesPaginator(
            DescribeReplicationConfigurationTemplatesRequest describeReplicationConfigurationTemplatesRequest) {
        return new DescribeReplicationConfigurationTemplatesPublisher(this,
                applyPaginatorUserAgent(describeReplicationConfigurationTemplatesRequest));
    }

    /**
     * <p>
     * Retrieves all SourceServers or multiple SourceServers by ID.
     * </p>
     *
     * @param describeSourceServersRequest
     * @return A Java Future containing the result of the DescribeSourceServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeSourceServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeSourceServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSourceServersResponse> describeSourceServers(
            DescribeSourceServersRequest describeSourceServersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSourceServersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSourceServers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSourceServersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSourceServersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSourceServersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSourceServersRequest, DescribeSourceServersResponse>()
                            .withOperationName("DescribeSourceServers")
                            .withMarshaller(new DescribeSourceServersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSourceServersRequest));
            CompletableFuture<DescribeSourceServersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all SourceServers or multiple SourceServers by ID.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeSourceServers(software.amazon.awssdk.services.mgn.model.DescribeSourceServersRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeSourceServersPublisher publisher = client.describeSourceServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeSourceServersPublisher publisher = client.describeSourceServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mgn.model.DescribeSourceServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mgn.model.DescribeSourceServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeSourceServers(software.amazon.awssdk.services.mgn.model.DescribeSourceServersRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeSourceServersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeSourceServers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeSourceServers" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeSourceServersPublisher describeSourceServersPaginator(DescribeSourceServersRequest describeSourceServersRequest) {
        return new DescribeSourceServersPublisher(this, applyPaginatorUserAgent(describeSourceServersRequest));
    }

    /**
     * <p>
     * Returns a list of the installed vCenter clients.
     * </p>
     *
     * @param describeVcenterClientsRequest
     * @return A Java Future containing the result of the DescribeVcenterClients operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeVcenterClients
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeVcenterClients" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVcenterClientsResponse> describeVcenterClients(
            DescribeVcenterClientsRequest describeVcenterClientsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVcenterClientsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVcenterClients");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVcenterClientsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVcenterClientsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVcenterClientsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVcenterClientsRequest, DescribeVcenterClientsResponse>()
                            .withOperationName("DescribeVcenterClients")
                            .withMarshaller(new DescribeVcenterClientsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeVcenterClientsRequest));
            CompletableFuture<DescribeVcenterClientsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the installed vCenter clients.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeVcenterClients(software.amazon.awssdk.services.mgn.model.DescribeVcenterClientsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeVcenterClientsPublisher publisher = client.describeVcenterClientsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mgn.paginators.DescribeVcenterClientsPublisher publisher = client.describeVcenterClientsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mgn.model.DescribeVcenterClientsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mgn.model.DescribeVcenterClientsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeVcenterClients(software.amazon.awssdk.services.mgn.model.DescribeVcenterClientsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeVcenterClientsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DescribeVcenterClients
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DescribeVcenterClients" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeVcenterClientsPublisher describeVcenterClientsPaginator(
            DescribeVcenterClientsRequest describeVcenterClientsRequest) {
        return new DescribeVcenterClientsPublisher(this, applyPaginatorUserAgent(describeVcenterClientsRequest));
    }

    /**
     * <p>
     * Disconnects specific Source Servers from Application Migration Service. Data replication is stopped immediately.
     * All AWS resources created by Application Migration Service for enabling the replication of these source servers
     * will be terminated / deleted within 90 minutes. Launched Test or Cutover instances will NOT be terminated. If the
     * agent on the source server has not been prevented from communicating with the Application Migration Service
     * service, then it will receive a command to uninstall itself (within approximately 10 minutes). The following
     * properties of the SourceServer will be changed immediately: dataReplicationInfo.dataReplicationState will be set
     * to DISCONNECTED; The totalStorageBytes property for each of dataReplicationInfo.replicatedDisks will be set to
     * zero; dataReplicationInfo.lagDuration and dataReplicationInfo.lagDuration will be nullified.
     * </p>
     *
     * @param disconnectFromServiceRequest
     * @return A Java Future containing the result of the DisconnectFromService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.DisconnectFromService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/DisconnectFromService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisconnectFromServiceResponse> disconnectFromService(
            DisconnectFromServiceRequest disconnectFromServiceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectFromServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectFromService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisconnectFromServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisconnectFromServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisconnectFromServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisconnectFromServiceRequest, DisconnectFromServiceResponse>()
                            .withOperationName("DisconnectFromService")
                            .withMarshaller(new DisconnectFromServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disconnectFromServiceRequest));
            CompletableFuture<DisconnectFromServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Finalizes the cutover immediately for specific Source Servers. All AWS resources created by Application Migration
     * Service for enabling the replication of these source servers will be terminated / deleted within 90 minutes.
     * Launched Test or Cutover instances will NOT be terminated. The AWS Replication Agent will receive a command to
     * uninstall itself (within 10 minutes). The following properties of the SourceServer will be changed immediately:
     * dataReplicationInfo.dataReplicationState will be changed to DISCONNECTED; The SourceServer.lifeCycle.state will
     * be changed to CUTOVER; The totalStorageBytes property fo each of dataReplicationInfo.replicatedDisks will be set
     * to zero; dataReplicationInfo.lagDuration and dataReplicationInfo.lagDuration will be nullified.
     * </p>
     *
     * @param finalizeCutoverRequest
     * @return A Java Future containing the result of the FinalizeCutover operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.FinalizeCutover
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/FinalizeCutover" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<FinalizeCutoverResponse> finalizeCutover(FinalizeCutoverRequest finalizeCutoverRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, finalizeCutoverRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "FinalizeCutover");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<FinalizeCutoverResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, FinalizeCutoverResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<FinalizeCutoverResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<FinalizeCutoverRequest, FinalizeCutoverResponse>()
                            .withOperationName("FinalizeCutover")
                            .withMarshaller(new FinalizeCutoverRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(finalizeCutoverRequest));
            CompletableFuture<FinalizeCutoverResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all LaunchConfigurations available, filtered by Source Server IDs.
     * </p>
     *
     * @param getLaunchConfigurationRequest
     * @return A Java Future containing the result of the GetLaunchConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.GetLaunchConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/GetLaunchConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLaunchConfigurationResponse> getLaunchConfiguration(
            GetLaunchConfigurationRequest getLaunchConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLaunchConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLaunchConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLaunchConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLaunchConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLaunchConfigurationRequest, GetLaunchConfigurationResponse>()
                            .withOperationName("GetLaunchConfiguration")
                            .withMarshaller(new GetLaunchConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLaunchConfigurationRequest));
            CompletableFuture<GetLaunchConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all ReplicationConfigurations, filtered by Source Server ID.
     * </p>
     *
     * @param getReplicationConfigurationRequest
     * @return A Java Future containing the result of the GetReplicationConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.GetReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/GetReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetReplicationConfigurationResponse> getReplicationConfiguration(
            GetReplicationConfigurationRequest getReplicationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReplicationConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReplicationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReplicationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReplicationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReplicationConfigurationRequest, GetReplicationConfigurationResponse>()
                            .withOperationName("GetReplicationConfiguration")
                            .withMarshaller(new GetReplicationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getReplicationConfigurationRequest));
            CompletableFuture<GetReplicationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Initialize Application Migration Service.
     * </p>
     *
     * @param initializeServiceRequest
     * @return A Java Future containing the result of the InitializeService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validate exception.</li>
     *         <li>AccessDeniedException Operating denied due to a file permission or access check error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.InitializeService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/InitializeService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<InitializeServiceResponse> initializeService(InitializeServiceRequest initializeServiceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initializeServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitializeService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InitializeServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InitializeServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InitializeServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InitializeServiceRequest, InitializeServiceResponse>()
                            .withOperationName("InitializeService")
                            .withMarshaller(new InitializeServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(initializeServiceRequest));
            CompletableFuture<InitializeServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all tags for your Application Migration Service resources.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ThrottlingException Reached throttling quota exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>AccessDeniedException Operating denied due to a file permission or access check error.</li>
     *         <li>InternalServerException The server encountered an unexpected condition that prevented it from
     *         fulfilling the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Archives specific Source Servers by setting the SourceServer.isArchived property to true for specified
     * SourceServers by ID. This command only works for SourceServers with a lifecycle. state which equals DISCONNECTED
     * or CUTOVER.
     * </p>
     *
     * @param markAsArchivedRequest
     * @return A Java Future containing the result of the MarkAsArchived operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.MarkAsArchived
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/MarkAsArchived" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<MarkAsArchivedResponse> markAsArchived(MarkAsArchivedRequest markAsArchivedRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, markAsArchivedRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MarkAsArchived");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MarkAsArchivedResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MarkAsArchivedResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MarkAsArchivedResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MarkAsArchivedRequest, MarkAsArchivedResponse>()
                            .withOperationName("MarkAsArchived")
                            .withMarshaller(new MarkAsArchivedRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(markAsArchivedRequest));
            CompletableFuture<MarkAsArchivedResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Causes the data replication initiation sequence to begin immediately upon next Handshake for specified
     * SourceServer IDs, regardless of when the previous initiation started. This command will not work if the
     * SourceServer is not stalled or is in a DISCONNECTED or STOPPED state.
     * </p>
     *
     * @param retryDataReplicationRequest
     * @return A Java Future containing the result of the RetryDataReplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.RetryDataReplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/RetryDataReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RetryDataReplicationResponse> retryDataReplication(
            RetryDataReplicationRequest retryDataReplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retryDataReplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetryDataReplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetryDataReplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RetryDataReplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RetryDataReplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetryDataReplicationRequest, RetryDataReplicationResponse>()
                            .withOperationName("RetryDataReplication")
                            .withMarshaller(new RetryDataReplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(retryDataReplicationRequest));
            CompletableFuture<RetryDataReplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Launches a Cutover Instance for specific Source Servers. This command starts a LAUNCH job whose initiatedBy
     * property is StartCutover and changes the SourceServer.lifeCycle.state property to CUTTING_OVER.
     * </p>
     *
     * @param startCutoverRequest
     * @return A Java Future containing the result of the StartCutover operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.StartCutover
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/StartCutover" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartCutoverResponse> startCutover(StartCutoverRequest startCutoverRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCutoverRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCutover");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCutoverResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartCutoverResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartCutoverResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartCutoverRequest, StartCutoverResponse>()
                            .withOperationName("StartCutover").withMarshaller(new StartCutoverRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startCutoverRequest));
            CompletableFuture<StartCutoverResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts replication for SNAPSHOT_SHIPPING agents.
     * </p>
     *
     * @param startReplicationRequest
     * @return A Java Future containing the result of the StartReplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ServiceQuotaExceededException The request could not be completed because its exceeded the service
     *         quota.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.StartReplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/StartReplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartReplicationResponse> startReplication(StartReplicationRequest startReplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startReplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartReplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartReplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartReplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartReplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartReplicationRequest, StartReplicationResponse>()
                            .withOperationName("StartReplication")
                            .withMarshaller(new StartReplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startReplicationRequest));
            CompletableFuture<StartReplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Launches a Test Instance for specific Source Servers. This command starts a LAUNCH job whose initiatedBy property
     * is StartTest and changes the SourceServer.lifeCycle.state property to TESTING.
     * </p>
     *
     * @param startTestRequest
     * @return A Java Future containing the result of the StartTest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.StartTest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/StartTest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartTestResponse> startTest(StartTestRequest startTestRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTestResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartTestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTestRequest, StartTestResponse>().withOperationName("StartTest")
                            .withMarshaller(new StartTestRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(startTestRequest));
            CompletableFuture<StartTestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or overwrites only the specified tags for the specified Application Migration Service resource or resources.
     * When you specify an existing tag key, the value is overwritten with the new value. Each resource can have a
     * maximum of 50 tags. Each tag consists of a key and optional value.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ThrottlingException Reached throttling quota exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>AccessDeniedException Operating denied due to a file permission or access check error.</li>
     *         <li>InternalServerException The server encountered an unexpected condition that prevented it from
     *         fulfilling the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a job that terminates specific launched EC2 Test and Cutover instances. This command will not work for any
     * Source Server with a lifecycle.state of TESTING, CUTTING_OVER, or CUTOVER.
     * </p>
     *
     * @param terminateTargetInstancesRequest
     * @return A Java Future containing the result of the TerminateTargetInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.TerminateTargetInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/TerminateTargetInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TerminateTargetInstancesResponse> terminateTargetInstances(
            TerminateTargetInstancesRequest terminateTargetInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateTargetInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateTargetInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TerminateTargetInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TerminateTargetInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TerminateTargetInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateTargetInstancesRequest, TerminateTargetInstancesResponse>()
                            .withOperationName("TerminateTargetInstances")
                            .withMarshaller(new TerminateTargetInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(terminateTargetInstancesRequest));
            CompletableFuture<TerminateTargetInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified set of tags from the specified set of Application Migration Service resources.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ThrottlingException Reached throttling quota exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>AccessDeniedException Operating denied due to a file permission or access check error.</li>
     *         <li>InternalServerException The server encountered an unexpected condition that prevented it from
     *         fulfilling the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates multiple LaunchConfigurations by Source Server ID.
     * </p>
     *
     * @param updateLaunchConfigurationRequest
     * @return A Java Future containing the result of the UpdateLaunchConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.UpdateLaunchConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/UpdateLaunchConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLaunchConfigurationResponse> updateLaunchConfiguration(
            UpdateLaunchConfigurationRequest updateLaunchConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLaunchConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLaunchConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLaunchConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLaunchConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLaunchConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLaunchConfigurationRequest, UpdateLaunchConfigurationResponse>()
                            .withOperationName("UpdateLaunchConfiguration")
                            .withMarshaller(new UpdateLaunchConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLaunchConfigurationRequest));
            CompletableFuture<UpdateLaunchConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to update multiple ReplicationConfigurations by Source Server ID.
     * </p>
     *
     * @param updateReplicationConfigurationRequest
     * @return A Java Future containing the result of the UpdateReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>AccessDeniedException Operating denied due to a file permission or access check error.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.UpdateReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/UpdateReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateReplicationConfigurationResponse> updateReplicationConfiguration(
            UpdateReplicationConfigurationRequest updateReplicationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateReplicationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReplicationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateReplicationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateReplicationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateReplicationConfigurationRequest, UpdateReplicationConfigurationResponse>()
                            .withOperationName("UpdateReplicationConfiguration")
                            .withMarshaller(new UpdateReplicationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateReplicationConfigurationRequest));
            CompletableFuture<UpdateReplicationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates multiple ReplicationConfigurationTemplates by ID.
     * </p>
     *
     * @param updateReplicationConfigurationTemplateRequest
     * @return A Java Future containing the result of the UpdateReplicationConfigurationTemplate operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>AccessDeniedException Operating denied due to a file permission or access check error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.UpdateReplicationConfigurationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/UpdateReplicationConfigurationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateReplicationConfigurationTemplateResponse> updateReplicationConfigurationTemplate(
            UpdateReplicationConfigurationTemplateRequest updateReplicationConfigurationTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateReplicationConfigurationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReplicationConfigurationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateReplicationConfigurationTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateReplicationConfigurationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateReplicationConfigurationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateReplicationConfigurationTemplateRequest, UpdateReplicationConfigurationTemplateResponse>()
                            .withOperationName("UpdateReplicationConfigurationTemplate")
                            .withMarshaller(new UpdateReplicationConfigurationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateReplicationConfigurationTemplateRequest));
            CompletableFuture<UpdateReplicationConfigurationTemplateResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to change between the AGENT_BASED replication type and the SNAPSHOT_SHIPPING replication type.
     * </p>
     *
     * @param updateSourceServerReplicationTypeRequest
     * @return A Java Future containing the result of the UpdateSourceServerReplicationType operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UninitializedAccountException Uninitialized account exception.</li>
     *         <li>ResourceNotFoundException Resource not found exception.</li>
     *         <li>ValidationException Validate exception.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MgnException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MgnAsyncClient.UpdateSourceServerReplicationType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mgn-2020-02-26/UpdateSourceServerReplicationType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSourceServerReplicationTypeResponse> updateSourceServerReplicationType(
            UpdateSourceServerReplicationTypeRequest updateSourceServerReplicationTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateSourceServerReplicationTypeRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mgn");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSourceServerReplicationType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSourceServerReplicationTypeResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateSourceServerReplicationTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSourceServerReplicationTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSourceServerReplicationTypeRequest, UpdateSourceServerReplicationTypeResponse>()
                            .withOperationName("UpdateSourceServerReplicationType")
                            .withMarshaller(new UpdateSourceServerReplicationTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSourceServerReplicationTypeRequest));
            CompletableFuture<UpdateSourceServerReplicationTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MgnException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UninitializedAccountException")
                                .exceptionBuilderSupplier(UninitializedAccountException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MgnRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
