/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchain.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Member configuration properties.
 * </p>
 * <p>
 * Applies only to Hyperledger Fabric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Member implements SdkPojo, Serializable, ToCopyableBuilder<Member.Builder, Member> {
    private static final SdkField<String> NETWORK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NetworkId").getter(getter(Member::networkId)).setter(setter(Builder::networkId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkId").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Member::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Member::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Member::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<MemberFrameworkAttributes> FRAMEWORK_ATTRIBUTES_FIELD = SdkField
            .<MemberFrameworkAttributes> builder(MarshallingType.SDK_POJO).memberName("FrameworkAttributes")
            .getter(getter(Member::frameworkAttributes)).setter(setter(Builder::frameworkAttributes))
            .constructor(MemberFrameworkAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FrameworkAttributes").build())
            .build();

    private static final SdkField<MemberLogPublishingConfiguration> LOG_PUBLISHING_CONFIGURATION_FIELD = SdkField
            .<MemberLogPublishingConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("LogPublishingConfiguration")
            .getter(getter(Member::logPublishingConfiguration))
            .setter(setter(Builder::logPublishingConfiguration))
            .constructor(MemberLogPublishingConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPublishingConfiguration").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Member::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate")
            .getter(getter(Member::creationDate))
            .setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(Member::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Member::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyArn").getter(getter(Member::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NETWORK_ID_FIELD, ID_FIELD,
            NAME_FIELD, DESCRIPTION_FIELD, FRAMEWORK_ATTRIBUTES_FIELD, LOG_PUBLISHING_CONFIGURATION_FIELD, STATUS_FIELD,
            CREATION_DATE_FIELD, TAGS_FIELD, ARN_FIELD, KMS_KEY_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String networkId;

    private final String id;

    private final String name;

    private final String description;

    private final MemberFrameworkAttributes frameworkAttributes;

    private final MemberLogPublishingConfiguration logPublishingConfiguration;

    private final String status;

    private final Instant creationDate;

    private final Map<String, String> tags;

    private final String arn;

    private final String kmsKeyArn;

    private Member(BuilderImpl builder) {
        this.networkId = builder.networkId;
        this.id = builder.id;
        this.name = builder.name;
        this.description = builder.description;
        this.frameworkAttributes = builder.frameworkAttributes;
        this.logPublishingConfiguration = builder.logPublishingConfiguration;
        this.status = builder.status;
        this.creationDate = builder.creationDate;
        this.tags = builder.tags;
        this.arn = builder.arn;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The unique identifier of the network to which the member belongs.
     * </p>
     * 
     * @return The unique identifier of the network to which the member belongs.
     */
    public final String networkId() {
        return networkId;
    }

    /**
     * <p>
     * The unique identifier of the member.
     * </p>
     * 
     * @return The unique identifier of the member.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the member.
     * </p>
     * 
     * @return The name of the member.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * An optional description for the member.
     * </p>
     * 
     * @return An optional description for the member.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Attributes relevant to a member for the blockchain framework that the Managed Blockchain network uses.
     * </p>
     * 
     * @return Attributes relevant to a member for the blockchain framework that the Managed Blockchain network uses.
     */
    public final MemberFrameworkAttributes frameworkAttributes() {
        return frameworkAttributes;
    }

    /**
     * <p>
     * Configuration properties for logging events associated with a member.
     * </p>
     * 
     * @return Configuration properties for logging events associated with a member.
     */
    public final MemberLogPublishingConfiguration logPublishingConfiguration() {
        return logPublishingConfiguration;
    }

    /**
     * <p>
     * The status of a member.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The member has been created and can participate in the network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The member is in the process of being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The member and all associated resources are in the process of being deleted. Either the
     * Amazon Web Services account that owns the member deleted it, or the member is being deleted as the result of an
     * <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The member can no longer participate on the network and all associated resources are
     * deleted. Either the Amazon Web Services account that owns the member deleted it, or the member is being deleted
     * as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected because it
     * cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was disabled
     * or deleted, or the grants on the key were revoked.
     * </p>
     * <p>
     * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for the
     * member resource to discover that the key is inaccessible. When a resource is in this state, we recommend deleting
     * and recreating the resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MemberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a member.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The member has been created and can participate in the network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation
     *         failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The member is in the process of being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The member and all associated resources are in the process of being deleted.
     *         Either the Amazon Web Services account that owns the member deleted it, or the member is being deleted as
     *         the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The member can no longer participate on the network and all associated resources
     *         are deleted. Either the Amazon Web Services account that owns the member deleted it, or the member is
     *         being deleted as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected
     *         because it cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS
     *         key was disabled or deleted, or the grants on the key were revoked.
     *         </p>
     *         <p>
     *         The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time
     *         for the member resource to discover that the key is inaccessible. When a resource is in this state, we
     *         recommend deleting and recreating the resource.
     *         </p>
     *         </li>
     * @see MemberStatus
     */
    public final MemberStatus status() {
        return MemberStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a member.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The member has been created and can participate in the network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The member is in the process of being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The member and all associated resources are in the process of being deleted. Either the
     * Amazon Web Services account that owns the member deleted it, or the member is being deleted as the result of an
     * <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The member can no longer participate on the network and all associated resources are
     * deleted. Either the Amazon Web Services account that owns the member deleted it, or the member is being deleted
     * as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected because it
     * cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was disabled
     * or deleted, or the grants on the key were revoked.
     * </p>
     * <p>
     * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for the
     * member resource to discover that the key is inaccessible. When a resource is in this state, we recommend deleting
     * and recreating the resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MemberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a member.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The member has been created and can participate in the network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation
     *         failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The member is in the process of being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The member and all associated resources are in the process of being deleted.
     *         Either the Amazon Web Services account that owns the member deleted it, or the member is being deleted as
     *         the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The member can no longer participate on the network and all associated resources
     *         are deleted. Either the Amazon Web Services account that owns the member deleted it, or the member is
     *         being deleted as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected
     *         because it cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS
     *         key was disabled or deleted, or the grants on the key were revoked.
     *         </p>
     *         <p>
     *         The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time
     *         for the member resource to discover that the key is inaccessible. When a resource is in this state, we
     *         recommend deleting and recreating the resource.
     *         </p>
     *         </li>
     * @see MemberStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time that the member was created.
     * </p>
     * 
     * @return The date and time that the member was created.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Tags assigned to the member. Tags consist of a key and optional value. For more information about tags, see <a
     * href
     * ="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags assigned to the member. Tags consist of a key and optional value. For more information about tags,
     *         see <a href=
     *         "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     *         >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the member. For more information about ARNs and their format, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     * (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the member. For more information about ARNs and their format, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     *         (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the member uses
     * for encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>, the member uses an
     * Amazon Web Services owned KMS key for encryption. This parameter is inherited by the nodes that this member owns.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
     * >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the
     *         member uses for encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>,
     *         the member uses an Amazon Web Services owned KMS key for encryption. This parameter is inherited by the
     *         nodes that this member owns.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
     *         >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(networkId());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(frameworkAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(logPublishingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Member)) {
            return false;
        }
        Member other = (Member) obj;
        return Objects.equals(networkId(), other.networkId()) && Objects.equals(id(), other.id())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(frameworkAttributes(), other.frameworkAttributes())
                && Objects.equals(logPublishingConfiguration(), other.logPublishingConfiguration())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationDate(), other.creationDate()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(arn(), other.arn())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Member").add("NetworkId", networkId()).add("Id", id()).add("Name", name())
                .add("Description", description()).add("FrameworkAttributes", frameworkAttributes())
                .add("LogPublishingConfiguration", logPublishingConfiguration()).add("Status", statusAsString())
                .add("CreationDate", creationDate()).add("Tags", hasTags() ? tags() : null).add("Arn", arn())
                .add("KmsKeyArn", kmsKeyArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NetworkId":
            return Optional.ofNullable(clazz.cast(networkId()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "FrameworkAttributes":
            return Optional.ofNullable(clazz.cast(frameworkAttributes()));
        case "LogPublishingConfiguration":
            return Optional.ofNullable(clazz.cast(logPublishingConfiguration()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Member, T> g) {
        return obj -> g.apply((Member) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Member> {
        /**
         * <p>
         * The unique identifier of the network to which the member belongs.
         * </p>
         * 
         * @param networkId
         *        The unique identifier of the network to which the member belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkId(String networkId);

        /**
         * <p>
         * The unique identifier of the member.
         * </p>
         * 
         * @param id
         *        The unique identifier of the member.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the member.
         * </p>
         * 
         * @param name
         *        The name of the member.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * An optional description for the member.
         * </p>
         * 
         * @param description
         *        An optional description for the member.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Attributes relevant to a member for the blockchain framework that the Managed Blockchain network uses.
         * </p>
         * 
         * @param frameworkAttributes
         *        Attributes relevant to a member for the blockchain framework that the Managed Blockchain network uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder frameworkAttributes(MemberFrameworkAttributes frameworkAttributes);

        /**
         * <p>
         * Attributes relevant to a member for the blockchain framework that the Managed Blockchain network uses.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemberFrameworkAttributes.Builder}
         * avoiding the need to create one manually via {@link MemberFrameworkAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemberFrameworkAttributes.Builder#build()} is called immediately
         * and its result is passed to {@link #frameworkAttributes(MemberFrameworkAttributes)}.
         * 
         * @param frameworkAttributes
         *        a consumer that will call methods on {@link MemberFrameworkAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #frameworkAttributes(MemberFrameworkAttributes)
         */
        default Builder frameworkAttributes(Consumer<MemberFrameworkAttributes.Builder> frameworkAttributes) {
            return frameworkAttributes(MemberFrameworkAttributes.builder().applyMutation(frameworkAttributes).build());
        }

        /**
         * <p>
         * Configuration properties for logging events associated with a member.
         * </p>
         * 
         * @param logPublishingConfiguration
         *        Configuration properties for logging events associated with a member.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingConfiguration(MemberLogPublishingConfiguration logPublishingConfiguration);

        /**
         * <p>
         * Configuration properties for logging events associated with a member.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemberLogPublishingConfiguration.Builder}
         * avoiding the need to create one manually via {@link MemberLogPublishingConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemberLogPublishingConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #logPublishingConfiguration(MemberLogPublishingConfiguration)}.
         * 
         * @param logPublishingConfiguration
         *        a consumer that will call methods on {@link MemberLogPublishingConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logPublishingConfiguration(MemberLogPublishingConfiguration)
         */
        default Builder logPublishingConfiguration(Consumer<MemberLogPublishingConfiguration.Builder> logPublishingConfiguration) {
            return logPublishingConfiguration(MemberLogPublishingConfiguration.builder()
                    .applyMutation(logPublishingConfiguration).build());
        }

        /**
         * <p>
         * The status of a member.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The member has been created and can participate in the network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation
         * failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The member is in the process of being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The member and all associated resources are in the process of being deleted. Either
         * the Amazon Web Services account that owns the member deleted it, or the member is being deleted as the result
         * of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The member can no longer participate on the network and all associated resources are
         * deleted. Either the Amazon Web Services account that owns the member deleted it, or the member is being
         * deleted as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected because
         * it cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was
         * disabled or deleted, or the grants on the key were revoked.
         * </p>
         * <p>
         * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for
         * the member resource to discover that the key is inaccessible. When a resource is in this state, we recommend
         * deleting and recreating the resource.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of a member.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The member has been created and can participate in the network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation
         *        failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The member is in the process of being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The member and all associated resources are in the process of being deleted.
         *        Either the Amazon Web Services account that owns the member deleted it, or the member is being deleted
         *        as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The member can no longer participate on the network and all associated
         *        resources are deleted. Either the Amazon Web Services account that owns the member deleted it, or the
         *        member is being deleted as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the
         *        member.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected
         *        because it cannot access the specified customer managed key in KMS for encryption at rest. Either the
         *        KMS key was disabled or deleted, or the grants on the key were revoked.
         *        </p>
         *        <p>
         *        The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some
         *        time for the member resource to discover that the key is inaccessible. When a resource is in this
         *        state, we recommend deleting and recreating the resource.
         *        </p>
         *        </li>
         * @see MemberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a member.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The member has been created and can participate in the network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation
         * failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The member is in the process of being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The member and all associated resources are in the process of being deleted. Either
         * the Amazon Web Services account that owns the member deleted it, or the member is being deleted as the result
         * of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The member can no longer participate on the network and all associated resources are
         * deleted. Either the Amazon Web Services account that owns the member deleted it, or the member is being
         * deleted as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected because
         * it cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was
         * disabled or deleted, or the grants on the key were revoked.
         * </p>
         * <p>
         * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for
         * the member resource to discover that the key is inaccessible. When a resource is in this state, we recommend
         * deleting and recreating the resource.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of a member.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The Amazon Web Services account is in the process of creating a member.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The member has been created and can participate in the network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a member and creation
         *        failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The member is in the process of being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The member and all associated resources are in the process of being deleted.
         *        Either the Amazon Web Services account that owns the member deleted it, or the member is being deleted
         *        as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the member.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The member can no longer participate on the network and all associated
         *        resources are deleted. Either the Amazon Web Services account that owns the member deleted it, or the
         *        member is being deleted as the result of an <code>APPROVED</code> <code>PROPOSAL</code> to remove the
         *        member.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The member is impaired and might not function as expected
         *        because it cannot access the specified customer managed key in KMS for encryption at rest. Either the
         *        KMS key was disabled or deleted, or the grants on the key were revoked.
         *        </p>
         *        <p>
         *        The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some
         *        time for the member resource to discover that the key is inaccessible. When a resource is in this
         *        state, we recommend deleting and recreating the resource.
         *        </p>
         *        </li>
         * @see MemberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberStatus
         */
        Builder status(MemberStatus status);

        /**
         * <p>
         * The date and time that the member was created.
         * </p>
         * 
         * @param creationDate
         *        The date and time that the member was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * Tags assigned to the member. Tags consist of a key and optional value. For more information about tags, see
         * <a
         * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html">
         * Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * </p>
         * 
         * @param tags
         *        Tags assigned to the member. Tags consist of a key and optional value. For more information about
         *        tags, see <a href=
         *        "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
         *        >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the member. For more information about ARNs and their format, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
         * (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the member. For more information about ARNs and their format, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the member
         * uses for encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>, the member
         * uses an Amazon Web Services owned KMS key for encryption. This parameter is inherited by the nodes that this
         * member owns.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
         * >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * </p>
         * 
         * @param kmsKeyArn
         *        The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the
         *        member uses for encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>
         *        , the member uses an Amazon Web Services owned KMS key for encryption. This parameter is inherited by
         *        the nodes that this member owns.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
         *        >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String networkId;

        private String id;

        private String name;

        private String description;

        private MemberFrameworkAttributes frameworkAttributes;

        private MemberLogPublishingConfiguration logPublishingConfiguration;

        private String status;

        private Instant creationDate;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String arn;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(Member model) {
            networkId(model.networkId);
            id(model.id);
            name(model.name);
            description(model.description);
            frameworkAttributes(model.frameworkAttributes);
            logPublishingConfiguration(model.logPublishingConfiguration);
            status(model.status);
            creationDate(model.creationDate);
            tags(model.tags);
            arn(model.arn);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getNetworkId() {
            return networkId;
        }

        public final void setNetworkId(String networkId) {
            this.networkId = networkId;
        }

        @Override
        public final Builder networkId(String networkId) {
            this.networkId = networkId;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final MemberFrameworkAttributes.Builder getFrameworkAttributes() {
            return frameworkAttributes != null ? frameworkAttributes.toBuilder() : null;
        }

        public final void setFrameworkAttributes(MemberFrameworkAttributes.BuilderImpl frameworkAttributes) {
            this.frameworkAttributes = frameworkAttributes != null ? frameworkAttributes.build() : null;
        }

        @Override
        public final Builder frameworkAttributes(MemberFrameworkAttributes frameworkAttributes) {
            this.frameworkAttributes = frameworkAttributes;
            return this;
        }

        public final MemberLogPublishingConfiguration.Builder getLogPublishingConfiguration() {
            return logPublishingConfiguration != null ? logPublishingConfiguration.toBuilder() : null;
        }

        public final void setLogPublishingConfiguration(MemberLogPublishingConfiguration.BuilderImpl logPublishingConfiguration) {
            this.logPublishingConfiguration = logPublishingConfiguration != null ? logPublishingConfiguration.build() : null;
        }

        @Override
        public final Builder logPublishingConfiguration(MemberLogPublishingConfiguration logPublishingConfiguration) {
            this.logPublishingConfiguration = logPublishingConfiguration;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MemberStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = OutputTagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = OutputTagMapCopier.copy(tags);
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        @Override
        public Member build() {
            return new Member(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
