/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchain;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.managedblockchain.model.AccessDeniedException;
import software.amazon.awssdk.services.managedblockchain.model.CreateAccessorRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateAccessorResponse;
import software.amazon.awssdk.services.managedblockchain.model.CreateMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.CreateNetworkRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateNetworkResponse;
import software.amazon.awssdk.services.managedblockchain.model.CreateNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.CreateProposalRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateProposalResponse;
import software.amazon.awssdk.services.managedblockchain.model.DeleteAccessorRequest;
import software.amazon.awssdk.services.managedblockchain.model.DeleteAccessorResponse;
import software.amazon.awssdk.services.managedblockchain.model.DeleteMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.DeleteMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.DeleteNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.DeleteNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetAccessorRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetAccessorResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetNetworkRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetNetworkResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetProposalRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetProposalResponse;
import software.amazon.awssdk.services.managedblockchain.model.IllegalActionException;
import software.amazon.awssdk.services.managedblockchain.model.InternalServiceErrorException;
import software.amazon.awssdk.services.managedblockchain.model.InvalidRequestException;
import software.amazon.awssdk.services.managedblockchain.model.ListAccessorsRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListAccessorsResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListInvitationsRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListInvitationsResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListMembersRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListMembersResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListNetworksRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListNetworksResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListNodesRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListNodesResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.managedblockchain.model.ManagedBlockchainException;
import software.amazon.awssdk.services.managedblockchain.model.ManagedBlockchainRequest;
import software.amazon.awssdk.services.managedblockchain.model.RejectInvitationRequest;
import software.amazon.awssdk.services.managedblockchain.model.RejectInvitationResponse;
import software.amazon.awssdk.services.managedblockchain.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.managedblockchain.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.managedblockchain.model.ResourceNotFoundException;
import software.amazon.awssdk.services.managedblockchain.model.ResourceNotReadyException;
import software.amazon.awssdk.services.managedblockchain.model.TagResourceRequest;
import software.amazon.awssdk.services.managedblockchain.model.TagResourceResponse;
import software.amazon.awssdk.services.managedblockchain.model.ThrottlingException;
import software.amazon.awssdk.services.managedblockchain.model.TooManyTagsException;
import software.amazon.awssdk.services.managedblockchain.model.UntagResourceRequest;
import software.amazon.awssdk.services.managedblockchain.model.UntagResourceResponse;
import software.amazon.awssdk.services.managedblockchain.model.UpdateMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.UpdateMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.UpdateNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.UpdateNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.VoteOnProposalRequest;
import software.amazon.awssdk.services.managedblockchain.model.VoteOnProposalResponse;
import software.amazon.awssdk.services.managedblockchain.paginators.ListAccessorsPublisher;
import software.amazon.awssdk.services.managedblockchain.paginators.ListInvitationsPublisher;
import software.amazon.awssdk.services.managedblockchain.paginators.ListMembersPublisher;
import software.amazon.awssdk.services.managedblockchain.paginators.ListNetworksPublisher;
import software.amazon.awssdk.services.managedblockchain.paginators.ListNodesPublisher;
import software.amazon.awssdk.services.managedblockchain.paginators.ListProposalVotesPublisher;
import software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsPublisher;
import software.amazon.awssdk.services.managedblockchain.transform.CreateAccessorRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.CreateMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.CreateNetworkRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.CreateNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.CreateProposalRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.DeleteAccessorRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.DeleteMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.DeleteNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetAccessorRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetNetworkRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetProposalRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListAccessorsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListNetworksRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListNodesRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListProposalVotesRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListProposalsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.RejectInvitationRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.UpdateMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.UpdateNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.VoteOnProposalRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ManagedBlockchainAsyncClient}.
 *
 * @see ManagedBlockchainAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultManagedBlockchainAsyncClient implements ManagedBlockchainAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultManagedBlockchainAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultManagedBlockchainAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <important>
     * <p>
     * The token based access feature is in preview release for Ethereum on Amazon Managed Blockchain and is subject to
     * change. We recommend that you use this feature only with test scenarios, and not in production environments.
     * </p>
     * </important>
     * <p>
     * Creates a new accessor for use with Managed Blockchain Ethereum nodes. An accessor object is a container that has
     * the information required for token based access to your Ethereum nodes.
     * </p>
     *
     * @param createAccessorRequest
     * @return A Java Future containing the result of the CreateAccessor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceAlreadyExistsException A resource request is issued for a resource that already exists.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>ResourceLimitExceededException The maximum number of resources of that type already exist. Ensure the
     *         resources requested are within the boundaries of the service edition and your account limits.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.CreateAccessor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateAccessor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessorResponse> createAccessor(CreateAccessorRequest createAccessorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAccessorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAccessorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAccessorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessorRequest, CreateAccessorResponse>()
                            .withOperationName("CreateAccessor")
                            .withMarshaller(new CreateAccessorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAccessorRequest));
            CompletableFuture<CreateAccessorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a member within a Managed Blockchain network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param createMemberRequest
     * @return A Java Future containing the result of the CreateMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ResourceAlreadyExistsException A resource request is issued for a resource that already exists.</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>ResourceLimitExceededException The maximum number of resources of that type already exist. Ensure the
     *         resources requested are within the boundaries of the service edition and your account limits.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>TooManyTagsException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.CreateMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMemberResponse> createMember(CreateMemberRequest createMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMemberRequest, CreateMemberResponse>()
                            .withOperationName("CreateMember").withMarshaller(new CreateMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMemberRequest));
            CompletableFuture<CreateMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new blockchain network using Amazon Managed Blockchain.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param createNetworkRequest
     * @return A Java Future containing the result of the CreateNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceAlreadyExistsException A resource request is issued for a resource that already exists.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>ResourceLimitExceededException The maximum number of resources of that type already exist. Ensure the
     *         resources requested are within the boundaries of the service edition and your account limits.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>TooManyTagsException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.CreateNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateNetwork"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNetworkResponse> createNetwork(CreateNetworkRequest createNetworkRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNetworkRequest, CreateNetworkResponse>()
                            .withOperationName("CreateNetwork")
                            .withMarshaller(new CreateNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createNetworkRequest));
            CompletableFuture<CreateNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a node on the specified blockchain network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param createNodeRequest
     * @return A Java Future containing the result of the CreateNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ResourceAlreadyExistsException A resource request is issued for a resource that already exists.</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>ResourceLimitExceededException The maximum number of resources of that type already exist. Ensure the
     *         resources requested are within the boundaries of the service edition and your account limits.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>TooManyTagsException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.CreateNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNodeResponse> createNode(CreateNodeRequest createNodeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNodeRequest, CreateNodeResponse>().withOperationName("CreateNode")
                            .withMarshaller(new CreateNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createNodeRequest));
            CompletableFuture<CreateNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a proposal for a change to the network that other members of the network can vote on, for example, a
     * proposal to add a new member to the network. Any member can create a proposal.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param createProposalRequest
     * @return A Java Future containing the result of the CreateProposal operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>TooManyTagsException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.CreateProposal
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateProposal"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProposalResponse> createProposal(CreateProposalRequest createProposalRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProposalRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProposal");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProposalResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateProposalResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProposalResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProposalRequest, CreateProposalResponse>()
                            .withOperationName("CreateProposal")
                            .withMarshaller(new CreateProposalRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProposalRequest));
            CompletableFuture<CreateProposalResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <important>
     * <p>
     * The token based access feature is in preview release for Ethereum on Amazon Managed Blockchain and is subject to
     * change. We recommend that you use this feature only with test scenarios, and not in production environments.
     * </p>
     * </important>
     * <p>
     * Deletes an accessor that your Amazon Web Services account owns. An accessor object is a container that has the
     * information required for token based access to your Ethereum nodes including, the <code>BILLING_TOKEN</code>.
     * After an accessor is deleted, the status of the accessor changes from <code>AVAILABLE</code> to
     * <code>PENDING_DELETION</code>. An accessor in the <code>PENDING_DELETION</code> state can’t be used for new
     * WebSocket requests or HTTP requests. However, WebSocket connections that were initiated while the accessor was in
     * the <code>AVAILABLE</code> state remain open until they expire (up to 2 hours).
     * </p>
     *
     * @param deleteAccessorRequest
     * @return A Java Future containing the result of the DeleteAccessor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.DeleteAccessor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/DeleteAccessor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessorResponse> deleteAccessor(DeleteAccessorRequest deleteAccessorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccessorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAccessorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAccessorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAccessorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessorRequest, DeleteAccessorResponse>()
                            .withOperationName("DeleteAccessor")
                            .withMarshaller(new DeleteAccessorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAccessorRequest));
            CompletableFuture<DeleteAccessorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a member. Deleting a member removes the member and all associated resources from the network.
     * <code>DeleteMember</code> can only be called for a specified <code>MemberId</code> if the principal performing
     * the action is associated with the Amazon Web Services account that owns the member. In all other cases, the
     * <code>DeleteMember</code> action is carried out as the result of an approved proposal to remove a member. If
     * <code>MemberId</code> is the last member in a network specified by the last Amazon Web Services account, the
     * network is deleted also.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param deleteMemberRequest
     * @return A Java Future containing the result of the DeleteMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.DeleteMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/DeleteMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMemberResponse> deleteMember(DeleteMemberRequest deleteMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMemberRequest, DeleteMemberResponse>()
                            .withOperationName("DeleteMember").withMarshaller(new DeleteMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMemberRequest));
            CompletableFuture<DeleteMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a node that your Amazon Web Services account owns. All data on the node is lost and cannot be recovered.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param deleteNodeRequest
     * @return A Java Future containing the result of the DeleteNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.DeleteNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/DeleteNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNodeResponse> deleteNode(DeleteNodeRequest deleteNodeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNodeRequest, DeleteNodeResponse>().withOperationName("DeleteNode")
                            .withMarshaller(new DeleteNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteNodeRequest));
            CompletableFuture<DeleteNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <important>
     * <p>
     * The token based access feature is in preview release for Ethereum on Amazon Managed Blockchain and is subject to
     * change. We recommend that you use this feature only with test scenarios, and not in production environments.
     * </p>
     * </important>
     * <p>
     * Returns detailed information about an accessor. An accessor object is a container that has the information
     * required for token based access to your Ethereum nodes.
     * </p>
     *
     * @param getAccessorRequest
     * @return A Java Future containing the result of the GetAccessor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.GetAccessor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetAccessor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessorResponse> getAccessor(GetAccessorRequest getAccessorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccessorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAccessorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccessorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessorRequest, GetAccessorResponse>()
                            .withOperationName("GetAccessor").withMarshaller(new GetAccessorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccessorRequest));
            CompletableFuture<GetAccessorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a member.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param getMemberRequest
     * @return A Java Future containing the result of the GetMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.GetMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetMember" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMemberResponse> getMember(GetMemberRequest getMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMemberRequest, GetMemberResponse>().withOperationName("GetMember")
                            .withMarshaller(new GetMemberRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMemberRequest));
            CompletableFuture<GetMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param getNetworkRequest
     * @return A Java Future containing the result of the GetNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.GetNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNetworkResponse> getNetwork(GetNetworkRequest getNetworkRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNetworkRequest, GetNetworkResponse>().withOperationName("GetNetwork")
                            .withMarshaller(new GetNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getNetworkRequest));
            CompletableFuture<GetNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a node.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param getNodeRequest
     * @return A Java Future containing the result of the GetNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.GetNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetNodeResponse> getNode(GetNodeRequest getNodeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNodeRequest, GetNodeResponse>().withOperationName("GetNode")
                            .withMarshaller(new GetNodeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getNodeRequest));
            CompletableFuture<GetNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a proposal.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param getProposalRequest
     * @return A Java Future containing the result of the GetProposal operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.GetProposal
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetProposal" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProposalResponse> getProposal(GetProposalRequest getProposalRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProposalRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProposal");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProposalResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProposalResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProposalResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProposalRequest, GetProposalResponse>()
                            .withOperationName("GetProposal").withMarshaller(new GetProposalRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProposalRequest));
            CompletableFuture<GetProposalResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <important>
     * <p>
     * The token based access feature is in preview release for Ethereum on Amazon Managed Blockchain and is subject to
     * change. We recommend that you use this feature only with test scenarios, and not in production environments.
     * </p>
     * </important>
     * <p>
     * Returns a list of the accessors and their properties. Accessor objects are containers that have the information
     * required for token based access to your Ethereum nodes.
     * </p>
     *
     * @param listAccessorsRequest
     * @return A Java Future containing the result of the ListAccessors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListAccessors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListAccessors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessorsResponse> listAccessors(ListAccessorsRequest listAccessorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccessorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAccessorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccessorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessorsRequest, ListAccessorsResponse>()
                            .withOperationName("ListAccessors")
                            .withMarshaller(new ListAccessorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAccessorsRequest));
            CompletableFuture<ListAccessorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <important>
     * <p>
     * The token based access feature is in preview release for Ethereum on Amazon Managed Blockchain and is subject to
     * change. We recommend that you use this feature only with test scenarios, and not in production environments.
     * </p>
     * </important>
     * <p>
     * Returns a list of the accessors and their properties. Accessor objects are containers that have the information
     * required for token based access to your Ethereum nodes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessors(software.amazon.awssdk.services.managedblockchain.model.ListAccessorsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListAccessorsPublisher publisher = client.listAccessorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListAccessorsPublisher publisher = client.listAccessorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.managedblockchain.model.ListAccessorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.managedblockchain.model.ListAccessorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessors(software.amazon.awssdk.services.managedblockchain.model.ListAccessorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAccessorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListAccessors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListAccessors"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAccessorsPublisher listAccessorsPaginator(ListAccessorsRequest listAccessorsRequest) {
        return new ListAccessorsPublisher(this, applyPaginatorUserAgent(listAccessorsRequest));
    }

    /**
     * <p>
     * Returns a list of all invitations for the current Amazon Web Services account.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listInvitationsRequest
     * @return A Java Future containing the result of the ListInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>ResourceLimitExceededException The maximum number of resources of that type already exist. Ensure the
     *         resources requested are within the boundaries of the service edition and your account limits.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvitationsResponse> listInvitations(ListInvitationsRequest listInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                            .withOperationName("ListInvitations")
                            .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInvitationsRequest));
            CompletableFuture<ListInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all invitations for the current Amazon Web Services account.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.managedblockchain.model.ListInvitationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.managedblockchain.model.ListInvitationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.managedblockchain.model.ListInvitationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.managedblockchain.model.ListInvitationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInvitationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>ResourceLimitExceededException The maximum number of resources of that type already exist. Ensure the
     *         resources requested are within the boundaries of the service edition and your account limits.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListInvitationsPublisher listInvitationsPaginator(ListInvitationsRequest listInvitationsRequest) {
        return new ListInvitationsPublisher(this, applyPaginatorUserAgent(listInvitationsRequest));
    }

    /**
     * <p>
     * Returns a list of the members in a network and properties of their configurations.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listMembersRequest
     * @return A Java Future containing the result of the ListMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListMembers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMembersResponse> listMembers(ListMembersRequest listMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                            .withOperationName("ListMembers").withMarshaller(new ListMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMembersRequest));
            CompletableFuture<ListMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the members in a network and properties of their configurations.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMembers(software.amazon.awssdk.services.managedblockchain.model.ListMembersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.managedblockchain.model.ListMembersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.managedblockchain.model.ListMembersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.managedblockchain.model.ListMembersRequest)} operation.</b>
     * </p>
     *
     * @param listMembersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListMembers" target="_top">AWS
     *      API Documentation</a>
     */
    public ListMembersPublisher listMembersPaginator(ListMembersRequest listMembersRequest) {
        return new ListMembersPublisher(this, applyPaginatorUserAgent(listMembersRequest));
    }

    /**
     * <p>
     * Returns information about the networks in which the current Amazon Web Services account participates.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param listNetworksRequest
     * @return A Java Future containing the result of the ListNetworks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListNetworks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNetworks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNetworksResponse> listNetworks(ListNetworksRequest listNetworksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNetworksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListNetworksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNetworksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNetworksRequest, ListNetworksResponse>()
                            .withOperationName("ListNetworks").withMarshaller(new ListNetworksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listNetworksRequest));
            CompletableFuture<ListNetworksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the networks in which the current Amazon Web Services account participates.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworks(software.amazon.awssdk.services.managedblockchain.model.ListNetworksRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNetworksPublisher publisher = client.listNetworksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNetworksPublisher publisher = client.listNetworksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.managedblockchain.model.ListNetworksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.managedblockchain.model.ListNetworksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworks(software.amazon.awssdk.services.managedblockchain.model.ListNetworksRequest)} operation.</b>
     * </p>
     *
     * @param listNetworksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListNetworks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNetworks"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNetworksPublisher listNetworksPaginator(ListNetworksRequest listNetworksRequest) {
        return new ListNetworksPublisher(this, applyPaginatorUserAgent(listNetworksRequest));
    }

    /**
     * <p>
     * Returns information about the nodes within a network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param listNodesRequest
     * @return A Java Future containing the result of the ListNodes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNodes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNodesResponse> listNodes(ListNodesRequest listNodesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNodes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNodesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListNodesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNodesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNodesRequest, ListNodesResponse>().withOperationName("ListNodes")
                            .withMarshaller(new ListNodesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNodesRequest));
            CompletableFuture<ListNodesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the nodes within a network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listNodes(software.amazon.awssdk.services.managedblockchain.model.ListNodesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNodesPublisher publisher = client.listNodesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNodesPublisher publisher = client.listNodesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.managedblockchain.model.ListNodesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.managedblockchain.model.ListNodesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNodes(software.amazon.awssdk.services.managedblockchain.model.ListNodesRequest)} operation.</b>
     * </p>
     *
     * @param listNodesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNodes" target="_top">AWS
     *      API Documentation</a>
     */
    public ListNodesPublisher listNodesPaginator(ListNodesRequest listNodesRequest) {
        return new ListNodesPublisher(this, applyPaginatorUserAgent(listNodesRequest));
    }

    /**
     * <p>
     * Returns the list of votes for a specified proposal, including the value of each vote and the unique identifier of
     * the member that cast the vote.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listProposalVotesRequest
     * @return A Java Future containing the result of the ListProposalVotes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListProposalVotes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposalVotes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProposalVotesResponse> listProposalVotes(ListProposalVotesRequest listProposalVotesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProposalVotesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProposalVotes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProposalVotesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProposalVotesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProposalVotesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProposalVotesRequest, ListProposalVotesResponse>()
                            .withOperationName("ListProposalVotes")
                            .withMarshaller(new ListProposalVotesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProposalVotesRequest));
            CompletableFuture<ListProposalVotesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of votes for a specified proposal, including the value of each vote and the unique identifier of
     * the member that cast the vote.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProposalVotes(software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalVotesPublisher publisher = client.listProposalVotesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalVotesPublisher publisher = client.listProposalVotesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProposalVotes(software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProposalVotesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListProposalVotes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposalVotes"
     *      target="_top">AWS API Documentation</a>
     */
    public ListProposalVotesPublisher listProposalVotesPaginator(ListProposalVotesRequest listProposalVotesRequest) {
        return new ListProposalVotesPublisher(this, applyPaginatorUserAgent(listProposalVotesRequest));
    }

    /**
     * <p>
     * Returns a list of proposals for the network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listProposalsRequest
     * @return A Java Future containing the result of the ListProposals operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListProposals
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposals"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProposalsResponse> listProposals(ListProposalsRequest listProposalsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProposalsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProposals");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProposalsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProposalsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProposalsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProposalsRequest, ListProposalsResponse>()
                            .withOperationName("ListProposals")
                            .withMarshaller(new ListProposalsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProposalsRequest));
            CompletableFuture<ListProposalsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of proposals for the network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProposals(software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsPublisher publisher = client.listProposalsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsPublisher publisher = client.listProposalsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProposals(software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProposalsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListProposals
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposals"
     *      target="_top">AWS API Documentation</a>
     */
    public ListProposalsPublisher listProposalsPaginator(ListProposalsRequest listProposalsRequest) {
        return new ListProposalsPublisher(this, applyPaginatorUserAgent(listProposalsRequest));
    }

    /**
     * <p>
     * Returns a list of tags for the specified resource. Each tag consists of a key and optional value.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects an invitation to join a network. This action can be called by a principal in an Amazon Web Services
     * account that has received an invitation to create a member and join a network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param rejectInvitationRequest
     * @return A Java Future containing the result of the RejectInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>IllegalActionException</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.RejectInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/RejectInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectInvitationResponse> rejectInvitation(RejectInvitationRequest rejectInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectInvitationRequest, RejectInvitationResponse>()
                            .withOperationName("RejectInvitation")
                            .withMarshaller(new RejectInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rejectInvitationRequest));
            CompletableFuture<RejectInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or overwrites the specified tags for the specified Amazon Managed Blockchain resource. Each tag consists of
     * a key and optional value.
     * </p>
     * <p>
     * When you specify a tag key that already exists, the tag value is overwritten with the new value. Use
     * <code>UntagResource</code> to remove tag keys.
     * </p>
     * <p>
     * A resource can have up to 50 tags. If you try to create more than 50 tags for a resource, your request fails and
     * returns an error.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>TooManyTagsException</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the Amazon Managed Blockchain resource.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ResourceNotReadyException The requested resource exists but isn't in a status that can complete the
     *         operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a member configuration with new parameters.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param updateMemberRequest
     * @return A Java Future containing the result of the UpdateMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.UpdateMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/UpdateMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMemberResponse> updateMember(UpdateMemberRequest updateMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMemberRequest, UpdateMemberResponse>()
                            .withOperationName("UpdateMember").withMarshaller(new UpdateMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMemberRequest));
            CompletableFuture<UpdateMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a node configuration with new parameters.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param updateNodeRequest
     * @return A Java Future containing the result of the UpdateNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.UpdateNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/UpdateNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNodeResponse> updateNode(UpdateNodeRequest updateNodeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNodeRequest, UpdateNodeResponse>().withOperationName("UpdateNode")
                            .withMarshaller(new UpdateNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateNodeRequest));
            CompletableFuture<UpdateNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Casts a vote for a specified <code>ProposalId</code> on behalf of a member. The member to vote as, specified by
     * <code>VoterMemberId</code>, must be in the same Amazon Web Services account as the principal that calls the
     * action.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param voteOnProposalRequest
     * @return A Java Future containing the result of the VoteOnProposal operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The action or operation requested is invalid. Verify that the action is typed
     *         correctly.</li>
     *         <li>IllegalActionException</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException A requested resource doesn't exist. It may have been deleted or referenced
     *         incorrectly.</li>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is creating resources that exceed your service
     *         limit for this resource type. Request a limit increase or delete unused resources if possible.</li>
     *         <li>InternalServiceErrorException The request processing has failed because of an unknown error,
     *         exception or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainAsyncClient.VoteOnProposal
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/VoteOnProposal"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<VoteOnProposalResponse> voteOnProposal(VoteOnProposalRequest voteOnProposalRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, voteOnProposalRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VoteOnProposal");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<VoteOnProposalResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, VoteOnProposalResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<VoteOnProposalResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<VoteOnProposalRequest, VoteOnProposalResponse>()
                            .withOperationName("VoteOnProposal")
                            .withMarshaller(new VoteOnProposalRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(voteOnProposalRequest));
            CompletableFuture<VoteOnProposalResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ManagedBlockchainException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotReadyException")
                                .exceptionBuilderSupplier(ResourceNotReadyException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalActionException")
                                .exceptionBuilderSupplier(IllegalActionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ManagedBlockchainRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
