/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.macie.model.AccessDeniedException;
import software.amazon.awssdk.services.macie.model.AssociateMemberAccountRequest;
import software.amazon.awssdk.services.macie.model.AssociateMemberAccountResponse;
import software.amazon.awssdk.services.macie.model.AssociateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.AssociateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.DisassociateMemberAccountRequest;
import software.amazon.awssdk.services.macie.model.DisassociateMemberAccountResponse;
import software.amazon.awssdk.services.macie.model.DisassociateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.DisassociateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.InternalException;
import software.amazon.awssdk.services.macie.model.InvalidInputException;
import software.amazon.awssdk.services.macie.model.LimitExceededException;
import software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest;
import software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse;
import software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.MacieException;
import software.amazon.awssdk.services.macie.model.UpdateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.UpdateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable;
import software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable;

/**
 * Service client for accessing Amazon Macie. This can be created using the static {@link #builder()} method.
 *
 * <fullname>Amazon Macie Classic</fullname>
 * <p>
 * Amazon Macie Classic is a security service that uses machine learning to automatically discover, classify, and
 * protect sensitive data in AWS. Macie Classic recognizes sensitive data such as personally identifiable information
 * (PII) or intellectual property, and provides you with dashboards and alerts that give visibility into how this data
 * is being accessed or moved. For more information, see the <a
 * href="https://docs.aws.amazon.com/macie/latest/userguide/what-is-macie.html">Amazon Macie Classic User Guide</a>.
 * </p>
 * <p>
 * A new Amazon Macie is now available with significant design improvements and additional features, at a lower price
 * and in most AWS Regions. We encourage you to explore and use the new and improved features, and benefit from the
 * reduced cost. To learn about features and pricing for the new Amazon Macie, see <a
 * href="https://aws.amazon.com/macie/">Amazon Macie</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface MacieClient extends SdkClient {
    String SERVICE_NAME = "macie";

    /**
     * Create a {@link MacieClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static MacieClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link MacieClient}.
     */
    static MacieClientBuilder builder() {
        return new DefaultMacieClientBuilder();
    }

    /**
     * <p>
     * Associates a specified AWS account with Amazon Macie Classic as a member account.
     * </p>
     *
     * @param associateMemberAccountRequest
     * @return Result of the AssociateMemberAccount operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.AssociateMemberAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateMemberAccount" target="_top">AWS
     *      API Documentation</a>
     */
    default AssociateMemberAccountResponse associateMemberAccount(AssociateMemberAccountRequest associateMemberAccountRequest)
            throws InvalidInputException, LimitExceededException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates a specified AWS account with Amazon Macie Classic as a member account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AssociateMemberAccountRequest.Builder} avoiding the
     * need to create one manually via {@link AssociateMemberAccountRequest#builder()}
     * </p>
     *
     * @param associateMemberAccountRequest
     *        A {@link Consumer} that will call methods on {@link AssociateMemberAccountRequest.Builder} to create a
     *        request.
     * @return Result of the AssociateMemberAccount operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.AssociateMemberAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateMemberAccount" target="_top">AWS
     *      API Documentation</a>
     */
    default AssociateMemberAccountResponse associateMemberAccount(
            Consumer<AssociateMemberAccountRequest.Builder> associateMemberAccountRequest) throws InvalidInputException,
            LimitExceededException, InternalException, AwsServiceException, SdkClientException, MacieException {
        return associateMemberAccount(AssociateMemberAccountRequest.builder().applyMutation(associateMemberAccountRequest)
                .build());
    }

    /**
     * <p>
     * Associates specified S3 resources with Amazon Macie Classic for monitoring and data classification. If
     * memberAccountId isn't specified, the action associates specified S3 resources with Macie Classic for the current
     * master account. If memberAccountId is specified, the action associates specified S3 resources with Macie Classic
     * for the specified member account.
     * </p>
     *
     * @param associateS3ResourcesRequest
     * @return Result of the AssociateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.AssociateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default AssociateS3ResourcesResponse associateS3Resources(AssociateS3ResourcesRequest associateS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, LimitExceededException, InternalException, AwsServiceException,
            SdkClientException, MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates specified S3 resources with Amazon Macie Classic for monitoring and data classification. If
     * memberAccountId isn't specified, the action associates specified S3 resources with Macie Classic for the current
     * master account. If memberAccountId is specified, the action associates specified S3 resources with Macie Classic
     * for the specified member account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AssociateS3ResourcesRequest.Builder} avoiding the
     * need to create one manually via {@link AssociateS3ResourcesRequest#builder()}
     * </p>
     *
     * @param associateS3ResourcesRequest
     *        A {@link Consumer} that will call methods on {@link AssociateS3ResourcesRequest.Builder} to create a
     *        request.
     * @return Result of the AssociateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.AssociateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default AssociateS3ResourcesResponse associateS3Resources(
            Consumer<AssociateS3ResourcesRequest.Builder> associateS3ResourcesRequest) throws InvalidInputException,
            AccessDeniedException, LimitExceededException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        return associateS3Resources(AssociateS3ResourcesRequest.builder().applyMutation(associateS3ResourcesRequest).build());
    }

    /**
     * <p>
     * Removes the specified member account from Amazon Macie Classic.
     * </p>
     *
     * @param disassociateMemberAccountRequest
     * @return Result of the DisassociateMemberAccount operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.DisassociateMemberAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateMemberAccount"
     *      target="_top">AWS API Documentation</a>
     */
    default DisassociateMemberAccountResponse disassociateMemberAccount(
            DisassociateMemberAccountRequest disassociateMemberAccountRequest) throws InvalidInputException, InternalException,
            AwsServiceException, SdkClientException, MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes the specified member account from Amazon Macie Classic.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisassociateMemberAccountRequest.Builder} avoiding
     * the need to create one manually via {@link DisassociateMemberAccountRequest#builder()}
     * </p>
     *
     * @param disassociateMemberAccountRequest
     *        A {@link Consumer} that will call methods on {@link DisassociateMemberAccountRequest.Builder} to create a
     *        request.
     * @return Result of the DisassociateMemberAccount operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.DisassociateMemberAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateMemberAccount"
     *      target="_top">AWS API Documentation</a>
     */
    default DisassociateMemberAccountResponse disassociateMemberAccount(
            Consumer<DisassociateMemberAccountRequest.Builder> disassociateMemberAccountRequest) throws InvalidInputException,
            InternalException, AwsServiceException, SdkClientException, MacieException {
        return disassociateMemberAccount(DisassociateMemberAccountRequest.builder()
                .applyMutation(disassociateMemberAccountRequest).build());
    }

    /**
     * <p>
     * Removes specified S3 resources from being monitored by Amazon Macie Classic. If memberAccountId isn't specified,
     * the action removes specified S3 resources from Macie Classic for the current master account. If memberAccountId
     * is specified, the action removes specified S3 resources from Macie Classic for the specified member account.
     * </p>
     *
     * @param disassociateS3ResourcesRequest
     * @return Result of the DisassociateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.DisassociateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateS3Resources" target="_top">AWS
     *      API Documentation</a>
     */
    default DisassociateS3ResourcesResponse disassociateS3Resources(DisassociateS3ResourcesRequest disassociateS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes specified S3 resources from being monitored by Amazon Macie Classic. If memberAccountId isn't specified,
     * the action removes specified S3 resources from Macie Classic for the current master account. If memberAccountId
     * is specified, the action removes specified S3 resources from Macie Classic for the specified member account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisassociateS3ResourcesRequest.Builder} avoiding
     * the need to create one manually via {@link DisassociateS3ResourcesRequest#builder()}
     * </p>
     *
     * @param disassociateS3ResourcesRequest
     *        A {@link Consumer} that will call methods on {@link DisassociateS3ResourcesRequest.Builder} to create a
     *        request.
     * @return Result of the DisassociateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.DisassociateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateS3Resources" target="_top">AWS
     *      API Documentation</a>
     */
    default DisassociateS3ResourcesResponse disassociateS3Resources(
            Consumer<DisassociateS3ResourcesRequest.Builder> disassociateS3ResourcesRequest) throws InvalidInputException,
            AccessDeniedException, InternalException, AwsServiceException, SdkClientException, MacieException {
        return disassociateS3Resources(DisassociateS3ResourcesRequest.builder().applyMutation(disassociateS3ResourcesRequest)
                .build());
    }

    /**
     * <p>
     * Lists all Amazon Macie Classic member accounts for the current Amazon Macie Classic master account.
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return Result of the ListMemberAccounts operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMemberAccountsResponse listMemberAccounts(ListMemberAccountsRequest listMemberAccountsRequest)
            throws InternalException, InvalidInputException, AwsServiceException, SdkClientException, MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all Amazon Macie Classic member accounts for the current Amazon Macie Classic master account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMemberAccountsRequest.Builder} avoiding the
     * need to create one manually via {@link ListMemberAccountsRequest#builder()}
     * </p>
     *
     * @param listMemberAccountsRequest
     *        A {@link Consumer} that will call methods on {@link ListMemberAccountsRequest.Builder} to create a
     *        request.
     * @return Result of the ListMemberAccounts operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMemberAccountsResponse listMemberAccounts(Consumer<ListMemberAccountsRequest.Builder> listMemberAccountsRequest)
            throws InternalException, InvalidInputException, AwsServiceException, SdkClientException, MacieException {
        return listMemberAccounts(ListMemberAccountsRequest.builder().applyMutation(listMemberAccountsRequest).build());
    }

    /**
     * <p>
     * Lists all Amazon Macie Classic member accounts for the current Amazon Macie Classic master account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client.listMemberAccountsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client
     *             .listMemberAccountsPaginator(request);
     *     for (software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client.listMemberAccountsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation.</b>
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMemberAccountsIterable listMemberAccountsPaginator(ListMemberAccountsRequest listMemberAccountsRequest)
            throws InternalException, InvalidInputException, AwsServiceException, SdkClientException, MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all Amazon Macie Classic member accounts for the current Amazon Macie Classic master account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client.listMemberAccountsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client
     *             .listMemberAccountsPaginator(request);
     *     for (software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client.listMemberAccountsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMemberAccountsRequest.Builder} avoiding the
     * need to create one manually via {@link ListMemberAccountsRequest#builder()}
     * </p>
     *
     * @param listMemberAccountsRequest
     *        A {@link Consumer} that will call methods on {@link ListMemberAccountsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMemberAccountsIterable listMemberAccountsPaginator(
            Consumer<ListMemberAccountsRequest.Builder> listMemberAccountsRequest) throws InternalException,
            InvalidInputException, AwsServiceException, SdkClientException, MacieException {
        return listMemberAccountsPaginator(ListMemberAccountsRequest.builder().applyMutation(listMemberAccountsRequest).build());
    }

    /**
     * <p>
     * Lists all the S3 resources associated with Amazon Macie Classic. If memberAccountId isn't specified, the action
     * lists the S3 resources associated with Amazon Macie Classic for the current master account. If memberAccountId is
     * specified, the action lists the S3 resources associated with Amazon Macie Classic for the specified member
     * account.
     * </p>
     *
     * @param listS3ResourcesRequest
     * @return Result of the ListS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default ListS3ResourcesResponse listS3Resources(ListS3ResourcesRequest listS3ResourcesRequest) throws InvalidInputException,
            AccessDeniedException, InternalException, AwsServiceException, SdkClientException, MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all the S3 resources associated with Amazon Macie Classic. If memberAccountId isn't specified, the action
     * lists the S3 resources associated with Amazon Macie Classic for the current master account. If memberAccountId is
     * specified, the action lists the S3 resources associated with Amazon Macie Classic for the specified member
     * account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListS3ResourcesRequest.Builder} avoiding the need
     * to create one manually via {@link ListS3ResourcesRequest#builder()}
     * </p>
     *
     * @param listS3ResourcesRequest
     *        A {@link Consumer} that will call methods on {@link ListS3ResourcesRequest.Builder} to create a request.
     * @return Result of the ListS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default ListS3ResourcesResponse listS3Resources(Consumer<ListS3ResourcesRequest.Builder> listS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        return listS3Resources(ListS3ResourcesRequest.builder().applyMutation(listS3ResourcesRequest).build());
    }

    /**
     * <p>
     * Lists all the S3 resources associated with Amazon Macie Classic. If memberAccountId isn't specified, the action
     * lists the S3 resources associated with Amazon Macie Classic for the current master account. If memberAccountId is
     * specified, the action lists the S3 resources associated with Amazon Macie Classic for the specified member
     * account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listS3ResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default ListS3ResourcesIterable listS3ResourcesPaginator(ListS3ResourcesRequest listS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all the S3 resources associated with Amazon Macie Classic. If memberAccountId isn't specified, the action
     * lists the S3 resources associated with Amazon Macie Classic for the current master account. If memberAccountId is
     * specified, the action lists the S3 resources associated with Amazon Macie Classic for the specified member
     * account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListS3ResourcesRequest.Builder} avoiding the need
     * to create one manually via {@link ListS3ResourcesRequest#builder()}
     * </p>
     *
     * @param listS3ResourcesRequest
     *        A {@link Consumer} that will call methods on {@link ListS3ResourcesRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default ListS3ResourcesIterable listS3ResourcesPaginator(Consumer<ListS3ResourcesRequest.Builder> listS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        return listS3ResourcesPaginator(ListS3ResourcesRequest.builder().applyMutation(listS3ResourcesRequest).build());
    }

    /**
     * <p>
     * Updates the classification types for the specified S3 resources. If memberAccountId isn't specified, the action
     * updates the classification types of the S3 resources associated with Amazon Macie Classic for the current master
     * account. If memberAccountId is specified, the action updates the classification types of the S3 resources
     * associated with Amazon Macie Classic for the specified member account.
     * </p>
     *
     * @param updateS3ResourcesRequest
     * @return Result of the UpdateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.UpdateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/UpdateS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateS3ResourcesResponse updateS3Resources(UpdateS3ResourcesRequest updateS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the classification types for the specified S3 resources. If memberAccountId isn't specified, the action
     * updates the classification types of the S3 resources associated with Amazon Macie Classic for the current master
     * account. If memberAccountId is specified, the action updates the classification types of the S3 resources
     * associated with Amazon Macie Classic for the specified member account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateS3ResourcesRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateS3ResourcesRequest#builder()}
     * </p>
     *
     * @param updateS3ResourcesRequest
     *        A {@link Consumer} that will call methods on {@link UpdateS3ResourcesRequest.Builder} to create a request.
     * @return Result of the UpdateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         The request was rejected because an invalid or out-of-range value was supplied for an input parameter.
     * @throws AccessDeniedException
     *         You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.UpdateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/UpdateS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateS3ResourcesResponse updateS3Resources(Consumer<UpdateS3ResourcesRequest.Builder> updateS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        return updateS3Resources(UpdateS3ResourcesRequest.builder().applyMutation(updateS3ResourcesRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("macie");
    }
}
