/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.machinelearning.MachineLearningAsyncClient;
import software.amazon.awssdk.services.machinelearning.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsResponse;
import software.amazon.awssdk.services.machinelearning.model.MachineLearningRequest;
import software.amazon.awssdk.services.machinelearning.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultMachineLearningAsyncWaiter implements MachineLearningAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final MachineLearningAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeDataSourcesResponse> dataSourceAvailableWaiter;

    private final AsyncWaiter<DescribeMlModelsResponse> mLModelAvailableWaiter;

    private final AsyncWaiter<DescribeEvaluationsResponse> evaluationAvailableWaiter;

    private final AsyncWaiter<DescribeBatchPredictionsResponse> batchPredictionAvailableWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultMachineLearningAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = MachineLearningAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.dataSourceAvailableWaiter = AsyncWaiter.builder(DescribeDataSourcesResponse.class)
                .acceptors(dataSourceAvailableWaiterAcceptors())
                .overrideConfiguration(dataSourceAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.mLModelAvailableWaiter = AsyncWaiter.builder(DescribeMlModelsResponse.class)
                .acceptors(mLModelAvailableWaiterAcceptors())
                .overrideConfiguration(mLModelAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.evaluationAvailableWaiter = AsyncWaiter.builder(DescribeEvaluationsResponse.class)
                .acceptors(evaluationAvailableWaiterAcceptors())
                .overrideConfiguration(evaluationAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.batchPredictionAvailableWaiter = AsyncWaiter.builder(DescribeBatchPredictionsResponse.class)
                .acceptors(batchPredictionAvailableWaiterAcceptors())
                .overrideConfiguration(batchPredictionAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBatchPredictionsResponse>> waitUntilBatchPredictionAvailable(
            DescribeBatchPredictionsRequest describeBatchPredictionsRequest) {
        return batchPredictionAvailableWaiter.runAsync(() -> client
                .describeBatchPredictions(applyWaitersUserAgent(describeBatchPredictionsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBatchPredictionsResponse>> waitUntilBatchPredictionAvailable(
            DescribeBatchPredictionsRequest describeBatchPredictionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return batchPredictionAvailableWaiter.runAsync(
                () -> client.describeBatchPredictions(applyWaitersUserAgent(describeBatchPredictionsRequest)),
                batchPredictionAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDataSourcesResponse>> waitUntilDataSourceAvailable(
            DescribeDataSourcesRequest describeDataSourcesRequest) {
        return dataSourceAvailableWaiter.runAsync(() -> client
                .describeDataSources(applyWaitersUserAgent(describeDataSourcesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDataSourcesResponse>> waitUntilDataSourceAvailable(
            DescribeDataSourcesRequest describeDataSourcesRequest, WaiterOverrideConfiguration overrideConfig) {
        return dataSourceAvailableWaiter.runAsync(
                () -> client.describeDataSources(applyWaitersUserAgent(describeDataSourcesRequest)),
                dataSourceAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEvaluationsResponse>> waitUntilEvaluationAvailable(
            DescribeEvaluationsRequest describeEvaluationsRequest) {
        return evaluationAvailableWaiter.runAsync(() -> client
                .describeEvaluations(applyWaitersUserAgent(describeEvaluationsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEvaluationsResponse>> waitUntilEvaluationAvailable(
            DescribeEvaluationsRequest describeEvaluationsRequest, WaiterOverrideConfiguration overrideConfig) {
        return evaluationAvailableWaiter.runAsync(
                () -> client.describeEvaluations(applyWaitersUserAgent(describeEvaluationsRequest)),
                evaluationAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMlModelsResponse>> waitUntilMLModelAvailable(
            DescribeMlModelsRequest describeMlModelsRequest) {
        return mLModelAvailableWaiter.runAsync(() -> client.describeMLModels(applyWaitersUserAgent(describeMlModelsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMlModelsResponse>> waitUntilMLModelAvailable(
            DescribeMlModelsRequest describeMlModelsRequest, WaiterOverrideConfiguration overrideConfig) {
        return mLModelAvailableWaiter.runAsync(() -> client.describeMLModels(applyWaitersUserAgent(describeMlModelsRequest)),
                mLModelAvailableWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeDataSourcesResponse>> dataSourceAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDataSourcesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Results").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "COMPLETED"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Results").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Results[].Status=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMlModelsResponse>> mLModelAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMlModelsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Results").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "COMPLETED"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Results").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Results[].Status=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEvaluationsResponse>> evaluationAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEvaluationsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Results").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "COMPLETED"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Results").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Results[].Status=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBatchPredictionsResponse>> batchPredictionAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBatchPredictionsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Results").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "COMPLETED"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Results").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Results[].Status=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration dataSourceAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration mLModelAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration evaluationAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration batchPredictionAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static MachineLearningAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends MachineLearningRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements MachineLearningAsyncWaiter.Builder {
        private MachineLearningAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public MachineLearningAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public MachineLearningAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public MachineLearningAsyncWaiter.Builder client(MachineLearningAsyncClient client) {
            this.client = client;
            return this;
        }

        public MachineLearningAsyncWaiter build() {
            return new DefaultMachineLearningAsyncWaiter(this);
        }
    }
}
