/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.machinelearning.model.AddTagsRequest;
import software.amazon.awssdk.services.machinelearning.model.AddTagsResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRdsRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRdsResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRedshiftRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRedshiftResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromS3Request;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromS3Response;
import software.amazon.awssdk.services.machinelearning.model.CreateEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateMlModelResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateRealtimeEndpointRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateRealtimeEndpointResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteDataSourceRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteDataSourceResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteMlModelResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteRealtimeEndpointRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteRealtimeEndpointResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteTagsRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteTagsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeTagsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeTagsResponse;
import software.amazon.awssdk.services.machinelearning.model.GetBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.GetBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.GetDataSourceRequest;
import software.amazon.awssdk.services.machinelearning.model.GetDataSourceResponse;
import software.amazon.awssdk.services.machinelearning.model.GetEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.GetEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.GetMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.GetMlModelResponse;
import software.amazon.awssdk.services.machinelearning.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.machinelearning.model.InternalServerException;
import software.amazon.awssdk.services.machinelearning.model.InvalidInputException;
import software.amazon.awssdk.services.machinelearning.model.InvalidTagException;
import software.amazon.awssdk.services.machinelearning.model.LimitExceededException;
import software.amazon.awssdk.services.machinelearning.model.MachineLearningException;
import software.amazon.awssdk.services.machinelearning.model.MachineLearningRequest;
import software.amazon.awssdk.services.machinelearning.model.PredictRequest;
import software.amazon.awssdk.services.machinelearning.model.PredictResponse;
import software.amazon.awssdk.services.machinelearning.model.PredictorNotMountedException;
import software.amazon.awssdk.services.machinelearning.model.ResourceNotFoundException;
import software.amazon.awssdk.services.machinelearning.model.TagLimitExceededException;
import software.amazon.awssdk.services.machinelearning.model.UpdateBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.UpdateDataSourceRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateDataSourceResponse;
import software.amazon.awssdk.services.machinelearning.model.UpdateEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.UpdateMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateMlModelResponse;
import software.amazon.awssdk.services.machinelearning.paginators.DescribeBatchPredictionsIterable;
import software.amazon.awssdk.services.machinelearning.paginators.DescribeDataSourcesIterable;
import software.amazon.awssdk.services.machinelearning.paginators.DescribeEvaluationsIterable;
import software.amazon.awssdk.services.machinelearning.paginators.DescribeMLModelsIterable;
import software.amazon.awssdk.services.machinelearning.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateDataSourceFromRdsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateDataSourceFromRedshiftRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateDataSourceFromS3RequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateRealtimeEndpointRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteDataSourceRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteRealtimeEndpointRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeBatchPredictionsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeDataSourcesRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeEvaluationsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeMlModelsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetDataSourceRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.PredictRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateDataSourceRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.waiters.MachineLearningWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MachineLearningClient}.
 *
 * @see MachineLearningClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMachineLearningClient implements MachineLearningClient {
    private static final Logger log = Logger.loggerFor(DefaultMachineLearningClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMachineLearningClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds one or more tags to an object, up to a limit of 10. Each tag consists of a key and an optional value. If you
     * add a tag using a key that is already associated with the ML object, <code>AddTags</code> updates the tag's
     * value.
     * </p>
     *
     * @param addTagsRequest
     * @return Result of the AddTags operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidTagException
     * @throws TagLimitExceededException
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.AddTags
     */
    @Override
    public AddTagsResponse addTags(AddTagsRequest addTagsRequest) throws InvalidInputException, InvalidTagException,
            TagLimitExceededException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTags");

            return clientHandler.execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>()
                    .withOperationName("AddTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new AddTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generates predictions for a group of observations. The observations to process exist in one or more data files
     * referenced by a <code>DataSource</code>. This operation creates a new <code>BatchPrediction</code>, and uses an
     * <code>MLModel</code> and the data files referenced by the <code>DataSource</code> as information sources.
     * </p>
     * <p>
     * <code>CreateBatchPrediction</code> is an asynchronous operation. In response to
     * <code>CreateBatchPrediction</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>BatchPrediction</code> status to <code>PENDING</code>. After the <code>BatchPrediction</code> completes,
     * Amazon ML sets the status to <code>COMPLETED</code>.
     * </p>
     * <p>
     * You can poll for status updates by using the <a>GetBatchPrediction</a> operation and checking the
     * <code>Status</code> parameter of the result. After the <code>COMPLETED</code> status appears, the results are
     * available in the location specified by the <code>OutputUri</code> parameter.
     * </p>
     *
     * @param createBatchPredictionRequest
     * @return Result of the CreateBatchPrediction operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws IdempotentParameterMismatchException
     *         A second request to use or change an object was not allowed. This can result from retrying a request
     *         using a parameter that was not present in the original request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.CreateBatchPrediction
     */
    @Override
    public CreateBatchPredictionResponse createBatchPrediction(CreateBatchPredictionRequest createBatchPredictionRequest)
            throws InvalidInputException, InternalServerException, IdempotentParameterMismatchException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateBatchPredictionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBatchPrediction");

            return clientHandler.execute(new ClientExecutionParams<CreateBatchPredictionRequest, CreateBatchPredictionResponse>()
                    .withOperationName("CreateBatchPrediction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBatchPredictionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBatchPredictionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>DataSource</code> object from an <a href="http://aws.amazon.com/rds/"> Amazon Relational Database
     * Service</a> (Amazon RDS). A <code>DataSource</code> references data that can be used to perform
     * <code>CreateMLModel</code>, <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * <code>CreateDataSourceFromRDS</code> is an asynchronous operation. In response to
     * <code>CreateDataSourceFromRDS</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>DataSource</code> status to <code>PENDING</code>. After the <code>DataSource</code> is created and ready
     * for use, Amazon ML sets the <code>Status</code> parameter to <code>COMPLETED</code>. <code>DataSource</code> in
     * the <code>COMPLETED</code> or <code>PENDING</code> state can be used only to perform
     * <code>&gt;CreateMLModel</code>&gt;, <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code>
     * operations.
     * </p>
     * <p>
     * If Amazon ML cannot accept the input source, it sets the <code>Status</code> parameter to <code>FAILED</code> and
     * includes an error message in the <code>Message</code> attribute of the <code>GetDataSource</code> operation
     * response.
     * </p>
     *
     * @param createDataSourceFromRdsRequest
     * @return Result of the CreateDataSourceFromRDS operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws IdempotentParameterMismatchException
     *         A second request to use or change an object was not allowed. This can result from retrying a request
     *         using a parameter that was not present in the original request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.CreateDataSourceFromRDS
     */
    @Override
    public CreateDataSourceFromRdsResponse createDataSourceFromRDS(CreateDataSourceFromRdsRequest createDataSourceFromRdsRequest)
            throws InvalidInputException, InternalServerException, IdempotentParameterMismatchException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataSourceFromRdsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataSourceFromRdsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceFromRdsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSourceFromRDS");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataSourceFromRdsRequest, CreateDataSourceFromRdsResponse>()
                            .withOperationName("CreateDataSourceFromRDS").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDataSourceFromRdsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataSourceFromRdsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>DataSource</code> from a database hosted on an Amazon Redshift cluster. A <code>DataSource</code>
     * references data that can be used to perform either <code>CreateMLModel</code>, <code>CreateEvaluation</code>, or
     * <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * <code>CreateDataSourceFromRedshift</code> is an asynchronous operation. In response to
     * <code>CreateDataSourceFromRedshift</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>DataSource</code> status to <code>PENDING</code>. After the <code>DataSource</code> is created and ready
     * for use, Amazon ML sets the <code>Status</code> parameter to <code>COMPLETED</code>. <code>DataSource</code> in
     * <code>COMPLETED</code> or <code>PENDING</code> states can be used to perform only <code>CreateMLModel</code>,
     * <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * If Amazon ML can't accept the input source, it sets the <code>Status</code> parameter to <code>FAILED</code> and
     * includes an error message in the <code>Message</code> attribute of the <code>GetDataSource</code> operation
     * response.
     * </p>
     * <p>
     * The observations should be contained in the database hosted on an Amazon Redshift cluster and should be specified
     * by a <code>SelectSqlQuery</code> query. Amazon ML executes an <code>Unload</code> command in Amazon Redshift to
     * transfer the result set of the <code>SelectSqlQuery</code> query to <code>S3StagingLocation</code>.
     * </p>
     * <p>
     * After the <code>DataSource</code> has been created, it's ready for use in evaluations and batch predictions. If
     * you plan to use the <code>DataSource</code> to train an <code>MLModel</code>, the <code>DataSource</code> also
     * requires a recipe. A recipe describes how each input variable will be used in training an <code>MLModel</code>.
     * Will the variable be included or excluded from training? Will the variable be manipulated; for example, will it
     * be combined with another variable or will it be split apart into word combinations? The recipe provides answers
     * to these questions.
     * </p>
     * <p>
     * You can't change an existing datasource, but you can copy and modify the settings from an existing Amazon
     * Redshift datasource to create a new datasource. To do so, call <code>GetDataSource</code> for an existing
     * datasource and copy the values to a <code>CreateDataSource</code> call. Change the settings that you want to
     * change and make sure that all required fields have the appropriate values.
     * </p>
     *
     * @param createDataSourceFromRedshiftRequest
     * @return Result of the CreateDataSourceFromRedshift operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws IdempotentParameterMismatchException
     *         A second request to use or change an object was not allowed. This can result from retrying a request
     *         using a parameter that was not present in the original request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.CreateDataSourceFromRedshift
     */
    @Override
    public CreateDataSourceFromRedshiftResponse createDataSourceFromRedshift(
            CreateDataSourceFromRedshiftRequest createDataSourceFromRedshiftRequest) throws InvalidInputException,
            InternalServerException, IdempotentParameterMismatchException, AwsServiceException, SdkClientException,
            MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataSourceFromRedshiftResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataSourceFromRedshiftResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceFromRedshiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSourceFromRedshift");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataSourceFromRedshiftRequest, CreateDataSourceFromRedshiftResponse>()
                            .withOperationName("CreateDataSourceFromRedshift").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDataSourceFromRedshiftRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataSourceFromRedshiftRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>DataSource</code> object. A <code>DataSource</code> references data that can be used to perform
     * <code>CreateMLModel</code>, <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * <code>CreateDataSourceFromS3</code> is an asynchronous operation. In response to
     * <code>CreateDataSourceFromS3</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>DataSource</code> status to <code>PENDING</code>. After the <code>DataSource</code> has been created and is
     * ready for use, Amazon ML sets the <code>Status</code> parameter to <code>COMPLETED</code>.
     * <code>DataSource</code> in the <code>COMPLETED</code> or <code>PENDING</code> state can be used to perform only
     * <code>CreateMLModel</code>, <code>CreateEvaluation</code> or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * If Amazon ML can't accept the input source, it sets the <code>Status</code> parameter to <code>FAILED</code> and
     * includes an error message in the <code>Message</code> attribute of the <code>GetDataSource</code> operation
     * response.
     * </p>
     * <p>
     * The observation data used in a <code>DataSource</code> should be ready to use; that is, it should have a
     * consistent structure, and missing data values should be kept to a minimum. The observation data must reside in
     * one or more .csv files in an Amazon Simple Storage Service (Amazon S3) location, along with a schema that
     * describes the data items by name and type. The same schema must be used for all of the data files referenced by
     * the <code>DataSource</code>.
     * </p>
     * <p>
     * After the <code>DataSource</code> has been created, it's ready to use in evaluations and batch predictions. If
     * you plan to use the <code>DataSource</code> to train an <code>MLModel</code>, the <code>DataSource</code> also
     * needs a recipe. A recipe describes how each input variable will be used in training an <code>MLModel</code>. Will
     * the variable be included or excluded from training? Will the variable be manipulated; for example, will it be
     * combined with another variable or will it be split apart into word combinations? The recipe provides answers to
     * these questions.
     * </p>
     *
     * @param createDataSourceFromS3Request
     * @return Result of the CreateDataSourceFromS3 operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws IdempotentParameterMismatchException
     *         A second request to use or change an object was not allowed. This can result from retrying a request
     *         using a parameter that was not present in the original request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.CreateDataSourceFromS3
     */
    @Override
    public CreateDataSourceFromS3Response createDataSourceFromS3(CreateDataSourceFromS3Request createDataSourceFromS3Request)
            throws InvalidInputException, InternalServerException, IdempotentParameterMismatchException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataSourceFromS3Response> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataSourceFromS3Response::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceFromS3Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSourceFromS3");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataSourceFromS3Request, CreateDataSourceFromS3Response>()
                            .withOperationName("CreateDataSourceFromS3").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDataSourceFromS3Request)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataSourceFromS3RequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new <code>Evaluation</code> of an <code>MLModel</code>. An <code>MLModel</code> is evaluated on a set
     * of observations associated to a <code>DataSource</code>. Like a <code>DataSource</code> for an
     * <code>MLModel</code>, the <code>DataSource</code> for an <code>Evaluation</code> contains values for the
     * <code>Target Variable</code>. The <code>Evaluation</code> compares the predicted result for each observation to
     * the actual outcome and provides a summary so that you know how effective the <code>MLModel</code> functions on
     * the test data. Evaluation generates a relevant performance metric, such as BinaryAUC, RegressionRMSE or
     * MulticlassAvgFScore based on the corresponding <code>MLModelType</code>: <code>BINARY</code>,
     * <code>REGRESSION</code> or <code>MULTICLASS</code>.
     * </p>
     * <p>
     * <code>CreateEvaluation</code> is an asynchronous operation. In response to <code>CreateEvaluation</code>, Amazon
     * Machine Learning (Amazon ML) immediately returns and sets the evaluation status to <code>PENDING</code>. After
     * the <code>Evaluation</code> is created and ready for use, Amazon ML sets the status to <code>COMPLETED</code>.
     * </p>
     * <p>
     * You can use the <code>GetEvaluation</code> operation to check progress of the evaluation during the creation
     * operation.
     * </p>
     *
     * @param createEvaluationRequest
     * @return Result of the CreateEvaluation operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws IdempotentParameterMismatchException
     *         A second request to use or change an object was not allowed. This can result from retrying a request
     *         using a parameter that was not present in the original request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.CreateEvaluation
     */
    @Override
    public CreateEvaluationResponse createEvaluation(CreateEvaluationRequest createEvaluationRequest)
            throws InvalidInputException, InternalServerException, IdempotentParameterMismatchException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEvaluationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEvaluation");

            return clientHandler.execute(new ClientExecutionParams<CreateEvaluationRequest, CreateEvaluationResponse>()
                    .withOperationName("CreateEvaluation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createEvaluationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEvaluationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new <code>MLModel</code> using the <code>DataSource</code> and the recipe as information sources.
     * </p>
     * <p>
     * An <code>MLModel</code> is nearly immutable. Users can update only the <code>MLModelName</code> and the
     * <code>ScoreThreshold</code> in an <code>MLModel</code> without creating a new <code>MLModel</code>.
     * </p>
     * <p>
     * <code>CreateMLModel</code> is an asynchronous operation. In response to <code>CreateMLModel</code>, Amazon
     * Machine Learning (Amazon ML) immediately returns and sets the <code>MLModel</code> status to <code>PENDING</code>
     * . After the <code>MLModel</code> has been created and ready is for use, Amazon ML sets the status to
     * <code>COMPLETED</code>.
     * </p>
     * <p>
     * You can use the <code>GetMLModel</code> operation to check the progress of the <code>MLModel</code> during the
     * creation operation.
     * </p>
     * <p>
     * <code>CreateMLModel</code> requires a <code>DataSource</code> with computed statistics, which can be created by
     * setting <code>ComputeStatistics</code> to <code>true</code> in <code>CreateDataSourceFromRDS</code>,
     * <code>CreateDataSourceFromS3</code>, or <code>CreateDataSourceFromRedshift</code> operations.
     * </p>
     *
     * @param createMlModelRequest
     * @return Result of the CreateMLModel operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws IdempotentParameterMismatchException
     *         A second request to use or change an object was not allowed. This can result from retrying a request
     *         using a parameter that was not present in the original request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.CreateMLModel
     */
    @Override
    public CreateMlModelResponse createMLModel(CreateMlModelRequest createMlModelRequest) throws InvalidInputException,
            InternalServerException, IdempotentParameterMismatchException, AwsServiceException, SdkClientException,
            MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMlModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMLModel");

            return clientHandler.execute(new ClientExecutionParams<CreateMlModelRequest, CreateMlModelResponse>()
                    .withOperationName("CreateMLModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createMlModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMlModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a real-time endpoint for the <code>MLModel</code>. The endpoint contains the URI of the
     * <code>MLModel</code>; that is, the location to send real-time prediction requests for the specified
     * <code>MLModel</code>.
     * </p>
     *
     * @param createRealtimeEndpointRequest
     * @return Result of the CreateRealtimeEndpoint operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.CreateRealtimeEndpoint
     */
    @Override
    public CreateRealtimeEndpointResponse createRealtimeEndpoint(CreateRealtimeEndpointRequest createRealtimeEndpointRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRealtimeEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRealtimeEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRealtimeEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRealtimeEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateRealtimeEndpointRequest, CreateRealtimeEndpointResponse>()
                            .withOperationName("CreateRealtimeEndpoint").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createRealtimeEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateRealtimeEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns the DELETED status to a <code>BatchPrediction</code>, rendering it unusable.
     * </p>
     * <p>
     * After using the <code>DeleteBatchPrediction</code> operation, you can use the <a>GetBatchPrediction</a> operation
     * to verify that the status of the <code>BatchPrediction</code> changed to DELETED.
     * </p>
     * <p>
     * <b>Caution:</b> The result of the <code>DeleteBatchPrediction</code> operation is irreversible.
     * </p>
     *
     * @param deleteBatchPredictionRequest
     * @return Result of the DeleteBatchPrediction operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DeleteBatchPrediction
     */
    @Override
    public DeleteBatchPredictionResponse deleteBatchPrediction(DeleteBatchPredictionRequest deleteBatchPredictionRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteBatchPredictionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBatchPrediction");

            return clientHandler.execute(new ClientExecutionParams<DeleteBatchPredictionRequest, DeleteBatchPredictionResponse>()
                    .withOperationName("DeleteBatchPrediction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBatchPredictionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBatchPredictionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns the DELETED status to a <code>DataSource</code>, rendering it unusable.
     * </p>
     * <p>
     * After using the <code>DeleteDataSource</code> operation, you can use the <a>GetDataSource</a> operation to verify
     * that the status of the <code>DataSource</code> changed to DELETED.
     * </p>
     * <p>
     * <b>Caution:</b> The results of the <code>DeleteDataSource</code> operation are irreversible.
     * </p>
     *
     * @param deleteDataSourceRequest
     * @return Result of the DeleteDataSource operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DeleteDataSource
     */
    @Override
    public DeleteDataSourceResponse deleteDataSource(DeleteDataSourceRequest deleteDataSourceRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDataSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataSource");

            return clientHandler.execute(new ClientExecutionParams<DeleteDataSourceRequest, DeleteDataSourceResponse>()
                    .withOperationName("DeleteDataSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDataSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns the <code>DELETED</code> status to an <code>Evaluation</code>, rendering it unusable.
     * </p>
     * <p>
     * After invoking the <code>DeleteEvaluation</code> operation, you can use the <code>GetEvaluation</code> operation
     * to verify that the status of the <code>Evaluation</code> changed to <code>DELETED</code>.
     * </p>
     * <p>
     * <b>Caution:</b> The results of the <code>DeleteEvaluation</code> operation are irreversible.
     * </p>
     *
     * @param deleteEvaluationRequest
     * @return Result of the DeleteEvaluation operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DeleteEvaluation
     */
    @Override
    public DeleteEvaluationResponse deleteEvaluation(DeleteEvaluationRequest deleteEvaluationRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEvaluationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEvaluation");

            return clientHandler.execute(new ClientExecutionParams<DeleteEvaluationRequest, DeleteEvaluationResponse>()
                    .withOperationName("DeleteEvaluation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteEvaluationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEvaluationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns the <code>DELETED</code> status to an <code>MLModel</code>, rendering it unusable.
     * </p>
     * <p>
     * After using the <code>DeleteMLModel</code> operation, you can use the <code>GetMLModel</code> operation to verify
     * that the status of the <code>MLModel</code> changed to DELETED.
     * </p>
     * <p>
     * <b>Caution:</b> The result of the <code>DeleteMLModel</code> operation is irreversible.
     * </p>
     *
     * @param deleteMlModelRequest
     * @return Result of the DeleteMLModel operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DeleteMLModel
     */
    @Override
    public DeleteMlModelResponse deleteMLModel(DeleteMlModelRequest deleteMlModelRequest) throws InvalidInputException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMlModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMLModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteMlModelRequest, DeleteMlModelResponse>()
                    .withOperationName("DeleteMLModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteMlModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMlModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a real time endpoint of an <code>MLModel</code>.
     * </p>
     *
     * @param deleteRealtimeEndpointRequest
     * @return Result of the DeleteRealtimeEndpoint operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DeleteRealtimeEndpoint
     */
    @Override
    public DeleteRealtimeEndpointResponse deleteRealtimeEndpoint(DeleteRealtimeEndpointRequest deleteRealtimeEndpointRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRealtimeEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRealtimeEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRealtimeEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRealtimeEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRealtimeEndpointRequest, DeleteRealtimeEndpointResponse>()
                            .withOperationName("DeleteRealtimeEndpoint").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteRealtimeEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRealtimeEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified tags associated with an ML object. After this operation is complete, you can't recover
     * deleted tags.
     * </p>
     * <p>
     * If you specify a tag that doesn't exist, Amazon ML ignores it.
     * </p>
     *
     * @param deleteTagsRequest
     * @return Result of the DeleteTags operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidTagException
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DeleteTags
     */
    @Override
    public DeleteTagsResponse deleteTags(DeleteTagsRequest deleteTagsRequest) throws InvalidInputException, InvalidTagException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>BatchPrediction</code> operations that match the search criteria in the request.
     * </p>
     *
     * @param describeBatchPredictionsRequest
     * @return Result of the DescribeBatchPredictions operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeBatchPredictions
     */
    @Override
    public DescribeBatchPredictionsResponse describeBatchPredictions(
            DescribeBatchPredictionsRequest describeBatchPredictionsRequest) throws InvalidInputException,
            InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBatchPredictionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBatchPredictionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBatchPredictionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBatchPredictions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBatchPredictionsRequest, DescribeBatchPredictionsResponse>()
                            .withOperationName("DescribeBatchPredictions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeBatchPredictionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBatchPredictionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>BatchPrediction</code> operations that match the search criteria in the request.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBatchPredictions(software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeBatchPredictionsIterable responses = client.describeBatchPredictionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.machinelearning.paginators.DescribeBatchPredictionsIterable responses = client
     *             .describeBatchPredictionsPaginator(request);
     *     for (software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeBatchPredictionsIterable responses = client.describeBatchPredictionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBatchPredictions(software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeBatchPredictionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeBatchPredictions
     */
    @Override
    public DescribeBatchPredictionsIterable describeBatchPredictionsPaginator(
            DescribeBatchPredictionsRequest describeBatchPredictionsRequest) throws InvalidInputException,
            InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        return new DescribeBatchPredictionsIterable(this, applyPaginatorUserAgent(describeBatchPredictionsRequest));
    }

    /**
     * <p>
     * Returns a list of <code>DataSource</code> that match the search criteria in the request.
     * </p>
     *
     * @param describeDataSourcesRequest
     * @return Result of the DescribeDataSources operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeDataSources
     */
    @Override
    public DescribeDataSourcesResponse describeDataSources(DescribeDataSourcesRequest describeDataSourcesRequest)
            throws InvalidInputException, InternalServerException, AwsServiceException, SdkClientException,
            MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDataSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDataSourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataSources");

            return clientHandler.execute(new ClientExecutionParams<DescribeDataSourcesRequest, DescribeDataSourcesResponse>()
                    .withOperationName("DescribeDataSources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDataSourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDataSourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>DataSource</code> that match the search criteria in the request.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDataSources(software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeDataSourcesIterable responses = client.describeDataSourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.machinelearning.paginators.DescribeDataSourcesIterable responses = client
     *             .describeDataSourcesPaginator(request);
     *     for (software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeDataSourcesIterable responses = client.describeDataSourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDataSources(software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDataSourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeDataSources
     */
    @Override
    public DescribeDataSourcesIterable describeDataSourcesPaginator(DescribeDataSourcesRequest describeDataSourcesRequest)
            throws InvalidInputException, InternalServerException, AwsServiceException, SdkClientException,
            MachineLearningException {
        return new DescribeDataSourcesIterable(this, applyPaginatorUserAgent(describeDataSourcesRequest));
    }

    /**
     * <p>
     * Returns a list of <code>DescribeEvaluations</code> that match the search criteria in the request.
     * </p>
     *
     * @param describeEvaluationsRequest
     * @return Result of the DescribeEvaluations operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeEvaluations
     */
    @Override
    public DescribeEvaluationsResponse describeEvaluations(DescribeEvaluationsRequest describeEvaluationsRequest)
            throws InvalidInputException, InternalServerException, AwsServiceException, SdkClientException,
            MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEvaluationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEvaluationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEvaluationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEvaluations");

            return clientHandler.execute(new ClientExecutionParams<DescribeEvaluationsRequest, DescribeEvaluationsResponse>()
                    .withOperationName("DescribeEvaluations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEvaluationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEvaluationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>DescribeEvaluations</code> that match the search criteria in the request.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEvaluations(software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeEvaluationsIterable responses = client.describeEvaluationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.machinelearning.paginators.DescribeEvaluationsIterable responses = client
     *             .describeEvaluationsPaginator(request);
     *     for (software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeEvaluationsIterable responses = client.describeEvaluationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEvaluations(software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEvaluationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeEvaluations
     */
    @Override
    public DescribeEvaluationsIterable describeEvaluationsPaginator(DescribeEvaluationsRequest describeEvaluationsRequest)
            throws InvalidInputException, InternalServerException, AwsServiceException, SdkClientException,
            MachineLearningException {
        return new DescribeEvaluationsIterable(this, applyPaginatorUserAgent(describeEvaluationsRequest));
    }

    /**
     * <p>
     * Returns a list of <code>MLModel</code> that match the search criteria in the request.
     * </p>
     *
     * @param describeMlModelsRequest
     * @return Result of the DescribeMLModels operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeMLModels
     */
    @Override
    public DescribeMlModelsResponse describeMLModels(DescribeMlModelsRequest describeMlModelsRequest)
            throws InvalidInputException, InternalServerException, AwsServiceException, SdkClientException,
            MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMlModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeMlModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMlModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMLModels");

            return clientHandler.execute(new ClientExecutionParams<DescribeMlModelsRequest, DescribeMlModelsResponse>()
                    .withOperationName("DescribeMLModels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeMlModelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeMlModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>MLModel</code> that match the search criteria in the request.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeMLModels(software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeMLModelsIterable responses = client.describeMLModelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.machinelearning.paginators.DescribeMLModelsIterable responses = client
     *             .describeMLModelsPaginator(request);
     *     for (software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.machinelearning.paginators.DescribeMLModelsIterable responses = client.describeMLModelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMLModels(software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeMlModelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeMLModels
     */
    @Override
    public DescribeMLModelsIterable describeMLModelsPaginator(DescribeMlModelsRequest describeMlModelsRequest)
            throws InvalidInputException, InternalServerException, AwsServiceException, SdkClientException,
            MachineLearningException {
        return new DescribeMLModelsIterable(this, applyPaginatorUserAgent(describeMlModelsRequest));
    }

    /**
     * <p>
     * Describes one or more of the tags for your Amazon ML object.
     * </p>
     *
     * @param describeTagsRequest
     * @return Result of the DescribeTags operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.DescribeTags
     */
    @Override
    public DescribeTagsResponse describeTags(DescribeTagsRequest describeTagsRequest) throws InvalidInputException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTags");

            return clientHandler.execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                    .withOperationName("DescribeTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeTagsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a <code>BatchPrediction</code> that includes detailed metadata, status, and data file information for a
     * <code>Batch Prediction</code> request.
     * </p>
     *
     * @param getBatchPredictionRequest
     * @return Result of the GetBatchPrediction operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.GetBatchPrediction
     */
    @Override
    public GetBatchPredictionResponse getBatchPrediction(GetBatchPredictionRequest getBatchPredictionRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBatchPredictionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBatchPrediction");

            return clientHandler.execute(new ClientExecutionParams<GetBatchPredictionRequest, GetBatchPredictionResponse>()
                    .withOperationName("GetBatchPrediction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getBatchPredictionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBatchPredictionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a <code>DataSource</code> that includes metadata and data file information, as well as the current status
     * of the <code>DataSource</code>.
     * </p>
     * <p>
     * <code>GetDataSource</code> provides results in normal or verbose format. The verbose format adds the schema
     * description and the list of files pointed to by the DataSource to the normal format.
     * </p>
     *
     * @param getDataSourceRequest
     * @return Result of the GetDataSource operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.GetDataSource
     */
    @Override
    public GetDataSourceResponse getDataSource(GetDataSourceRequest getDataSourceRequest) throws InvalidInputException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDataSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataSource");

            return clientHandler.execute(new ClientExecutionParams<GetDataSourceRequest, GetDataSourceResponse>()
                    .withOperationName("GetDataSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDataSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDataSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an <code>Evaluation</code> that includes metadata as well as the current status of the
     * <code>Evaluation</code>.
     * </p>
     *
     * @param getEvaluationRequest
     * @return Result of the GetEvaluation operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.GetEvaluation
     */
    @Override
    public GetEvaluationResponse getEvaluation(GetEvaluationRequest getEvaluationRequest) throws InvalidInputException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEvaluationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEvaluation");

            return clientHandler.execute(new ClientExecutionParams<GetEvaluationRequest, GetEvaluationResponse>()
                    .withOperationName("GetEvaluation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEvaluationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEvaluationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an <code>MLModel</code> that includes detailed metadata, data source information, and the current status
     * of the <code>MLModel</code>.
     * </p>
     * <p>
     * <code>GetMLModel</code> provides results in normal or verbose format.
     * </p>
     *
     * @param getMlModelRequest
     * @return Result of the GetMLModel operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.GetMLModel
     */
    @Override
    public GetMlModelResponse getMLModel(GetMlModelRequest getMlModelRequest) throws InvalidInputException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMlModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMLModel");

            return clientHandler
                    .execute(new ClientExecutionParams<GetMlModelRequest, GetMlModelResponse>().withOperationName("GetMLModel")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getMlModelRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetMlModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generates a prediction for the observation using the specified <code>ML Model</code>.
     * </p>
     * <p>
     * <b>Note:</b> Not all response parameters will be populated. Whether a response parameter is populated depends on
     * the type of model requested.
     * </p>
     *
     * @param predictRequest
     * @return Result of the Predict operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws LimitExceededException
     *         The subscriber exceeded the maximum number of operations. This exception can occur when listing objects
     *         such as <code>DataSource</code>.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws PredictorNotMountedException
     *         The exception is thrown when a predict request is made to an unmounted <code>MLModel</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.Predict
     */
    @Override
    public PredictResponse predict(PredictRequest predictRequest) throws InvalidInputException, ResourceNotFoundException,
            LimitExceededException, InternalServerException, PredictorNotMountedException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PredictResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PredictResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, predictRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Predict");

            return clientHandler.execute(new ClientExecutionParams<PredictRequest, PredictResponse>()
                    .withOperationName("Predict").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(predictRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new PredictRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the <code>BatchPredictionName</code> of a <code>BatchPrediction</code>.
     * </p>
     * <p>
     * You can use the <code>GetBatchPrediction</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateBatchPredictionRequest
     * @return Result of the UpdateBatchPrediction operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.UpdateBatchPrediction
     */
    @Override
    public UpdateBatchPredictionResponse updateBatchPrediction(UpdateBatchPredictionRequest updateBatchPredictionRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateBatchPredictionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBatchPrediction");

            return clientHandler.execute(new ClientExecutionParams<UpdateBatchPredictionRequest, UpdateBatchPredictionResponse>()
                    .withOperationName("UpdateBatchPrediction").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateBatchPredictionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBatchPredictionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the <code>DataSourceName</code> of a <code>DataSource</code>.
     * </p>
     * <p>
     * You can use the <code>GetDataSource</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateDataSourceRequest
     * @return Result of the UpdateDataSource operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.UpdateDataSource
     */
    @Override
    public UpdateDataSourceResponse updateDataSource(UpdateDataSourceRequest updateDataSourceRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDataSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataSource");

            return clientHandler.execute(new ClientExecutionParams<UpdateDataSourceRequest, UpdateDataSourceResponse>()
                    .withOperationName("UpdateDataSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDataSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDataSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the <code>EvaluationName</code> of an <code>Evaluation</code>.
     * </p>
     * <p>
     * You can use the <code>GetEvaluation</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateEvaluationRequest
     * @return Result of the UpdateEvaluation operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.UpdateEvaluation
     */
    @Override
    public UpdateEvaluationResponse updateEvaluation(UpdateEvaluationRequest updateEvaluationRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEvaluationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEvaluation");

            return clientHandler.execute(new ClientExecutionParams<UpdateEvaluationRequest, UpdateEvaluationResponse>()
                    .withOperationName("UpdateEvaluation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateEvaluationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEvaluationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the <code>MLModelName</code> and the <code>ScoreThreshold</code> of an <code>MLModel</code>.
     * </p>
     * <p>
     * You can use the <code>GetMLModel</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateMlModelRequest
     * @return Result of the UpdateMLModel operation returned by the service.
     * @throws InvalidInputException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws ResourceNotFoundException
     *         A specified resource cannot be located.
     * @throws InternalServerException
     *         An error on the server occurred when trying to process a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MachineLearningException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MachineLearningClient.UpdateMLModel
     */
    @Override
    public UpdateMlModelResponse updateMLModel(UpdateMlModelRequest updateMlModelRequest) throws InvalidInputException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, MachineLearningException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateMlModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMLModel");

            return clientHandler.execute(new ClientExecutionParams<UpdateMlModelRequest, UpdateMlModelResponse>()
                    .withOperationName("UpdateMLModel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateMlModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateMlModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link MachineLearningWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link MachineLearningWaiter}
     */
    @Override
    public MachineLearningWaiter waiter() {
        return MachineLearningWaiter.builder().client(this).build();
    }

    private <T extends MachineLearningRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MachineLearningException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PredictorNotMountedException")
                                .exceptionBuilderSupplier(PredictorNotMountedException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
