/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The data specification of an Amazon Relational Database Service (Amazon RDS) <code>DataSource</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RDSDataSpec implements SdkPojo, Serializable, ToCopyableBuilder<RDSDataSpec.Builder, RDSDataSpec> {
    private static final SdkField<RDSDatabase> DATABASE_INFORMATION_FIELD = SdkField
            .<RDSDatabase> builder(MarshallingType.SDK_POJO).memberName("DatabaseInformation")
            .getter(getter(RDSDataSpec::databaseInformation)).setter(setter(Builder::databaseInformation))
            .constructor(RDSDatabase::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseInformation").build())
            .build();

    private static final SdkField<String> SELECT_SQL_QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SelectSqlQuery").getter(getter(RDSDataSpec::selectSqlQuery)).setter(setter(Builder::selectSqlQuery))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectSqlQuery").build()).build();

    private static final SdkField<RDSDatabaseCredentials> DATABASE_CREDENTIALS_FIELD = SdkField
            .<RDSDatabaseCredentials> builder(MarshallingType.SDK_POJO).memberName("DatabaseCredentials")
            .getter(getter(RDSDataSpec::databaseCredentials)).setter(setter(Builder::databaseCredentials))
            .constructor(RDSDatabaseCredentials::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseCredentials").build())
            .build();

    private static final SdkField<String> S3_STAGING_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3StagingLocation").getter(getter(RDSDataSpec::s3StagingLocation))
            .setter(setter(Builder::s3StagingLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3StagingLocation").build()).build();

    private static final SdkField<String> DATA_REARRANGEMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataRearrangement").getter(getter(RDSDataSpec::dataRearrangement))
            .setter(setter(Builder::dataRearrangement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRearrangement").build()).build();

    private static final SdkField<String> DATA_SCHEMA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataSchema").getter(getter(RDSDataSpec::dataSchema)).setter(setter(Builder::dataSchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSchema").build()).build();

    private static final SdkField<String> DATA_SCHEMA_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataSchemaUri").getter(getter(RDSDataSpec::dataSchemaUri)).setter(setter(Builder::dataSchemaUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSchemaUri").build()).build();

    private static final SdkField<String> RESOURCE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceRole").getter(getter(RDSDataSpec::resourceRole)).setter(setter(Builder::resourceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceRole").build()).build();

    private static final SdkField<String> SERVICE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceRole").getter(getter(RDSDataSpec::serviceRole)).setter(setter(Builder::serviceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceRole").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetId").getter(getter(RDSDataSpec::subnetId)).setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(RDSDataSpec::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATABASE_INFORMATION_FIELD,
            SELECT_SQL_QUERY_FIELD, DATABASE_CREDENTIALS_FIELD, S3_STAGING_LOCATION_FIELD, DATA_REARRANGEMENT_FIELD,
            DATA_SCHEMA_FIELD, DATA_SCHEMA_URI_FIELD, RESOURCE_ROLE_FIELD, SERVICE_ROLE_FIELD, SUBNET_ID_FIELD,
            SECURITY_GROUP_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final RDSDatabase databaseInformation;

    private final String selectSqlQuery;

    private final RDSDatabaseCredentials databaseCredentials;

    private final String s3StagingLocation;

    private final String dataRearrangement;

    private final String dataSchema;

    private final String dataSchemaUri;

    private final String resourceRole;

    private final String serviceRole;

    private final String subnetId;

    private final List<String> securityGroupIds;

    private RDSDataSpec(BuilderImpl builder) {
        this.databaseInformation = builder.databaseInformation;
        this.selectSqlQuery = builder.selectSqlQuery;
        this.databaseCredentials = builder.databaseCredentials;
        this.s3StagingLocation = builder.s3StagingLocation;
        this.dataRearrangement = builder.dataRearrangement;
        this.dataSchema = builder.dataSchema;
        this.dataSchemaUri = builder.dataSchemaUri;
        this.resourceRole = builder.resourceRole;
        this.serviceRole = builder.serviceRole;
        this.subnetId = builder.subnetId;
        this.securityGroupIds = builder.securityGroupIds;
    }

    /**
     * <p>
     * Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
     * </p>
     * 
     * @return Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
     */
    public final RDSDatabase databaseInformation() {
        return databaseInformation;
    }

    /**
     * <p>
     * The query that is used to retrieve the observation data for the <code>DataSource</code>.
     * </p>
     * 
     * @return The query that is used to retrieve the observation data for the <code>DataSource</code>.
     */
    public final String selectSqlQuery() {
        return selectSqlQuery;
    }

    /**
     * <p>
     * The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS database.
     * </p>
     * 
     * @return The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS
     *         database.
     */
    public final RDSDatabaseCredentials databaseCredentials() {
        return databaseCredentials;
    }

    /**
     * <p>
     * The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
     * <code>SelectSqlQuery</code> is stored in this location.
     * </p>
     * 
     * @return The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
     *         <code>SelectSqlQuery</code> is stored in this location.
     */
    public final String s3StagingLocation() {
        return s3StagingLocation;
    }

    /**
     * <p>
     * A JSON string that represents the splitting and rearrangement processing to be applied to a
     * <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the input data
     * is used to create the <code>Datasource</code>.
     * </p>
     * <p>
     * There are multiple parameters that control what data is used to create a datasource:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>percentBegin</code> </b>
     * </p>
     * <p>
     * Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the Datasource.
     * If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data
     * when creating the datasource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>percentEnd</code> </b>
     * </p>
     * <p>
     * Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource. If you do
     * not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data when
     * creating the datasource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>complement</code> </b>
     * </p>
     * <p>
     * The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the range of
     * <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The <code>complement</code>
     * parameter is useful if you need to create complementary datasources for training and evaluation. To create a
     * complementary datasource, use the same values for <code>percentBegin</code> and <code>percentEnd</code>, along
     * with the <code>complement</code> parameter.
     * </p>
     * <p>
     * For example, the following two datasources do not share any data, and can be used to train and evaluate a model.
     * The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
     * </p>
     * <p>
     * Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
     * </p>
     * <p>
     * Datasource for training: <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>strategy</code> </b>
     * </p>
     * <p>
     * To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
     * </p>
     * <p>
     * The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon ML
     * takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code> parameters for
     * the datasource, in the order that the records appear in the input data.
     * </p>
     * <p>
     * The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
     * evaluation datasources:
     * </p>
     * <p>
     * Datasource for evaluation:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
     * </p>
     * <p>
     * Datasource for training:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
     * </p>
     * <p>
     * To randomly split the input data into the proportions indicated by the percentBegin and percentEnd parameters,
     * set the <code>strategy</code> parameter to <code>random</code> and provide a string that is used as the seed
     * value for the random data splitting (for example, you can use the S3 path to your data as the random seed
     * string). If you choose the random split strategy, Amazon ML assigns each row of data a pseudo-random number
     * between 0 and 100, and then selects the rows that have an assigned number between <code>percentBegin</code> and
     * <code>percentEnd</code>. Pseudo-random numbers are assigned using both the input seed string value and the byte
     * offset as a seed, so changing the data results in a different split. Any existing ordering is preserved. The
     * random splitting strategy ensures that variables in the training and evaluation data are distributed similarly.
     * It is useful in the cases where the input data may have an implicit sort order, which would otherwise result in
     * training and evaluation datasources containing non-similar data records.
     * </p>
     * <p>
     * The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training and
     * evaluation datasources:
     * </p>
     * <p>
     * Datasource for evaluation:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
     * </p>
     * <p>
     * Datasource for training:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return A JSON string that represents the splitting and rearrangement processing to be applied to a
     *         <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the
     *         input data is used to create the <code>Datasource</code>.</p>
     *         <p>
     *         There are multiple parameters that control what data is used to create a datasource:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>percentBegin</code> </b>
     *         </p>
     *         <p>
     *         Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
     *         Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
     *         includes all of the data when creating the datasource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>percentEnd</code> </b>
     *         </p>
     *         <p>
     *         Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource.
     *         If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of
     *         the data when creating the datasource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>complement</code> </b>
     *         </p>
     *         <p>
     *         The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the
     *         range of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The
     *         <code>complement</code> parameter is useful if you need to create complementary datasources for training
     *         and evaluation. To create a complementary datasource, use the same values for <code>percentBegin</code>
     *         and <code>percentEnd</code>, along with the <code>complement</code> parameter.
     *         </p>
     *         <p>
     *         For example, the following two datasources do not share any data, and can be used to train and evaluate a
     *         model. The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
     *         </p>
     *         <p>
     *         Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>strategy</code> </b>
     *         </p>
     *         <p>
     *         To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
     *         </p>
     *         <p>
     *         The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon
     *         ML takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code>
     *         parameters for the datasource, in the order that the records appear in the input data.
     *         </p>
     *         <p>
     *         The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
     *         evaluation datasources:
     *         </p>
     *         <p>
     *         Datasource for evaluation:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
     *         </p>
     *         <p>
     *         To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
     *         parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that is
     *         used as the seed value for the random data splitting (for example, you can use the S3 path to your data
     *         as the random seed string). If you choose the random split strategy, Amazon ML assigns each row of data a
     *         pseudo-random number between 0 and 100, and then selects the rows that have an assigned number between
     *         <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are assigned using both the
     *         input seed string value and the byte offset as a seed, so changing the data results in a different split.
     *         Any existing ordering is preserved. The random splitting strategy ensures that variables in the training
     *         and evaluation data are distributed similarly. It is useful in the cases where the input data may have an
     *         implicit sort order, which would otherwise result in training and evaluation datasources containing
     *         non-similar data records.
     *         </p>
     *         <p>
     *         The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training
     *         and evaluation datasources:
     *         </p>
     *         <p>
     *         Datasource for evaluation:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
     *         </p>
     *         </li>
     */
    public final String dataRearrangement() {
        return dataRearrangement;
    }

    /**
     * <p>
     * A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The <code>DataSchema</code>
     * defines the structure of the observation data in the data file(s) referenced in the <code>DataSource</code>.
     * </p>
     * <p>
     * A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
     * </p>
     * <p>
     * Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
     * <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following format to
     * define your <code>DataSchema</code>.
     * </p>
     * <p>
     * { "version": "1.0",
     * </p>
     * <p>
     * "recordAnnotationFieldName": "F1",
     * </p>
     * <p>
     * "recordWeightFieldName": "F2",
     * </p>
     * <p>
     * "targetFieldName": "F3",
     * </p>
     * <p>
     * "dataFormat": "CSV",
     * </p>
     * <p>
     * "dataFileContainsHeader": true,
     * </p>
     * <p>
     * "attributes": [
     * </p>
     * <p>
     * { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName": "F3",
     * "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5", "fieldType":
     * "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7", "fieldType":
     * "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
     * </p>
     * <p>
     * "excludedVariableNames": [ "F6" ] }
     * </p>
     * 
     * @return A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The
     *         <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced in
     *         the <code>DataSource</code>.</p>
     *         <p>
     *         A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
     *         </p>
     *         <p>
     *         Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
     *         <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following
     *         format to define your <code>DataSchema</code>.
     *         </p>
     *         <p>
     *         { "version": "1.0",
     *         </p>
     *         <p>
     *         "recordAnnotationFieldName": "F1",
     *         </p>
     *         <p>
     *         "recordWeightFieldName": "F2",
     *         </p>
     *         <p>
     *         "targetFieldName": "F3",
     *         </p>
     *         <p>
     *         "dataFormat": "CSV",
     *         </p>
     *         <p>
     *         "dataFileContainsHeader": true,
     *         </p>
     *         <p>
     *         "attributes": [
     *         </p>
     *         <p>
     *         { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName":
     *         "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5",
     *         "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7",
     *         "fieldType": "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
     *         </p>
     *         <p>
     *         "excludedVariableNames": [ "F6" ] }
     */
    public final String dataSchema() {
        return dataSchema;
    }

    /**
     * <p>
     * The Amazon S3 location of the <code>DataSchema</code>.
     * </p>
     * 
     * @return The Amazon S3 location of the <code>DataSchema</code>.
     */
    public final String dataSchemaUri() {
        return dataSchemaUri;
    }

    /**
     * <p>
     * The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2) instance to
     * carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information, see <a
     * href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a> for
     * data pipelines.
     * </p>
     * 
     * @return The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2)
     *         instance to carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information, see
     *         <a href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role
     *         templates</a> for data pipelines.
     */
    public final String resourceRole() {
        return resourceRole;
    }

    /**
     * <p>
     * The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the copy task
     * from Amazon RDS to Amazon S3. For more information, see <a
     * href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a> for
     * data pipelines.
     * </p>
     * 
     * @return The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the
     *         copy task from Amazon RDS to Amazon S3. For more information, see <a
     *         href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role
     *         templates</a> for data pipelines.
     */
    public final String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data Pipeline to carry
     * out the copy task from Amazon RDS to Amazon S3.
     * </p>
     * 
     * @return The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data Pipeline
     *         to carry out the copy task from Amazon RDS to Amazon S3.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are appropriate
     * ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data Pipeline to carry out
     * the copy operation from Amazon RDS to an Amazon S3 task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are
     *         appropriate ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data
     *         Pipeline to carry out the copy operation from Amazon RDS to an Amazon S3 task.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(databaseInformation());
        hashCode = 31 * hashCode + Objects.hashCode(selectSqlQuery());
        hashCode = 31 * hashCode + Objects.hashCode(databaseCredentials());
        hashCode = 31 * hashCode + Objects.hashCode(s3StagingLocation());
        hashCode = 31 * hashCode + Objects.hashCode(dataRearrangement());
        hashCode = 31 * hashCode + Objects.hashCode(dataSchema());
        hashCode = 31 * hashCode + Objects.hashCode(dataSchemaUri());
        hashCode = 31 * hashCode + Objects.hashCode(resourceRole());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RDSDataSpec)) {
            return false;
        }
        RDSDataSpec other = (RDSDataSpec) obj;
        return Objects.equals(databaseInformation(), other.databaseInformation())
                && Objects.equals(selectSqlQuery(), other.selectSqlQuery())
                && Objects.equals(databaseCredentials(), other.databaseCredentials())
                && Objects.equals(s3StagingLocation(), other.s3StagingLocation())
                && Objects.equals(dataRearrangement(), other.dataRearrangement())
                && Objects.equals(dataSchema(), other.dataSchema()) && Objects.equals(dataSchemaUri(), other.dataSchemaUri())
                && Objects.equals(resourceRole(), other.resourceRole()) && Objects.equals(serviceRole(), other.serviceRole())
                && Objects.equals(subnetId(), other.subnetId()) && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RDSDataSpec").add("DatabaseInformation", databaseInformation())
                .add("SelectSqlQuery", selectSqlQuery()).add("DatabaseCredentials", databaseCredentials())
                .add("S3StagingLocation", s3StagingLocation()).add("DataRearrangement", dataRearrangement())
                .add("DataSchema", dataSchema()).add("DataSchemaUri", dataSchemaUri()).add("ResourceRole", resourceRole())
                .add("ServiceRole", serviceRole()).add("SubnetId", subnetId())
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DatabaseInformation":
            return Optional.ofNullable(clazz.cast(databaseInformation()));
        case "SelectSqlQuery":
            return Optional.ofNullable(clazz.cast(selectSqlQuery()));
        case "DatabaseCredentials":
            return Optional.ofNullable(clazz.cast(databaseCredentials()));
        case "S3StagingLocation":
            return Optional.ofNullable(clazz.cast(s3StagingLocation()));
        case "DataRearrangement":
            return Optional.ofNullable(clazz.cast(dataRearrangement()));
        case "DataSchema":
            return Optional.ofNullable(clazz.cast(dataSchema()));
        case "DataSchemaUri":
            return Optional.ofNullable(clazz.cast(dataSchemaUri()));
        case "ResourceRole":
            return Optional.ofNullable(clazz.cast(resourceRole()));
        case "ServiceRole":
            return Optional.ofNullable(clazz.cast(serviceRole()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RDSDataSpec, T> g) {
        return obj -> g.apply((RDSDataSpec) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RDSDataSpec> {
        /**
         * <p>
         * Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
         * </p>
         * 
         * @param databaseInformation
         *        Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseInformation(RDSDatabase databaseInformation);

        /**
         * <p>
         * Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
         * </p>
         * This is a convenience method that creates an instance of the {@link RDSDatabase.Builder} avoiding the need to
         * create one manually via {@link RDSDatabase#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RDSDatabase.Builder#build()} is called immediately and its result
         * is passed to {@link #databaseInformation(RDSDatabase)}.
         * 
         * @param databaseInformation
         *        a consumer that will call methods on {@link RDSDatabase.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseInformation(RDSDatabase)
         */
        default Builder databaseInformation(Consumer<RDSDatabase.Builder> databaseInformation) {
            return databaseInformation(RDSDatabase.builder().applyMutation(databaseInformation).build());
        }

        /**
         * <p>
         * The query that is used to retrieve the observation data for the <code>DataSource</code>.
         * </p>
         * 
         * @param selectSqlQuery
         *        The query that is used to retrieve the observation data for the <code>DataSource</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectSqlQuery(String selectSqlQuery);

        /**
         * <p>
         * The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS database.
         * </p>
         * 
         * @param databaseCredentials
         *        The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS
         *        database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseCredentials(RDSDatabaseCredentials databaseCredentials);

        /**
         * <p>
         * The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS database.
         * </p>
         * This is a convenience method that creates an instance of the {@link RDSDatabaseCredentials.Builder} avoiding
         * the need to create one manually via {@link RDSDatabaseCredentials#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RDSDatabaseCredentials.Builder#build()} is called immediately and
         * its result is passed to {@link #databaseCredentials(RDSDatabaseCredentials)}.
         * 
         * @param databaseCredentials
         *        a consumer that will call methods on {@link RDSDatabaseCredentials.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseCredentials(RDSDatabaseCredentials)
         */
        default Builder databaseCredentials(Consumer<RDSDatabaseCredentials.Builder> databaseCredentials) {
            return databaseCredentials(RDSDatabaseCredentials.builder().applyMutation(databaseCredentials).build());
        }

        /**
         * <p>
         * The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
         * <code>SelectSqlQuery</code> is stored in this location.
         * </p>
         * 
         * @param s3StagingLocation
         *        The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
         *        <code>SelectSqlQuery</code> is stored in this location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3StagingLocation(String s3StagingLocation);

        /**
         * <p>
         * A JSON string that represents the splitting and rearrangement processing to be applied to a
         * <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the input
         * data is used to create the <code>Datasource</code>.
         * </p>
         * <p>
         * There are multiple parameters that control what data is used to create a datasource:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>percentBegin</code> </b>
         * </p>
         * <p>
         * Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
         * Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes
         * all of the data when creating the datasource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>percentEnd</code> </b>
         * </p>
         * <p>
         * Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource. If
         * you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data
         * when creating the datasource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>complement</code> </b>
         * </p>
         * <p>
         * The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the range
         * of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The <code>complement</code>
         * parameter is useful if you need to create complementary datasources for training and evaluation. To create a
         * complementary datasource, use the same values for <code>percentBegin</code> and <code>percentEnd</code>,
         * along with the <code>complement</code> parameter.
         * </p>
         * <p>
         * For example, the following two datasources do not share any data, and can be used to train and evaluate a
         * model. The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
         * </p>
         * <p>
         * Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
         * </p>
         * <p>
         * Datasource for training: <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>strategy</code> </b>
         * </p>
         * <p>
         * To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
         * </p>
         * <p>
         * The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon ML
         * takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code> parameters
         * for the datasource, in the order that the records appear in the input data.
         * </p>
         * <p>
         * The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
         * evaluation datasources:
         * </p>
         * <p>
         * Datasource for evaluation:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
         * </p>
         * <p>
         * Datasource for training:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
         * </p>
         * <p>
         * To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
         * parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that is used
         * as the seed value for the random data splitting (for example, you can use the S3 path to your data as the
         * random seed string). If you choose the random split strategy, Amazon ML assigns each row of data a
         * pseudo-random number between 0 and 100, and then selects the rows that have an assigned number between
         * <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are assigned using both the
         * input seed string value and the byte offset as a seed, so changing the data results in a different split. Any
         * existing ordering is preserved. The random splitting strategy ensures that variables in the training and
         * evaluation data are distributed similarly. It is useful in the cases where the input data may have an
         * implicit sort order, which would otherwise result in training and evaluation datasources containing
         * non-similar data records.
         * </p>
         * <p>
         * The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training and
         * evaluation datasources:
         * </p>
         * <p>
         * Datasource for evaluation:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
         * </p>
         * <p>
         * Datasource for training:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param dataRearrangement
         *        A JSON string that represents the splitting and rearrangement processing to be applied to a
         *        <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the
         *        input data is used to create the <code>Datasource</code>.</p>
         *        <p>
         *        There are multiple parameters that control what data is used to create a datasource:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>percentBegin</code> </b>
         *        </p>
         *        <p>
         *        Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
         *        Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
         *        includes all of the data when creating the datasource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>percentEnd</code> </b>
         *        </p>
         *        <p>
         *        Use <code>percentEnd</code> to indicate the end of the range of the data used to create the
         *        Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
         *        includes all of the data when creating the datasource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>complement</code> </b>
         *        </p>
         *        <p>
         *        The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the
         *        range of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The
         *        <code>complement</code> parameter is useful if you need to create complementary datasources for
         *        training and evaluation. To create a complementary datasource, use the same values for
         *        <code>percentBegin</code> and <code>percentEnd</code>, along with the <code>complement</code>
         *        parameter.
         *        </p>
         *        <p>
         *        For example, the following two datasources do not share any data, and can be used to train and
         *        evaluate a model. The first datasource has 25 percent of the data, and the second one has 75 percent
         *        of the data.
         *        </p>
         *        <p>
         *        Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>strategy</code> </b>
         *        </p>
         *        <p>
         *        To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
         *        </p>
         *        <p>
         *        The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that
         *        Amazon ML takes all of the data records between the <code>percentBegin</code> and
         *        <code>percentEnd</code> parameters for the datasource, in the order that the records appear in the
         *        input data.
         *        </p>
         *        <p>
         *        The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training
         *        and evaluation datasources:
         *        </p>
         *        <p>
         *        Datasource for evaluation:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
         *        </p>
         *        <p>
         *        To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
         *        parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that
         *        is used as the seed value for the random data splitting (for example, you can use the S3 path to your
         *        data as the random seed string). If you choose the random split strategy, Amazon ML assigns each row
         *        of data a pseudo-random number between 0 and 100, and then selects the rows that have an assigned
         *        number between <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are
         *        assigned using both the input seed string value and the byte offset as a seed, so changing the data
         *        results in a different split. Any existing ordering is preserved. The random splitting strategy
         *        ensures that variables in the training and evaluation data are distributed similarly. It is useful in
         *        the cases where the input data may have an implicit sort order, which would otherwise result in
         *        training and evaluation datasources containing non-similar data records.
         *        </p>
         *        <p>
         *        The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered
         *        training and evaluation datasources:
         *        </p>
         *        <p>
         *        Datasource for evaluation:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRearrangement(String dataRearrangement);

        /**
         * <p>
         * A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The
         * <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced in the
         * <code>DataSource</code>.
         * </p>
         * <p>
         * A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
         * </p>
         * <p>
         * Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
         * <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following format
         * to define your <code>DataSchema</code>.
         * </p>
         * <p>
         * { "version": "1.0",
         * </p>
         * <p>
         * "recordAnnotationFieldName": "F1",
         * </p>
         * <p>
         * "recordWeightFieldName": "F2",
         * </p>
         * <p>
         * "targetFieldName": "F3",
         * </p>
         * <p>
         * "dataFormat": "CSV",
         * </p>
         * <p>
         * "dataFileContainsHeader": true,
         * </p>
         * <p>
         * "attributes": [
         * </p>
         * <p>
         * { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName":
         * "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5",
         * "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7", "fieldType":
         * "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
         * </p>
         * <p>
         * "excludedVariableNames": [ "F6" ] }
         * </p>
         * 
         * @param dataSchema
         *        A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The
         *        <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced
         *        in the <code>DataSource</code>.</p>
         *        <p>
         *        A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
         *        </p>
         *        <p>
         *        Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
         *        <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following
         *        format to define your <code>DataSchema</code>.
         *        </p>
         *        <p>
         *        { "version": "1.0",
         *        </p>
         *        <p>
         *        "recordAnnotationFieldName": "F1",
         *        </p>
         *        <p>
         *        "recordWeightFieldName": "F2",
         *        </p>
         *        <p>
         *        "targetFieldName": "F3",
         *        </p>
         *        <p>
         *        "dataFormat": "CSV",
         *        </p>
         *        <p>
         *        "dataFileContainsHeader": true,
         *        </p>
         *        <p>
         *        "attributes": [
         *        </p>
         *        <p>
         *        { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, {
         *        "fieldName": "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, {
         *        "fieldName": "F5", "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, {
         *        "fieldName": "F7", "fieldType": "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType":
         *        "WEIGHTED_STRING_SEQUENCE" } ],
         *        </p>
         *        <p>
         *        "excludedVariableNames": [ "F6" ] }
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSchema(String dataSchema);

        /**
         * <p>
         * The Amazon S3 location of the <code>DataSchema</code>.
         * </p>
         * 
         * @param dataSchemaUri
         *        The Amazon S3 location of the <code>DataSchema</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSchemaUri(String dataSchemaUri);

        /**
         * <p>
         * The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2) instance
         * to carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information, see <a
         * href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a>
         * for data pipelines.
         * </p>
         * 
         * @param resourceRole
         *        The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2)
         *        instance to carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information,
         *        see <a href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role
         *        templates</a> for data pipelines.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRole(String resourceRole);

        /**
         * <p>
         * The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the copy
         * task from Amazon RDS to Amazon S3. For more information, see <a
         * href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a>
         * for data pipelines.
         * </p>
         * 
         * @param serviceRole
         *        The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the
         *        copy task from Amazon RDS to Amazon S3. For more information, see <a
         *        href="https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role
         *        templates</a> for data pipelines.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data Pipeline to
         * carry out the copy task from Amazon RDS to Amazon S3.
         * </p>
         * 
         * @param subnetId
         *        The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data
         *        Pipeline to carry out the copy task from Amazon RDS to Amazon S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are appropriate
         * ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data Pipeline to carry
         * out the copy operation from Amazon RDS to an Amazon S3 task.
         * </p>
         * 
         * @param securityGroupIds
         *        The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are
         *        appropriate ingress rules set up to allow access to the RDS DB instance. This attribute is used by
         *        Data Pipeline to carry out the copy operation from Amazon RDS to an Amazon S3 task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are appropriate
         * ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data Pipeline to carry
         * out the copy operation from Amazon RDS to an Amazon S3 task.
         * </p>
         * 
         * @param securityGroupIds
         *        The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are
         *        appropriate ingress rules set up to allow access to the RDS DB instance. This attribute is used by
         *        Data Pipeline to carry out the copy operation from Amazon RDS to an Amazon S3 task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);
    }

    static final class BuilderImpl implements Builder {
        private RDSDatabase databaseInformation;

        private String selectSqlQuery;

        private RDSDatabaseCredentials databaseCredentials;

        private String s3StagingLocation;

        private String dataRearrangement;

        private String dataSchema;

        private String dataSchemaUri;

        private String resourceRole;

        private String serviceRole;

        private String subnetId;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RDSDataSpec model) {
            databaseInformation(model.databaseInformation);
            selectSqlQuery(model.selectSqlQuery);
            databaseCredentials(model.databaseCredentials);
            s3StagingLocation(model.s3StagingLocation);
            dataRearrangement(model.dataRearrangement);
            dataSchema(model.dataSchema);
            dataSchemaUri(model.dataSchemaUri);
            resourceRole(model.resourceRole);
            serviceRole(model.serviceRole);
            subnetId(model.subnetId);
            securityGroupIds(model.securityGroupIds);
        }

        public final RDSDatabase.Builder getDatabaseInformation() {
            return databaseInformation != null ? databaseInformation.toBuilder() : null;
        }

        public final void setDatabaseInformation(RDSDatabase.BuilderImpl databaseInformation) {
            this.databaseInformation = databaseInformation != null ? databaseInformation.build() : null;
        }

        @Override
        public final Builder databaseInformation(RDSDatabase databaseInformation) {
            this.databaseInformation = databaseInformation;
            return this;
        }

        public final String getSelectSqlQuery() {
            return selectSqlQuery;
        }

        public final void setSelectSqlQuery(String selectSqlQuery) {
            this.selectSqlQuery = selectSqlQuery;
        }

        @Override
        public final Builder selectSqlQuery(String selectSqlQuery) {
            this.selectSqlQuery = selectSqlQuery;
            return this;
        }

        public final RDSDatabaseCredentials.Builder getDatabaseCredentials() {
            return databaseCredentials != null ? databaseCredentials.toBuilder() : null;
        }

        public final void setDatabaseCredentials(RDSDatabaseCredentials.BuilderImpl databaseCredentials) {
            this.databaseCredentials = databaseCredentials != null ? databaseCredentials.build() : null;
        }

        @Override
        public final Builder databaseCredentials(RDSDatabaseCredentials databaseCredentials) {
            this.databaseCredentials = databaseCredentials;
            return this;
        }

        public final String getS3StagingLocation() {
            return s3StagingLocation;
        }

        public final void setS3StagingLocation(String s3StagingLocation) {
            this.s3StagingLocation = s3StagingLocation;
        }

        @Override
        public final Builder s3StagingLocation(String s3StagingLocation) {
            this.s3StagingLocation = s3StagingLocation;
            return this;
        }

        public final String getDataRearrangement() {
            return dataRearrangement;
        }

        public final void setDataRearrangement(String dataRearrangement) {
            this.dataRearrangement = dataRearrangement;
        }

        @Override
        public final Builder dataRearrangement(String dataRearrangement) {
            this.dataRearrangement = dataRearrangement;
            return this;
        }

        public final String getDataSchema() {
            return dataSchema;
        }

        public final void setDataSchema(String dataSchema) {
            this.dataSchema = dataSchema;
        }

        @Override
        public final Builder dataSchema(String dataSchema) {
            this.dataSchema = dataSchema;
            return this;
        }

        public final String getDataSchemaUri() {
            return dataSchemaUri;
        }

        public final void setDataSchemaUri(String dataSchemaUri) {
            this.dataSchemaUri = dataSchemaUri;
        }

        @Override
        public final Builder dataSchemaUri(String dataSchemaUri) {
            this.dataSchemaUri = dataSchemaUri;
            return this;
        }

        public final String getResourceRole() {
            return resourceRole;
        }

        public final void setResourceRole(String resourceRole) {
            this.resourceRole = resourceRole;
        }

        @Override
        public final Builder resourceRole(String resourceRole) {
            this.resourceRole = resourceRole;
            return this;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = EDPSecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = EDPSecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        @Override
        public RDSDataSpec build() {
            return new RDSDataSpec(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
