/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the data specification of an Amazon Redshift <code>DataSource</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RedshiftDataSpec implements SdkPojo, Serializable,
        ToCopyableBuilder<RedshiftDataSpec.Builder, RedshiftDataSpec> {
    private static final SdkField<RedshiftDatabase> DATABASE_INFORMATION_FIELD = SdkField
            .<RedshiftDatabase> builder(MarshallingType.SDK_POJO).memberName("DatabaseInformation")
            .getter(getter(RedshiftDataSpec::databaseInformation)).setter(setter(Builder::databaseInformation))
            .constructor(RedshiftDatabase::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseInformation").build())
            .build();

    private static final SdkField<String> SELECT_SQL_QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SelectSqlQuery").getter(getter(RedshiftDataSpec::selectSqlQuery))
            .setter(setter(Builder::selectSqlQuery))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectSqlQuery").build()).build();

    private static final SdkField<RedshiftDatabaseCredentials> DATABASE_CREDENTIALS_FIELD = SdkField
            .<RedshiftDatabaseCredentials> builder(MarshallingType.SDK_POJO).memberName("DatabaseCredentials")
            .getter(getter(RedshiftDataSpec::databaseCredentials)).setter(setter(Builder::databaseCredentials))
            .constructor(RedshiftDatabaseCredentials::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseCredentials").build())
            .build();

    private static final SdkField<String> S3_STAGING_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3StagingLocation").getter(getter(RedshiftDataSpec::s3StagingLocation))
            .setter(setter(Builder::s3StagingLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3StagingLocation").build()).build();

    private static final SdkField<String> DATA_REARRANGEMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataRearrangement").getter(getter(RedshiftDataSpec::dataRearrangement))
            .setter(setter(Builder::dataRearrangement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRearrangement").build()).build();

    private static final SdkField<String> DATA_SCHEMA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataSchema").getter(getter(RedshiftDataSpec::dataSchema)).setter(setter(Builder::dataSchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSchema").build()).build();

    private static final SdkField<String> DATA_SCHEMA_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataSchemaUri").getter(getter(RedshiftDataSpec::dataSchemaUri)).setter(setter(Builder::dataSchemaUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSchemaUri").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATABASE_INFORMATION_FIELD,
            SELECT_SQL_QUERY_FIELD, DATABASE_CREDENTIALS_FIELD, S3_STAGING_LOCATION_FIELD, DATA_REARRANGEMENT_FIELD,
            DATA_SCHEMA_FIELD, DATA_SCHEMA_URI_FIELD));

    private static final long serialVersionUID = 1L;

    private final RedshiftDatabase databaseInformation;

    private final String selectSqlQuery;

    private final RedshiftDatabaseCredentials databaseCredentials;

    private final String s3StagingLocation;

    private final String dataRearrangement;

    private final String dataSchema;

    private final String dataSchemaUri;

    private RedshiftDataSpec(BuilderImpl builder) {
        this.databaseInformation = builder.databaseInformation;
        this.selectSqlQuery = builder.selectSqlQuery;
        this.databaseCredentials = builder.databaseCredentials;
        this.s3StagingLocation = builder.s3StagingLocation;
        this.dataRearrangement = builder.dataRearrangement;
        this.dataSchema = builder.dataSchema;
        this.dataSchemaUri = builder.dataSchemaUri;
    }

    /**
     * <p>
     * Describes the <code>DatabaseName</code> and <code>ClusterIdentifier</code> for an Amazon Redshift
     * <code>DataSource</code>.
     * </p>
     * 
     * @return Describes the <code>DatabaseName</code> and <code>ClusterIdentifier</code> for an Amazon Redshift
     *         <code>DataSource</code>.
     */
    public RedshiftDatabase databaseInformation() {
        return databaseInformation;
    }

    /**
     * <p>
     * Describes the SQL Query to execute on an Amazon Redshift database for an Amazon Redshift <code>DataSource</code>.
     * </p>
     * 
     * @return Describes the SQL Query to execute on an Amazon Redshift database for an Amazon Redshift
     *         <code>DataSource</code>.
     */
    public String selectSqlQuery() {
        return selectSqlQuery;
    }

    /**
     * <p>
     * Describes AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon Redshift
     * database.
     * </p>
     * 
     * @return Describes AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon
     *         Redshift database.
     */
    public RedshiftDatabaseCredentials databaseCredentials() {
        return databaseCredentials;
    }

    /**
     * <p>
     * Describes an Amazon S3 location to store the result set of the <code>SelectSqlQuery</code> query.
     * </p>
     * 
     * @return Describes an Amazon S3 location to store the result set of the <code>SelectSqlQuery</code> query.
     */
    public String s3StagingLocation() {
        return s3StagingLocation;
    }

    /**
     * <p>
     * A JSON string that represents the splitting and rearrangement processing to be applied to a
     * <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the input data
     * is used to create the <code>Datasource</code>.
     * </p>
     * <p>
     * There are multiple parameters that control what data is used to create a datasource:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b><code>percentBegin</code></b>
     * </p>
     * <p>
     * Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the Datasource.
     * If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data
     * when creating the datasource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b><code>percentEnd</code></b>
     * </p>
     * <p>
     * Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource. If you do
     * not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data when
     * creating the datasource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b><code>complement</code></b>
     * </p>
     * <p>
     * The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the range of
     * <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The <code>complement</code>
     * parameter is useful if you need to create complementary datasources for training and evaluation. To create a
     * complementary datasource, use the same values for <code>percentBegin</code> and <code>percentEnd</code>, along
     * with the <code>complement</code> parameter.
     * </p>
     * <p>
     * For example, the following two datasources do not share any data, and can be used to train and evaluate a model.
     * The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
     * </p>
     * <p>
     * Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
     * </p>
     * <p>
     * Datasource for training: <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b><code>strategy</code></b>
     * </p>
     * <p>
     * To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
     * </p>
     * <p>
     * The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon ML
     * takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code> parameters for
     * the datasource, in the order that the records appear in the input data.
     * </p>
     * <p>
     * The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
     * evaluation datasources:
     * </p>
     * <p>
     * Datasource for evaluation:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
     * </p>
     * <p>
     * Datasource for training:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
     * </p>
     * <p>
     * To randomly split the input data into the proportions indicated by the percentBegin and percentEnd parameters,
     * set the <code>strategy</code> parameter to <code>random</code> and provide a string that is used as the seed
     * value for the random data splitting (for example, you can use the S3 path to your data as the random seed
     * string). If you choose the random split strategy, Amazon ML assigns each row of data a pseudo-random number
     * between 0 and 100, and then selects the rows that have an assigned number between <code>percentBegin</code> and
     * <code>percentEnd</code>. Pseudo-random numbers are assigned using both the input seed string value and the byte
     * offset as a seed, so changing the data results in a different split. Any existing ordering is preserved. The
     * random splitting strategy ensures that variables in the training and evaluation data are distributed similarly.
     * It is useful in the cases where the input data may have an implicit sort order, which would otherwise result in
     * training and evaluation datasources containing non-similar data records.
     * </p>
     * <p>
     * The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training and
     * evaluation datasources:
     * </p>
     * <p>
     * Datasource for evaluation:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
     * </p>
     * <p>
     * Datasource for training:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return A JSON string that represents the splitting and rearrangement processing to be applied to a
     *         <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the
     *         input data is used to create the <code>Datasource</code>.</p>
     *         <p>
     *         There are multiple parameters that control what data is used to create a datasource:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b><code>percentBegin</code></b>
     *         </p>
     *         <p>
     *         Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
     *         Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
     *         includes all of the data when creating the datasource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b><code>percentEnd</code></b>
     *         </p>
     *         <p>
     *         Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource.
     *         If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of
     *         the data when creating the datasource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b><code>complement</code></b>
     *         </p>
     *         <p>
     *         The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the
     *         range of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The
     *         <code>complement</code> parameter is useful if you need to create complementary datasources for training
     *         and evaluation. To create a complementary datasource, use the same values for <code>percentBegin</code>
     *         and <code>percentEnd</code>, along with the <code>complement</code> parameter.
     *         </p>
     *         <p>
     *         For example, the following two datasources do not share any data, and can be used to train and evaluate a
     *         model. The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
     *         </p>
     *         <p>
     *         Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b><code>strategy</code></b>
     *         </p>
     *         <p>
     *         To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
     *         </p>
     *         <p>
     *         The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon
     *         ML takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code>
     *         parameters for the datasource, in the order that the records appear in the input data.
     *         </p>
     *         <p>
     *         The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
     *         evaluation datasources:
     *         </p>
     *         <p>
     *         Datasource for evaluation:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
     *         </p>
     *         <p>
     *         To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
     *         parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that is
     *         used as the seed value for the random data splitting (for example, you can use the S3 path to your data
     *         as the random seed string). If you choose the random split strategy, Amazon ML assigns each row of data a
     *         pseudo-random number between 0 and 100, and then selects the rows that have an assigned number between
     *         <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are assigned using both the
     *         input seed string value and the byte offset as a seed, so changing the data results in a different split.
     *         Any existing ordering is preserved. The random splitting strategy ensures that variables in the training
     *         and evaluation data are distributed similarly. It is useful in the cases where the input data may have an
     *         implicit sort order, which would otherwise result in training and evaluation datasources containing
     *         non-similar data records.
     *         </p>
     *         <p>
     *         The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training
     *         and evaluation datasources:
     *         </p>
     *         <p>
     *         Datasource for evaluation:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
     *         </p>
     *         </li>
     */
    public String dataRearrangement() {
        return dataRearrangement;
    }

    /**
     * <p>
     * A JSON string that represents the schema for an Amazon Redshift <code>DataSource</code>. The
     * <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced in the
     * <code>DataSource</code>.
     * </p>
     * <p>
     * A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>.
     * </p>
     * <p>
     * Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
     * <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following format to
     * define your <code>DataSchema</code>.
     * </p>
     * <p>
     * { "version": "1.0",
     * </p>
     * <p>
     * "recordAnnotationFieldName": "F1",
     * </p>
     * <p>
     * "recordWeightFieldName": "F2",
     * </p>
     * <p>
     * "targetFieldName": "F3",
     * </p>
     * <p>
     * "dataFormat": "CSV",
     * </p>
     * <p>
     * "dataFileContainsHeader": true,
     * </p>
     * <p>
     * "attributes": [
     * </p>
     * <p>
     * { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName": "F3",
     * "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5", "fieldType":
     * "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7", "fieldType":
     * "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
     * </p>
     * <p>
     * "excludedVariableNames": [ "F6" ] }
     * </p>
     * 
     * @return A JSON string that represents the schema for an Amazon Redshift <code>DataSource</code>. The
     *         <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced in
     *         the <code>DataSource</code>.</p>
     *         <p>
     *         A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>.
     *         </p>
     *         <p>
     *         Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
     *         <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following
     *         format to define your <code>DataSchema</code>.
     *         </p>
     *         <p>
     *         { "version": "1.0",
     *         </p>
     *         <p>
     *         "recordAnnotationFieldName": "F1",
     *         </p>
     *         <p>
     *         "recordWeightFieldName": "F2",
     *         </p>
     *         <p>
     *         "targetFieldName": "F3",
     *         </p>
     *         <p>
     *         "dataFormat": "CSV",
     *         </p>
     *         <p>
     *         "dataFileContainsHeader": true,
     *         </p>
     *         <p>
     *         "attributes": [
     *         </p>
     *         <p>
     *         { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName":
     *         "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5",
     *         "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7",
     *         "fieldType": "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
     *         </p>
     *         <p>
     *         "excludedVariableNames": [ "F6" ] }
     */
    public String dataSchema() {
        return dataSchema;
    }

    /**
     * <p>
     * Describes the schema location for an Amazon Redshift <code>DataSource</code>.
     * </p>
     * 
     * @return Describes the schema location for an Amazon Redshift <code>DataSource</code>.
     */
    public String dataSchemaUri() {
        return dataSchemaUri;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(databaseInformation());
        hashCode = 31 * hashCode + Objects.hashCode(selectSqlQuery());
        hashCode = 31 * hashCode + Objects.hashCode(databaseCredentials());
        hashCode = 31 * hashCode + Objects.hashCode(s3StagingLocation());
        hashCode = 31 * hashCode + Objects.hashCode(dataRearrangement());
        hashCode = 31 * hashCode + Objects.hashCode(dataSchema());
        hashCode = 31 * hashCode + Objects.hashCode(dataSchemaUri());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RedshiftDataSpec)) {
            return false;
        }
        RedshiftDataSpec other = (RedshiftDataSpec) obj;
        return Objects.equals(databaseInformation(), other.databaseInformation())
                && Objects.equals(selectSqlQuery(), other.selectSqlQuery())
                && Objects.equals(databaseCredentials(), other.databaseCredentials())
                && Objects.equals(s3StagingLocation(), other.s3StagingLocation())
                && Objects.equals(dataRearrangement(), other.dataRearrangement())
                && Objects.equals(dataSchema(), other.dataSchema()) && Objects.equals(dataSchemaUri(), other.dataSchemaUri());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RedshiftDataSpec").add("DatabaseInformation", databaseInformation())
                .add("SelectSqlQuery", selectSqlQuery()).add("DatabaseCredentials", databaseCredentials())
                .add("S3StagingLocation", s3StagingLocation()).add("DataRearrangement", dataRearrangement())
                .add("DataSchema", dataSchema()).add("DataSchemaUri", dataSchemaUri()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DatabaseInformation":
            return Optional.ofNullable(clazz.cast(databaseInformation()));
        case "SelectSqlQuery":
            return Optional.ofNullable(clazz.cast(selectSqlQuery()));
        case "DatabaseCredentials":
            return Optional.ofNullable(clazz.cast(databaseCredentials()));
        case "S3StagingLocation":
            return Optional.ofNullable(clazz.cast(s3StagingLocation()));
        case "DataRearrangement":
            return Optional.ofNullable(clazz.cast(dataRearrangement()));
        case "DataSchema":
            return Optional.ofNullable(clazz.cast(dataSchema()));
        case "DataSchemaUri":
            return Optional.ofNullable(clazz.cast(dataSchemaUri()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RedshiftDataSpec, T> g) {
        return obj -> g.apply((RedshiftDataSpec) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RedshiftDataSpec> {
        /**
         * <p>
         * Describes the <code>DatabaseName</code> and <code>ClusterIdentifier</code> for an Amazon Redshift
         * <code>DataSource</code>.
         * </p>
         * 
         * @param databaseInformation
         *        Describes the <code>DatabaseName</code> and <code>ClusterIdentifier</code> for an Amazon Redshift
         *        <code>DataSource</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseInformation(RedshiftDatabase databaseInformation);

        /**
         * <p>
         * Describes the <code>DatabaseName</code> and <code>ClusterIdentifier</code> for an Amazon Redshift
         * <code>DataSource</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link RedshiftDatabase.Builder} avoiding the need to
         * create one manually via {@link RedshiftDatabase#builder()}.
         *
         * When the {@link Consumer} completes, {@link RedshiftDatabase.Builder#build()} is called immediately and its
         * result is passed to {@link #databaseInformation(RedshiftDatabase)}.
         * 
         * @param databaseInformation
         *        a consumer that will call methods on {@link RedshiftDatabase.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseInformation(RedshiftDatabase)
         */
        default Builder databaseInformation(Consumer<RedshiftDatabase.Builder> databaseInformation) {
            return databaseInformation(RedshiftDatabase.builder().applyMutation(databaseInformation).build());
        }

        /**
         * <p>
         * Describes the SQL Query to execute on an Amazon Redshift database for an Amazon Redshift
         * <code>DataSource</code>.
         * </p>
         * 
         * @param selectSqlQuery
         *        Describes the SQL Query to execute on an Amazon Redshift database for an Amazon Redshift
         *        <code>DataSource</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectSqlQuery(String selectSqlQuery);

        /**
         * <p>
         * Describes AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon Redshift
         * database.
         * </p>
         * 
         * @param databaseCredentials
         *        Describes AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon
         *        Redshift database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseCredentials(RedshiftDatabaseCredentials databaseCredentials);

        /**
         * <p>
         * Describes AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon Redshift
         * database.
         * </p>
         * This is a convenience that creates an instance of the {@link RedshiftDatabaseCredentials.Builder} avoiding
         * the need to create one manually via {@link RedshiftDatabaseCredentials#builder()}.
         *
         * When the {@link Consumer} completes, {@link RedshiftDatabaseCredentials.Builder#build()} is called
         * immediately and its result is passed to {@link #databaseCredentials(RedshiftDatabaseCredentials)}.
         * 
         * @param databaseCredentials
         *        a consumer that will call methods on {@link RedshiftDatabaseCredentials.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseCredentials(RedshiftDatabaseCredentials)
         */
        default Builder databaseCredentials(Consumer<RedshiftDatabaseCredentials.Builder> databaseCredentials) {
            return databaseCredentials(RedshiftDatabaseCredentials.builder().applyMutation(databaseCredentials).build());
        }

        /**
         * <p>
         * Describes an Amazon S3 location to store the result set of the <code>SelectSqlQuery</code> query.
         * </p>
         * 
         * @param s3StagingLocation
         *        Describes an Amazon S3 location to store the result set of the <code>SelectSqlQuery</code> query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3StagingLocation(String s3StagingLocation);

        /**
         * <p>
         * A JSON string that represents the splitting and rearrangement processing to be applied to a
         * <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the input
         * data is used to create the <code>Datasource</code>.
         * </p>
         * <p>
         * There are multiple parameters that control what data is used to create a datasource:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b><code>percentBegin</code></b>
         * </p>
         * <p>
         * Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
         * Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes
         * all of the data when creating the datasource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b><code>percentEnd</code></b>
         * </p>
         * <p>
         * Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource. If
         * you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data
         * when creating the datasource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b><code>complement</code></b>
         * </p>
         * <p>
         * The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the range
         * of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The <code>complement</code>
         * parameter is useful if you need to create complementary datasources for training and evaluation. To create a
         * complementary datasource, use the same values for <code>percentBegin</code> and <code>percentEnd</code>,
         * along with the <code>complement</code> parameter.
         * </p>
         * <p>
         * For example, the following two datasources do not share any data, and can be used to train and evaluate a
         * model. The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
         * </p>
         * <p>
         * Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
         * </p>
         * <p>
         * Datasource for training: <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b><code>strategy</code></b>
         * </p>
         * <p>
         * To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
         * </p>
         * <p>
         * The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon ML
         * takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code> parameters
         * for the datasource, in the order that the records appear in the input data.
         * </p>
         * <p>
         * The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
         * evaluation datasources:
         * </p>
         * <p>
         * Datasource for evaluation:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
         * </p>
         * <p>
         * Datasource for training:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
         * </p>
         * <p>
         * To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
         * parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that is used
         * as the seed value for the random data splitting (for example, you can use the S3 path to your data as the
         * random seed string). If you choose the random split strategy, Amazon ML assigns each row of data a
         * pseudo-random number between 0 and 100, and then selects the rows that have an assigned number between
         * <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are assigned using both the
         * input seed string value and the byte offset as a seed, so changing the data results in a different split. Any
         * existing ordering is preserved. The random splitting strategy ensures that variables in the training and
         * evaluation data are distributed similarly. It is useful in the cases where the input data may have an
         * implicit sort order, which would otherwise result in training and evaluation datasources containing
         * non-similar data records.
         * </p>
         * <p>
         * The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training and
         * evaluation datasources:
         * </p>
         * <p>
         * Datasource for evaluation:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
         * </p>
         * <p>
         * Datasource for training:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param dataRearrangement
         *        A JSON string that represents the splitting and rearrangement processing to be applied to a
         *        <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the
         *        input data is used to create the <code>Datasource</code>.</p>
         *        <p>
         *        There are multiple parameters that control what data is used to create a datasource:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b><code>percentBegin</code></b>
         *        </p>
         *        <p>
         *        Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
         *        Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
         *        includes all of the data when creating the datasource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b><code>percentEnd</code></b>
         *        </p>
         *        <p>
         *        Use <code>percentEnd</code> to indicate the end of the range of the data used to create the
         *        Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
         *        includes all of the data when creating the datasource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b><code>complement</code></b>
         *        </p>
         *        <p>
         *        The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the
         *        range of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The
         *        <code>complement</code> parameter is useful if you need to create complementary datasources for
         *        training and evaluation. To create a complementary datasource, use the same values for
         *        <code>percentBegin</code> and <code>percentEnd</code>, along with the <code>complement</code>
         *        parameter.
         *        </p>
         *        <p>
         *        For example, the following two datasources do not share any data, and can be used to train and
         *        evaluate a model. The first datasource has 25 percent of the data, and the second one has 75 percent
         *        of the data.
         *        </p>
         *        <p>
         *        Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b><code>strategy</code></b>
         *        </p>
         *        <p>
         *        To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
         *        </p>
         *        <p>
         *        The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that
         *        Amazon ML takes all of the data records between the <code>percentBegin</code> and
         *        <code>percentEnd</code> parameters for the datasource, in the order that the records appear in the
         *        input data.
         *        </p>
         *        <p>
         *        The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training
         *        and evaluation datasources:
         *        </p>
         *        <p>
         *        Datasource for evaluation:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
         *        </p>
         *        <p>
         *        To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
         *        parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that
         *        is used as the seed value for the random data splitting (for example, you can use the S3 path to your
         *        data as the random seed string). If you choose the random split strategy, Amazon ML assigns each row
         *        of data a pseudo-random number between 0 and 100, and then selects the rows that have an assigned
         *        number between <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are
         *        assigned using both the input seed string value and the byte offset as a seed, so changing the data
         *        results in a different split. Any existing ordering is preserved. The random splitting strategy
         *        ensures that variables in the training and evaluation data are distributed similarly. It is useful in
         *        the cases where the input data may have an implicit sort order, which would otherwise result in
         *        training and evaluation datasources containing non-similar data records.
         *        </p>
         *        <p>
         *        The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered
         *        training and evaluation datasources:
         *        </p>
         *        <p>
         *        Datasource for evaluation:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRearrangement(String dataRearrangement);

        /**
         * <p>
         * A JSON string that represents the schema for an Amazon Redshift <code>DataSource</code>. The
         * <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced in the
         * <code>DataSource</code>.
         * </p>
         * <p>
         * A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>.
         * </p>
         * <p>
         * Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
         * <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following format
         * to define your <code>DataSchema</code>.
         * </p>
         * <p>
         * { "version": "1.0",
         * </p>
         * <p>
         * "recordAnnotationFieldName": "F1",
         * </p>
         * <p>
         * "recordWeightFieldName": "F2",
         * </p>
         * <p>
         * "targetFieldName": "F3",
         * </p>
         * <p>
         * "dataFormat": "CSV",
         * </p>
         * <p>
         * "dataFileContainsHeader": true,
         * </p>
         * <p>
         * "attributes": [
         * </p>
         * <p>
         * { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName":
         * "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5",
         * "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7", "fieldType":
         * "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
         * </p>
         * <p>
         * "excludedVariableNames": [ "F6" ] }
         * </p>
         * 
         * @param dataSchema
         *        A JSON string that represents the schema for an Amazon Redshift <code>DataSource</code>. The
         *        <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced
         *        in the <code>DataSource</code>.</p>
         *        <p>
         *        A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>.
         *        </p>
         *        <p>
         *        Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
         *        <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following
         *        format to define your <code>DataSchema</code>.
         *        </p>
         *        <p>
         *        { "version": "1.0",
         *        </p>
         *        <p>
         *        "recordAnnotationFieldName": "F1",
         *        </p>
         *        <p>
         *        "recordWeightFieldName": "F2",
         *        </p>
         *        <p>
         *        "targetFieldName": "F3",
         *        </p>
         *        <p>
         *        "dataFormat": "CSV",
         *        </p>
         *        <p>
         *        "dataFileContainsHeader": true,
         *        </p>
         *        <p>
         *        "attributes": [
         *        </p>
         *        <p>
         *        { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, {
         *        "fieldName": "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, {
         *        "fieldName": "F5", "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, {
         *        "fieldName": "F7", "fieldType": "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType":
         *        "WEIGHTED_STRING_SEQUENCE" } ],
         *        </p>
         *        <p>
         *        "excludedVariableNames": [ "F6" ] }
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSchema(String dataSchema);

        /**
         * <p>
         * Describes the schema location for an Amazon Redshift <code>DataSource</code>.
         * </p>
         * 
         * @param dataSchemaUri
         *        Describes the schema location for an Amazon Redshift <code>DataSource</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSchemaUri(String dataSchemaUri);
    }

    static final class BuilderImpl implements Builder {
        private RedshiftDatabase databaseInformation;

        private String selectSqlQuery;

        private RedshiftDatabaseCredentials databaseCredentials;

        private String s3StagingLocation;

        private String dataRearrangement;

        private String dataSchema;

        private String dataSchemaUri;

        private BuilderImpl() {
        }

        private BuilderImpl(RedshiftDataSpec model) {
            databaseInformation(model.databaseInformation);
            selectSqlQuery(model.selectSqlQuery);
            databaseCredentials(model.databaseCredentials);
            s3StagingLocation(model.s3StagingLocation);
            dataRearrangement(model.dataRearrangement);
            dataSchema(model.dataSchema);
            dataSchemaUri(model.dataSchemaUri);
        }

        public final RedshiftDatabase.Builder getDatabaseInformation() {
            return databaseInformation != null ? databaseInformation.toBuilder() : null;
        }

        @Override
        public final Builder databaseInformation(RedshiftDatabase databaseInformation) {
            this.databaseInformation = databaseInformation;
            return this;
        }

        public final void setDatabaseInformation(RedshiftDatabase.BuilderImpl databaseInformation) {
            this.databaseInformation = databaseInformation != null ? databaseInformation.build() : null;
        }

        public final String getSelectSqlQuery() {
            return selectSqlQuery;
        }

        @Override
        public final Builder selectSqlQuery(String selectSqlQuery) {
            this.selectSqlQuery = selectSqlQuery;
            return this;
        }

        public final void setSelectSqlQuery(String selectSqlQuery) {
            this.selectSqlQuery = selectSqlQuery;
        }

        public final RedshiftDatabaseCredentials.Builder getDatabaseCredentials() {
            return databaseCredentials != null ? databaseCredentials.toBuilder() : null;
        }

        @Override
        public final Builder databaseCredentials(RedshiftDatabaseCredentials databaseCredentials) {
            this.databaseCredentials = databaseCredentials;
            return this;
        }

        public final void setDatabaseCredentials(RedshiftDatabaseCredentials.BuilderImpl databaseCredentials) {
            this.databaseCredentials = databaseCredentials != null ? databaseCredentials.build() : null;
        }

        public final String getS3StagingLocation() {
            return s3StagingLocation;
        }

        @Override
        public final Builder s3StagingLocation(String s3StagingLocation) {
            this.s3StagingLocation = s3StagingLocation;
            return this;
        }

        public final void setS3StagingLocation(String s3StagingLocation) {
            this.s3StagingLocation = s3StagingLocation;
        }

        public final String getDataRearrangement() {
            return dataRearrangement;
        }

        @Override
        public final Builder dataRearrangement(String dataRearrangement) {
            this.dataRearrangement = dataRearrangement;
            return this;
        }

        public final void setDataRearrangement(String dataRearrangement) {
            this.dataRearrangement = dataRearrangement;
        }

        public final String getDataSchema() {
            return dataSchema;
        }

        @Override
        public final Builder dataSchema(String dataSchema) {
            this.dataSchema = dataSchema;
            return this;
        }

        public final void setDataSchema(String dataSchema) {
            this.dataSchema = dataSchema;
        }

        public final String getDataSchemaUri() {
            return dataSchemaUri;
        }

        @Override
        public final Builder dataSchemaUri(String dataSchemaUri) {
            this.dataSchemaUri = dataSchemaUri;
            return this;
        }

        public final void setDataSchemaUri(String dataSchemaUri) {
            this.dataSchemaUri = dataSchemaUri;
        }

        @Override
        public RedshiftDataSpec build() {
            return new RedshiftDataSpec(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
