/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.machinelearning.transform.RDSDataSpecMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The data specification of an Amazon Relational Database Service (Amazon RDS) <code>DataSource</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class RDSDataSpec implements StructuredPojo, ToCopyableBuilder<RDSDataSpec.Builder, RDSDataSpec> {
    private final RDSDatabase databaseInformation;

    private final String selectSqlQuery;

    private final RDSDatabaseCredentials databaseCredentials;

    private final String s3StagingLocation;

    private final String dataRearrangement;

    private final String dataSchema;

    private final String dataSchemaUri;

    private final String resourceRole;

    private final String serviceRole;

    private final String subnetId;

    private final List<String> securityGroupIds;

    private RDSDataSpec(BuilderImpl builder) {
        this.databaseInformation = builder.databaseInformation;
        this.selectSqlQuery = builder.selectSqlQuery;
        this.databaseCredentials = builder.databaseCredentials;
        this.s3StagingLocation = builder.s3StagingLocation;
        this.dataRearrangement = builder.dataRearrangement;
        this.dataSchema = builder.dataSchema;
        this.dataSchemaUri = builder.dataSchemaUri;
        this.resourceRole = builder.resourceRole;
        this.serviceRole = builder.serviceRole;
        this.subnetId = builder.subnetId;
        this.securityGroupIds = builder.securityGroupIds;
    }

    /**
     * <p>
     * Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
     * </p>
     * 
     * @return Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
     */
    public RDSDatabase databaseInformation() {
        return databaseInformation;
    }

    /**
     * <p>
     * The query that is used to retrieve the observation data for the <code>DataSource</code>.
     * </p>
     * 
     * @return The query that is used to retrieve the observation data for the <code>DataSource</code>.
     */
    public String selectSqlQuery() {
        return selectSqlQuery;
    }

    /**
     * <p>
     * The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS database.
     * </p>
     * 
     * @return The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS
     *         database.
     */
    public RDSDatabaseCredentials databaseCredentials() {
        return databaseCredentials;
    }

    /**
     * <p>
     * The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
     * <code>SelectSqlQuery</code> is stored in this location.
     * </p>
     * 
     * @return The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
     *         <code>SelectSqlQuery</code> is stored in this location.
     */
    public String s3StagingLocation() {
        return s3StagingLocation;
    }

    /**
     * <p>
     * A JSON string that represents the splitting and rearrangement processing to be applied to a
     * <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the input data
     * is used to create the <code>Datasource</code>.
     * </p>
     * <p>
     * There are multiple parameters that control what data is used to create a datasource:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b><code>percentBegin</code></b>
     * </p>
     * <p>
     * Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the Datasource.
     * If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data
     * when creating the datasource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b><code>percentEnd</code></b>
     * </p>
     * <p>
     * Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource. If you do
     * not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data when
     * creating the datasource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b><code>complement</code></b>
     * </p>
     * <p>
     * The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the range of
     * <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The <code>complement</code>
     * parameter is useful if you need to create complementary datasources for training and evaluation. To create a
     * complementary datasource, use the same values for <code>percentBegin</code> and <code>percentEnd</code>, along
     * with the <code>complement</code> parameter.
     * </p>
     * <p>
     * For example, the following two datasources do not share any data, and can be used to train and evaluate a model.
     * The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
     * </p>
     * <p>
     * Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
     * </p>
     * <p>
     * Datasource for training: <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <b><code>strategy</code></b>
     * </p>
     * <p>
     * To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
     * </p>
     * <p>
     * The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon ML
     * takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code> parameters for
     * the datasource, in the order that the records appear in the input data.
     * </p>
     * <p>
     * The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
     * evaluation datasources:
     * </p>
     * <p>
     * Datasource for evaluation:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
     * </p>
     * <p>
     * Datasource for training:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
     * </p>
     * <p>
     * To randomly split the input data into the proportions indicated by the percentBegin and percentEnd parameters,
     * set the <code>strategy</code> parameter to <code>random</code> and provide a string that is used as the seed
     * value for the random data splitting (for example, you can use the S3 path to your data as the random seed
     * string). If you choose the random split strategy, Amazon ML assigns each row of data a pseudo-random number
     * between 0 and 100, and then selects the rows that have an assigned number between <code>percentBegin</code> and
     * <code>percentEnd</code>. Pseudo-random numbers are assigned using both the input seed string value and the byte
     * offset as a seed, so changing the data results in a different split. Any existing ordering is preserved. The
     * random splitting strategy ensures that variables in the training and evaluation data are distributed similarly.
     * It is useful in the cases where the input data may have an implicit sort order, which would otherwise result in
     * training and evaluation datasources containing non-similar data records.
     * </p>
     * <p>
     * The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training and
     * evaluation datasources:
     * </p>
     * <p>
     * Datasource for evaluation:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
     * </p>
     * <p>
     * Datasource for training:
     * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return A JSON string that represents the splitting and rearrangement processing to be applied to a
     *         <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the
     *         input data is used to create the <code>Datasource</code>.</p>
     *         <p>
     *         There are multiple parameters that control what data is used to create a datasource:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b><code>percentBegin</code></b>
     *         </p>
     *         <p>
     *         Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
     *         Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
     *         includes all of the data when creating the datasource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b><code>percentEnd</code></b>
     *         </p>
     *         <p>
     *         Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource.
     *         If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of
     *         the data when creating the datasource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b><code>complement</code></b>
     *         </p>
     *         <p>
     *         The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the
     *         range of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The
     *         <code>complement</code> parameter is useful if you need to create complementary datasources for training
     *         and evaluation. To create a complementary datasource, use the same values for <code>percentBegin</code>
     *         and <code>percentEnd</code>, along with the <code>complement</code> parameter.
     *         </p>
     *         <p>
     *         For example, the following two datasources do not share any data, and can be used to train and evaluate a
     *         model. The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
     *         </p>
     *         <p>
     *         Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b><code>strategy</code></b>
     *         </p>
     *         <p>
     *         To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
     *         </p>
     *         <p>
     *         The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon
     *         ML takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code>
     *         parameters for the datasource, in the order that the records appear in the input data.
     *         </p>
     *         <p>
     *         The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
     *         evaluation datasources:
     *         </p>
     *         <p>
     *         Datasource for evaluation:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
     *         </p>
     *         <p>
     *         To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
     *         parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that is
     *         used as the seed value for the random data splitting (for example, you can use the S3 path to your data
     *         as the random seed string). If you choose the random split strategy, Amazon ML assigns each row of data a
     *         pseudo-random number between 0 and 100, and then selects the rows that have an assigned number between
     *         <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are assigned using both the
     *         input seed string value and the byte offset as a seed, so changing the data results in a different split.
     *         Any existing ordering is preserved. The random splitting strategy ensures that variables in the training
     *         and evaluation data are distributed similarly. It is useful in the cases where the input data may have an
     *         implicit sort order, which would otherwise result in training and evaluation datasources containing
     *         non-similar data records.
     *         </p>
     *         <p>
     *         The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training
     *         and evaluation datasources:
     *         </p>
     *         <p>
     *         Datasource for evaluation:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
     *         </p>
     *         <p>
     *         Datasource for training:
     *         <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
     *         </p>
     *         </li>
     */
    public String dataRearrangement() {
        return dataRearrangement;
    }

    /**
     * <p>
     * A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The <code>DataSchema</code>
     * defines the structure of the observation data in the data file(s) referenced in the <code>DataSource</code>.
     * </p>
     * <p>
     * A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
     * </p>
     * <p>
     * Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
     * <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following format to
     * define your <code>DataSchema</code>.
     * </p>
     * <p>
     * { "version": "1.0",
     * </p>
     * <p>
     * "recordAnnotationFieldName": "F1",
     * </p>
     * <p>
     * "recordWeightFieldName": "F2",
     * </p>
     * <p>
     * "targetFieldName": "F3",
     * </p>
     * <p>
     * "dataFormat": "CSV",
     * </p>
     * <p>
     * "dataFileContainsHeader": true,
     * </p>
     * <p>
     * "attributes": [
     * </p>
     * <p>
     * { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName": "F3",
     * "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5", "fieldType":
     * "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7", "fieldType":
     * "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
     * </p>
     * <p>
     * "excludedVariableNames": [ "F6" ] }
     * </p>
     * <?oxy_insert_end>
     * 
     * @return A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The
     *         <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced in
     *         the <code>DataSource</code>.</p>
     *         <p>
     *         A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
     *         </p>
     *         <p>
     *         Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
     *         <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following
     *         format to define your <code>DataSchema</code>.
     *         </p>
     *         <p>
     *         { "version": "1.0",
     *         </p>
     *         <p>
     *         "recordAnnotationFieldName": "F1",
     *         </p>
     *         <p>
     *         "recordWeightFieldName": "F2",
     *         </p>
     *         <p>
     *         "targetFieldName": "F3",
     *         </p>
     *         <p>
     *         "dataFormat": "CSV",
     *         </p>
     *         <p>
     *         "dataFileContainsHeader": true,
     *         </p>
     *         <p>
     *         "attributes": [
     *         </p>
     *         <p>
     *         { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName":
     *         "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5",
     *         "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7",
     *         "fieldType": "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
     *         </p>
     *         <p>
     *         "excludedVariableNames": [ "F6" ] }
     *         </p>
     */
    public String dataSchema() {
        return dataSchema;
    }

    /**
     * <p>
     * The Amazon S3 location of the <code>DataSchema</code>.
     * </p>
     * 
     * @return The Amazon S3 location of the <code>DataSchema</code>.
     */
    public String dataSchemaUri() {
        return dataSchemaUri;
    }

    /**
     * <p>
     * The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2) instance to
     * carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information, see <a
     * href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a> for
     * data pipelines.
     * </p>
     * 
     * @return The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2)
     *         instance to carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information, see
     *         <a href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role
     *         templates</a> for data pipelines.
     */
    public String resourceRole() {
        return resourceRole;
    }

    /**
     * <p>
     * The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the copy task
     * from Amazon RDS to Amazon S3. For more information, see <a
     * href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a> for
     * data pipelines.
     * </p>
     * 
     * @return The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the
     *         copy task from Amazon RDS to Amazon S3. For more information, see <a
     *         href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a>
     *         for data pipelines.
     */
    public String serviceRole() {
        return serviceRole;
    }

    /**
     * <p>
     * The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data Pipeline to carry
     * out the copy task from Amazon RDS to Amazon S3.
     * </p>
     * 
     * @return The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data Pipeline
     *         to carry out the copy task from Amazon RDS to Amazon S3.
     */
    public String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are appropriate
     * ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data Pipeline to carry out
     * the copy operation from Amazon RDS to an Amazon S3 task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are
     *         appropriate ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data
     *         Pipeline to carry out the copy operation from Amazon RDS to an Amazon S3 task.
     */
    public List<String> securityGroupIds() {
        return securityGroupIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((databaseInformation() == null) ? 0 : databaseInformation().hashCode());
        hashCode = 31 * hashCode + ((selectSqlQuery() == null) ? 0 : selectSqlQuery().hashCode());
        hashCode = 31 * hashCode + ((databaseCredentials() == null) ? 0 : databaseCredentials().hashCode());
        hashCode = 31 * hashCode + ((s3StagingLocation() == null) ? 0 : s3StagingLocation().hashCode());
        hashCode = 31 * hashCode + ((dataRearrangement() == null) ? 0 : dataRearrangement().hashCode());
        hashCode = 31 * hashCode + ((dataSchema() == null) ? 0 : dataSchema().hashCode());
        hashCode = 31 * hashCode + ((dataSchemaUri() == null) ? 0 : dataSchemaUri().hashCode());
        hashCode = 31 * hashCode + ((resourceRole() == null) ? 0 : resourceRole().hashCode());
        hashCode = 31 * hashCode + ((serviceRole() == null) ? 0 : serviceRole().hashCode());
        hashCode = 31 * hashCode + ((subnetId() == null) ? 0 : subnetId().hashCode());
        hashCode = 31 * hashCode + ((securityGroupIds() == null) ? 0 : securityGroupIds().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RDSDataSpec)) {
            return false;
        }
        RDSDataSpec other = (RDSDataSpec) obj;
        if (other.databaseInformation() == null ^ this.databaseInformation() == null) {
            return false;
        }
        if (other.databaseInformation() != null && !other.databaseInformation().equals(this.databaseInformation())) {
            return false;
        }
        if (other.selectSqlQuery() == null ^ this.selectSqlQuery() == null) {
            return false;
        }
        if (other.selectSqlQuery() != null && !other.selectSqlQuery().equals(this.selectSqlQuery())) {
            return false;
        }
        if (other.databaseCredentials() == null ^ this.databaseCredentials() == null) {
            return false;
        }
        if (other.databaseCredentials() != null && !other.databaseCredentials().equals(this.databaseCredentials())) {
            return false;
        }
        if (other.s3StagingLocation() == null ^ this.s3StagingLocation() == null) {
            return false;
        }
        if (other.s3StagingLocation() != null && !other.s3StagingLocation().equals(this.s3StagingLocation())) {
            return false;
        }
        if (other.dataRearrangement() == null ^ this.dataRearrangement() == null) {
            return false;
        }
        if (other.dataRearrangement() != null && !other.dataRearrangement().equals(this.dataRearrangement())) {
            return false;
        }
        if (other.dataSchema() == null ^ this.dataSchema() == null) {
            return false;
        }
        if (other.dataSchema() != null && !other.dataSchema().equals(this.dataSchema())) {
            return false;
        }
        if (other.dataSchemaUri() == null ^ this.dataSchemaUri() == null) {
            return false;
        }
        if (other.dataSchemaUri() != null && !other.dataSchemaUri().equals(this.dataSchemaUri())) {
            return false;
        }
        if (other.resourceRole() == null ^ this.resourceRole() == null) {
            return false;
        }
        if (other.resourceRole() != null && !other.resourceRole().equals(this.resourceRole())) {
            return false;
        }
        if (other.serviceRole() == null ^ this.serviceRole() == null) {
            return false;
        }
        if (other.serviceRole() != null && !other.serviceRole().equals(this.serviceRole())) {
            return false;
        }
        if (other.subnetId() == null ^ this.subnetId() == null) {
            return false;
        }
        if (other.subnetId() != null && !other.subnetId().equals(this.subnetId())) {
            return false;
        }
        if (other.securityGroupIds() == null ^ this.securityGroupIds() == null) {
            return false;
        }
        if (other.securityGroupIds() != null && !other.securityGroupIds().equals(this.securityGroupIds())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (databaseInformation() != null) {
            sb.append("DatabaseInformation: ").append(databaseInformation()).append(",");
        }
        if (selectSqlQuery() != null) {
            sb.append("SelectSqlQuery: ").append(selectSqlQuery()).append(",");
        }
        if (databaseCredentials() != null) {
            sb.append("DatabaseCredentials: ").append(databaseCredentials()).append(",");
        }
        if (s3StagingLocation() != null) {
            sb.append("S3StagingLocation: ").append(s3StagingLocation()).append(",");
        }
        if (dataRearrangement() != null) {
            sb.append("DataRearrangement: ").append(dataRearrangement()).append(",");
        }
        if (dataSchema() != null) {
            sb.append("DataSchema: ").append(dataSchema()).append(",");
        }
        if (dataSchemaUri() != null) {
            sb.append("DataSchemaUri: ").append(dataSchemaUri()).append(",");
        }
        if (resourceRole() != null) {
            sb.append("ResourceRole: ").append(resourceRole()).append(",");
        }
        if (serviceRole() != null) {
            sb.append("ServiceRole: ").append(serviceRole()).append(",");
        }
        if (subnetId() != null) {
            sb.append("SubnetId: ").append(subnetId()).append(",");
        }
        if (securityGroupIds() != null) {
            sb.append("SecurityGroupIds: ").append(securityGroupIds()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DatabaseInformation":
            return Optional.of(clazz.cast(databaseInformation()));
        case "SelectSqlQuery":
            return Optional.of(clazz.cast(selectSqlQuery()));
        case "DatabaseCredentials":
            return Optional.of(clazz.cast(databaseCredentials()));
        case "S3StagingLocation":
            return Optional.of(clazz.cast(s3StagingLocation()));
        case "DataRearrangement":
            return Optional.of(clazz.cast(dataRearrangement()));
        case "DataSchema":
            return Optional.of(clazz.cast(dataSchema()));
        case "DataSchemaUri":
            return Optional.of(clazz.cast(dataSchemaUri()));
        case "ResourceRole":
            return Optional.of(clazz.cast(resourceRole()));
        case "ServiceRole":
            return Optional.of(clazz.cast(serviceRole()));
        case "SubnetId":
            return Optional.of(clazz.cast(subnetId()));
        case "SecurityGroupIds":
            return Optional.of(clazz.cast(securityGroupIds()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        RDSDataSpecMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, RDSDataSpec> {
        /**
         * <p>
         * Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
         * </p>
         * 
         * @param databaseInformation
         *        Describes the <code>DatabaseName</code> and <code>InstanceIdentifier</code> of an Amazon RDS database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseInformation(RDSDatabase databaseInformation);

        /**
         * <p>
         * The query that is used to retrieve the observation data for the <code>DataSource</code>.
         * </p>
         * 
         * @param selectSqlQuery
         *        The query that is used to retrieve the observation data for the <code>DataSource</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectSqlQuery(String selectSqlQuery);

        /**
         * <p>
         * The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS database.
         * </p>
         * 
         * @param databaseCredentials
         *        The AWS Identity and Access Management (IAM) credentials that are used connect to the Amazon RDS
         *        database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseCredentials(RDSDatabaseCredentials databaseCredentials);

        /**
         * <p>
         * The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
         * <code>SelectSqlQuery</code> is stored in this location.
         * </p>
         * 
         * @param s3StagingLocation
         *        The Amazon S3 location for staging Amazon RDS data. The data retrieved from Amazon RDS using
         *        <code>SelectSqlQuery</code> is stored in this location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3StagingLocation(String s3StagingLocation);

        /**
         * <p>
         * A JSON string that represents the splitting and rearrangement processing to be applied to a
         * <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the input
         * data is used to create the <code>Datasource</code>.
         * </p>
         * <p>
         * There are multiple parameters that control what data is used to create a datasource:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b><code>percentBegin</code></b>
         * </p>
         * <p>
         * Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
         * Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes
         * all of the data when creating the datasource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b><code>percentEnd</code></b>
         * </p>
         * <p>
         * Use <code>percentEnd</code> to indicate the end of the range of the data used to create the Datasource. If
         * you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML includes all of the data
         * when creating the datasource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b><code>complement</code></b>
         * </p>
         * <p>
         * The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the range
         * of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The <code>complement</code>
         * parameter is useful if you need to create complementary datasources for training and evaluation. To create a
         * complementary datasource, use the same values for <code>percentBegin</code> and <code>percentEnd</code>,
         * along with the <code>complement</code> parameter.
         * </p>
         * <p>
         * For example, the following two datasources do not share any data, and can be used to train and evaluate a
         * model. The first datasource has 25 percent of the data, and the second one has 75 percent of the data.
         * </p>
         * <p>
         * Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
         * </p>
         * <p>
         * Datasource for training: <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <b><code>strategy</code></b>
         * </p>
         * <p>
         * To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
         * </p>
         * <p>
         * The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that Amazon ML
         * takes all of the data records between the <code>percentBegin</code> and <code>percentEnd</code> parameters
         * for the datasource, in the order that the records appear in the input data.
         * </p>
         * <p>
         * The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training and
         * evaluation datasources:
         * </p>
         * <p>
         * Datasource for evaluation:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
         * </p>
         * <p>
         * Datasource for training:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
         * </p>
         * <p>
         * To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
         * parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that is used
         * as the seed value for the random data splitting (for example, you can use the S3 path to your data as the
         * random seed string). If you choose the random split strategy, Amazon ML assigns each row of data a
         * pseudo-random number between 0 and 100, and then selects the rows that have an assigned number between
         * <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are assigned using both the
         * input seed string value and the byte offset as a seed, so changing the data results in a different split. Any
         * existing ordering is preserved. The random splitting strategy ensures that variables in the training and
         * evaluation data are distributed similarly. It is useful in the cases where the input data may have an
         * implicit sort order, which would otherwise result in training and evaluation datasources containing
         * non-similar data records.
         * </p>
         * <p>
         * The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered training and
         * evaluation datasources:
         * </p>
         * <p>
         * Datasource for evaluation:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
         * </p>
         * <p>
         * Datasource for training:
         * <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param dataRearrangement
         *        A JSON string that represents the splitting and rearrangement processing to be applied to a
         *        <code>DataSource</code>. If the <code>DataRearrangement</code> parameter is not provided, all of the
         *        input data is used to create the <code>Datasource</code>.</p>
         *        <p>
         *        There are multiple parameters that control what data is used to create a datasource:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b><code>percentBegin</code></b>
         *        </p>
         *        <p>
         *        Use <code>percentBegin</code> to indicate the beginning of the range of the data used to create the
         *        Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
         *        includes all of the data when creating the datasource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b><code>percentEnd</code></b>
         *        </p>
         *        <p>
         *        Use <code>percentEnd</code> to indicate the end of the range of the data used to create the
         *        Datasource. If you do not include <code>percentBegin</code> and <code>percentEnd</code>, Amazon ML
         *        includes all of the data when creating the datasource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b><code>complement</code></b>
         *        </p>
         *        <p>
         *        The <code>complement</code> parameter instructs Amazon ML to use the data that is not included in the
         *        range of <code>percentBegin</code> to <code>percentEnd</code> to create a datasource. The
         *        <code>complement</code> parameter is useful if you need to create complementary datasources for
         *        training and evaluation. To create a complementary datasource, use the same values for
         *        <code>percentBegin</code> and <code>percentEnd</code>, along with the <code>complement</code>
         *        parameter.
         *        </p>
         *        <p>
         *        For example, the following two datasources do not share any data, and can be used to train and
         *        evaluate a model. The first datasource has 25 percent of the data, and the second one has 75 percent
         *        of the data.
         *        </p>
         *        <p>
         *        Datasource for evaluation: <code>{"splitting":{"percentBegin":0, "percentEnd":25}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":0, "percentEnd":25, "complement":"true"}}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b><code>strategy</code></b>
         *        </p>
         *        <p>
         *        To change how Amazon ML splits the data for a datasource, use the <code>strategy</code> parameter.
         *        </p>
         *        <p>
         *        The default value for the <code>strategy</code> parameter is <code>sequential</code>, meaning that
         *        Amazon ML takes all of the data records between the <code>percentBegin</code> and
         *        <code>percentEnd</code> parameters for the datasource, in the order that the records appear in the
         *        input data.
         *        </p>
         *        <p>
         *        The following two <code>DataRearrangement</code> lines are examples of sequentially ordered training
         *        and evaluation datasources:
         *        </p>
         *        <p>
         *        Datasource for evaluation:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential"}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"sequential", "complement":"true"}}</code>
         *        </p>
         *        <p>
         *        To randomly split the input data into the proportions indicated by the percentBegin and percentEnd
         *        parameters, set the <code>strategy</code> parameter to <code>random</code> and provide a string that
         *        is used as the seed value for the random data splitting (for example, you can use the S3 path to your
         *        data as the random seed string). If you choose the random split strategy, Amazon ML assigns each row
         *        of data a pseudo-random number between 0 and 100, and then selects the rows that have an assigned
         *        number between <code>percentBegin</code> and <code>percentEnd</code>. Pseudo-random numbers are
         *        assigned using both the input seed string value and the byte offset as a seed, so changing the data
         *        results in a different split. Any existing ordering is preserved. The random splitting strategy
         *        ensures that variables in the training and evaluation data are distributed similarly. It is useful in
         *        the cases where the input data may have an implicit sort order, which would otherwise result in
         *        training and evaluation datasources containing non-similar data records.
         *        </p>
         *        <p>
         *        The following two <code>DataRearrangement</code> lines are examples of non-sequentially ordered
         *        training and evaluation datasources:
         *        </p>
         *        <p>
         *        Datasource for evaluation:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv"}}</code>
         *        </p>
         *        <p>
         *        Datasource for training:
         *        <code>{"splitting":{"percentBegin":70, "percentEnd":100, "strategy":"random", "randomSeed"="s3://my_s3_path/bucket/file.csv", "complement":"true"}}</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRearrangement(String dataRearrangement);

        /**
         * <p>
         * A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The
         * <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced in the
         * <code>DataSource</code>.
         * </p>
         * <p>
         * A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
         * </p>
         * <p>
         * Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
         * <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following format
         * to define your <code>DataSchema</code>.
         * </p>
         * <p>
         * { "version": "1.0",
         * </p>
         * <p>
         * "recordAnnotationFieldName": "F1",
         * </p>
         * <p>
         * "recordWeightFieldName": "F2",
         * </p>
         * <p>
         * "targetFieldName": "F3",
         * </p>
         * <p>
         * "dataFormat": "CSV",
         * </p>
         * <p>
         * "dataFileContainsHeader": true,
         * </p>
         * <p>
         * "attributes": [
         * </p>
         * <p>
         * { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, { "fieldName":
         * "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, { "fieldName": "F5",
         * "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, { "fieldName": "F7", "fieldType":
         * "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType": "WEIGHTED_STRING_SEQUENCE" } ],
         * </p>
         * <p>
         * "excludedVariableNames": [ "F6" ] }
         * </p>
         * <?oxy_insert_end>
         * 
         * @param dataSchema
         *        A JSON string that represents the schema for an Amazon RDS <code>DataSource</code>. The
         *        <code>DataSchema</code> defines the structure of the observation data in the data file(s) referenced
         *        in the <code>DataSource</code>.</p>
         *        <p>
         *        A <code>DataSchema</code> is not required if you specify a <code>DataSchemaUri</code>
         *        </p>
         *        <p>
         *        Define your <code>DataSchema</code> as a series of key-value pairs. <code>attributes</code> and
         *        <code>excludedVariableNames</code> have an array of key-value pairs for their value. Use the following
         *        format to define your <code>DataSchema</code>.
         *        </p>
         *        <p>
         *        { "version": "1.0",
         *        </p>
         *        <p>
         *        "recordAnnotationFieldName": "F1",
         *        </p>
         *        <p>
         *        "recordWeightFieldName": "F2",
         *        </p>
         *        <p>
         *        "targetFieldName": "F3",
         *        </p>
         *        <p>
         *        "dataFormat": "CSV",
         *        </p>
         *        <p>
         *        "dataFileContainsHeader": true,
         *        </p>
         *        <p>
         *        "attributes": [
         *        </p>
         *        <p>
         *        { "fieldName": "F1", "fieldType": "TEXT" }, { "fieldName": "F2", "fieldType": "NUMERIC" }, {
         *        "fieldName": "F3", "fieldType": "CATEGORICAL" }, { "fieldName": "F4", "fieldType": "NUMERIC" }, {
         *        "fieldName": "F5", "fieldType": "CATEGORICAL" }, { "fieldName": "F6", "fieldType": "TEXT" }, {
         *        "fieldName": "F7", "fieldType": "WEIGHTED_INT_SEQUENCE" }, { "fieldName": "F8", "fieldType":
         *        "WEIGHTED_STRING_SEQUENCE" } ],
         *        </p>
         *        <p>
         *        "excludedVariableNames": [ "F6" ] }
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSchema(String dataSchema);

        /**
         * <p>
         * The Amazon S3 location of the <code>DataSchema</code>.
         * </p>
         * 
         * @param dataSchemaUri
         *        The Amazon S3 location of the <code>DataSchema</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSchemaUri(String dataSchemaUri);

        /**
         * <p>
         * The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2) instance
         * to carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information, see <a
         * href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a> for
         * data pipelines.
         * </p>
         * 
         * @param resourceRole
         *        The role (DataPipelineDefaultResourceRole) assumed by an Amazon Elastic Compute Cloud (Amazon EC2)
         *        instance to carry out the copy operation from Amazon RDS to an Amazon S3 task. For more information,
         *        see <a href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role
         *        templates</a> for data pipelines.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRole(String resourceRole);

        /**
         * <p>
         * The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the copy
         * task from Amazon RDS to Amazon S3. For more information, see <a
         * href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role templates</a> for
         * data pipelines.
         * </p>
         * 
         * @param serviceRole
         *        The role (DataPipelineDefaultRole) assumed by AWS Data Pipeline service to monitor the progress of the
         *        copy task from Amazon RDS to Amazon S3. For more information, see <a
         *        href="http://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-roles.html">Role
         *        templates</a> for data pipelines.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data Pipeline to
         * carry out the copy task from Amazon RDS to Amazon S3.
         * </p>
         * 
         * @param subnetId
         *        The subnet ID to be used to access a VPC-based RDS DB instance. This attribute is used by Data
         *        Pipeline to carry out the copy task from Amazon RDS to Amazon S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are appropriate
         * ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data Pipeline to carry
         * out the copy operation from Amazon RDS to an Amazon S3 task.
         * </p>
         * 
         * @param securityGroupIds
         *        The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are
         *        appropriate ingress rules set up to allow access to the RDS DB instance. This attribute is used by
         *        Data Pipeline to carry out the copy operation from Amazon RDS to an Amazon S3 task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are appropriate
         * ingress rules set up to allow access to the RDS DB instance. This attribute is used by Data Pipeline to carry
         * out the copy operation from Amazon RDS to an Amazon S3 task.
         * </p>
         * 
         * @param securityGroupIds
         *        The security group IDs to be used to access a VPC-based RDS DB instance. Ensure that there are
         *        appropriate ingress rules set up to allow access to the RDS DB instance. This attribute is used by
         *        Data Pipeline to carry out the copy operation from Amazon RDS to an Amazon S3 task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);
    }

    static final class BuilderImpl implements Builder {
        private RDSDatabase databaseInformation;

        private String selectSqlQuery;

        private RDSDatabaseCredentials databaseCredentials;

        private String s3StagingLocation;

        private String dataRearrangement;

        private String dataSchema;

        private String dataSchemaUri;

        private String resourceRole;

        private String serviceRole;

        private String subnetId;

        private List<String> securityGroupIds;

        private BuilderImpl() {
        }

        private BuilderImpl(RDSDataSpec model) {
            databaseInformation(model.databaseInformation);
            selectSqlQuery(model.selectSqlQuery);
            databaseCredentials(model.databaseCredentials);
            s3StagingLocation(model.s3StagingLocation);
            dataRearrangement(model.dataRearrangement);
            dataSchema(model.dataSchema);
            dataSchemaUri(model.dataSchemaUri);
            resourceRole(model.resourceRole);
            serviceRole(model.serviceRole);
            subnetId(model.subnetId);
            securityGroupIds(model.securityGroupIds);
        }

        public final RDSDatabase.Builder getDatabaseInformation() {
            return databaseInformation != null ? databaseInformation.toBuilder() : null;
        }

        @Override
        public final Builder databaseInformation(RDSDatabase databaseInformation) {
            this.databaseInformation = databaseInformation;
            return this;
        }

        public final void setDatabaseInformation(RDSDatabase.BuilderImpl databaseInformation) {
            this.databaseInformation = databaseInformation != null ? databaseInformation.build() : null;
        }

        public final String getSelectSqlQuery() {
            return selectSqlQuery;
        }

        @Override
        public final Builder selectSqlQuery(String selectSqlQuery) {
            this.selectSqlQuery = selectSqlQuery;
            return this;
        }

        public final void setSelectSqlQuery(String selectSqlQuery) {
            this.selectSqlQuery = selectSqlQuery;
        }

        public final RDSDatabaseCredentials.Builder getDatabaseCredentials() {
            return databaseCredentials != null ? databaseCredentials.toBuilder() : null;
        }

        @Override
        public final Builder databaseCredentials(RDSDatabaseCredentials databaseCredentials) {
            this.databaseCredentials = databaseCredentials;
            return this;
        }

        public final void setDatabaseCredentials(RDSDatabaseCredentials.BuilderImpl databaseCredentials) {
            this.databaseCredentials = databaseCredentials != null ? databaseCredentials.build() : null;
        }

        public final String getS3StagingLocation() {
            return s3StagingLocation;
        }

        @Override
        public final Builder s3StagingLocation(String s3StagingLocation) {
            this.s3StagingLocation = s3StagingLocation;
            return this;
        }

        public final void setS3StagingLocation(String s3StagingLocation) {
            this.s3StagingLocation = s3StagingLocation;
        }

        public final String getDataRearrangement() {
            return dataRearrangement;
        }

        @Override
        public final Builder dataRearrangement(String dataRearrangement) {
            this.dataRearrangement = dataRearrangement;
            return this;
        }

        public final void setDataRearrangement(String dataRearrangement) {
            this.dataRearrangement = dataRearrangement;
        }

        public final String getDataSchema() {
            return dataSchema;
        }

        @Override
        public final Builder dataSchema(String dataSchema) {
            this.dataSchema = dataSchema;
            return this;
        }

        public final void setDataSchema(String dataSchema) {
            this.dataSchema = dataSchema;
        }

        public final String getDataSchemaUri() {
            return dataSchemaUri;
        }

        @Override
        public final Builder dataSchemaUri(String dataSchemaUri) {
            this.dataSchemaUri = dataSchemaUri;
            return this;
        }

        public final void setDataSchemaUri(String dataSchemaUri) {
            this.dataSchemaUri = dataSchemaUri;
        }

        public final String getResourceRole() {
            return resourceRole;
        }

        @Override
        public final Builder resourceRole(String resourceRole) {
            this.resourceRole = resourceRole;
            return this;
        }

        public final void setResourceRole(String resourceRole) {
            this.resourceRole = resourceRole;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        public final Collection<String> getSecurityGroupIds() {
            return securityGroupIds;
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = EDPSecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = EDPSecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public RDSDataSpec build() {
            return new RDSDataSpec(this);
        }
    }
}
