/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.services.machinelearning.transform.PredictionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The output from a <code>Predict</code> operation:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Details</code> - Contains the following attributes:
 * <code>DetailsAttributes.PREDICTIVE_MODEL_TYPE - REGRESSION | BINARY | MULTICLASS</code>
 * <code>DetailsAttributes.ALGORITHM - SGD</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>PredictedLabel</code> - Present for either a <code>BINARY</code> or <code>MULTICLASS</code>
 * <code>MLModel</code> request.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>PredictedScores</code> - Contains the raw classification score corresponding to each label.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>PredictedValue</code> - Present for a <code>REGRESSION</code> <code>MLModel</code> request.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public class Prediction implements StructuredPojo, ToCopyableBuilder<Prediction.Builder, Prediction> {
    private final String predictedLabel;

    private final Float predictedValue;

    private final Map<String, Float> predictedScores;

    private final Map<String, String> details;

    private Prediction(BuilderImpl builder) {
        this.predictedLabel = builder.predictedLabel;
        this.predictedValue = builder.predictedValue;
        this.predictedScores = builder.predictedScores;
        this.details = builder.details;
    }

    /**
     * <p>
     * The prediction label for either a <code>BINARY</code> or <code>MULTICLASS</code> <code>MLModel</code>.
     * </p>
     * 
     * @return The prediction label for either a <code>BINARY</code> or <code>MULTICLASS</code> <code>MLModel</code>.
     */
    public String predictedLabel() {
        return predictedLabel;
    }

    /**
     * The prediction value for <code>REGRESSION</code> <code>MLModel</code>.
     * 
     * @return The prediction value for <code>REGRESSION</code> <code>MLModel</code>.
     */
    public Float predictedValue() {
        return predictedValue;
    }

    /**
     * Returns the value of the PredictedScores property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The value of the PredictedScores property for this object.
     */
    public Map<String, Float> predictedScores() {
        return predictedScores;
    }

    /**
     * Returns the value of the Details property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The value of the Details property for this object.
     */
    public Map<DetailsAttributes, String> details() {
        return TypeConverter.convert(details, DetailsAttributes::fromValue, Function.identity(),
                (k, v) -> !Objects.equals(k, DetailsAttributes.UNKNOWN_TO_SDK_VERSION));
    }

    /**
     * Returns the value of the Details property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The value of the Details property for this object.
     */
    public Map<String, String> detailsAsStrings() {
        return details;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(predictedLabel());
        hashCode = 31 * hashCode + Objects.hashCode(predictedValue());
        hashCode = 31 * hashCode + Objects.hashCode(predictedScores());
        hashCode = 31 * hashCode + Objects.hashCode(detailsAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Prediction)) {
            return false;
        }
        Prediction other = (Prediction) obj;
        return Objects.equals(predictedLabel(), other.predictedLabel())
                && Objects.equals(predictedValue(), other.predictedValue())
                && Objects.equals(predictedScores(), other.predictedScores())
                && Objects.equals(detailsAsStrings(), other.detailsAsStrings());
    }

    @Override
    public String toString() {
        return ToString.builder("Prediction").add("PredictedLabel", predictedLabel()).add("PredictedValue", predictedValue())
                .add("PredictedScores", predictedScores()).add("Details", detailsAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "predictedLabel":
            return Optional.of(clazz.cast(predictedLabel()));
        case "predictedValue":
            return Optional.of(clazz.cast(predictedValue()));
        case "predictedScores":
            return Optional.of(clazz.cast(predictedScores()));
        case "details":
            return Optional.of(clazz.cast(detailsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        PredictionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Prediction> {
        /**
         * <p>
         * The prediction label for either a <code>BINARY</code> or <code>MULTICLASS</code> <code>MLModel</code>.
         * </p>
         * 
         * @param predictedLabel
         *        The prediction label for either a <code>BINARY</code> or <code>MULTICLASS</code> <code>MLModel</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predictedLabel(String predictedLabel);

        /**
         * The prediction value for <code>REGRESSION</code> <code>MLModel</code>.
         * 
         * @param predictedValue
         *        The prediction value for <code>REGRESSION</code> <code>MLModel</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predictedValue(Float predictedValue);

        /**
         * Sets the value of the PredictedScores property for this object.
         *
         * @param predictedScores
         *        The new value for the PredictedScores property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predictedScores(Map<String, Float> predictedScores);

        /**
         * Sets the value of the Details property for this object.
         *
         * @param details
         *        The new value for the Details property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(Map<String, String> details);
    }

    static final class BuilderImpl implements Builder {
        private String predictedLabel;

        private Float predictedValue;

        private Map<String, Float> predictedScores;

        private Map<String, String> details;

        private BuilderImpl() {
        }

        private BuilderImpl(Prediction model) {
            predictedLabel(model.predictedLabel);
            predictedValue(model.predictedValue);
            predictedScores(model.predictedScores);
            details(model.details);
        }

        public final String getPredictedLabel() {
            return predictedLabel;
        }

        @Override
        public final Builder predictedLabel(String predictedLabel) {
            this.predictedLabel = predictedLabel;
            return this;
        }

        public final void setPredictedLabel(String predictedLabel) {
            this.predictedLabel = predictedLabel;
        }

        public final Float getPredictedValue() {
            return predictedValue;
        }

        @Override
        public final Builder predictedValue(Float predictedValue) {
            this.predictedValue = predictedValue;
            return this;
        }

        public final void setPredictedValue(Float predictedValue) {
            this.predictedValue = predictedValue;
        }

        public final Map<String, Float> getPredictedScores() {
            return predictedScores;
        }

        @Override
        public final Builder predictedScores(Map<String, Float> predictedScores) {
            this.predictedScores = ScoreValuePerLabelMapCopier.copy(predictedScores);
            return this;
        }

        public final void setPredictedScores(Map<String, Float> predictedScores) {
            this.predictedScores = ScoreValuePerLabelMapCopier.copy(predictedScores);
        }

        public final Map<String, String> getDetails() {
            return details;
        }

        @Override
        public final Builder details(Map<String, String> details) {
            this.details = DetailsMapCopier.copy(details);
            return this;
        }

        public final void setDetails(Map<String, String> details) {
            this.details = DetailsMapCopier.copy(details);
        }

        @Override
        public Prediction build() {
            return new Prediction(this);
        }
    }
}
