/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.machinelearning.transform.EvaluationMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of <code>GetEvaluation</code> operation.
 * </p>
 * <p>
 * The content consists of the detailed metadata and data file information and the current status of the
 * <code>Evaluation</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Evaluation implements StructuredPojo, ToCopyableBuilder<Evaluation.Builder, Evaluation> {
    private final String evaluationId;

    private final String mlModelId;

    private final String evaluationDataSourceId;

    private final String inputDataLocationS3;

    private final String createdByIamUser;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private final String name;

    private final String status;

    private final PerformanceMetrics performanceMetrics;

    private final String message;

    private final Long computeTime;

    private final Instant finishedAt;

    private final Instant startedAt;

    private Evaluation(BuilderImpl builder) {
        this.evaluationId = builder.evaluationId;
        this.mlModelId = builder.mlModelId;
        this.evaluationDataSourceId = builder.evaluationDataSourceId;
        this.inputDataLocationS3 = builder.inputDataLocationS3;
        this.createdByIamUser = builder.createdByIamUser;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.name = builder.name;
        this.status = builder.status;
        this.performanceMetrics = builder.performanceMetrics;
        this.message = builder.message;
        this.computeTime = builder.computeTime;
        this.finishedAt = builder.finishedAt;
        this.startedAt = builder.startedAt;
    }

    /**
     * <p>
     * The ID that is assigned to the <code>Evaluation</code> at creation.
     * </p>
     * 
     * @return The ID that is assigned to the <code>Evaluation</code> at creation.
     */
    public String evaluationId() {
        return evaluationId;
    }

    /**
     * <p>
     * The ID of the <code>MLModel</code> that is the focus of the evaluation.
     * </p>
     * 
     * @return The ID of the <code>MLModel</code> that is the focus of the evaluation.
     */
    public String mlModelId() {
        return mlModelId;
    }

    /**
     * <p>
     * The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
     * </p>
     * 
     * @return The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
     */
    public String evaluationDataSourceId() {
        return evaluationDataSourceId;
    }

    /**
     * <p>
     * The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the evaluation.
     * </p>
     * 
     * @return The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the
     *         evaluation.
     */
    public String inputDataLocationS3() {
        return inputDataLocationS3;
    }

    /**
     * <p>
     * The AWS user account that invoked the evaluation. The account type can be either an AWS root account or an AWS
     * Identity and Access Management (IAM) user account.
     * </p>
     * 
     * @return The AWS user account that invoked the evaluation. The account type can be either an AWS root account or
     *         an AWS Identity and Access Management (IAM) user account.
     */
    public String createdByIamUser() {
        return createdByIamUser;
    }

    /**
     * <p>
     * The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
     * </p>
     * 
     * @return The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
     * </p>
     * 
     * @return The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
     */
    public Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * <p>
     * A user-supplied name or description of the <code>Evaluation</code>.
     * </p>
     * 
     * @return A user-supplied name or description of the <code>Evaluation</code>.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The status of the evaluation. This element can have one of the following values:
     * </p>
     * <ul>
     * <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     * <code>MLModel</code>.</li>
     * <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
     * <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is not
     * usable.</li>
     * <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
     * <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EntityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the evaluation. This element can have one of the following values:</p>
     *         <ul>
     *         <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     *         <code>MLModel</code>.</li>
     *         <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
     *         <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It
     *         is not usable.</li>
     *         <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
     *         <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
     * @see EntityStatus
     */
    public EntityStatus status() {
        return EntityStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the evaluation. This element can have one of the following values:
     * </p>
     * <ul>
     * <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     * <code>MLModel</code>.</li>
     * <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
     * <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is not
     * usable.</li>
     * <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
     * <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EntityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the evaluation. This element can have one of the following values:</p>
     *         <ul>
     *         <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     *         <code>MLModel</code>.</li>
     *         <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
     *         <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It
     *         is not usable.</li>
     *         <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
     *         <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
     * @see EntityStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
     * <code>DataSource</code>. One of the following metrics is returned, based on the type of the <code>MLModel</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to measure
     * performance. RMSE measures the difference between predicted and actual values for a single variable.
     * </p>
     * </li>
     * <li>
     * <p>
     * MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure performance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about performance metrics, please see the <a
     * href="http://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer Guide</a>.
     * </p>
     * 
     * @return Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
     *         <code>DataSource</code>. One of the following metrics is returned, based on the type of the
     *         <code>MLModel</code>: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
     *         performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to
     *         measure performance. RMSE measures the difference between predicted and actual values for a single
     *         variable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure
     *         performance.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about performance metrics, please see the <a
     *         href="http://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer Guide</a>.
     */
    public PerformanceMetrics performanceMetrics() {
        return performanceMetrics;
    }

    /**
     * <p>
     * A description of the most recent details about evaluating the <code>MLModel</code>.
     * </p>
     * 
     * @return A description of the most recent details about evaluating the <code>MLModel</code>.
     */
    public String message() {
        return message;
    }

    /**
     * Returns the value of the ComputeTime property for this object.
     * 
     * @return The value of the ComputeTime property for this object.
     */
    public Long computeTime() {
        return computeTime;
    }

    /**
     * Returns the value of the FinishedAt property for this object.
     * 
     * @return The value of the FinishedAt property for this object.
     */
    public Instant finishedAt() {
        return finishedAt;
    }

    /**
     * Returns the value of the StartedAt property for this object.
     * 
     * @return The value of the StartedAt property for this object.
     */
    public Instant startedAt() {
        return startedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(evaluationId());
        hashCode = 31 * hashCode + Objects.hashCode(mlModelId());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationDataSourceId());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataLocationS3());
        hashCode = 31 * hashCode + Objects.hashCode(createdByIamUser());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(performanceMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(computeTime());
        hashCode = 31 * hashCode + Objects.hashCode(finishedAt());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Evaluation)) {
            return false;
        }
        Evaluation other = (Evaluation) obj;
        return Objects.equals(evaluationId(), other.evaluationId()) && Objects.equals(mlModelId(), other.mlModelId())
                && Objects.equals(evaluationDataSourceId(), other.evaluationDataSourceId())
                && Objects.equals(inputDataLocationS3(), other.inputDataLocationS3())
                && Objects.equals(createdByIamUser(), other.createdByIamUser()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt()) && Objects.equals(name(), other.name())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(performanceMetrics(), other.performanceMetrics()) && Objects.equals(message(), other.message())
                && Objects.equals(computeTime(), other.computeTime()) && Objects.equals(finishedAt(), other.finishedAt())
                && Objects.equals(startedAt(), other.startedAt());
    }

    @Override
    public String toString() {
        return ToString.builder("Evaluation").add("EvaluationId", evaluationId()).add("MLModelId", mlModelId())
                .add("EvaluationDataSourceId", evaluationDataSourceId()).add("InputDataLocationS3", inputDataLocationS3())
                .add("CreatedByIamUser", createdByIamUser()).add("CreatedAt", createdAt()).add("LastUpdatedAt", lastUpdatedAt())
                .add("Name", name()).add("Status", statusAsString()).add("PerformanceMetrics", performanceMetrics())
                .add("Message", message()).add("ComputeTime", computeTime()).add("FinishedAt", finishedAt())
                .add("StartedAt", startedAt()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EvaluationId":
            return Optional.of(clazz.cast(evaluationId()));
        case "MLModelId":
            return Optional.of(clazz.cast(mlModelId()));
        case "EvaluationDataSourceId":
            return Optional.of(clazz.cast(evaluationDataSourceId()));
        case "InputDataLocationS3":
            return Optional.of(clazz.cast(inputDataLocationS3()));
        case "CreatedByIamUser":
            return Optional.of(clazz.cast(createdByIamUser()));
        case "CreatedAt":
            return Optional.of(clazz.cast(createdAt()));
        case "LastUpdatedAt":
            return Optional.of(clazz.cast(lastUpdatedAt()));
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "Status":
            return Optional.of(clazz.cast(statusAsString()));
        case "PerformanceMetrics":
            return Optional.of(clazz.cast(performanceMetrics()));
        case "Message":
            return Optional.of(clazz.cast(message()));
        case "ComputeTime":
            return Optional.of(clazz.cast(computeTime()));
        case "FinishedAt":
            return Optional.of(clazz.cast(finishedAt()));
        case "StartedAt":
            return Optional.of(clazz.cast(startedAt()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        EvaluationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Evaluation> {
        /**
         * <p>
         * The ID that is assigned to the <code>Evaluation</code> at creation.
         * </p>
         * 
         * @param evaluationId
         *        The ID that is assigned to the <code>Evaluation</code> at creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationId(String evaluationId);

        /**
         * <p>
         * The ID of the <code>MLModel</code> that is the focus of the evaluation.
         * </p>
         * 
         * @param mlModelId
         *        The ID of the <code>MLModel</code> that is the focus of the evaluation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mlModelId(String mlModelId);

        /**
         * <p>
         * The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
         * </p>
         * 
         * @param evaluationDataSourceId
         *        The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationDataSourceId(String evaluationDataSourceId);

        /**
         * <p>
         * The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the evaluation.
         * </p>
         * 
         * @param inputDataLocationS3
         *        The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the
         *        evaluation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataLocationS3(String inputDataLocationS3);

        /**
         * <p>
         * The AWS user account that invoked the evaluation. The account type can be either an AWS root account or an
         * AWS Identity and Access Management (IAM) user account.
         * </p>
         * 
         * @param createdByIamUser
         *        The AWS user account that invoked the evaluation. The account type can be either an AWS root account
         *        or an AWS Identity and Access Management (IAM) user account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdByIamUser(String createdByIamUser);

        /**
         * <p>
         * The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
         * </p>
         * 
         * @param createdAt
         *        The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * A user-supplied name or description of the <code>Evaluation</code>.
         * </p>
         * 
         * @param name
         *        A user-supplied name or description of the <code>Evaluation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The status of the evaluation. This element can have one of the following values:
         * </p>
         * <ul>
         * <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         * <code>MLModel</code>.</li>
         * <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
         * <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is
         * not usable.</li>
         * <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
         * <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
         * </ul>
         * 
         * @param status
         *        The status of the evaluation. This element can have one of the following values:</p>
         *        <ul>
         *        <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         *        <code>MLModel</code>.</li>
         *        <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
         *        <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion.
         *        It is not usable.</li>
         *        <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
         *        <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
         * @see EntityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the evaluation. This element can have one of the following values:
         * </p>
         * <ul>
         * <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         * <code>MLModel</code>.</li>
         * <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
         * <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is
         * not usable.</li>
         * <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
         * <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
         * </ul>
         * 
         * @param status
         *        The status of the evaluation. This element can have one of the following values:</p>
         *        <ul>
         *        <li> <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         *        <code>MLModel</code>.</li>
         *        <li> <code>INPROGRESS</code> - The evaluation is underway.</li>
         *        <li> <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion.
         *        It is not usable.</li>
         *        <li> <code>COMPLETED</code> - The evaluation process completed successfully.</li>
         *        <li> <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.</li>
         * @see EntityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityStatus
         */
        Builder status(EntityStatus status);

        /**
         * <p>
         * Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
         * <code>DataSource</code>. One of the following metrics is returned, based on the type of the
         * <code>MLModel</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
         * performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to measure
         * performance. RMSE measures the difference between predicted and actual values for a single variable.
         * </p>
         * </li>
         * <li>
         * <p>
         * MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure performance.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about performance metrics, please see the <a
         * href="http://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer Guide</a>.
         * </p>
         * 
         * @param performanceMetrics
         *        Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
         *        <code>DataSource</code>. One of the following metrics is returned, based on the type of the
         *        <code>MLModel</code>: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
         *        performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to
         *        measure performance. RMSE measures the difference between predicted and actual values for a single
         *        variable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure
         *        performance.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about performance metrics, please see the <a
         *        href="http://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceMetrics(PerformanceMetrics performanceMetrics);

        /**
         * <p>
         * Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
         * <code>DataSource</code>. One of the following metrics is returned, based on the type of the
         * <code>MLModel</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
         * performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to measure
         * performance. RMSE measures the difference between predicted and actual values for a single variable.
         * </p>
         * </li>
         * <li>
         * <p>
         * MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure performance.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about performance metrics, please see the <a
         * href="http://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link PerformanceMetrics.Builder} avoiding the need to
         * create one manually via {@link PerformanceMetrics#builder()}.
         *
         * When the {@link Consumer} completes, {@link PerformanceMetrics.Builder#build()} is called immediately and its
         * result is passed to {@link #performanceMetrics(PerformanceMetrics)}.
         * 
         * @param performanceMetrics
         *        a consumer that will call methods on {@link PerformanceMetrics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #performanceMetrics(PerformanceMetrics)
         */
        default Builder performanceMetrics(Consumer<PerformanceMetrics.Builder> performanceMetrics) {
            return performanceMetrics(PerformanceMetrics.builder().apply(performanceMetrics).build());
        }

        /**
         * <p>
         * A description of the most recent details about evaluating the <code>MLModel</code>.
         * </p>
         * 
         * @param message
         *        A description of the most recent details about evaluating the <code>MLModel</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * Sets the value of the ComputeTime property for this object.
         *
         * @param computeTime
         *        The new value for the ComputeTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeTime(Long computeTime);

        /**
         * Sets the value of the FinishedAt property for this object.
         *
         * @param finishedAt
         *        The new value for the FinishedAt property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder finishedAt(Instant finishedAt);

        /**
         * Sets the value of the StartedAt property for this object.
         *
         * @param startedAt
         *        The new value for the StartedAt property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);
    }

    static final class BuilderImpl implements Builder {
        private String evaluationId;

        private String mlModelId;

        private String evaluationDataSourceId;

        private String inputDataLocationS3;

        private String createdByIamUser;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private String name;

        private String status;

        private PerformanceMetrics performanceMetrics;

        private String message;

        private Long computeTime;

        private Instant finishedAt;

        private Instant startedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(Evaluation model) {
            evaluationId(model.evaluationId);
            mlModelId(model.mlModelId);
            evaluationDataSourceId(model.evaluationDataSourceId);
            inputDataLocationS3(model.inputDataLocationS3);
            createdByIamUser(model.createdByIamUser);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
            name(model.name);
            status(model.status);
            performanceMetrics(model.performanceMetrics);
            message(model.message);
            computeTime(model.computeTime);
            finishedAt(model.finishedAt);
            startedAt(model.startedAt);
        }

        public final String getEvaluationId() {
            return evaluationId;
        }

        @Override
        public final Builder evaluationId(String evaluationId) {
            this.evaluationId = evaluationId;
            return this;
        }

        public final void setEvaluationId(String evaluationId) {
            this.evaluationId = evaluationId;
        }

        public final String getMLModelId() {
            return mlModelId;
        }

        @Override
        public final Builder mlModelId(String mlModelId) {
            this.mlModelId = mlModelId;
            return this;
        }

        public final void setMLModelId(String mlModelId) {
            this.mlModelId = mlModelId;
        }

        public final String getEvaluationDataSourceId() {
            return evaluationDataSourceId;
        }

        @Override
        public final Builder evaluationDataSourceId(String evaluationDataSourceId) {
            this.evaluationDataSourceId = evaluationDataSourceId;
            return this;
        }

        public final void setEvaluationDataSourceId(String evaluationDataSourceId) {
            this.evaluationDataSourceId = evaluationDataSourceId;
        }

        public final String getInputDataLocationS3() {
            return inputDataLocationS3;
        }

        @Override
        public final Builder inputDataLocationS3(String inputDataLocationS3) {
            this.inputDataLocationS3 = inputDataLocationS3;
            return this;
        }

        public final void setInputDataLocationS3(String inputDataLocationS3) {
            this.inputDataLocationS3 = inputDataLocationS3;
        }

        public final String getCreatedByIamUser() {
            return createdByIamUser;
        }

        @Override
        public final Builder createdByIamUser(String createdByIamUser) {
            this.createdByIamUser = createdByIamUser;
            return this;
        }

        public final void setCreatedByIamUser(String createdByIamUser) {
            this.createdByIamUser = createdByIamUser;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(EntityStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final PerformanceMetrics.Builder getPerformanceMetrics() {
            return performanceMetrics != null ? performanceMetrics.toBuilder() : null;
        }

        @Override
        public final Builder performanceMetrics(PerformanceMetrics performanceMetrics) {
            this.performanceMetrics = performanceMetrics;
            return this;
        }

        public final void setPerformanceMetrics(PerformanceMetrics.BuilderImpl performanceMetrics) {
            this.performanceMetrics = performanceMetrics != null ? performanceMetrics.build() : null;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Long getComputeTime() {
            return computeTime;
        }

        @Override
        public final Builder computeTime(Long computeTime) {
            this.computeTime = computeTime;
            return this;
        }

        public final void setComputeTime(Long computeTime) {
            this.computeTime = computeTime;
        }

        public final Instant getFinishedAt() {
            return finishedAt;
        }

        @Override
        public final Builder finishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
            return this;
        }

        public final void setFinishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public Evaluation build() {
            return new Evaluation(this);
        }
    }
}
