/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutGeofenceRequest extends LocationRequest implements
        ToCopyableBuilder<PutGeofenceRequest.Builder, PutGeofenceRequest> {
    private static final SdkField<String> COLLECTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CollectionName").getter(getter(PutGeofenceRequest::collectionName))
            .setter(setter(Builder::collectionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("CollectionName").build()).build();

    private static final SdkField<String> GEOFENCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GeofenceId").getter(getter(PutGeofenceRequest::geofenceId)).setter(setter(Builder::geofenceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("GeofenceId").build()).build();

    private static final SdkField<Map<String, String>> GEOFENCE_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("GeofenceProperties")
            .getter(getter(PutGeofenceRequest::geofenceProperties))
            .setter(setter(Builder::geofenceProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeofenceProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<GeofenceGeometry> GEOMETRY_FIELD = SdkField
            .<GeofenceGeometry> builder(MarshallingType.SDK_POJO).memberName("Geometry")
            .getter(getter(PutGeofenceRequest::geometry)).setter(setter(Builder::geometry))
            .constructor(GeofenceGeometry::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geometry").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COLLECTION_NAME_FIELD,
            GEOFENCE_ID_FIELD, GEOFENCE_PROPERTIES_FIELD, GEOMETRY_FIELD));

    private final String collectionName;

    private final String geofenceId;

    private final Map<String, String> geofenceProperties;

    private final GeofenceGeometry geometry;

    private PutGeofenceRequest(BuilderImpl builder) {
        super(builder);
        this.collectionName = builder.collectionName;
        this.geofenceId = builder.geofenceId;
        this.geofenceProperties = builder.geofenceProperties;
        this.geometry = builder.geometry;
    }

    /**
     * <p>
     * The geofence collection to store the geofence in.
     * </p>
     * 
     * @return The geofence collection to store the geofence in.
     */
    public final String collectionName() {
        return collectionName;
    }

    /**
     * <p>
     * An identifier for the geofence. For example, <code>ExampleGeofence-1</code>.
     * </p>
     * 
     * @return An identifier for the geofence. For example, <code>ExampleGeofence-1</code>.
     */
    public final String geofenceId() {
        return geofenceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the GeofenceProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGeofenceProperties() {
        return geofenceProperties != null && !(geofenceProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Associates one of more properties with the geofence. A property is a key-value pair stored with the geofence and
     * added to any geofence event triggered with that geofence.
     * </p>
     * <p>
     * Format: <code>"key" : "value"</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGeofenceProperties} method.
     * </p>
     * 
     * @return Associates one of more properties with the geofence. A property is a key-value pair stored with the
     *         geofence and added to any geofence event triggered with that geofence.</p>
     *         <p>
     *         Format: <code>"key" : "value"</code>
     */
    public final Map<String, String> geofenceProperties() {
        return geofenceProperties;
    }

    /**
     * <p>
     * Contains the details to specify the position of the geofence. Can be either a polygon or a circle. Including both
     * will return a validation error.
     * </p>
     * <note>
     * <p>
     * Each <a href="https://docs.aws.amazon.com/location-geofences/latest/APIReference/API_GeofenceGeometry.html">
     * geofence polygon</a> can have a maximum of 1,000 vertices.
     * </p>
     * </note>
     * 
     * @return Contains the details to specify the position of the geofence. Can be either a polygon or a circle.
     *         Including both will return a validation error.</p> <note>
     *         <p>
     *         Each <a
     *         href="https://docs.aws.amazon.com/location-geofences/latest/APIReference/API_GeofenceGeometry.html">
     *         geofence polygon</a> can have a maximum of 1,000 vertices.
     *         </p>
     */
    public final GeofenceGeometry geometry() {
        return geometry;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(collectionName());
        hashCode = 31 * hashCode + Objects.hashCode(geofenceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasGeofenceProperties() ? geofenceProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(geometry());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutGeofenceRequest)) {
            return false;
        }
        PutGeofenceRequest other = (PutGeofenceRequest) obj;
        return Objects.equals(collectionName(), other.collectionName()) && Objects.equals(geofenceId(), other.geofenceId())
                && hasGeofenceProperties() == other.hasGeofenceProperties()
                && Objects.equals(geofenceProperties(), other.geofenceProperties())
                && Objects.equals(geometry(), other.geometry());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutGeofenceRequest").add("CollectionName", collectionName()).add("GeofenceId", geofenceId())
                .add("GeofenceProperties", geofenceProperties() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Geometry", geometry()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CollectionName":
            return Optional.ofNullable(clazz.cast(collectionName()));
        case "GeofenceId":
            return Optional.ofNullable(clazz.cast(geofenceId()));
        case "GeofenceProperties":
            return Optional.ofNullable(clazz.cast(geofenceProperties()));
        case "Geometry":
            return Optional.ofNullable(clazz.cast(geometry()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutGeofenceRequest, T> g) {
        return obj -> g.apply((PutGeofenceRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LocationRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutGeofenceRequest> {
        /**
         * <p>
         * The geofence collection to store the geofence in.
         * </p>
         * 
         * @param collectionName
         *        The geofence collection to store the geofence in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collectionName(String collectionName);

        /**
         * <p>
         * An identifier for the geofence. For example, <code>ExampleGeofence-1</code>.
         * </p>
         * 
         * @param geofenceId
         *        An identifier for the geofence. For example, <code>ExampleGeofence-1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geofenceId(String geofenceId);

        /**
         * <p>
         * Associates one of more properties with the geofence. A property is a key-value pair stored with the geofence
         * and added to any geofence event triggered with that geofence.
         * </p>
         * <p>
         * Format: <code>"key" : "value"</code>
         * </p>
         * 
         * @param geofenceProperties
         *        Associates one of more properties with the geofence. A property is a key-value pair stored with the
         *        geofence and added to any geofence event triggered with that geofence.</p>
         *        <p>
         *        Format: <code>"key" : "value"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geofenceProperties(Map<String, String> geofenceProperties);

        /**
         * <p>
         * Contains the details to specify the position of the geofence. Can be either a polygon or a circle. Including
         * both will return a validation error.
         * </p>
         * <note>
         * <p>
         * Each <a href="https://docs.aws.amazon.com/location-geofences/latest/APIReference/API_GeofenceGeometry.html">
         * geofence polygon</a> can have a maximum of 1,000 vertices.
         * </p>
         * </note>
         * 
         * @param geometry
         *        Contains the details to specify the position of the geofence. Can be either a polygon or a circle.
         *        Including both will return a validation error.</p> <note>
         *        <p>
         *        Each <a
         *        href="https://docs.aws.amazon.com/location-geofences/latest/APIReference/API_GeofenceGeometry.html">
         *        geofence polygon</a> can have a maximum of 1,000 vertices.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometry(GeofenceGeometry geometry);

        /**
         * <p>
         * Contains the details to specify the position of the geofence. Can be either a polygon or a circle. Including
         * both will return a validation error.
         * </p>
         * <note>
         * <p>
         * Each <a href="https://docs.aws.amazon.com/location-geofences/latest/APIReference/API_GeofenceGeometry.html">
         * geofence polygon</a> can have a maximum of 1,000 vertices.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link GeofenceGeometry.Builder}
         * avoiding the need to create one manually via {@link GeofenceGeometry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GeofenceGeometry.Builder#build()} is called immediately and its
         * result is passed to {@link #geometry(GeofenceGeometry)}.
         * 
         * @param geometry
         *        a consumer that will call methods on {@link GeofenceGeometry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geometry(GeofenceGeometry)
         */
        default Builder geometry(Consumer<GeofenceGeometry.Builder> geometry) {
            return geometry(GeofenceGeometry.builder().applyMutation(geometry).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LocationRequest.BuilderImpl implements Builder {
        private String collectionName;

        private String geofenceId;

        private Map<String, String> geofenceProperties = DefaultSdkAutoConstructMap.getInstance();

        private GeofenceGeometry geometry;

        private BuilderImpl() {
        }

        private BuilderImpl(PutGeofenceRequest model) {
            super(model);
            collectionName(model.collectionName);
            geofenceId(model.geofenceId);
            geofenceProperties(model.geofenceProperties);
            geometry(model.geometry);
        }

        public final String getCollectionName() {
            return collectionName;
        }

        public final void setCollectionName(String collectionName) {
            this.collectionName = collectionName;
        }

        @Override
        public final Builder collectionName(String collectionName) {
            this.collectionName = collectionName;
            return this;
        }

        public final String getGeofenceId() {
            return geofenceId;
        }

        public final void setGeofenceId(String geofenceId) {
            this.geofenceId = geofenceId;
        }

        @Override
        public final Builder geofenceId(String geofenceId) {
            this.geofenceId = geofenceId;
            return this;
        }

        public final Map<String, String> getGeofenceProperties() {
            if (geofenceProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return geofenceProperties;
        }

        public final void setGeofenceProperties(Map<String, String> geofenceProperties) {
            this.geofenceProperties = PropertyMapCopier.copy(geofenceProperties);
        }

        @Override
        public final Builder geofenceProperties(Map<String, String> geofenceProperties) {
            this.geofenceProperties = PropertyMapCopier.copy(geofenceProperties);
            return this;
        }

        public final GeofenceGeometry.Builder getGeometry() {
            return geometry != null ? geometry.toBuilder() : null;
        }

        public final void setGeometry(GeofenceGeometry.BuilderImpl geometry) {
            this.geometry = geometry != null ? geometry.build() : null;
        }

        @Override
        public final Builder geometry(GeofenceGeometry geometry) {
            this.geometry = geometry;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutGeofenceRequest build() {
            return new PutGeofenceRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
