/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.endpoints.internal;

import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.location.endpoints.LocationEndpointParams;
import software.amazon.awssdk.services.location.endpoints.LocationEndpointProvider;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class LocationResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return context.request();
        }
        LocationEndpointProvider provider = (LocationEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            Endpoint result = provider.resolveEndpoint(ruleParams(context, executionAttributes)).join();
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        context.request());
                if (hostPrefix.isPresent()) {
                    result = AwsEndpointProviderUtils.addHostPrefix(result, hostPrefix.get());
                }
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, result);
            return context.request();
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    private static LocationEndpointParams ruleParams(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        LocationEndpointParams.Builder builder = LocationEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), context.request());
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        return builder.build();
    }

    private static void setContextParams(LocationEndpointParams.Builder params, String operationName, SdkRequest request) {
    }

    private static void setStaticContextParams(LocationEndpointParams.Builder params, String operationName) {
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        switch (operationName) {
        case "AssociateTrackerConsumer": {
            return Optional.of("tracking.");
        }
        case "BatchDeleteDevicePositionHistory": {
            return Optional.of("tracking.");
        }
        case "BatchDeleteGeofence": {
            return Optional.of("geofencing.");
        }
        case "BatchEvaluateGeofences": {
            return Optional.of("geofencing.");
        }
        case "BatchGetDevicePosition": {
            return Optional.of("tracking.");
        }
        case "BatchPutGeofence": {
            return Optional.of("geofencing.");
        }
        case "BatchUpdateDevicePosition": {
            return Optional.of("tracking.");
        }
        case "CalculateRoute": {
            return Optional.of("routes.");
        }
        case "CalculateRouteMatrix": {
            return Optional.of("routes.");
        }
        case "CreateGeofenceCollection": {
            return Optional.of("geofencing.");
        }
        case "CreateMap": {
            return Optional.of("maps.");
        }
        case "CreatePlaceIndex": {
            return Optional.of("places.");
        }
        case "CreateRouteCalculator": {
            return Optional.of("routes.");
        }
        case "CreateTracker": {
            return Optional.of("tracking.");
        }
        case "DeleteGeofenceCollection": {
            return Optional.of("geofencing.");
        }
        case "DeleteMap": {
            return Optional.of("maps.");
        }
        case "DeletePlaceIndex": {
            return Optional.of("places.");
        }
        case "DeleteRouteCalculator": {
            return Optional.of("routes.");
        }
        case "DeleteTracker": {
            return Optional.of("tracking.");
        }
        case "DescribeGeofenceCollection": {
            return Optional.of("geofencing.");
        }
        case "DescribeMap": {
            return Optional.of("maps.");
        }
        case "DescribePlaceIndex": {
            return Optional.of("places.");
        }
        case "DescribeRouteCalculator": {
            return Optional.of("routes.");
        }
        case "DescribeTracker": {
            return Optional.of("tracking.");
        }
        case "DisassociateTrackerConsumer": {
            return Optional.of("tracking.");
        }
        case "GetDevicePosition": {
            return Optional.of("tracking.");
        }
        case "GetDevicePositionHistory": {
            return Optional.of("tracking.");
        }
        case "GetGeofence": {
            return Optional.of("geofencing.");
        }
        case "GetMapGlyphs": {
            return Optional.of("maps.");
        }
        case "GetMapSprites": {
            return Optional.of("maps.");
        }
        case "GetMapStyleDescriptor": {
            return Optional.of("maps.");
        }
        case "GetMapTile": {
            return Optional.of("maps.");
        }
        case "GetPlace": {
            return Optional.of("places.");
        }
        case "ListDevicePositions": {
            return Optional.of("tracking.");
        }
        case "ListGeofenceCollections": {
            return Optional.of("geofencing.");
        }
        case "ListGeofences": {
            return Optional.of("geofencing.");
        }
        case "ListMaps": {
            return Optional.of("maps.");
        }
        case "ListPlaceIndexes": {
            return Optional.of("places.");
        }
        case "ListRouteCalculators": {
            return Optional.of("routes.");
        }
        case "ListTagsForResource": {
            return Optional.of("metadata.");
        }
        case "ListTrackerConsumers": {
            return Optional.of("tracking.");
        }
        case "ListTrackers": {
            return Optional.of("tracking.");
        }
        case "PutGeofence": {
            return Optional.of("geofencing.");
        }
        case "SearchPlaceIndexForPosition": {
            return Optional.of("places.");
        }
        case "SearchPlaceIndexForSuggestions": {
            return Optional.of("places.");
        }
        case "SearchPlaceIndexForText": {
            return Optional.of("places.");
        }
        case "TagResource": {
            return Optional.of("metadata.");
        }
        case "UntagResource": {
            return Optional.of("metadata.");
        }
        case "UpdateGeofenceCollection": {
            return Optional.of("geofencing.");
        }
        case "UpdateMap": {
            return Optional.of("maps.");
        }
        case "UpdatePlaceIndex": {
            return Optional.of("places.");
        }
        case "UpdateRouteCalculator": {
            return Optional.of("routes.");
        }
        case "UpdateTracker": {
            return Optional.of("tracking.");
        }
        default:
            return Optional.empty();
        }
    }
}
