/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the device position details.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DevicePosition implements SdkPojo, Serializable, ToCopyableBuilder<DevicePosition.Builder, DevicePosition> {
    private static final SdkField<String> DEVICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeviceId").getter(getter(DevicePosition::deviceId)).setter(setter(Builder::deviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceId").build()).build();

    private static final SdkField<List<Double>> POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Position")
            .getter(getter(DevicePosition::position))
            .setter(setter(Builder::position))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Position").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> RECEIVED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReceivedTime")
            .getter(getter(DevicePosition::receivedTime))
            .setter(setter(Builder::receivedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReceivedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> SAMPLE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SampleTime")
            .getter(getter(DevicePosition::sampleTime))
            .setter(setter(Builder::sampleTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEVICE_ID_FIELD,
            POSITION_FIELD, RECEIVED_TIME_FIELD, SAMPLE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deviceId;

    private final List<Double> position;

    private final Instant receivedTime;

    private final Instant sampleTime;

    private DevicePosition(BuilderImpl builder) {
        this.deviceId = builder.deviceId;
        this.position = builder.position;
        this.receivedTime = builder.receivedTime;
        this.sampleTime = builder.sampleTime;
    }

    /**
     * <p>
     * The device whose position you retrieved.
     * </p>
     * 
     * @return The device whose position you retrieved.
     */
    public final String deviceId() {
        return deviceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Position property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPosition() {
        return position != null && !(position instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The last known device position.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPosition} method.
     * </p>
     * 
     * @return The last known device position.
     */
    public final List<Double> position() {
        return position;
    }

    /**
     * <p>
     * The timestamp for when the tracker resource received the device position in <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     * </p>
     * 
     * @return The timestamp for when the tracker resource received the device position in <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     */
    public final Instant receivedTime() {
        return receivedTime;
    }

    /**
     * <p>
     * The timestamp at which the device's position was determined. Uses <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     * </p>
     * 
     * @return The timestamp at which the device's position was determined. Uses <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     */
    public final Instant sampleTime() {
        return sampleTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deviceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasPosition() ? position() : null);
        hashCode = 31 * hashCode + Objects.hashCode(receivedTime());
        hashCode = 31 * hashCode + Objects.hashCode(sampleTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DevicePosition)) {
            return false;
        }
        DevicePosition other = (DevicePosition) obj;
        return Objects.equals(deviceId(), other.deviceId()) && hasPosition() == other.hasPosition()
                && Objects.equals(position(), other.position()) && Objects.equals(receivedTime(), other.receivedTime())
                && Objects.equals(sampleTime(), other.sampleTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DevicePosition").add("DeviceId", deviceId())
                .add("Position", position() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ReceivedTime", receivedTime()).add("SampleTime", sampleTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeviceId":
            return Optional.ofNullable(clazz.cast(deviceId()));
        case "Position":
            return Optional.ofNullable(clazz.cast(position()));
        case "ReceivedTime":
            return Optional.ofNullable(clazz.cast(receivedTime()));
        case "SampleTime":
            return Optional.ofNullable(clazz.cast(sampleTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DevicePosition, T> g) {
        return obj -> g.apply((DevicePosition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DevicePosition> {
        /**
         * <p>
         * The device whose position you retrieved.
         * </p>
         * 
         * @param deviceId
         *        The device whose position you retrieved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceId(String deviceId);

        /**
         * <p>
         * The last known device position.
         * </p>
         * 
         * @param position
         *        The last known device position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Collection<Double> position);

        /**
         * <p>
         * The last known device position.
         * </p>
         * 
         * @param position
         *        The last known device position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Double... position);

        /**
         * <p>
         * The timestamp for when the tracker resource received the device position in <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * </p>
         * 
         * @param receivedTime
         *        The timestamp for when the tracker resource received the device position in <a
         *        href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder receivedTime(Instant receivedTime);

        /**
         * <p>
         * The timestamp at which the device's position was determined. Uses <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * </p>
         * 
         * @param sampleTime
         *        The timestamp at which the device's position was determined. Uses <a
         *        href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleTime(Instant sampleTime);
    }

    static final class BuilderImpl implements Builder {
        private String deviceId;

        private List<Double> position = DefaultSdkAutoConstructList.getInstance();

        private Instant receivedTime;

        private Instant sampleTime;

        private BuilderImpl() {
        }

        private BuilderImpl(DevicePosition model) {
            deviceId(model.deviceId);
            position(model.position);
            receivedTime(model.receivedTime);
            sampleTime(model.sampleTime);
        }

        public final String getDeviceId() {
            return deviceId;
        }

        public final void setDeviceId(String deviceId) {
            this.deviceId = deviceId;
        }

        @Override
        @Transient
        public final Builder deviceId(String deviceId) {
            this.deviceId = deviceId;
            return this;
        }

        public final Collection<Double> getPosition() {
            if (position instanceof SdkAutoConstructList) {
                return null;
            }
            return position;
        }

        public final void setPosition(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
        }

        @Override
        @Transient
        public final Builder position(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder position(Double... position) {
            position(Arrays.asList(position));
            return this;
        }

        public final Instant getReceivedTime() {
            return receivedTime;
        }

        public final void setReceivedTime(Instant receivedTime) {
            this.receivedTime = receivedTime;
        }

        @Override
        @Transient
        public final Builder receivedTime(Instant receivedTime) {
            this.receivedTime = receivedTime;
            return this;
        }

        public final Instant getSampleTime() {
            return sampleTime;
        }

        public final void setSampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
        }

        @Override
        @Transient
        public final Builder sampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
            return this;
        }

        @Override
        public DevicePosition build() {
            return new DevicePosition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
