/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.location.model.AccessDeniedException;
import software.amazon.awssdk.services.location.model.AssociateTrackerConsumerRequest;
import software.amazon.awssdk.services.location.model.AssociateTrackerConsumerResponse;
import software.amazon.awssdk.services.location.model.BatchDeleteGeofenceRequest;
import software.amazon.awssdk.services.location.model.BatchDeleteGeofenceResponse;
import software.amazon.awssdk.services.location.model.BatchEvaluateGeofencesRequest;
import software.amazon.awssdk.services.location.model.BatchEvaluateGeofencesResponse;
import software.amazon.awssdk.services.location.model.BatchGetDevicePositionRequest;
import software.amazon.awssdk.services.location.model.BatchGetDevicePositionResponse;
import software.amazon.awssdk.services.location.model.BatchPutGeofenceRequest;
import software.amazon.awssdk.services.location.model.BatchPutGeofenceResponse;
import software.amazon.awssdk.services.location.model.BatchUpdateDevicePositionRequest;
import software.amazon.awssdk.services.location.model.BatchUpdateDevicePositionResponse;
import software.amazon.awssdk.services.location.model.ConflictException;
import software.amazon.awssdk.services.location.model.CreateGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.CreateGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.CreateMapRequest;
import software.amazon.awssdk.services.location.model.CreateMapResponse;
import software.amazon.awssdk.services.location.model.CreatePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.CreatePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.CreateTrackerRequest;
import software.amazon.awssdk.services.location.model.CreateTrackerResponse;
import software.amazon.awssdk.services.location.model.DeleteGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.DeleteGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.DeleteMapRequest;
import software.amazon.awssdk.services.location.model.DeleteMapResponse;
import software.amazon.awssdk.services.location.model.DeletePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.DeletePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.DeleteTrackerRequest;
import software.amazon.awssdk.services.location.model.DeleteTrackerResponse;
import software.amazon.awssdk.services.location.model.DescribeGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.DescribeGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.DescribeMapRequest;
import software.amazon.awssdk.services.location.model.DescribeMapResponse;
import software.amazon.awssdk.services.location.model.DescribePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.DescribePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.DescribeTrackerRequest;
import software.amazon.awssdk.services.location.model.DescribeTrackerResponse;
import software.amazon.awssdk.services.location.model.DisassociateTrackerConsumerRequest;
import software.amazon.awssdk.services.location.model.DisassociateTrackerConsumerResponse;
import software.amazon.awssdk.services.location.model.GetDevicePositionHistoryRequest;
import software.amazon.awssdk.services.location.model.GetDevicePositionHistoryResponse;
import software.amazon.awssdk.services.location.model.GetDevicePositionRequest;
import software.amazon.awssdk.services.location.model.GetDevicePositionResponse;
import software.amazon.awssdk.services.location.model.GetGeofenceRequest;
import software.amazon.awssdk.services.location.model.GetGeofenceResponse;
import software.amazon.awssdk.services.location.model.GetMapGlyphsRequest;
import software.amazon.awssdk.services.location.model.GetMapGlyphsResponse;
import software.amazon.awssdk.services.location.model.GetMapSpritesRequest;
import software.amazon.awssdk.services.location.model.GetMapSpritesResponse;
import software.amazon.awssdk.services.location.model.GetMapStyleDescriptorRequest;
import software.amazon.awssdk.services.location.model.GetMapStyleDescriptorResponse;
import software.amazon.awssdk.services.location.model.GetMapTileRequest;
import software.amazon.awssdk.services.location.model.GetMapTileResponse;
import software.amazon.awssdk.services.location.model.InternalServerException;
import software.amazon.awssdk.services.location.model.ListGeofenceCollectionsRequest;
import software.amazon.awssdk.services.location.model.ListGeofenceCollectionsResponse;
import software.amazon.awssdk.services.location.model.ListGeofencesRequest;
import software.amazon.awssdk.services.location.model.ListGeofencesResponse;
import software.amazon.awssdk.services.location.model.ListMapsRequest;
import software.amazon.awssdk.services.location.model.ListMapsResponse;
import software.amazon.awssdk.services.location.model.ListPlaceIndexesRequest;
import software.amazon.awssdk.services.location.model.ListPlaceIndexesResponse;
import software.amazon.awssdk.services.location.model.ListTrackerConsumersRequest;
import software.amazon.awssdk.services.location.model.ListTrackerConsumersResponse;
import software.amazon.awssdk.services.location.model.ListTrackersRequest;
import software.amazon.awssdk.services.location.model.ListTrackersResponse;
import software.amazon.awssdk.services.location.model.LocationException;
import software.amazon.awssdk.services.location.model.LocationRequest;
import software.amazon.awssdk.services.location.model.PutGeofenceRequest;
import software.amazon.awssdk.services.location.model.PutGeofenceResponse;
import software.amazon.awssdk.services.location.model.ResourceNotFoundException;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForPositionRequest;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForPositionResponse;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForTextRequest;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForTextResponse;
import software.amazon.awssdk.services.location.model.ThrottlingException;
import software.amazon.awssdk.services.location.model.ValidationException;
import software.amazon.awssdk.services.location.paginators.GetDevicePositionHistoryIterable;
import software.amazon.awssdk.services.location.paginators.ListGeofenceCollectionsIterable;
import software.amazon.awssdk.services.location.paginators.ListGeofencesIterable;
import software.amazon.awssdk.services.location.paginators.ListMapsIterable;
import software.amazon.awssdk.services.location.paginators.ListPlaceIndexesIterable;
import software.amazon.awssdk.services.location.paginators.ListTrackerConsumersIterable;
import software.amazon.awssdk.services.location.paginators.ListTrackersIterable;
import software.amazon.awssdk.services.location.transform.AssociateTrackerConsumerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchDeleteGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchEvaluateGeofencesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchGetDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchPutGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchUpdateDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreatePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeletePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DisassociateTrackerConsumerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetDevicePositionHistoryRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapGlyphsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapSpritesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapStyleDescriptorRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapTileRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListGeofenceCollectionsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListGeofencesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListMapsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListPlaceIndexesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListTrackerConsumersRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListTrackersRequestMarshaller;
import software.amazon.awssdk.services.location.transform.PutGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.SearchPlaceIndexForPositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.SearchPlaceIndexForTextRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LocationClient}.
 *
 * @see LocationClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLocationClient implements LocationClient {
    private static final Logger log = Logger.loggerFor(DefaultLocationClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLocationClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an association between a geofence collection and a tracker resource. This allows the tracker resource to
     * communicate location data to the linked geofence collection.
     * </p>
     * <note>
     * <p>
     * Currently not supported — Cross-account configurations, such as creating associations between a tracker resource
     * in one account and a geofence collection in another account.
     * </p>
     * </note>
     *
     * @param associateTrackerConsumerRequest
     * @return Result of the AssociateTrackerConsumer operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws ConflictException
     *         The request was unsuccessful due to a conflict.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.AssociateTrackerConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/AssociateTrackerConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateTrackerConsumerResponse associateTrackerConsumer(
            AssociateTrackerConsumerRequest associateTrackerConsumerRequest) throws InternalServerException,
            ResourceNotFoundException, ConflictException, AccessDeniedException, ValidationException, ThrottlingException,
            AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateTrackerConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateTrackerConsumerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateTrackerConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateTrackerConsumer");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateTrackerConsumerRequest, AssociateTrackerConsumerResponse>()
                            .withOperationName("AssociateTrackerConsumer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(associateTrackerConsumerRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateTrackerConsumerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a batch of geofences from a geofence collection.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You can't undo this action.
     * </p>
     * </note>
     *
     * @param batchDeleteGeofenceRequest
     * @return Result of the BatchDeleteGeofence operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.BatchDeleteGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchDeleteGeofence" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchDeleteGeofenceResponse batchDeleteGeofence(BatchDeleteGeofenceRequest batchDeleteGeofenceRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeleteGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchDeleteGeofenceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteGeofence");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler.execute(new ClientExecutionParams<BatchDeleteGeofenceRequest, BatchDeleteGeofenceResponse>()
                    .withOperationName("BatchDeleteGeofence").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(batchDeleteGeofenceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchDeleteGeofenceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Evaluates device positions against the geofence geometries from a given geofence collection. The evaluation
     * determines if the device has entered or exited a geofenced area, which publishes ENTER or EXIT geofence events to
     * Amazon EventBridge.
     * </p>
     * <note>
     * <p>
     * The last geofence that a device was observed within, if any, is tracked for 30 days after the most recent device
     * position update
     * </p>
     * </note>
     *
     * @param batchEvaluateGeofencesRequest
     * @return Result of the BatchEvaluateGeofences operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.BatchEvaluateGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchEvaluateGeofences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchEvaluateGeofencesResponse batchEvaluateGeofences(BatchEvaluateGeofencesRequest batchEvaluateGeofencesRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchEvaluateGeofencesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchEvaluateGeofencesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEvaluateGeofencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEvaluateGeofences");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler
                    .execute(new ClientExecutionParams<BatchEvaluateGeofencesRequest, BatchEvaluateGeofencesResponse>()
                            .withOperationName("BatchEvaluateGeofences").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchEvaluateGeofencesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchEvaluateGeofencesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A batch request to retrieve all device positions.
     * </p>
     *
     * @param batchGetDevicePositionRequest
     * @return Result of the BatchGetDevicePosition operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.BatchGetDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchGetDevicePosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchGetDevicePositionResponse batchGetDevicePosition(BatchGetDevicePositionRequest batchGetDevicePositionRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchGetDevicePositionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetDevicePosition");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler
                    .execute(new ClientExecutionParams<BatchGetDevicePositionRequest, BatchGetDevicePositionResponse>()
                            .withOperationName("BatchGetDevicePosition").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchGetDevicePositionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchGetDevicePositionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A batch request for storing geofence geometries into a given geofence collection.
     * </p>
     *
     * @param batchPutGeofenceRequest
     * @return Result of the BatchPutGeofence operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.BatchPutGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchPutGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchPutGeofenceResponse batchPutGeofence(BatchPutGeofenceRequest batchPutGeofenceRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchPutGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchPutGeofenceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutGeofence");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler.execute(new ClientExecutionParams<BatchPutGeofenceRequest, BatchPutGeofenceResponse>()
                    .withOperationName("BatchPutGeofence").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(batchPutGeofenceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchPutGeofenceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uploads position update data for one or more devices to a tracker resource. Amazon Location uses the data when
     * reporting the last known device position and position history.
     * </p>
     * <note>
     * <p>
     * Only one position update is stored per sample time. Location data is sampled at a fixed rate of one position per
     * 30-second interval, and retained for one year before it is deleted.
     * </p>
     * </note>
     *
     * @param batchUpdateDevicePositionRequest
     * @return Result of the BatchUpdateDevicePosition operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.BatchUpdateDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchUpdateDevicePosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchUpdateDevicePositionResponse batchUpdateDevicePosition(
            BatchUpdateDevicePositionRequest batchUpdateDevicePositionRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchUpdateDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchUpdateDevicePositionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateDevicePosition");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateDevicePositionRequest, BatchUpdateDevicePositionResponse>()
                            .withOperationName("BatchUpdateDevicePosition").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchUpdateDevicePositionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchUpdateDevicePositionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a geofence collection, which manages and stores geofences.
     * </p>
     *
     * @param createGeofenceCollectionRequest
     * @return Result of the CreateGeofenceCollection operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ConflictException
     *         The request was unsuccessful due to a conflict.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.CreateGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateGeofenceCollectionResponse createGeofenceCollection(
            CreateGeofenceCollectionRequest createGeofenceCollectionRequest) throws InternalServerException, ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateGeofenceCollectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGeofenceCollection");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler
                    .execute(new ClientExecutionParams<CreateGeofenceCollectionRequest, CreateGeofenceCollectionResponse>()
                            .withOperationName("CreateGeofenceCollection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createGeofenceCollectionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateGeofenceCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a map resource in your AWS account, which provides map tiles of different styles sourced from global
     * location data providers.
     * </p>
     * <note>
     * <p>
     * By using Maps, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using. For more information, see the <a
     * href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon Location Service.
     * </p>
     * </note>
     *
     * @param createMapRequest
     * @return Result of the CreateMap operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ConflictException
     *         The request was unsuccessful due to a conflict.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.CreateMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateMapResponse createMap(CreateMapRequest createMapRequest) throws InternalServerException, ConflictException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMapResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMap");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<CreateMapRequest, CreateMapResponse>()
                    .withOperationName("CreateMap").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(createMapRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMapRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Place index resource in your AWS account, which supports Places functions with geospatial data sourced
     * from your chosen data provider.
     * </p>
     * <note>
     * <p>
     * By using Places, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using.
     * </p>
     * <p>
     * Because of licensing limitations, you may not use HERE to store results for locations in Japan. For more
     * information, see the <a href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon Location
     * Service.
     * </p>
     * </note>
     *
     * @param createPlaceIndexRequest
     * @return Result of the CreatePlaceIndex operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ConflictException
     *         The request was unsuccessful due to a conflict.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.CreatePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreatePlaceIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePlaceIndexResponse createPlaceIndex(CreatePlaceIndexRequest createPlaceIndexRequest)
            throws InternalServerException, ConflictException, AccessDeniedException, ValidationException, ThrottlingException,
            AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePlaceIndexResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlaceIndex");
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            return clientHandler.execute(new ClientExecutionParams<CreatePlaceIndexRequest, CreatePlaceIndexResponse>()
                    .withOperationName("CreatePlaceIndex").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(createPlaceIndexRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePlaceIndexRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a tracker resource in your AWS account, which lets you retrieve current and historical location of
     * devices.
     * </p>
     *
     * @param createTrackerRequest
     * @return Result of the CreateTracker operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ConflictException
     *         The request was unsuccessful due to a conflict.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.CreateTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTrackerResponse createTracker(CreateTrackerRequest createTrackerRequest) throws InternalServerException,
            ConflictException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTrackerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTracker");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler.execute(new ClientExecutionParams<CreateTrackerRequest, CreateTrackerResponse>()
                    .withOperationName("CreateTracker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(createTrackerRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTrackerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a geofence collection from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You can't undo this action. If the geofence collection is the
     * target of a tracker resource, the devices will no longer be monitored.
     * </p>
     * </note>
     *
     * @param deleteGeofenceCollectionRequest
     * @return Result of the DeleteGeofenceCollection operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DeleteGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteGeofenceCollectionResponse deleteGeofenceCollection(
            DeleteGeofenceCollectionRequest deleteGeofenceCollectionRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteGeofenceCollectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGeofenceCollection");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteGeofenceCollectionRequest, DeleteGeofenceCollectionResponse>()
                            .withOperationName("DeleteGeofenceCollection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deleteGeofenceCollectionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteGeofenceCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a map resource from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You cannot undo this action. If the map is being used in an
     * application, the map may not render.
     * </p>
     * </note>
     *
     * @param deleteMapRequest
     * @return Result of the DeleteMap operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DeleteMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteMapResponse deleteMap(DeleteMapRequest deleteMapRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMapResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMap");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<DeleteMapRequest, DeleteMapResponse>()
                    .withOperationName("DeleteMap").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(deleteMapRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMapRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Place index resource from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You cannot undo this action.
     * </p>
     * </note>
     *
     * @param deletePlaceIndexRequest
     * @return Result of the DeletePlaceIndex operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DeletePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeletePlaceIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePlaceIndexResponse deletePlaceIndex(DeletePlaceIndexRequest deletePlaceIndexRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePlaceIndexResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlaceIndex");
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            return clientHandler.execute(new ClientExecutionParams<DeletePlaceIndexRequest, DeletePlaceIndexResponse>()
                    .withOperationName("DeletePlaceIndex").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(deletePlaceIndexRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePlaceIndexRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a tracker resource from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You can't undo this action. If the tracker resource is in use, you
     * may encounter an error. Make sure that the target resource is not a dependency for your applications.
     * </p>
     * </note>
     *
     * @param deleteTrackerRequest
     * @return Result of the DeleteTracker operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DeleteTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTrackerResponse deleteTracker(DeleteTrackerRequest deleteTrackerRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTrackerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTracker");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler.execute(new ClientExecutionParams<DeleteTrackerRequest, DeleteTrackerResponse>()
                    .withOperationName("DeleteTracker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(deleteTrackerRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTrackerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the geofence collection details.
     * </p>
     *
     * @param describeGeofenceCollectionRequest
     * @return Result of the DescribeGeofenceCollection operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DescribeGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeGeofenceCollectionResponse describeGeofenceCollection(
            DescribeGeofenceCollectionRequest describeGeofenceCollectionRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeGeofenceCollectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGeofenceCollection");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeGeofenceCollectionRequest, DescribeGeofenceCollectionResponse>()
                            .withOperationName("DescribeGeofenceCollection").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeGeofenceCollectionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeGeofenceCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the map resource details.
     * </p>
     *
     * @param describeMapRequest
     * @return Result of the DescribeMap operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DescribeMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeMapResponse describeMap(DescribeMapRequest describeMapRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeMapResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMap");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<DescribeMapRequest, DescribeMapResponse>()
                    .withOperationName("DescribeMap").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(describeMapRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeMapRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the Place index resource details.
     * </p>
     *
     * @param describePlaceIndexRequest
     * @return Result of the DescribePlaceIndex operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DescribePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribePlaceIndex" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribePlaceIndexResponse describePlaceIndex(DescribePlaceIndexRequest describePlaceIndexRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePlaceIndexResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePlaceIndex");
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            return clientHandler.execute(new ClientExecutionParams<DescribePlaceIndexRequest, DescribePlaceIndexResponse>()
                    .withOperationName("DescribePlaceIndex").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(describePlaceIndexRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribePlaceIndexRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the tracker resource details.
     * </p>
     *
     * @param describeTrackerRequest
     * @return Result of the DescribeTracker operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DescribeTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeTrackerResponse describeTracker(DescribeTrackerRequest describeTrackerRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTrackerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTracker");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler.execute(new ClientExecutionParams<DescribeTrackerRequest, DescribeTrackerResponse>()
                    .withOperationName("DescribeTracker").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(describeTrackerRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTrackerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association between a tracker resource and a geofence collection.
     * </p>
     * <note>
     * <p>
     * Once you unlink a tracker resource from a geofence collection, the tracker positions will no longer be
     * automatically evaluated against geofences.
     * </p>
     * </note>
     *
     * @param disassociateTrackerConsumerRequest
     * @return Result of the DisassociateTrackerConsumer operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.DisassociateTrackerConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DisassociateTrackerConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateTrackerConsumerResponse disassociateTrackerConsumer(
            DisassociateTrackerConsumerRequest disassociateTrackerConsumerRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateTrackerConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateTrackerConsumerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateTrackerConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateTrackerConsumer");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateTrackerConsumerRequest, DisassociateTrackerConsumerResponse>()
                            .withOperationName("DisassociateTrackerConsumer").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(disassociateTrackerConsumerRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateTrackerConsumerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a device's most recent position according to its sample time.
     * </p>
     * <note>
     * <p>
     * Device positions are deleted after one year.
     * </p>
     * </note>
     *
     * @param getDevicePositionRequest
     * @return Result of the GetDevicePosition operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePosition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDevicePositionResponse getDevicePosition(GetDevicePositionRequest getDevicePositionRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDevicePositionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevicePosition");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler.execute(new ClientExecutionParams<GetDevicePositionRequest, GetDevicePositionResponse>()
                    .withOperationName("GetDevicePosition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getDevicePositionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDevicePositionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the device position history from a tracker resource within a specified range of time.
     * </p>
     * <note>
     * <p>
     * Device positions are deleted after 1 year.
     * </p>
     * </note>
     *
     * @param getDevicePositionHistoryRequest
     * @return Result of the GetDevicePositionHistory operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetDevicePositionHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePositionHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDevicePositionHistoryResponse getDevicePositionHistory(
            GetDevicePositionHistoryRequest getDevicePositionHistoryRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDevicePositionHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDevicePositionHistoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDevicePositionHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevicePositionHistory");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler
                    .execute(new ClientExecutionParams<GetDevicePositionHistoryRequest, GetDevicePositionHistoryResponse>()
                            .withOperationName("GetDevicePositionHistory").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getDevicePositionHistoryRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDevicePositionHistoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the device position history from a tracker resource within a specified range of time.
     * </p>
     * <note>
     * <p>
     * Device positions are deleted after 1 year.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #getDevicePositionHistory(software.amazon.awssdk.services.location.model.GetDevicePositionHistoryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.GetDevicePositionHistoryIterable responses = client.getDevicePositionHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.location.paginators.GetDevicePositionHistoryIterable responses = client
     *             .getDevicePositionHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.location.model.GetDevicePositionHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.GetDevicePositionHistoryIterable responses = client.getDevicePositionHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getDevicePositionHistory(software.amazon.awssdk.services.location.model.GetDevicePositionHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getDevicePositionHistoryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetDevicePositionHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePositionHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDevicePositionHistoryIterable getDevicePositionHistoryPaginator(
            GetDevicePositionHistoryRequest getDevicePositionHistoryRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        return new GetDevicePositionHistoryIterable(this, applyPaginatorUserAgent(getDevicePositionHistoryRequest));
    }

    /**
     * <p>
     * Retrieves the geofence details from a geofence collection.
     * </p>
     *
     * @param getGeofenceRequest
     * @return Result of the GetGeofence operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGeofenceResponse getGeofence(GetGeofenceRequest getGeofenceRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGeofenceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGeofence");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler.execute(new ClientExecutionParams<GetGeofenceRequest, GetGeofenceResponse>()
                    .withOperationName("GetGeofence").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getGeofenceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGeofenceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves glyphs used to display labels on a map.
     * </p>
     *
     * @param getMapGlyphsRequest
     * @return Result of the GetMapGlyphs operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetMapGlyphs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapGlyphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMapGlyphsResponse getMapGlyphs(GetMapGlyphsRequest getMapGlyphsRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetMapGlyphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMapGlyphsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapGlyphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapGlyphs");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<GetMapGlyphsRequest, GetMapGlyphsResponse>()
                    .withOperationName("GetMapGlyphs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getMapGlyphsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMapGlyphsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the sprite sheet corresponding to a map resource. The sprite sheet is a PNG image paired with a JSON
     * document describing the offsets of individual icons that will be displayed on a rendered map.
     * </p>
     *
     * @param getMapSpritesRequest
     * @return Result of the GetMapSprites operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetMapSprites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapSprites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMapSpritesResponse getMapSprites(GetMapSpritesRequest getMapSpritesRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetMapSpritesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMapSpritesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapSpritesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapSprites");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<GetMapSpritesRequest, GetMapSpritesResponse>()
                    .withOperationName("GetMapSprites").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getMapSpritesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMapSpritesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the map style descriptor from a map resource.
     * </p>
     * <p>
     * The style descriptor contains speciﬁcations on how features render on a map. For example, what data to display,
     * what order to display the data in, and the style for the data. Style descriptors follow the Mapbox Style
     * Specification.
     * </p>
     *
     * @param getMapStyleDescriptorRequest
     * @return Result of the GetMapStyleDescriptor operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetMapStyleDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapStyleDescriptor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMapStyleDescriptorResponse getMapStyleDescriptor(GetMapStyleDescriptorRequest getMapStyleDescriptorRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetMapStyleDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetMapStyleDescriptorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapStyleDescriptorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapStyleDescriptor");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<GetMapStyleDescriptorRequest, GetMapStyleDescriptorResponse>()
                    .withOperationName("GetMapStyleDescriptor").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getMapStyleDescriptorRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMapStyleDescriptorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a vector data tile from the map resource. Map tiles are used by clients to render a map. They are
     * addressed using a grid arrangement with an X coordinate, Y coordinate, and Z (zoom) level.
     * </p>
     * <p>
     * The origin (0, 0) is the top left of the map. Increasing the zoom level by 1 doubles both the X and Y dimensions,
     * so a tile containing data for the entire world at (0/0/0) will be split into 4 tiles at zoom 1 (1/0/0, 1/0/1,
     * 1/1/0, 1/1/1).
     * </p>
     *
     * @param getMapTileRequest
     * @return Result of the GetMapTile operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.GetMapTile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapTile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMapTileResponse getMapTile(GetMapTileRequest getMapTileRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetMapTileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMapTileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapTileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapTile");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<GetMapTileRequest, GetMapTileResponse>()
                    .withOperationName("GetMapTile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getMapTileRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMapTileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists geofence collections in your AWS account.
     * </p>
     *
     * @param listGeofenceCollectionsRequest
     * @return Result of the ListGeofenceCollections operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListGeofenceCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofenceCollections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGeofenceCollectionsResponse listGeofenceCollections(ListGeofenceCollectionsRequest listGeofenceCollectionsRequest)
            throws InternalServerException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGeofenceCollectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGeofenceCollectionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeofenceCollectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeofenceCollections");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler
                    .execute(new ClientExecutionParams<ListGeofenceCollectionsRequest, ListGeofenceCollectionsResponse>()
                            .withOperationName("ListGeofenceCollections").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listGeofenceCollectionsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListGeofenceCollectionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists geofence collections in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGeofenceCollections(software.amazon.awssdk.services.location.model.ListGeofenceCollectionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofenceCollectionsIterable responses = client.listGeofenceCollectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.location.paginators.ListGeofenceCollectionsIterable responses = client
     *             .listGeofenceCollectionsPaginator(request);
     *     for (software.amazon.awssdk.services.location.model.ListGeofenceCollectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofenceCollectionsIterable responses = client.listGeofenceCollectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGeofenceCollections(software.amazon.awssdk.services.location.model.ListGeofenceCollectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGeofenceCollectionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListGeofenceCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofenceCollections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGeofenceCollectionsIterable listGeofenceCollectionsPaginator(
            ListGeofenceCollectionsRequest listGeofenceCollectionsRequest) throws InternalServerException, AccessDeniedException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        return new ListGeofenceCollectionsIterable(this, applyPaginatorUserAgent(listGeofenceCollectionsRequest));
    }

    /**
     * <p>
     * Lists geofences stored in a given geofence collection.
     * </p>
     *
     * @param listGeofencesRequest
     * @return Result of the ListGeofences operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofences" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGeofencesResponse listGeofences(ListGeofencesRequest listGeofencesRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGeofencesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGeofencesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeofencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeofences");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler.execute(new ClientExecutionParams<ListGeofencesRequest, ListGeofencesResponse>()
                    .withOperationName("ListGeofences").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listGeofencesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGeofencesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists geofences stored in a given geofence collection.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGeofences(software.amazon.awssdk.services.location.model.ListGeofencesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofencesIterable responses = client.listGeofencesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.location.paginators.ListGeofencesIterable responses = client.listGeofencesPaginator(request);
     *     for (software.amazon.awssdk.services.location.model.ListGeofencesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofencesIterable responses = client.listGeofencesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGeofences(software.amazon.awssdk.services.location.model.ListGeofencesRequest)} operation.</b>
     * </p>
     *
     * @param listGeofencesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofences" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGeofencesIterable listGeofencesPaginator(ListGeofencesRequest listGeofencesRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        return new ListGeofencesIterable(this, applyPaginatorUserAgent(listGeofencesRequest));
    }

    /**
     * <p>
     * Lists map resources in your AWS account.
     * </p>
     *
     * @param listMapsRequest
     * @return Result of the ListMaps operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListMaps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListMaps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMapsResponse listMaps(ListMapsRequest listMapsRequest) throws InternalServerException, AccessDeniedException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMapsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMapsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMapsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMaps");
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            return clientHandler.execute(new ClientExecutionParams<ListMapsRequest, ListMapsResponse>()
                    .withOperationName("ListMaps").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listMapsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMapsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists map resources in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMaps(software.amazon.awssdk.services.location.model.ListMapsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListMapsIterable responses = client.listMapsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.location.paginators.ListMapsIterable responses = client.listMapsPaginator(request);
     *     for (software.amazon.awssdk.services.location.model.ListMapsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListMapsIterable responses = client.listMapsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMaps(software.amazon.awssdk.services.location.model.ListMapsRequest)} operation.</b>
     * </p>
     *
     * @param listMapsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListMaps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListMaps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMapsIterable listMapsPaginator(ListMapsRequest listMapsRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            LocationException {
        return new ListMapsIterable(this, applyPaginatorUserAgent(listMapsRequest));
    }

    /**
     * <p>
     * Lists Place index resources in your AWS account.
     * </p>
     *
     * @param listPlaceIndexesRequest
     * @return Result of the ListPlaceIndexes operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListPlaceIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListPlaceIndexes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPlaceIndexesResponse listPlaceIndexes(ListPlaceIndexesRequest listPlaceIndexesRequest)
            throws InternalServerException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPlaceIndexesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPlaceIndexesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlaceIndexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlaceIndexes");
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            return clientHandler.execute(new ClientExecutionParams<ListPlaceIndexesRequest, ListPlaceIndexesResponse>()
                    .withOperationName("ListPlaceIndexes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listPlaceIndexesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPlaceIndexesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists Place index resources in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlaceIndexes(software.amazon.awssdk.services.location.model.ListPlaceIndexesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListPlaceIndexesIterable responses = client.listPlaceIndexesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.location.paginators.ListPlaceIndexesIterable responses = client
     *             .listPlaceIndexesPaginator(request);
     *     for (software.amazon.awssdk.services.location.model.ListPlaceIndexesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListPlaceIndexesIterable responses = client.listPlaceIndexesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlaceIndexes(software.amazon.awssdk.services.location.model.ListPlaceIndexesRequest)} operation.</b>
     * </p>
     *
     * @param listPlaceIndexesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListPlaceIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListPlaceIndexes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPlaceIndexesIterable listPlaceIndexesPaginator(ListPlaceIndexesRequest listPlaceIndexesRequest)
            throws InternalServerException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        return new ListPlaceIndexesIterable(this, applyPaginatorUserAgent(listPlaceIndexesRequest));
    }

    /**
     * <p>
     * Lists geofence collections currently associated to the given tracker resource.
     * </p>
     *
     * @param listTrackerConsumersRequest
     * @return Result of the ListTrackerConsumers operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListTrackerConsumers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackerConsumers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTrackerConsumersResponse listTrackerConsumers(ListTrackerConsumersRequest listTrackerConsumersRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTrackerConsumersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTrackerConsumersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrackerConsumersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrackerConsumers");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler.execute(new ClientExecutionParams<ListTrackerConsumersRequest, ListTrackerConsumersResponse>()
                    .withOperationName("ListTrackerConsumers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listTrackerConsumersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTrackerConsumersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists geofence collections currently associated to the given tracker resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTrackerConsumers(software.amazon.awssdk.services.location.model.ListTrackerConsumersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackerConsumersIterable responses = client.listTrackerConsumersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.location.paginators.ListTrackerConsumersIterable responses = client
     *             .listTrackerConsumersPaginator(request);
     *     for (software.amazon.awssdk.services.location.model.ListTrackerConsumersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackerConsumersIterable responses = client.listTrackerConsumersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTrackerConsumers(software.amazon.awssdk.services.location.model.ListTrackerConsumersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTrackerConsumersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListTrackerConsumers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackerConsumers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTrackerConsumersIterable listTrackerConsumersPaginator(ListTrackerConsumersRequest listTrackerConsumersRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        return new ListTrackerConsumersIterable(this, applyPaginatorUserAgent(listTrackerConsumersRequest));
    }

    /**
     * <p>
     * Lists tracker resources in your AWS account.
     * </p>
     *
     * @param listTrackersRequest
     * @return Result of the ListTrackers operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListTrackers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTrackersResponse listTrackers(ListTrackersRequest listTrackersRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTrackersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTrackersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrackersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrackers");
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            return clientHandler.execute(new ClientExecutionParams<ListTrackersRequest, ListTrackersResponse>()
                    .withOperationName("ListTrackers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listTrackersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTrackersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tracker resources in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTrackers(software.amazon.awssdk.services.location.model.ListTrackersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackersIterable responses = client.listTrackersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.location.paginators.ListTrackersIterable responses = client.listTrackersPaginator(request);
     *     for (software.amazon.awssdk.services.location.model.ListTrackersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackersIterable responses = client.listTrackersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTrackers(software.amazon.awssdk.services.location.model.ListTrackersRequest)} operation.</b>
     * </p>
     *
     * @param listTrackersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.ListTrackers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTrackersIterable listTrackersPaginator(ListTrackersRequest listTrackersRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            LocationException {
        return new ListTrackersIterable(this, applyPaginatorUserAgent(listTrackersRequest));
    }

    /**
     * <p>
     * Stores a geofence geometry in a given geofence collection, or updates the geometry of an existing geofence if a
     * geofence ID is included in the request.
     * </p>
     *
     * @param putGeofenceRequest
     * @return Result of the PutGeofence operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws ConflictException
     *         The request was unsuccessful due to a conflict.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.PutGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/PutGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutGeofenceResponse putGeofence(PutGeofenceRequest putGeofenceRequest) throws InternalServerException,
            ResourceNotFoundException, ConflictException, AccessDeniedException, ValidationException, ThrottlingException,
            AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutGeofenceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutGeofence");
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            return clientHandler.execute(new ClientExecutionParams<PutGeofenceRequest, PutGeofenceResponse>()
                    .withOperationName("PutGeofence").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(putGeofenceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutGeofenceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Reverse geocodes a given coordinate and returns a legible address. Allows you to search for Places or points of
     * interest near a given position.
     * </p>
     * <note>
     * <p>
     * By using Places, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using.
     * </p>
     * <p>
     * Because of licensing limitations, you may not use HERE to store results for locations in Japan. For more
     * information, see the <a href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon Location
     * Service.
     * </p>
     * </note>
     *
     * @param searchPlaceIndexForPositionRequest
     * @return Result of the SearchPlaceIndexForPosition operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.SearchPlaceIndexForPosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/SearchPlaceIndexForPosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchPlaceIndexForPositionResponse searchPlaceIndexForPosition(
            SearchPlaceIndexForPositionRequest searchPlaceIndexForPositionRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchPlaceIndexForPositionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchPlaceIndexForPositionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchPlaceIndexForPositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchPlaceIndexForPosition");
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            return clientHandler
                    .execute(new ClientExecutionParams<SearchPlaceIndexForPositionRequest, SearchPlaceIndexForPositionResponse>()
                            .withOperationName("SearchPlaceIndexForPosition").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(searchPlaceIndexForPositionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SearchPlaceIndexForPositionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Geocodes free-form text, such as an address, name, city, or region to allow you to search for Places or points of
     * interest.
     * </p>
     * <p>
     * Includes the option to apply additional parameters to narrow your list of results.
     * </p>
     * <note>
     * <p>
     * You can search for places near a given position using <code>BiasPosition</code>, or filter results within a
     * bounding box using <code>FilterBBox</code>. Providing both parameters simultaneously returns an error.
     * </p>
     * </note> <note>
     * <p>
     * By using Places, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using.
     * </p>
     * <p>
     * Also, when using HERE as your data provider, you may not (a) use HERE Places for Asset Management, or (b) select
     * the <code>Storage</code> option for the <code>IntendedUse</code> parameter when requesting Places in Japan. For
     * more information, see the <a href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon
     * Location Service.
     * </p>
     * </note>
     *
     * @param searchPlaceIndexForTextRequest
     * @return Result of the SearchPlaceIndexForText operation returned by the service.
     * @throws InternalServerException
     *         The request has failed to process because of an unknown server error, exception, or failure.
     * @throws ResourceNotFoundException
     *         The resource that you've entered was not found in your AWS account.
     * @throws AccessDeniedException
     *         The request was denied due to insufficient access or permission. Check with an administrator to verify
     *         your permissions.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LocationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LocationClient.SearchPlaceIndexForText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/SearchPlaceIndexForText"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchPlaceIndexForTextResponse searchPlaceIndexForText(SearchPlaceIndexForTextRequest searchPlaceIndexForTextRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, LocationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchPlaceIndexForTextResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchPlaceIndexForTextResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchPlaceIndexForTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchPlaceIndexForText");
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            return clientHandler
                    .execute(new ClientExecutionParams<SearchPlaceIndexForTextRequest, SearchPlaceIndexForTextResponse>()
                            .withOperationName("SearchPlaceIndexForText").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withInput(searchPlaceIndexForTextRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SearchPlaceIndexForTextRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LocationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends LocationRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
