/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.StreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lexruntime.internal.LexRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.lexruntime.model.BadGatewayException;
import software.amazon.awssdk.services.lexruntime.model.BadRequestException;
import software.amazon.awssdk.services.lexruntime.model.ConflictException;
import software.amazon.awssdk.services.lexruntime.model.DeleteSessionRequest;
import software.amazon.awssdk.services.lexruntime.model.DeleteSessionResponse;
import software.amazon.awssdk.services.lexruntime.model.DependencyFailedException;
import software.amazon.awssdk.services.lexruntime.model.GetSessionRequest;
import software.amazon.awssdk.services.lexruntime.model.GetSessionResponse;
import software.amazon.awssdk.services.lexruntime.model.InternalFailureException;
import software.amazon.awssdk.services.lexruntime.model.LexRuntimeException;
import software.amazon.awssdk.services.lexruntime.model.LexRuntimeRequest;
import software.amazon.awssdk.services.lexruntime.model.LimitExceededException;
import software.amazon.awssdk.services.lexruntime.model.LoopDetectedException;
import software.amazon.awssdk.services.lexruntime.model.NotAcceptableException;
import software.amazon.awssdk.services.lexruntime.model.NotFoundException;
import software.amazon.awssdk.services.lexruntime.model.PostContentRequest;
import software.amazon.awssdk.services.lexruntime.model.PostContentResponse;
import software.amazon.awssdk.services.lexruntime.model.PostTextRequest;
import software.amazon.awssdk.services.lexruntime.model.PostTextResponse;
import software.amazon.awssdk.services.lexruntime.model.PutSessionRequest;
import software.amazon.awssdk.services.lexruntime.model.PutSessionResponse;
import software.amazon.awssdk.services.lexruntime.model.RequestTimeoutException;
import software.amazon.awssdk.services.lexruntime.model.UnsupportedMediaTypeException;
import software.amazon.awssdk.services.lexruntime.transform.DeleteSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.PostContentRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.PostTextRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.PutSessionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LexRuntimeClient}.
 *
 * @see LexRuntimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLexRuntimeClient implements LexRuntimeClient {
    private static final Logger log = Logger.loggerFor(DefaultLexRuntimeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLexRuntimeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     *
     * @param deleteSessionRequest
     * @return Result of the DeleteSession operation returned by the service.
     * @throws NotFoundException
     *         The resource (such as the Amazon Lex bot or an alias) that is referred to is not found.
     * @throws BadRequestException
     *         Request validation failed, there is no usable message in the context, or the bot build failed, is still
     *         in progress, or contains unbuilt changes.
     * @throws LimitExceededException
     *         Exceeded a limit.
     * @throws InternalFailureException
     *         Internal service error. Retry the call.
     * @throws ConflictException
     *         Two clients are using the same AWS account, Amazon Lex bot, and user ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeClient.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/DeleteSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSessionResponse deleteSession(DeleteSessionRequest deleteSessionRequest) throws NotFoundException,
            BadRequestException, LimitExceededException, InternalFailureException, ConflictException, AwsServiceException,
            SdkClientException, LexRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSession");

            return clientHandler.execute(new ClientExecutionParams<DeleteSessionRequest, DeleteSessionResponse>()
                    .withOperationName("DeleteSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user ID.
     * </p>
     *
     * @param getSessionRequest
     * @return Result of the GetSession operation returned by the service.
     * @throws NotFoundException
     *         The resource (such as the Amazon Lex bot or an alias) that is referred to is not found.
     * @throws BadRequestException
     *         Request validation failed, there is no usable message in the context, or the bot build failed, is still
     *         in progress, or contains unbuilt changes.
     * @throws LimitExceededException
     *         Exceeded a limit.
     * @throws InternalFailureException
     *         Internal service error. Retry the call.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSessionResponse getSession(GetSessionRequest getSessionRequest) throws NotFoundException, BadRequestException,
            LimitExceededException, InternalFailureException, AwsServiceException, SdkClientException, LexRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");

            return clientHandler.execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>()
                    .withOperationName("GetSession").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSessionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends user input (text or speech) to Amazon Lex. Clients use this API to send text and audio requests to Amazon
     * Lex at runtime. Amazon Lex interprets the user input using the machine learning model that it built for the bot.
     * </p>
     * <p>
     * The <code>PostContent</code> operation supports audio input at 8kHz and 16kHz. You can use 8kHz audio to achieve
     * higher speech recognition accuracy in telephone audio applications.
     * </p>
     * <p>
     * In response, Amazon Lex returns the next message to convey to the user. Consider the following example messages:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a user input "I would like a pizza," Amazon Lex might return a response with a message eliciting slot data
     * (for example, <code>PizzaSize</code>): "What size pizza would you like?".
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user provides all of the pizza order information, Amazon Lex might return a response with a message to
     * get user confirmation: "Order the pizza?".
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user replies "Yes" to the confirmation prompt, Amazon Lex might return a conclusion statement:
     * "Thank you, your cheese pizza has been ordered.".
     * </p>
     * </li>
     * </ul>
     * <p>
     * Not all Amazon Lex messages require a response from the user. For example, conclusion statements do not require a
     * response. Some messages require only a yes or no response. In addition to the <code>message</code>, Amazon Lex
     * provides additional context about the message in the response that you can use to enhance client behavior, such
     * as displaying the appropriate client user interface. Consider the following examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the message is to elicit slot data, Amazon Lex returns the following context information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-lex-dialog-state</code> header set to <code>ElicitSlot</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-lex-intent-name</code> header set to the intent name in the current context
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-lex-slot-to-elicit</code> header set to the slot name for which the <code>message</code> is eliciting
     * information
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-lex-slots</code> header set to a map of slots configured for the intent with their current values
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * If the message is a confirmation prompt, the <code>x-amz-lex-dialog-state</code> header is set to
     * <code>Confirmation</code> and the <code>x-amz-lex-slot-to-elicit</code> header is omitted.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the message is a clarification prompt configured for the intent, indicating that the user intent is not
     * understood, the <code>x-amz-dialog-state</code> header is set to <code>ElicitIntent</code> and the
     * <code>x-amz-slot-to-elicit</code> header is omitted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In addition, Amazon Lex also returns your application-specific <code>sessionAttributes</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/context-mgmt.html">Managing Conversation
     * Context</a>.
     * </p>
     *
     * @param postContentRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>Content-Type</code> HTTP
     *        header.
     *        </p>
     *        <p>
     *        You can stream audio data to Amazon Lex or you can create a local buffer that captures all of the audio
     *        data before sending. In general, you get better performance if you stream audio data rather than buffering
     *        the data locally.
     *        </p>
     *        '
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled PostContentResponse
     *        and an InputStream to the response content are provided as parameters to the callback. The callback may
     *        return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The prompt (or statement) to convey to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex did not understand the user intent, it sends the <code>clarificationPrompt</code>
     *        configured for the bot. If the intent requires confirmation before taking the fulfillment action, it sends
     *        the <code>confirmationPrompt</code>. Another example: Suppose that the Lambda function successfully
     *        fulfilled the intent, and sent a message to convey to the user. Then Amazon Lex sends that message in the
     *        response.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws NotFoundException
     *         The resource (such as the Amazon Lex bot or an alias) that is referred to is not found.
     * @throws BadRequestException
     *         Request validation failed, there is no usable message in the context, or the bot build failed, is still
     *         in progress, or contains unbuilt changes.
     * @throws LimitExceededException
     *         Exceeded a limit.
     * @throws InternalFailureException
     *         Internal service error. Retry the call.
     * @throws ConflictException
     *         Two clients are using the same AWS account, Amazon Lex bot, and user ID.
     * @throws UnsupportedMediaTypeException
     *         The Content-Type header (<code>PostContent</code> API) has an invalid value.
     * @throws NotAcceptableException
     *         The accept header in the request does not have a valid value.
     * @throws RequestTimeoutException
     *         The input speech is too long.
     * @throws DependencyFailedException
     *         One of the dependencies, such as AWS Lambda or Amazon Polly, threw an exception. For example, </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Amazon Lex does not have sufficient permissions to call a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a Lambda function takes longer than 30 seconds to execute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a fulfillment Lambda function returns a <code>Delegate</code> dialog action without removing any slot
     *         values.
     *         </p>
     *         </li>
     * @throws BadGatewayException
     *         Either the Amazon Lex bot is still building, or one of the dependent services (Amazon Polly, AWS Lambda)
     *         failed with an internal service error.
     * @throws LoopDetectedException
     *         This exception is not used.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeClient.PostContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/PostContent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT postContent(PostContentRequest postContentRequest, RequestBody requestBody,
            ResponseTransformer<PostContentResponse, ReturnT> responseTransformer) throws NotFoundException, BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, UnsupportedMediaTypeException,
            NotAcceptableException, RequestTimeoutException, DependencyFailedException, BadGatewayException,
            LoopDetectedException, AwsServiceException, SdkClientException, LexRuntimeException {
        postContentRequest = applySignerOverride(postContentRequest, Aws4UnsignedPayloadSigner.create());
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<PostContentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PostContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postContentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostContent");

            return clientHandler.execute(
                    new ClientExecutionParams<PostContentRequest, PostContentResponse>()
                            .withOperationName("PostContent")
                            .withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(postContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withRequestBody(requestBody)
                            .withMarshaller(
                                    StreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new PostContentRequestMarshaller(protocolFactory))
                                            .requestBody(requestBody).transferEncoding(true).build()), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. Client applications can use this API to send requests to Amazon Lex at runtime.
     * Amazon Lex then interprets the user input using the machine learning model it built for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex returns the next <code>message</code> to convey to the user an optional
     * <code>responseCard</code> to display. Consider the following example messages:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a user input "I would like a pizza", Amazon Lex might return a response with a message eliciting slot data
     * (for example, PizzaSize): "What size pizza would you like?"
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user provides all of the pizza order information, Amazon Lex might return a response with a message to
     * obtain user confirmation "Proceed with the pizza order?".
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user replies to a confirmation prompt with a "yes", Amazon Lex might return a conclusion statement:
     * "Thank you, your cheese pizza has been ordered.".
     * </p>
     * </li>
     * </ul>
     * <p>
     * Not all Amazon Lex messages require a user response. For example, a conclusion statement does not require a
     * response. Some messages require only a "yes" or "no" user response. In addition to the <code>message</code>,
     * Amazon Lex provides additional context about the message in the response that you might use to enhance client
     * behavior, for example, to display the appropriate client user interface. These are the <code>slotToElicit</code>,
     * <code>dialogState</code>, <code>intentName</code>, and <code>slots</code> fields in the response. Consider the
     * following examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the message is to elicit slot data, Amazon Lex returns the following context information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>dialogState</code> set to ElicitSlot
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>intentName</code> set to the intent name in the current context
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>slotToElicit</code> set to the slot name for which the <code>message</code> is eliciting information
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>slots</code> set to a map of slots, configured for the intent, with currently known values
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * If the message is a confirmation prompt, the <code>dialogState</code> is set to ConfirmIntent and
     * <code>SlotToElicit</code> is set to null.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the message is a clarification prompt (configured for the intent) that indicates that user intent is not
     * understood, the <code>dialogState</code> is set to ElicitIntent and <code>slotToElicit</code> is set to null.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In addition, Amazon Lex also returns your application-specific <code>sessionAttributes</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/context-mgmt.html">Managing Conversation
     * Context</a>.
     * </p>
     *
     * @param postTextRequest
     * @return Result of the PostText operation returned by the service.
     * @throws NotFoundException
     *         The resource (such as the Amazon Lex bot or an alias) that is referred to is not found.
     * @throws BadRequestException
     *         Request validation failed, there is no usable message in the context, or the bot build failed, is still
     *         in progress, or contains unbuilt changes.
     * @throws LimitExceededException
     *         Exceeded a limit.
     * @throws InternalFailureException
     *         Internal service error. Retry the call.
     * @throws ConflictException
     *         Two clients are using the same AWS account, Amazon Lex bot, and user ID.
     * @throws DependencyFailedException
     *         One of the dependencies, such as AWS Lambda or Amazon Polly, threw an exception. For example, </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Amazon Lex does not have sufficient permissions to call a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a Lambda function takes longer than 30 seconds to execute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a fulfillment Lambda function returns a <code>Delegate</code> dialog action without removing any slot
     *         values.
     *         </p>
     *         </li>
     * @throws BadGatewayException
     *         Either the Amazon Lex bot is still building, or one of the dependent services (Amazon Polly, AWS Lambda)
     *         failed with an internal service error.
     * @throws LoopDetectedException
     *         This exception is not used.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeClient.PostText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/PostText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PostTextResponse postText(PostTextRequest postTextRequest) throws NotFoundException, BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, DependencyFailedException, BadGatewayException,
            LoopDetectedException, AwsServiceException, SdkClientException, LexRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PostTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PostTextResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostText");

            return clientHandler.execute(new ClientExecutionParams<PostTextRequest, PostTextResponse>()
                    .withOperationName("PostText").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(postTextRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PostTextRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex bot. Use this operation to enable your
     * application to set the state of the bot.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/how-session-api.html">Managing
     * Sessions</a>.
     * </p>
     *
     * @param putSessionRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled PutSessionResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws NotFoundException
     *         The resource (such as the Amazon Lex bot or an alias) that is referred to is not found.
     * @throws BadRequestException
     *         Request validation failed, there is no usable message in the context, or the bot build failed, is still
     *         in progress, or contains unbuilt changes.
     * @throws LimitExceededException
     *         Exceeded a limit.
     * @throws InternalFailureException
     *         Internal service error. Retry the call.
     * @throws ConflictException
     *         Two clients are using the same AWS account, Amazon Lex bot, and user ID.
     * @throws NotAcceptableException
     *         The accept header in the request does not have a valid value.
     * @throws DependencyFailedException
     *         One of the dependencies, such as AWS Lambda or Amazon Polly, threw an exception. For example, </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Amazon Lex does not have sufficient permissions to call a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a Lambda function takes longer than 30 seconds to execute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a fulfillment Lambda function returns a <code>Delegate</code> dialog action without removing any slot
     *         values.
     *         </p>
     *         </li>
     * @throws BadGatewayException
     *         Either the Amazon Lex bot is still building, or one of the dependent services (Amazon Polly, AWS Lambda)
     *         failed with an internal service error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexRuntimeClient.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT putSession(PutSessionRequest putSessionRequest,
            ResponseTransformer<PutSessionResponse, ReturnT> responseTransformer) throws NotFoundException, BadRequestException,
            LimitExceededException, InternalFailureException, ConflictException, NotAcceptableException,
            DependencyFailedException, BadGatewayException, AwsServiceException, SdkClientException, LexRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<PutSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSession");

            return clientHandler.execute(
                    new ClientExecutionParams<PutSessionRequest, PutSessionResponse>().withOperationName("PutSession")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(putSessionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutSessionRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends LexRuntimeRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        LexRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new LexRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LexRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestTimeoutException")
                                .exceptionBuilderSupplier(RequestTimeoutException::builder).httpStatusCode(408).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailedException")
                                .exceptionBuilderSupplier(DependencyFailedException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedMediaTypeException")
                                .exceptionBuilderSupplier(UnsupportedMediaTypeException::builder).httpStatusCode(415).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAcceptableException")
                                .exceptionBuilderSupplier(NotAcceptableException::builder).httpStatusCode(406).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LoopDetectedException")
                                .exceptionBuilderSupplier(LoopDetectedException::builder).httpStatusCode(508).build());
    }

    @Override
    public final LexRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new LexRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
