/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PostTextResponse extends LexRuntimeResponse implements
        ToCopyableBuilder<PostTextResponse.Builder, PostTextResponse> {
    private static final SdkField<String> INTENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("intentName").getter(getter(PostTextResponse::intentName)).setter(setter(Builder::intentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("intentName").build()).build();

    private static final SdkField<IntentConfidence> NLU_INTENT_CONFIDENCE_FIELD = SdkField
            .<IntentConfidence> builder(MarshallingType.SDK_POJO).memberName("nluIntentConfidence")
            .getter(getter(PostTextResponse::nluIntentConfidence)).setter(setter(Builder::nluIntentConfidence))
            .constructor(IntentConfidence::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nluIntentConfidence").build())
            .build();

    private static final SdkField<List<PredictedIntent>> ALTERNATIVE_INTENTS_FIELD = SdkField
            .<List<PredictedIntent>> builder(MarshallingType.LIST)
            .memberName("alternativeIntents")
            .getter(getter(PostTextResponse::alternativeIntents))
            .setter(setter(Builder::alternativeIntents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alternativeIntents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PredictedIntent> builder(MarshallingType.SDK_POJO)
                                            .constructor(PredictedIntent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> SLOTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("slots")
            .getter(getter(PostTextResponse::slots))
            .setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slots").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> SESSION_ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("sessionAttributes")
            .getter(getter(PostTextResponse::sessionAttributes))
            .setter(setter(Builder::sessionAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(PostTextResponse::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<SentimentResponse> SENTIMENT_RESPONSE_FIELD = SdkField
            .<SentimentResponse> builder(MarshallingType.SDK_POJO).memberName("sentimentResponse")
            .getter(getter(PostTextResponse::sentimentResponse)).setter(setter(Builder::sentimentResponse))
            .constructor(SentimentResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sentimentResponse").build()).build();

    private static final SdkField<String> MESSAGE_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("messageFormat").getter(getter(PostTextResponse::messageFormatAsString))
            .setter(setter(Builder::messageFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messageFormat").build()).build();

    private static final SdkField<String> DIALOG_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dialogState").getter(getter(PostTextResponse::dialogStateAsString)).setter(setter(Builder::dialogState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dialogState").build()).build();

    private static final SdkField<String> SLOT_TO_ELICIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("slotToElicit").getter(getter(PostTextResponse::slotToElicit)).setter(setter(Builder::slotToElicit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotToElicit").build()).build();

    private static final SdkField<ResponseCard> RESPONSE_CARD_FIELD = SdkField.<ResponseCard> builder(MarshallingType.SDK_POJO)
            .memberName("responseCard").getter(getter(PostTextResponse::responseCard)).setter(setter(Builder::responseCard))
            .constructor(ResponseCard::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseCard").build()).build();

    private static final SdkField<String> SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sessionId").getter(getter(PostTextResponse::sessionId)).setter(setter(Builder::sessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionId").build()).build();

    private static final SdkField<String> BOT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("botVersion").getter(getter(PostTextResponse::botVersion)).setter(setter(Builder::botVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botVersion").build()).build();

    private static final SdkField<List<ActiveContext>> ACTIVE_CONTEXTS_FIELD = SdkField
            .<List<ActiveContext>> builder(MarshallingType.LIST)
            .memberName("activeContexts")
            .getter(getter(PostTextResponse::activeContexts))
            .setter(setter(Builder::activeContexts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("activeContexts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ActiveContext> builder(MarshallingType.SDK_POJO)
                                            .constructor(ActiveContext::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INTENT_NAME_FIELD,
            NLU_INTENT_CONFIDENCE_FIELD, ALTERNATIVE_INTENTS_FIELD, SLOTS_FIELD, SESSION_ATTRIBUTES_FIELD, MESSAGE_FIELD,
            SENTIMENT_RESPONSE_FIELD, MESSAGE_FORMAT_FIELD, DIALOG_STATE_FIELD, SLOT_TO_ELICIT_FIELD, RESPONSE_CARD_FIELD,
            SESSION_ID_FIELD, BOT_VERSION_FIELD, ACTIVE_CONTEXTS_FIELD));

    private final String intentName;

    private final IntentConfidence nluIntentConfidence;

    private final List<PredictedIntent> alternativeIntents;

    private final Map<String, String> slots;

    private final Map<String, String> sessionAttributes;

    private final String message;

    private final SentimentResponse sentimentResponse;

    private final String messageFormat;

    private final String dialogState;

    private final String slotToElicit;

    private final ResponseCard responseCard;

    private final String sessionId;

    private final String botVersion;

    private final List<ActiveContext> activeContexts;

    private PostTextResponse(BuilderImpl builder) {
        super(builder);
        this.intentName = builder.intentName;
        this.nluIntentConfidence = builder.nluIntentConfidence;
        this.alternativeIntents = builder.alternativeIntents;
        this.slots = builder.slots;
        this.sessionAttributes = builder.sessionAttributes;
        this.message = builder.message;
        this.sentimentResponse = builder.sentimentResponse;
        this.messageFormat = builder.messageFormat;
        this.dialogState = builder.dialogState;
        this.slotToElicit = builder.slotToElicit;
        this.responseCard = builder.responseCard;
        this.sessionId = builder.sessionId;
        this.botVersion = builder.botVersion;
        this.activeContexts = builder.activeContexts;
    }

    /**
     * <p>
     * The current user intent that Amazon Lex is aware of.
     * </p>
     * 
     * @return The current user intent that Amazon Lex is aware of.
     */
    public String intentName() {
        return intentName;
    }

    /**
     * <p>
     * Provides a score that indicates how confident Amazon Lex is that the returned intent is the one that matches the
     * user's intent. The score is between 0.0 and 1.0. For more information, see <a
     * href="https://docs.aws.amazon.com/lex/latest/dg/confidence-scores.html">Confidence Scores</a>.
     * </p>
     * <p>
     * The score is a relative score, not an absolute score. The score may change based on improvements to Amazon Lex.
     * </p>
     * 
     * @return Provides a score that indicates how confident Amazon Lex is that the returned intent is the one that
     *         matches the user's intent. The score is between 0.0 and 1.0. For more information, see <a
     *         href="https://docs.aws.amazon.com/lex/latest/dg/confidence-scores.html">Confidence Scores</a>.</p>
     *         <p>
     *         The score is a relative score, not an absolute score. The score may change based on improvements to
     *         Amazon Lex.
     */
    public IntentConfidence nluIntentConfidence() {
        return nluIntentConfidence;
    }

    /**
     * Returns true if the AlternativeIntents property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAlternativeIntents() {
        return alternativeIntents != null && !(alternativeIntents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One to four alternative intents that may be applicable to the user's intent.
     * </p>
     * <p>
     * Each alternative includes a score that indicates how confident Amazon Lex is that the intent matches the user's
     * intent. The intents are sorted by the confidence score.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAlternativeIntents()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One to four alternative intents that may be applicable to the user's intent.</p>
     *         <p>
     *         Each alternative includes a score that indicates how confident Amazon Lex is that the intent matches the
     *         user's intent. The intents are sorted by the confidence score.
     */
    public List<PredictedIntent> alternativeIntents() {
        return alternativeIntents;
    }

    /**
     * Returns true if the Slots property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSlots() {
        return slots != null && !(slots instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The intent slots that Amazon Lex detected from the user input in the conversation.
     * </p>
     * <p>
     * Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is determined
     * by the <code>valueSelectionStrategy</code> selected when the slot type was created or updated. If
     * <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value provided by the user is
     * returned, if the user value is similar to the slot values. If <code>valueSelectionStrategy</code> is set to
     * <code>TOP_RESOLUTION</code> Amazon Lex returns the first value in the resolution list or, if there is no
     * resolution list, null. If you don't specify a <code>valueSelectionStrategy</code>, the default is
     * <code>ORIGINAL_VALUE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSlots()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The intent slots that Amazon Lex detected from the user input in the conversation. </p>
     *         <p>
     *         Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is
     *         determined by the <code>valueSelectionStrategy</code> selected when the slot type was created or updated.
     *         If <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value provided by the
     *         user is returned, if the user value is similar to the slot values. If <code>valueSelectionStrategy</code>
     *         is set to <code>TOP_RESOLUTION</code> Amazon Lex returns the first value in the resolution list or, if
     *         there is no resolution list, null. If you don't specify a <code>valueSelectionStrategy</code>, the
     *         default is <code>ORIGINAL_VALUE</code>.
     */
    public Map<String, String> slots() {
        return slots;
    }

    /**
     * Returns true if the SessionAttributes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSessionAttributes() {
        return sessionAttributes != null && !(sessionAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of key-value pairs representing the session-specific context information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSessionAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of key-value pairs representing the session-specific context information.
     */
    public Map<String, String> sessionAttributes() {
        return sessionAttributes;
    }

    /**
     * <p>
     * The message to convey to the user. The message can come from the bot's configuration or from a Lambda function.
     * </p>
     * <p>
     * If the intent is not configured with a Lambda function, or if the Lambda function returned <code>Delegate</code>
     * as the <code>dialogAction.type</code> its response, Amazon Lex decides on the next course of action and selects
     * an appropriate message from the bot's configuration based on the current interaction context. For example, if
     * Amazon Lex isn't able to understand user input, it uses a clarification prompt message.
     * </p>
     * <p>
     * When you create an intent you can assign messages to groups. When messages are assigned to groups Amazon Lex
     * returns one message from each group in the response. The message field is an escaped JSON string containing the
     * messages. For more information about the structure of the JSON string returned, see <a>msg-prompts-formats</a>.
     * </p>
     * <p>
     * If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
     * </p>
     * 
     * @return The message to convey to the user. The message can come from the bot's configuration or from a Lambda
     *         function.</p>
     *         <p>
     *         If the intent is not configured with a Lambda function, or if the Lambda function returned
     *         <code>Delegate</code> as the <code>dialogAction.type</code> its response, Amazon Lex decides on the next
     *         course of action and selects an appropriate message from the bot's configuration based on the current
     *         interaction context. For example, if Amazon Lex isn't able to understand user input, it uses a
     *         clarification prompt message.
     *         </p>
     *         <p>
     *         When you create an intent you can assign messages to groups. When messages are assigned to groups Amazon
     *         Lex returns one message from each group in the response. The message field is an escaped JSON string
     *         containing the messages. For more information about the structure of the JSON string returned, see
     *         <a>msg-prompts-formats</a>.
     *         </p>
     *         <p>
     *         If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
     */
    public String message() {
        return message;
    }

    /**
     * <p>
     * The sentiment expressed in and utterance.
     * </p>
     * <p>
     * When the bot is configured to send utterances to Amazon Comprehend for sentiment analysis, this field contains
     * the result of the analysis.
     * </p>
     * 
     * @return The sentiment expressed in and utterance.</p>
     *         <p>
     *         When the bot is configured to send utterances to Amazon Comprehend for sentiment analysis, this field
     *         contains the result of the analysis.
     */
    public SentimentResponse sentimentResponse() {
        return sentimentResponse;
    }

    /**
     * <p>
     * The format of the response message. One of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PlainText</code> - The message contains plain UTF-8 text.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSML</code> - The message contains text formatted for voice output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
     * groups that messages were assigned to when the intent was created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormatType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageFormatAsString}.
     * </p>
     * 
     * @return The format of the response message. One of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PlainText</code> - The message contains plain UTF-8 text.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SSML</code> - The message contains text formatted for voice output.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from
     *         the groups that messages were assigned to when the intent was created.
     *         </p>
     *         </li>
     * @see MessageFormatType
     */
    public MessageFormatType messageFormat() {
        return MessageFormatType.fromValue(messageFormat);
    }

    /**
     * <p>
     * The format of the response message. One of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PlainText</code> - The message contains plain UTF-8 text.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSML</code> - The message contains text formatted for voice output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
     * groups that messages were assigned to when the intent was created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormatType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageFormatAsString}.
     * </p>
     * 
     * @return The format of the response message. One of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PlainText</code> - The message contains plain UTF-8 text.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SSML</code> - The message contains text formatted for voice output.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from
     *         the groups that messages were assigned to when the intent was created.
     *         </p>
     *         </li>
     * @see MessageFormatType
     */
    public String messageFormatAsString() {
        return messageFormat;
    }

    /**
     * <p>
     * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     * </p>
     * <p>
     * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user intent
     * from this utterance, it will return this dialogState.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     * </p>
     * <p>
     * For example, Amazon Lex wants user confirmation before fulfilling an intent.
     * </p>
     * <p>
     * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
     * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
     * OrderDrink).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     * </p>
     * <p>
     * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
     * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in the
     * response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information appropriately.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully fulfilled
     * the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - Conveys that the conversation with the user failed.
     * </p>
     * <p>
     * This can happen for various reasons including that the user did not provide an appropriate response to prompts
     * from the service (you can configure how many times Amazon Lex can prompt a user for specific information), or the
     * Lambda function failed to fulfill the intent.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dialogState} will
     * return {@link DialogState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dialogStateAsString}.
     * </p>
     * 
     * @return Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     *         <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     *         </p>
     *         <p>
     *         For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
     *         user intent from this utterance, it will return this dialogState.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     *         </p>
     *         <p>
     *         For example, Amazon Lex wants user confirmation before fulfilling an intent.
     *         </p>
     *         <p>
     *         Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     *         "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
     *         additional information (in these examples, update the crust type slot value, or change intent from
     *         OrderPizza to OrderDrink).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     *         </p>
     *         <p>
     *         For example, suppose that in the response Amazon Lex sends this message:
     *         "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
     *         might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon Lex
     *         can process such additional information appropriately.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
     *         fulfilled the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - Conveys that the conversation with the user failed.
     *         </p>
     *         <p>
     *         This can happen for various reasons including that the user did not provide an appropriate response to
     *         prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
     *         information), or the Lambda function failed to fulfill the intent.
     *         </p>
     *         </li>
     * @see DialogState
     */
    public DialogState dialogState() {
        return DialogState.fromValue(dialogState);
    }

    /**
     * <p>
     * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     * </p>
     * <p>
     * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user intent
     * from this utterance, it will return this dialogState.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     * </p>
     * <p>
     * For example, Amazon Lex wants user confirmation before fulfilling an intent.
     * </p>
     * <p>
     * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
     * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
     * OrderDrink).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     * </p>
     * <p>
     * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
     * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in the
     * response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information appropriately.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully fulfilled
     * the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - Conveys that the conversation with the user failed.
     * </p>
     * <p>
     * This can happen for various reasons including that the user did not provide an appropriate response to prompts
     * from the service (you can configure how many times Amazon Lex can prompt a user for specific information), or the
     * Lambda function failed to fulfill the intent.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dialogState} will
     * return {@link DialogState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dialogStateAsString}.
     * </p>
     * 
     * @return Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     *         <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     *         </p>
     *         <p>
     *         For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
     *         user intent from this utterance, it will return this dialogState.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     *         </p>
     *         <p>
     *         For example, Amazon Lex wants user confirmation before fulfilling an intent.
     *         </p>
     *         <p>
     *         Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     *         "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
     *         additional information (in these examples, update the crust type slot value, or change intent from
     *         OrderPizza to OrderDrink).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     *         </p>
     *         <p>
     *         For example, suppose that in the response Amazon Lex sends this message:
     *         "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
     *         might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon Lex
     *         can process such additional information appropriately.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
     *         fulfilled the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - Conveys that the conversation with the user failed.
     *         </p>
     *         <p>
     *         This can happen for various reasons including that the user did not provide an appropriate response to
     *         prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
     *         information), or the Lambda function failed to fulfill the intent.
     *         </p>
     *         </li>
     * @see DialogState
     */
    public String dialogStateAsString() {
        return dialogState;
    }

    /**
     * <p>
     * If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for which Amazon
     * Lex is eliciting a value.
     * </p>
     * 
     * @return If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for which
     *         Amazon Lex is eliciting a value.
     */
    public String slotToElicit() {
        return slotToElicit;
    }

    /**
     * <p>
     * Represents the options that the user has to respond to the current prompt. Response Card can come from the bot
     * configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a code hook (Lambda
     * function).
     * </p>
     * 
     * @return Represents the options that the user has to respond to the current prompt. Response Card can come from
     *         the bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a
     *         code hook (Lambda function).
     */
    public ResponseCard responseCard() {
        return responseCard;
    }

    /**
     * <p>
     * A unique identifier for the session.
     * </p>
     * 
     * @return A unique identifier for the session.
     */
    public String sessionId() {
        return sessionId;
    }

    /**
     * <p>
     * The version of the bot that responded to the conversation. You can use this information to help determine if one
     * version of a bot is performing better than another version.
     * </p>
     * 
     * @return The version of the bot that responded to the conversation. You can use this information to help determine
     *         if one version of a bot is performing better than another version.
     */
    public String botVersion() {
        return botVersion;
    }

    /**
     * Returns true if the ActiveContexts property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasActiveContexts() {
        return activeContexts != null && !(activeContexts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of active contexts for the session. A context can be set when an intent is fulfilled or by calling the
     * <code>PostContent</code>, <code>PostText</code>, or <code>PutSession</code> operation.
     * </p>
     * <p>
     * You can use a context to control the intents that can follow up an intent, or to modify the operation of your
     * application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasActiveContexts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of active contexts for the session. A context can be set when an intent is fulfilled or by calling
     *         the <code>PostContent</code>, <code>PostText</code>, or <code>PutSession</code> operation.</p>
     *         <p>
     *         You can use a context to control the intents that can follow up an intent, or to modify the operation of
     *         your application.
     */
    public List<ActiveContext> activeContexts() {
        return activeContexts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(intentName());
        hashCode = 31 * hashCode + Objects.hashCode(nluIntentConfidence());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlternativeIntents() ? alternativeIntents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSlots() ? slots() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSessionAttributes() ? sessionAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(sentimentResponse());
        hashCode = 31 * hashCode + Objects.hashCode(messageFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dialogStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(slotToElicit());
        hashCode = 31 * hashCode + Objects.hashCode(responseCard());
        hashCode = 31 * hashCode + Objects.hashCode(sessionId());
        hashCode = 31 * hashCode + Objects.hashCode(botVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasActiveContexts() ? activeContexts() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PostTextResponse)) {
            return false;
        }
        PostTextResponse other = (PostTextResponse) obj;
        return Objects.equals(intentName(), other.intentName())
                && Objects.equals(nluIntentConfidence(), other.nluIntentConfidence())
                && hasAlternativeIntents() == other.hasAlternativeIntents()
                && Objects.equals(alternativeIntents(), other.alternativeIntents()) && hasSlots() == other.hasSlots()
                && Objects.equals(slots(), other.slots()) && hasSessionAttributes() == other.hasSessionAttributes()
                && Objects.equals(sessionAttributes(), other.sessionAttributes()) && Objects.equals(message(), other.message())
                && Objects.equals(sentimentResponse(), other.sentimentResponse())
                && Objects.equals(messageFormatAsString(), other.messageFormatAsString())
                && Objects.equals(dialogStateAsString(), other.dialogStateAsString())
                && Objects.equals(slotToElicit(), other.slotToElicit()) && Objects.equals(responseCard(), other.responseCard())
                && Objects.equals(sessionId(), other.sessionId()) && Objects.equals(botVersion(), other.botVersion())
                && hasActiveContexts() == other.hasActiveContexts() && Objects.equals(activeContexts(), other.activeContexts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PostTextResponse").add("IntentName", intentName())
                .add("NluIntentConfidence", nluIntentConfidence())
                .add("AlternativeIntents", hasAlternativeIntents() ? alternativeIntents() : null)
                .add("Slots", slots() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SessionAttributes", sessionAttributes() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Message", message() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SentimentResponse", sentimentResponse()).add("MessageFormat", messageFormatAsString())
                .add("DialogState", dialogStateAsString()).add("SlotToElicit", slotToElicit())
                .add("ResponseCard", responseCard()).add("SessionId", sessionId()).add("BotVersion", botVersion())
                .add("ActiveContexts", activeContexts() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "intentName":
            return Optional.ofNullable(clazz.cast(intentName()));
        case "nluIntentConfidence":
            return Optional.ofNullable(clazz.cast(nluIntentConfidence()));
        case "alternativeIntents":
            return Optional.ofNullable(clazz.cast(alternativeIntents()));
        case "slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "sessionAttributes":
            return Optional.ofNullable(clazz.cast(sessionAttributes()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "sentimentResponse":
            return Optional.ofNullable(clazz.cast(sentimentResponse()));
        case "messageFormat":
            return Optional.ofNullable(clazz.cast(messageFormatAsString()));
        case "dialogState":
            return Optional.ofNullable(clazz.cast(dialogStateAsString()));
        case "slotToElicit":
            return Optional.ofNullable(clazz.cast(slotToElicit()));
        case "responseCard":
            return Optional.ofNullable(clazz.cast(responseCard()));
        case "sessionId":
            return Optional.ofNullable(clazz.cast(sessionId()));
        case "botVersion":
            return Optional.ofNullable(clazz.cast(botVersion()));
        case "activeContexts":
            return Optional.ofNullable(clazz.cast(activeContexts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PostTextResponse, T> g) {
        return obj -> g.apply((PostTextResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LexRuntimeResponse.Builder, SdkPojo, CopyableBuilder<Builder, PostTextResponse> {
        /**
         * <p>
         * The current user intent that Amazon Lex is aware of.
         * </p>
         * 
         * @param intentName
         *        The current user intent that Amazon Lex is aware of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intentName(String intentName);

        /**
         * <p>
         * Provides a score that indicates how confident Amazon Lex is that the returned intent is the one that matches
         * the user's intent. The score is between 0.0 and 1.0. For more information, see <a
         * href="https://docs.aws.amazon.com/lex/latest/dg/confidence-scores.html">Confidence Scores</a>.
         * </p>
         * <p>
         * The score is a relative score, not an absolute score. The score may change based on improvements to Amazon
         * Lex.
         * </p>
         * 
         * @param nluIntentConfidence
         *        Provides a score that indicates how confident Amazon Lex is that the returned intent is the one that
         *        matches the user's intent. The score is between 0.0 and 1.0. For more information, see <a
         *        href="https://docs.aws.amazon.com/lex/latest/dg/confidence-scores.html">Confidence Scores</a>.</p>
         *        <p>
         *        The score is a relative score, not an absolute score. The score may change based on improvements to
         *        Amazon Lex.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nluIntentConfidence(IntentConfidence nluIntentConfidence);

        /**
         * <p>
         * Provides a score that indicates how confident Amazon Lex is that the returned intent is the one that matches
         * the user's intent. The score is between 0.0 and 1.0. For more information, see <a
         * href="https://docs.aws.amazon.com/lex/latest/dg/confidence-scores.html">Confidence Scores</a>.
         * </p>
         * <p>
         * The score is a relative score, not an absolute score. The score may change based on improvements to Amazon
         * Lex.
         * </p>
         * This is a convenience that creates an instance of the {@link IntentConfidence.Builder} avoiding the need to
         * create one manually via {@link IntentConfidence#builder()}.
         *
         * When the {@link Consumer} completes, {@link IntentConfidence.Builder#build()} is called immediately and its
         * result is passed to {@link #nluIntentConfidence(IntentConfidence)}.
         * 
         * @param nluIntentConfidence
         *        a consumer that will call methods on {@link IntentConfidence.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nluIntentConfidence(IntentConfidence)
         */
        default Builder nluIntentConfidence(Consumer<IntentConfidence.Builder> nluIntentConfidence) {
            return nluIntentConfidence(IntentConfidence.builder().applyMutation(nluIntentConfidence).build());
        }

        /**
         * <p>
         * One to four alternative intents that may be applicable to the user's intent.
         * </p>
         * <p>
         * Each alternative includes a score that indicates how confident Amazon Lex is that the intent matches the
         * user's intent. The intents are sorted by the confidence score.
         * </p>
         * 
         * @param alternativeIntents
         *        One to four alternative intents that may be applicable to the user's intent.</p>
         *        <p>
         *        Each alternative includes a score that indicates how confident Amazon Lex is that the intent matches
         *        the user's intent. The intents are sorted by the confidence score.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternativeIntents(Collection<PredictedIntent> alternativeIntents);

        /**
         * <p>
         * One to four alternative intents that may be applicable to the user's intent.
         * </p>
         * <p>
         * Each alternative includes a score that indicates how confident Amazon Lex is that the intent matches the
         * user's intent. The intents are sorted by the confidence score.
         * </p>
         * 
         * @param alternativeIntents
         *        One to four alternative intents that may be applicable to the user's intent.</p>
         *        <p>
         *        Each alternative includes a score that indicates how confident Amazon Lex is that the intent matches
         *        the user's intent. The intents are sorted by the confidence score.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternativeIntents(PredictedIntent... alternativeIntents);

        /**
         * <p>
         * One to four alternative intents that may be applicable to the user's intent.
         * </p>
         * <p>
         * Each alternative includes a score that indicates how confident Amazon Lex is that the intent matches the
         * user's intent. The intents are sorted by the confidence score.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PredictedIntent>.Builder} avoiding the need
         * to create one manually via {@link List<PredictedIntent>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PredictedIntent>.Builder#build()} is called immediately and
         * its result is passed to {@link #alternativeIntents(List<PredictedIntent>)}.
         * 
         * @param alternativeIntents
         *        a consumer that will call methods on {@link List<PredictedIntent>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alternativeIntents(List<PredictedIntent>)
         */
        Builder alternativeIntents(Consumer<PredictedIntent.Builder>... alternativeIntents);

        /**
         * <p>
         * The intent slots that Amazon Lex detected from the user input in the conversation.
         * </p>
         * <p>
         * Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is
         * determined by the <code>valueSelectionStrategy</code> selected when the slot type was created or updated. If
         * <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value provided by the user is
         * returned, if the user value is similar to the slot values. If <code>valueSelectionStrategy</code> is set to
         * <code>TOP_RESOLUTION</code> Amazon Lex returns the first value in the resolution list or, if there is no
         * resolution list, null. If you don't specify a <code>valueSelectionStrategy</code>, the default is
         * <code>ORIGINAL_VALUE</code>.
         * </p>
         * 
         * @param slots
         *        The intent slots that Amazon Lex detected from the user input in the conversation. </p>
         *        <p>
         *        Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is
         *        determined by the <code>valueSelectionStrategy</code> selected when the slot type was created or
         *        updated. If <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value
         *        provided by the user is returned, if the user value is similar to the slot values. If
         *        <code>valueSelectionStrategy</code> is set to <code>TOP_RESOLUTION</code> Amazon Lex returns the first
         *        value in the resolution list or, if there is no resolution list, null. If you don't specify a
         *        <code>valueSelectionStrategy</code>, the default is <code>ORIGINAL_VALUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(Map<String, String> slots);

        /**
         * <p>
         * A map of key-value pairs representing the session-specific context information.
         * </p>
         * 
         * @param sessionAttributes
         *        A map of key-value pairs representing the session-specific context information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionAttributes(Map<String, String> sessionAttributes);

        /**
         * <p>
         * The message to convey to the user. The message can come from the bot's configuration or from a Lambda
         * function.
         * </p>
         * <p>
         * If the intent is not configured with a Lambda function, or if the Lambda function returned
         * <code>Delegate</code> as the <code>dialogAction.type</code> its response, Amazon Lex decides on the next
         * course of action and selects an appropriate message from the bot's configuration based on the current
         * interaction context. For example, if Amazon Lex isn't able to understand user input, it uses a clarification
         * prompt message.
         * </p>
         * <p>
         * When you create an intent you can assign messages to groups. When messages are assigned to groups Amazon Lex
         * returns one message from each group in the response. The message field is an escaped JSON string containing
         * the messages. For more information about the structure of the JSON string returned, see
         * <a>msg-prompts-formats</a>.
         * </p>
         * <p>
         * If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
         * </p>
         * 
         * @param message
         *        The message to convey to the user. The message can come from the bot's configuration or from a Lambda
         *        function.</p>
         *        <p>
         *        If the intent is not configured with a Lambda function, or if the Lambda function returned
         *        <code>Delegate</code> as the <code>dialogAction.type</code> its response, Amazon Lex decides on the
         *        next course of action and selects an appropriate message from the bot's configuration based on the
         *        current interaction context. For example, if Amazon Lex isn't able to understand user input, it uses a
         *        clarification prompt message.
         *        </p>
         *        <p>
         *        When you create an intent you can assign messages to groups. When messages are assigned to groups
         *        Amazon Lex returns one message from each group in the response. The message field is an escaped JSON
         *        string containing the messages. For more information about the structure of the JSON string returned,
         *        see <a>msg-prompts-formats</a>.
         *        </p>
         *        <p>
         *        If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The sentiment expressed in and utterance.
         * </p>
         * <p>
         * When the bot is configured to send utterances to Amazon Comprehend for sentiment analysis, this field
         * contains the result of the analysis.
         * </p>
         * 
         * @param sentimentResponse
         *        The sentiment expressed in and utterance.</p>
         *        <p>
         *        When the bot is configured to send utterances to Amazon Comprehend for sentiment analysis, this field
         *        contains the result of the analysis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sentimentResponse(SentimentResponse sentimentResponse);

        /**
         * <p>
         * The sentiment expressed in and utterance.
         * </p>
         * <p>
         * When the bot is configured to send utterances to Amazon Comprehend for sentiment analysis, this field
         * contains the result of the analysis.
         * </p>
         * This is a convenience that creates an instance of the {@link SentimentResponse.Builder} avoiding the need to
         * create one manually via {@link SentimentResponse#builder()}.
         *
         * When the {@link Consumer} completes, {@link SentimentResponse.Builder#build()} is called immediately and its
         * result is passed to {@link #sentimentResponse(SentimentResponse)}.
         * 
         * @param sentimentResponse
         *        a consumer that will call methods on {@link SentimentResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sentimentResponse(SentimentResponse)
         */
        default Builder sentimentResponse(Consumer<SentimentResponse.Builder> sentimentResponse) {
            return sentimentResponse(SentimentResponse.builder().applyMutation(sentimentResponse).build());
        }

        /**
         * <p>
         * The format of the response message. One of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PlainText</code> - The message contains plain UTF-8 text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSML</code> - The message contains text formatted for voice output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
         * groups that messages were assigned to when the intent was created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param messageFormat
         *        The format of the response message. One of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PlainText</code> - The message contains plain UTF-8 text.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SSML</code> - The message contains text formatted for voice output.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Composite</code> - The message contains an escaped JSON object containing one or more messages
         *        from the groups that messages were assigned to when the intent was created.
         *        </p>
         *        </li>
         * @see MessageFormatType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormatType
         */
        Builder messageFormat(String messageFormat);

        /**
         * <p>
         * The format of the response message. One of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PlainText</code> - The message contains plain UTF-8 text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSML</code> - The message contains text formatted for voice output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
         * groups that messages were assigned to when the intent was created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param messageFormat
         *        The format of the response message. One of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PlainText</code> - The message contains plain UTF-8 text.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SSML</code> - The message contains text formatted for voice output.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Composite</code> - The message contains an escaped JSON object containing one or more messages
         *        from the groups that messages were assigned to when the intent was created.
         *        </p>
         *        </li>
         * @see MessageFormatType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormatType
         */
        Builder messageFormat(MessageFormatType messageFormat);

        /**
         * <p>
         * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
         * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         * </p>
         * <p>
         * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user
         * intent from this utterance, it will return this dialogState.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         * </p>
         * <p>
         * For example, Amazon Lex wants user confirmation before fulfilling an intent.
         * </p>
         * <p>
         * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
         * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
         * OrderDrink).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         * </p>
         * <p>
         * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
         * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in
         * the response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information
         * appropriately.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         * fulfilled the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - Conveys that the conversation with the user failed.
         * </p>
         * <p>
         * This can happen for various reasons including that the user did not provide an appropriate response to
         * prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         * information), or the Lambda function failed to fulfill the intent.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dialogState
         *        Identifies the current state of the user interaction. Amazon Lex returns one of the following values
         *        as <code>dialogState</code>. The client can optionally use this information to customize the user
         *        interface. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         *        </p>
         *        <p>
         *        For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
         *        user intent from this utterance, it will return this dialogState.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         *        </p>
         *        <p>
         *        For example, Amazon Lex wants user confirmation before fulfilling an intent.
         *        </p>
         *        <p>
         *        Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         *        "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
         *        additional information (in these examples, update the crust type slot value, or change intent from
         *        OrderPizza to OrderDrink).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         *        </p>
         *        <p>
         *        For example, suppose that in the response Amazon Lex sends this message:
         *        "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
         *        might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon
         *        Lex can process such additional information appropriately.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         *        fulfilled the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - Conveys that the conversation with the user failed.
         *        </p>
         *        <p>
         *        This can happen for various reasons including that the user did not provide an appropriate response to
         *        prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         *        information), or the Lambda function failed to fulfill the intent.
         *        </p>
         *        </li>
         * @see DialogState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogState
         */
        Builder dialogState(String dialogState);

        /**
         * <p>
         * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
         * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         * </p>
         * <p>
         * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user
         * intent from this utterance, it will return this dialogState.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         * </p>
         * <p>
         * For example, Amazon Lex wants user confirmation before fulfilling an intent.
         * </p>
         * <p>
         * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
         * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
         * OrderDrink).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         * </p>
         * <p>
         * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
         * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in
         * the response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information
         * appropriately.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         * fulfilled the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - Conveys that the conversation with the user failed.
         * </p>
         * <p>
         * This can happen for various reasons including that the user did not provide an appropriate response to
         * prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         * information), or the Lambda function failed to fulfill the intent.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dialogState
         *        Identifies the current state of the user interaction. Amazon Lex returns one of the following values
         *        as <code>dialogState</code>. The client can optionally use this information to customize the user
         *        interface. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         *        </p>
         *        <p>
         *        For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
         *        user intent from this utterance, it will return this dialogState.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         *        </p>
         *        <p>
         *        For example, Amazon Lex wants user confirmation before fulfilling an intent.
         *        </p>
         *        <p>
         *        Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         *        "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
         *        additional information (in these examples, update the crust type slot value, or change intent from
         *        OrderPizza to OrderDrink).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         *        </p>
         *        <p>
         *        For example, suppose that in the response Amazon Lex sends this message:
         *        "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
         *        might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon
         *        Lex can process such additional information appropriately.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         *        fulfilled the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - Conveys that the conversation with the user failed.
         *        </p>
         *        <p>
         *        This can happen for various reasons including that the user did not provide an appropriate response to
         *        prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         *        information), or the Lambda function failed to fulfill the intent.
         *        </p>
         *        </li>
         * @see DialogState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogState
         */
        Builder dialogState(DialogState dialogState);

        /**
         * <p>
         * If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for which
         * Amazon Lex is eliciting a value.
         * </p>
         * 
         * @param slotToElicit
         *        If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for
         *        which Amazon Lex is eliciting a value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotToElicit(String slotToElicit);

        /**
         * <p>
         * Represents the options that the user has to respond to the current prompt. Response Card can come from the
         * bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a code hook
         * (Lambda function).
         * </p>
         * 
         * @param responseCard
         *        Represents the options that the user has to respond to the current prompt. Response Card can come from
         *        the bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a
         *        code hook (Lambda function).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCard(ResponseCard responseCard);

        /**
         * <p>
         * Represents the options that the user has to respond to the current prompt. Response Card can come from the
         * bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a code hook
         * (Lambda function).
         * </p>
         * This is a convenience that creates an instance of the {@link ResponseCard.Builder} avoiding the need to
         * create one manually via {@link ResponseCard#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResponseCard.Builder#build()} is called immediately and its
         * result is passed to {@link #responseCard(ResponseCard)}.
         * 
         * @param responseCard
         *        a consumer that will call methods on {@link ResponseCard.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #responseCard(ResponseCard)
         */
        default Builder responseCard(Consumer<ResponseCard.Builder> responseCard) {
            return responseCard(ResponseCard.builder().applyMutation(responseCard).build());
        }

        /**
         * <p>
         * A unique identifier for the session.
         * </p>
         * 
         * @param sessionId
         *        A unique identifier for the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionId(String sessionId);

        /**
         * <p>
         * The version of the bot that responded to the conversation. You can use this information to help determine if
         * one version of a bot is performing better than another version.
         * </p>
         * 
         * @param botVersion
         *        The version of the bot that responded to the conversation. You can use this information to help
         *        determine if one version of a bot is performing better than another version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botVersion(String botVersion);

        /**
         * <p>
         * A list of active contexts for the session. A context can be set when an intent is fulfilled or by calling the
         * <code>PostContent</code>, <code>PostText</code>, or <code>PutSession</code> operation.
         * </p>
         * <p>
         * You can use a context to control the intents that can follow up an intent, or to modify the operation of your
         * application.
         * </p>
         * 
         * @param activeContexts
         *        A list of active contexts for the session. A context can be set when an intent is fulfilled or by
         *        calling the <code>PostContent</code>, <code>PostText</code>, or <code>PutSession</code> operation.</p>
         *        <p>
         *        You can use a context to control the intents that can follow up an intent, or to modify the operation
         *        of your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeContexts(Collection<ActiveContext> activeContexts);

        /**
         * <p>
         * A list of active contexts for the session. A context can be set when an intent is fulfilled or by calling the
         * <code>PostContent</code>, <code>PostText</code>, or <code>PutSession</code> operation.
         * </p>
         * <p>
         * You can use a context to control the intents that can follow up an intent, or to modify the operation of your
         * application.
         * </p>
         * 
         * @param activeContexts
         *        A list of active contexts for the session. A context can be set when an intent is fulfilled or by
         *        calling the <code>PostContent</code>, <code>PostText</code>, or <code>PutSession</code> operation.</p>
         *        <p>
         *        You can use a context to control the intents that can follow up an intent, or to modify the operation
         *        of your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeContexts(ActiveContext... activeContexts);

        /**
         * <p>
         * A list of active contexts for the session. A context can be set when an intent is fulfilled or by calling the
         * <code>PostContent</code>, <code>PostText</code>, or <code>PutSession</code> operation.
         * </p>
         * <p>
         * You can use a context to control the intents that can follow up an intent, or to modify the operation of your
         * application.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ActiveContext>.Builder} avoiding the need
         * to create one manually via {@link List<ActiveContext>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ActiveContext>.Builder#build()} is called immediately and
         * its result is passed to {@link #activeContexts(List<ActiveContext>)}.
         * 
         * @param activeContexts
         *        a consumer that will call methods on {@link List<ActiveContext>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #activeContexts(List<ActiveContext>)
         */
        Builder activeContexts(Consumer<ActiveContext.Builder>... activeContexts);
    }

    static final class BuilderImpl extends LexRuntimeResponse.BuilderImpl implements Builder {
        private String intentName;

        private IntentConfidence nluIntentConfidence;

        private List<PredictedIntent> alternativeIntents = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> slots = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> sessionAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String message;

        private SentimentResponse sentimentResponse;

        private String messageFormat;

        private String dialogState;

        private String slotToElicit;

        private ResponseCard responseCard;

        private String sessionId;

        private String botVersion;

        private List<ActiveContext> activeContexts = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PostTextResponse model) {
            super(model);
            intentName(model.intentName);
            nluIntentConfidence(model.nluIntentConfidence);
            alternativeIntents(model.alternativeIntents);
            slots(model.slots);
            sessionAttributes(model.sessionAttributes);
            message(model.message);
            sentimentResponse(model.sentimentResponse);
            messageFormat(model.messageFormat);
            dialogState(model.dialogState);
            slotToElicit(model.slotToElicit);
            responseCard(model.responseCard);
            sessionId(model.sessionId);
            botVersion(model.botVersion);
            activeContexts(model.activeContexts);
        }

        public final String getIntentName() {
            return intentName;
        }

        @Override
        public final Builder intentName(String intentName) {
            this.intentName = intentName;
            return this;
        }

        public final void setIntentName(String intentName) {
            this.intentName = intentName;
        }

        public final IntentConfidence.Builder getNluIntentConfidence() {
            return nluIntentConfidence != null ? nluIntentConfidence.toBuilder() : null;
        }

        @Override
        public final Builder nluIntentConfidence(IntentConfidence nluIntentConfidence) {
            this.nluIntentConfidence = nluIntentConfidence;
            return this;
        }

        public final void setNluIntentConfidence(IntentConfidence.BuilderImpl nluIntentConfidence) {
            this.nluIntentConfidence = nluIntentConfidence != null ? nluIntentConfidence.build() : null;
        }

        public final Collection<PredictedIntent.Builder> getAlternativeIntents() {
            if (alternativeIntents instanceof SdkAutoConstructList) {
                return null;
            }
            return alternativeIntents != null ? alternativeIntents.stream().map(PredictedIntent::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder alternativeIntents(Collection<PredictedIntent> alternativeIntents) {
            this.alternativeIntents = IntentListCopier.copy(alternativeIntents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alternativeIntents(PredictedIntent... alternativeIntents) {
            alternativeIntents(Arrays.asList(alternativeIntents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alternativeIntents(Consumer<PredictedIntent.Builder>... alternativeIntents) {
            alternativeIntents(Stream.of(alternativeIntents).map(c -> PredictedIntent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAlternativeIntents(Collection<PredictedIntent.BuilderImpl> alternativeIntents) {
            this.alternativeIntents = IntentListCopier.copyFromBuilder(alternativeIntents);
        }

        public final Map<String, String> getSlots() {
            if (slots instanceof SdkAutoConstructMap) {
                return null;
            }
            return slots;
        }

        @Override
        public final Builder slots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
            return this;
        }

        public final void setSlots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
        }

        public final Map<String, String> getSessionAttributes() {
            if (sessionAttributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return sessionAttributes;
        }

        @Override
        public final Builder sessionAttributes(Map<String, String> sessionAttributes) {
            this.sessionAttributes = StringMapCopier.copy(sessionAttributes);
            return this;
        }

        public final void setSessionAttributes(Map<String, String> sessionAttributes) {
            this.sessionAttributes = StringMapCopier.copy(sessionAttributes);
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final SentimentResponse.Builder getSentimentResponse() {
            return sentimentResponse != null ? sentimentResponse.toBuilder() : null;
        }

        @Override
        public final Builder sentimentResponse(SentimentResponse sentimentResponse) {
            this.sentimentResponse = sentimentResponse;
            return this;
        }

        public final void setSentimentResponse(SentimentResponse.BuilderImpl sentimentResponse) {
            this.sentimentResponse = sentimentResponse != null ? sentimentResponse.build() : null;
        }

        public final String getMessageFormat() {
            return messageFormat;
        }

        @Override
        public final Builder messageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
            return this;
        }

        @Override
        public final Builder messageFormat(MessageFormatType messageFormat) {
            this.messageFormat(messageFormat == null ? null : messageFormat.toString());
            return this;
        }

        public final void setMessageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
        }

        public final String getDialogState() {
            return dialogState;
        }

        @Override
        public final Builder dialogState(String dialogState) {
            this.dialogState = dialogState;
            return this;
        }

        @Override
        public final Builder dialogState(DialogState dialogState) {
            this.dialogState(dialogState == null ? null : dialogState.toString());
            return this;
        }

        public final void setDialogState(String dialogState) {
            this.dialogState = dialogState;
        }

        public final String getSlotToElicit() {
            return slotToElicit;
        }

        @Override
        public final Builder slotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
            return this;
        }

        public final void setSlotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
        }

        public final ResponseCard.Builder getResponseCard() {
            return responseCard != null ? responseCard.toBuilder() : null;
        }

        @Override
        public final Builder responseCard(ResponseCard responseCard) {
            this.responseCard = responseCard;
            return this;
        }

        public final void setResponseCard(ResponseCard.BuilderImpl responseCard) {
            this.responseCard = responseCard != null ? responseCard.build() : null;
        }

        public final String getSessionId() {
            return sessionId;
        }

        @Override
        public final Builder sessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public final void setSessionId(String sessionId) {
            this.sessionId = sessionId;
        }

        public final String getBotVersion() {
            return botVersion;
        }

        @Override
        public final Builder botVersion(String botVersion) {
            this.botVersion = botVersion;
            return this;
        }

        public final void setBotVersion(String botVersion) {
            this.botVersion = botVersion;
        }

        public final Collection<ActiveContext.Builder> getActiveContexts() {
            if (activeContexts instanceof SdkAutoConstructList) {
                return null;
            }
            return activeContexts != null ? activeContexts.stream().map(ActiveContext::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder activeContexts(Collection<ActiveContext> activeContexts) {
            this.activeContexts = ActiveContextsListCopier.copy(activeContexts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder activeContexts(ActiveContext... activeContexts) {
            activeContexts(Arrays.asList(activeContexts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder activeContexts(Consumer<ActiveContext.Builder>... activeContexts) {
            activeContexts(Stream.of(activeContexts).map(c -> ActiveContext.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setActiveContexts(Collection<ActiveContext.BuilderImpl> activeContexts) {
            this.activeContexts = ActiveContextsListCopier.copyFromBuilder(activeContexts);
        }

        @Override
        public PostTextResponse build() {
            return new PostTextResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
