/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the next action that the bot should take in its interaction with the user and provides information about
 * the context in which the action takes place. Use the <code>DialogAction</code> data type to set the interaction to a
 * specific state, or to return the interaction to a previous state.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DialogAction implements SdkPojo, Serializable, ToCopyableBuilder<DialogAction.Builder, DialogAction> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(DialogAction::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> INTENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("intentName").getter(getter(DialogAction::intentName)).setter(setter(Builder::intentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("intentName").build()).build();

    private static final SdkField<Map<String, String>> SLOTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("slots")
            .getter(getter(DialogAction::slots))
            .setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slots").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> SLOT_TO_ELICIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("slotToElicit").getter(getter(DialogAction::slotToElicit)).setter(setter(Builder::slotToElicit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotToElicit").build()).build();

    private static final SdkField<String> FULFILLMENT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("fulfillmentState").getter(getter(DialogAction::fulfillmentStateAsString))
            .setter(setter(Builder::fulfillmentState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fulfillmentState").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(DialogAction::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<String> MESSAGE_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("messageFormat").getter(getter(DialogAction::messageFormatAsString))
            .setter(setter(Builder::messageFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messageFormat").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, INTENT_NAME_FIELD,
            SLOTS_FIELD, SLOT_TO_ELICIT_FIELD, FULFILLMENT_STATE_FIELD, MESSAGE_FIELD, MESSAGE_FORMAT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String intentName;

    private final Map<String, String> slots;

    private final String slotToElicit;

    private final String fulfillmentState;

    private final String message;

    private final String messageFormat;

    private DialogAction(BuilderImpl builder) {
        this.type = builder.type;
        this.intentName = builder.intentName;
        this.slots = builder.slots;
        this.slotToElicit = builder.slotToElicit;
        this.fulfillmentState = builder.fulfillmentState;
        this.message = builder.message;
        this.messageFormat = builder.messageFormat;
    }

    /**
     * <p>
     * The next action that the bot should take in its interaction with the user. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     * fulfilled. This is a yes/no question such as "Place the order?"
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the statement
     * "Your order has been placed" does not require a response.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Delegate</code> - The next action is determined by Amazon Lex.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DialogActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The next action that the bot should take in its interaction with the user. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     *         fulfilled. This is a yes/no question such as "Place the order?"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
     *         statement "Your order has been placed" does not require a response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Delegate</code> - The next action is determined by Amazon Lex.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     *         </p>
     *         </li>
     * @see DialogActionType
     */
    public DialogActionType type() {
        return DialogActionType.fromValue(type);
    }

    /**
     * <p>
     * The next action that the bot should take in its interaction with the user. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     * fulfilled. This is a yes/no question such as "Place the order?"
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the statement
     * "Your order has been placed" does not require a response.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Delegate</code> - The next action is determined by Amazon Lex.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DialogActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The next action that the bot should take in its interaction with the user. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     *         fulfilled. This is a yes/no question such as "Place the order?"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
     *         statement "Your order has been placed" does not require a response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Delegate</code> - The next action is determined by Amazon Lex.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     *         </p>
     *         </li>
     * @see DialogActionType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The name of the intent.
     * </p>
     * 
     * @return The name of the intent.
     */
    public String intentName() {
        return intentName;
    }

    /**
     * Returns true if the Slots property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSlots() {
        return slots != null && !(slots instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Map of the slots that have been gathered and their values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSlots()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Map of the slots that have been gathered and their values.
     */
    public Map<String, String> slots() {
        return slots;
    }

    /**
     * <p>
     * The name of the slot that should be elicited from the user.
     * </p>
     * 
     * @return The name of the slot that should be elicited from the user.
     */
    public String slotToElicit() {
        return slotToElicit;
    }

    /**
     * <p>
     * The fulfillment state of the intent. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
     * ready to be fulfilled by the client application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fulfillmentState}
     * will return {@link FulfillmentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fulfillmentStateAsString}.
     * </p>
     * 
     * @return The fulfillment state of the intent. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
     *         intent ready to be fulfilled by the client application.
     *         </p>
     *         </li>
     * @see FulfillmentState
     */
    public FulfillmentState fulfillmentState() {
        return FulfillmentState.fromValue(fulfillmentState);
    }

    /**
     * <p>
     * The fulfillment state of the intent. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
     * ready to be fulfilled by the client application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fulfillmentState}
     * will return {@link FulfillmentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fulfillmentStateAsString}.
     * </p>
     * 
     * @return The fulfillment state of the intent. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
     *         intent ready to be fulfilled by the client application.
     *         </p>
     *         </li>
     * @see FulfillmentState
     */
    public String fulfillmentStateAsString() {
        return fulfillmentState;
    }

    /**
     * <p>
     * The message that should be shown to the user. If you don't specify a message, Amazon Lex will use the message
     * configured for the intent.
     * </p>
     * 
     * @return The message that should be shown to the user. If you don't specify a message, Amazon Lex will use the
     *         message configured for the intent.
     */
    public String message() {
        return message;
    }

    /**
     * <ul>
     * <li>
     * <p>
     * <code>PlainText</code> - The message contains plain UTF-8 text.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CustomPayload</code> - The message is a custom format for the client.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSML</code> - The message contains text formatted for voice output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages. For more
     * information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message
     * Groups</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormatType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageFormatAsString}.
     * </p>
     * 
     * @return <li>
     *         <p>
     *         <code>PlainText</code> - The message contains plain UTF-8 text.
     *         </p>
     *         </li> <li>
     *         <p>
     *         <code>CustomPayload</code> - The message is a custom format for the client.
     *         </p>
     *         </li> <li>
     *         <p>
     *         <code>SSML</code> - The message contains text formatted for voice output.
     *         </p>
     *         </li> <li>
     *         <p>
     *         <code>Composite</code> - The message contains an escaped JSON object containing one or more messages. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message Groups</a>.
     *         </p>
     *         </li>
     * @see MessageFormatType
     */
    public MessageFormatType messageFormat() {
        return MessageFormatType.fromValue(messageFormat);
    }

    /**
     * <ul>
     * <li>
     * <p>
     * <code>PlainText</code> - The message contains plain UTF-8 text.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CustomPayload</code> - The message is a custom format for the client.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSML</code> - The message contains text formatted for voice output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages. For more
     * information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message
     * Groups</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormatType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageFormatAsString}.
     * </p>
     * 
     * @return <li>
     *         <p>
     *         <code>PlainText</code> - The message contains plain UTF-8 text.
     *         </p>
     *         </li> <li>
     *         <p>
     *         <code>CustomPayload</code> - The message is a custom format for the client.
     *         </p>
     *         </li> <li>
     *         <p>
     *         <code>SSML</code> - The message contains text formatted for voice output.
     *         </p>
     *         </li> <li>
     *         <p>
     *         <code>Composite</code> - The message contains an escaped JSON object containing one or more messages. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message Groups</a>.
     *         </p>
     *         </li>
     * @see MessageFormatType
     */
    public String messageFormatAsString() {
        return messageFormat;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(intentName());
        hashCode = 31 * hashCode + Objects.hashCode(slots());
        hashCode = 31 * hashCode + Objects.hashCode(slotToElicit());
        hashCode = 31 * hashCode + Objects.hashCode(fulfillmentStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(messageFormatAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DialogAction)) {
            return false;
        }
        DialogAction other = (DialogAction) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(intentName(), other.intentName())
                && Objects.equals(slots(), other.slots()) && Objects.equals(slotToElicit(), other.slotToElicit())
                && Objects.equals(fulfillmentStateAsString(), other.fulfillmentStateAsString())
                && Objects.equals(message(), other.message())
                && Objects.equals(messageFormatAsString(), other.messageFormatAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DialogAction").add("Type", typeAsString()).add("IntentName", intentName())
                .add("Slots", slots() == null ? null : "*** Sensitive Data Redacted ***").add("SlotToElicit", slotToElicit())
                .add("FulfillmentState", fulfillmentStateAsString())
                .add("Message", message() == null ? null : "*** Sensitive Data Redacted ***")
                .add("MessageFormat", messageFormatAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "intentName":
            return Optional.ofNullable(clazz.cast(intentName()));
        case "slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "slotToElicit":
            return Optional.ofNullable(clazz.cast(slotToElicit()));
        case "fulfillmentState":
            return Optional.ofNullable(clazz.cast(fulfillmentStateAsString()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "messageFormat":
            return Optional.ofNullable(clazz.cast(messageFormatAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DialogAction, T> g) {
        return obj -> g.apply((DialogAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DialogAction> {
        /**
         * <p>
         * The next action that the bot should take in its interaction with the user. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
         * fulfilled. This is a yes/no question such as "Place the order?"
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         * statement "Your order has been placed" does not require a response.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Delegate</code> - The next action is determined by Amazon Lex.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The next action that the bot should take in its interaction with the user. The possible values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to
         *        be fulfilled. This is a yes/no question such as "Place the order?"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         *        statement "Your order has been placed" does not require a response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Delegate</code> - The next action is determined by Amazon Lex.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         *        </p>
         *        </li>
         * @see DialogActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogActionType
         */
        Builder type(String type);

        /**
         * <p>
         * The next action that the bot should take in its interaction with the user. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
         * fulfilled. This is a yes/no question such as "Place the order?"
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         * statement "Your order has been placed" does not require a response.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Delegate</code> - The next action is determined by Amazon Lex.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The next action that the bot should take in its interaction with the user. The possible values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to
         *        be fulfilled. This is a yes/no question such as "Place the order?"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         *        statement "Your order has been placed" does not require a response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Delegate</code> - The next action is determined by Amazon Lex.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         *        </p>
         *        </li>
         * @see DialogActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogActionType
         */
        Builder type(DialogActionType type);

        /**
         * <p>
         * The name of the intent.
         * </p>
         * 
         * @param intentName
         *        The name of the intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intentName(String intentName);

        /**
         * <p>
         * Map of the slots that have been gathered and their values.
         * </p>
         * 
         * @param slots
         *        Map of the slots that have been gathered and their values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(Map<String, String> slots);

        /**
         * <p>
         * The name of the slot that should be elicited from the user.
         * </p>
         * 
         * @param slotToElicit
         *        The name of the slot that should be elicited from the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotToElicit(String slotToElicit);

        /**
         * <p>
         * The fulfillment state of the intent. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
         * ready to be fulfilled by the client application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param fulfillmentState
         *        The fulfillment state of the intent. The possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
         *        intent ready to be fulfilled by the client application.
         *        </p>
         *        </li>
         * @see FulfillmentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FulfillmentState
         */
        Builder fulfillmentState(String fulfillmentState);

        /**
         * <p>
         * The fulfillment state of the intent. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
         * ready to be fulfilled by the client application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param fulfillmentState
         *        The fulfillment state of the intent. The possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
         *        intent ready to be fulfilled by the client application.
         *        </p>
         *        </li>
         * @see FulfillmentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FulfillmentState
         */
        Builder fulfillmentState(FulfillmentState fulfillmentState);

        /**
         * <p>
         * The message that should be shown to the user. If you don't specify a message, Amazon Lex will use the message
         * configured for the intent.
         * </p>
         * 
         * @param message
         *        The message that should be shown to the user. If you don't specify a message, Amazon Lex will use the
         *        message configured for the intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <ul>
         * <li>
         * <p>
         * <code>PlainText</code> - The message contains plain UTF-8 text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CustomPayload</code> - The message is a custom format for the client.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSML</code> - The message contains text formatted for voice output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message Groups</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param messageFormat
         *        <li>
         *        <p>
         *        <code>PlainText</code> - The message contains plain UTF-8 text.
         *        </p>
         *        </li> <li>
         *        <p>
         *        <code>CustomPayload</code> - The message is a custom format for the client.
         *        </p>
         *        </li> <li>
         *        <p>
         *        <code>SSML</code> - The message contains text formatted for voice output.
         *        </p>
         *        </li> <li>
         *        <p>
         *        <code>Composite</code> - The message contains an escaped JSON object containing one or more messages.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message Groups</a>.
         *        </p>
         *        </li>
         * @see MessageFormatType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormatType
         */
        Builder messageFormat(String messageFormat);

        /**
         * <ul>
         * <li>
         * <p>
         * <code>PlainText</code> - The message contains plain UTF-8 text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CustomPayload</code> - The message is a custom format for the client.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSML</code> - The message contains text formatted for voice output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message Groups</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param messageFormat
         *        <li>
         *        <p>
         *        <code>PlainText</code> - The message contains plain UTF-8 text.
         *        </p>
         *        </li> <li>
         *        <p>
         *        <code>CustomPayload</code> - The message is a custom format for the client.
         *        </p>
         *        </li> <li>
         *        <p>
         *        <code>SSML</code> - The message contains text formatted for voice output.
         *        </p>
         *        </li> <li>
         *        <p>
         *        <code>Composite</code> - The message contains an escaped JSON object containing one or more messages.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/lex/latest/dg/howitworks-manage-prompts.html">Message Groups</a>.
         *        </p>
         *        </li>
         * @see MessageFormatType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormatType
         */
        Builder messageFormat(MessageFormatType messageFormat);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String intentName;

        private Map<String, String> slots = DefaultSdkAutoConstructMap.getInstance();

        private String slotToElicit;

        private String fulfillmentState;

        private String message;

        private String messageFormat;

        private BuilderImpl() {
        }

        private BuilderImpl(DialogAction model) {
            type(model.type);
            intentName(model.intentName);
            slots(model.slots);
            slotToElicit(model.slotToElicit);
            fulfillmentState(model.fulfillmentState);
            message(model.message);
            messageFormat(model.messageFormat);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DialogActionType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getIntentName() {
            return intentName;
        }

        @Override
        public final Builder intentName(String intentName) {
            this.intentName = intentName;
            return this;
        }

        public final void setIntentName(String intentName) {
            this.intentName = intentName;
        }

        public final Map<String, String> getSlots() {
            return slots;
        }

        @Override
        public final Builder slots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
            return this;
        }

        public final void setSlots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
        }

        public final String getSlotToElicit() {
            return slotToElicit;
        }

        @Override
        public final Builder slotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
            return this;
        }

        public final void setSlotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
        }

        public final String getFulfillmentState() {
            return fulfillmentState;
        }

        @Override
        public final Builder fulfillmentState(String fulfillmentState) {
            this.fulfillmentState = fulfillmentState;
            return this;
        }

        @Override
        public final Builder fulfillmentState(FulfillmentState fulfillmentState) {
            this.fulfillmentState(fulfillmentState == null ? null : fulfillmentState.toString());
            return this;
        }

        public final void setFulfillmentState(String fulfillmentState) {
            this.fulfillmentState = fulfillmentState;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final String getMessageFormat() {
            return messageFormat;
        }

        @Override
        public final Builder messageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
            return this;
        }

        @Override
        public final Builder messageFormat(MessageFormatType messageFormat) {
            this.messageFormat(messageFormat == null ? null : messageFormat.toString());
            return this;
        }

        public final void setMessageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
        }

        @Override
        public DialogAction build() {
            return new DialogAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
