/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An intent that Amazon Lex suggests satisfies the user's intent. Includes the name of the intent, the confidence that
 * Amazon Lex has that the user's intent is satisfied, and the slots defined for the intent.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PredictedIntent implements SdkPojo, Serializable, ToCopyableBuilder<PredictedIntent.Builder, PredictedIntent> {
    private static final SdkField<String> INTENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("intentName").getter(getter(PredictedIntent::intentName)).setter(setter(Builder::intentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("intentName").build()).build();

    private static final SdkField<IntentConfidence> NLU_INTENT_CONFIDENCE_FIELD = SdkField
            .<IntentConfidence> builder(MarshallingType.SDK_POJO).memberName("nluIntentConfidence")
            .getter(getter(PredictedIntent::nluIntentConfidence)).setter(setter(Builder::nluIntentConfidence))
            .constructor(IntentConfidence::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nluIntentConfidence").build())
            .build();

    private static final SdkField<Map<String, String>> SLOTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("slots")
            .getter(getter(PredictedIntent::slots))
            .setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slots").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INTENT_NAME_FIELD,
            NLU_INTENT_CONFIDENCE_FIELD, SLOTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String intentName;

    private final IntentConfidence nluIntentConfidence;

    private final Map<String, String> slots;

    private PredictedIntent(BuilderImpl builder) {
        this.intentName = builder.intentName;
        this.nluIntentConfidence = builder.nluIntentConfidence;
        this.slots = builder.slots;
    }

    /**
     * <p>
     * The name of the intent that Amazon Lex suggests satisfies the user's intent.
     * </p>
     * 
     * @return The name of the intent that Amazon Lex suggests satisfies the user's intent.
     */
    public String intentName() {
        return intentName;
    }

    /**
     * <p>
     * Indicates how confident Amazon Lex is that an intent satisfies the user's intent.
     * </p>
     * 
     * @return Indicates how confident Amazon Lex is that an intent satisfies the user's intent.
     */
    public IntentConfidence nluIntentConfidence() {
        return nluIntentConfidence;
    }

    /**
     * Returns true if the Slots property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSlots() {
        return slots != null && !(slots instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The slot and slot values associated with the predicted intent.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSlots()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The slot and slot values associated with the predicted intent.
     */
    public Map<String, String> slots() {
        return slots;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(intentName());
        hashCode = 31 * hashCode + Objects.hashCode(nluIntentConfidence());
        hashCode = 31 * hashCode + Objects.hashCode(slots());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PredictedIntent)) {
            return false;
        }
        PredictedIntent other = (PredictedIntent) obj;
        return Objects.equals(intentName(), other.intentName())
                && Objects.equals(nluIntentConfidence(), other.nluIntentConfidence()) && Objects.equals(slots(), other.slots());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PredictedIntent").add("IntentName", intentName())
                .add("NluIntentConfidence", nluIntentConfidence())
                .add("Slots", slots() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "intentName":
            return Optional.ofNullable(clazz.cast(intentName()));
        case "nluIntentConfidence":
            return Optional.ofNullable(clazz.cast(nluIntentConfidence()));
        case "slots":
            return Optional.ofNullable(clazz.cast(slots()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PredictedIntent, T> g) {
        return obj -> g.apply((PredictedIntent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PredictedIntent> {
        /**
         * <p>
         * The name of the intent that Amazon Lex suggests satisfies the user's intent.
         * </p>
         * 
         * @param intentName
         *        The name of the intent that Amazon Lex suggests satisfies the user's intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intentName(String intentName);

        /**
         * <p>
         * Indicates how confident Amazon Lex is that an intent satisfies the user's intent.
         * </p>
         * 
         * @param nluIntentConfidence
         *        Indicates how confident Amazon Lex is that an intent satisfies the user's intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nluIntentConfidence(IntentConfidence nluIntentConfidence);

        /**
         * <p>
         * Indicates how confident Amazon Lex is that an intent satisfies the user's intent.
         * </p>
         * This is a convenience that creates an instance of the {@link IntentConfidence.Builder} avoiding the need to
         * create one manually via {@link IntentConfidence#builder()}.
         *
         * When the {@link Consumer} completes, {@link IntentConfidence.Builder#build()} is called immediately and its
         * result is passed to {@link #nluIntentConfidence(IntentConfidence)}.
         * 
         * @param nluIntentConfidence
         *        a consumer that will call methods on {@link IntentConfidence.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nluIntentConfidence(IntentConfidence)
         */
        default Builder nluIntentConfidence(Consumer<IntentConfidence.Builder> nluIntentConfidence) {
            return nluIntentConfidence(IntentConfidence.builder().applyMutation(nluIntentConfidence).build());
        }

        /**
         * <p>
         * The slot and slot values associated with the predicted intent.
         * </p>
         * 
         * @param slots
         *        The slot and slot values associated with the predicted intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(Map<String, String> slots);
    }

    static final class BuilderImpl implements Builder {
        private String intentName;

        private IntentConfidence nluIntentConfidence;

        private Map<String, String> slots = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PredictedIntent model) {
            intentName(model.intentName);
            nluIntentConfidence(model.nluIntentConfidence);
            slots(model.slots);
        }

        public final String getIntentName() {
            return intentName;
        }

        @Override
        public final Builder intentName(String intentName) {
            this.intentName = intentName;
            return this;
        }

        public final void setIntentName(String intentName) {
            this.intentName = intentName;
        }

        public final IntentConfidence.Builder getNluIntentConfidence() {
            return nluIntentConfidence != null ? nluIntentConfidence.toBuilder() : null;
        }

        @Override
        public final Builder nluIntentConfidence(IntentConfidence nluIntentConfidence) {
            this.nluIntentConfidence = nluIntentConfidence;
            return this;
        }

        public final void setNluIntentConfidence(IntentConfidence.BuilderImpl nluIntentConfidence) {
            this.nluIntentConfidence = nluIntentConfidence != null ? nluIntentConfidence.build() : null;
        }

        public final Map<String, String> getSlots() {
            return slots;
        }

        @Override
        public final Builder slots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
            return this;
        }

        public final void setSlots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
        }

        @Override
        public PredictedIntent build() {
            return new PredictedIntent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
