/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the state of an intent. You can use this information to get the current state of an intent
 * so that you can process the intent, or so that you can return the intent to its previous state.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IntentSummary implements SdkPojo, Serializable, ToCopyableBuilder<IntentSummary.Builder, IntentSummary> {
    private static final SdkField<String> INTENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IntentSummary::intentName)).setter(setter(Builder::intentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("intentName").build()).build();

    private static final SdkField<String> CHECKPOINT_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IntentSummary::checkpointLabel)).setter(setter(Builder::checkpointLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("checkpointLabel").build()).build();

    private static final SdkField<Map<String, String>> SLOTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(IntentSummary::slots))
            .setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slots").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CONFIRMATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IntentSummary::confirmationStatusAsString)).setter(setter(Builder::confirmationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("confirmationStatus").build())
            .build();

    private static final SdkField<String> DIALOG_ACTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IntentSummary::dialogActionTypeAsString)).setter(setter(Builder::dialogActionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dialogActionType").build()).build();

    private static final SdkField<String> FULFILLMENT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IntentSummary::fulfillmentStateAsString)).setter(setter(Builder::fulfillmentState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fulfillmentState").build()).build();

    private static final SdkField<String> SLOT_TO_ELICIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IntentSummary::slotToElicit)).setter(setter(Builder::slotToElicit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotToElicit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INTENT_NAME_FIELD,
            CHECKPOINT_LABEL_FIELD, SLOTS_FIELD, CONFIRMATION_STATUS_FIELD, DIALOG_ACTION_TYPE_FIELD, FULFILLMENT_STATE_FIELD,
            SLOT_TO_ELICIT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String intentName;

    private final String checkpointLabel;

    private final Map<String, String> slots;

    private final String confirmationStatus;

    private final String dialogActionType;

    private final String fulfillmentState;

    private final String slotToElicit;

    private IntentSummary(BuilderImpl builder) {
        this.intentName = builder.intentName;
        this.checkpointLabel = builder.checkpointLabel;
        this.slots = builder.slots;
        this.confirmationStatus = builder.confirmationStatus;
        this.dialogActionType = builder.dialogActionType;
        this.fulfillmentState = builder.fulfillmentState;
        this.slotToElicit = builder.slotToElicit;
    }

    /**
     * <p>
     * The name of the intent.
     * </p>
     * 
     * @return The name of the intent.
     */
    public String intentName() {
        return intentName;
    }

    /**
     * <p>
     * A user-defined label that identifies a particular intent. You can use this label to return to a previous intent.
     * </p>
     * <p>
     * Use the <code>checkpointLabelFilter</code> parameter of the <code>GetSessionRequest</code> operation to filter
     * the intents returned by the operation to those with only the specified label.
     * </p>
     * 
     * @return A user-defined label that identifies a particular intent. You can use this label to return to a previous
     *         intent. </p>
     *         <p>
     *         Use the <code>checkpointLabelFilter</code> parameter of the <code>GetSessionRequest</code> operation to
     *         filter the intents returned by the operation to those with only the specified label.
     */
    public String checkpointLabel() {
        return checkpointLabel;
    }

    /**
     * <p>
     * Map of the slots that have been gathered and their values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Map of the slots that have been gathered and their values.
     */
    public Map<String, String> slots() {
        return slots;
    }

    /**
     * <p>
     * The status of the intent after the user responds to the confirmation prompt. If the user confirms the intent,
     * Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon Lex sets this value
     * to <code>Denied</code>. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the intent is
     * complete and that it is ready to be fulfilled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Denied</code> - The user has responded "No" to the confirmation prompt.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but did not
     * confirm or deny the prompt.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #confirmationStatus} will return {@link ConfirmationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #confirmationStatusAsString}.
     * </p>
     * 
     * @return The status of the intent after the user responds to the confirmation prompt. If the user confirms the
     *         intent, Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon Lex
     *         sets this value to <code>Denied</code>. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the
     *         intent is complete and that it is ready to be fulfilled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Denied</code> - The user has responded "No" to the confirmation prompt.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but did
     *         not confirm or deny the prompt.
     *         </p>
     *         </li>
     * @see ConfirmationStatus
     */
    public ConfirmationStatus confirmationStatus() {
        return ConfirmationStatus.fromValue(confirmationStatus);
    }

    /**
     * <p>
     * The status of the intent after the user responds to the confirmation prompt. If the user confirms the intent,
     * Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon Lex sets this value
     * to <code>Denied</code>. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the intent is
     * complete and that it is ready to be fulfilled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Denied</code> - The user has responded "No" to the confirmation prompt.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but did not
     * confirm or deny the prompt.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #confirmationStatus} will return {@link ConfirmationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #confirmationStatusAsString}.
     * </p>
     * 
     * @return The status of the intent after the user responds to the confirmation prompt. If the user confirms the
     *         intent, Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon Lex
     *         sets this value to <code>Denied</code>. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the
     *         intent is complete and that it is ready to be fulfilled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Denied</code> - The user has responded "No" to the confirmation prompt.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but did
     *         not confirm or deny the prompt.
     *         </p>
     *         </li>
     * @see ConfirmationStatus
     */
    public String confirmationStatusAsString() {
        return confirmationStatus;
    }

    /**
     * <p>
     * The next action that the bot should take in its interaction with the user. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     * fulfilled. This is a yes/no question such as "Place the order?"
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the statement
     * "Your order has been placed" does not require a response.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dialogActionType}
     * will return {@link DialogActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #dialogActionTypeAsString}.
     * </p>
     * 
     * @return The next action that the bot should take in its interaction with the user. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     *         fulfilled. This is a yes/no question such as "Place the order?"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
     *         statement "Your order has been placed" does not require a response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     *         </p>
     *         </li>
     * @see DialogActionType
     */
    public DialogActionType dialogActionType() {
        return DialogActionType.fromValue(dialogActionType);
    }

    /**
     * <p>
     * The next action that the bot should take in its interaction with the user. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     * fulfilled. This is a yes/no question such as "Place the order?"
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the statement
     * "Your order has been placed" does not require a response.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dialogActionType}
     * will return {@link DialogActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #dialogActionTypeAsString}.
     * </p>
     * 
     * @return The next action that the bot should take in its interaction with the user. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
     *         fulfilled. This is a yes/no question such as "Place the order?"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
     *         statement "Your order has been placed" does not require a response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
     *         </p>
     *         </li>
     * @see DialogActionType
     */
    public String dialogActionTypeAsString() {
        return dialogActionType;
    }

    /**
     * <p>
     * The fulfillment state of the intent. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
     * ready to be fulfilled by the client application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fulfillmentState}
     * will return {@link FulfillmentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fulfillmentStateAsString}.
     * </p>
     * 
     * @return The fulfillment state of the intent. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
     *         intent ready to be fulfilled by the client application.
     *         </p>
     *         </li>
     * @see FulfillmentState
     */
    public FulfillmentState fulfillmentState() {
        return FulfillmentState.fromValue(fulfillmentState);
    }

    /**
     * <p>
     * The fulfillment state of the intent. The possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
     * ready to be fulfilled by the client application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fulfillmentState}
     * will return {@link FulfillmentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fulfillmentStateAsString}.
     * </p>
     * 
     * @return The fulfillment state of the intent. The possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
     *         intent ready to be fulfilled by the client application.
     *         </p>
     *         </li>
     * @see FulfillmentState
     */
    public String fulfillmentStateAsString() {
        return fulfillmentState;
    }

    /**
     * <p>
     * The next slot to elicit from the user. If there is not slot to elicit, the field is blank.
     * </p>
     * 
     * @return The next slot to elicit from the user. If there is not slot to elicit, the field is blank.
     */
    public String slotToElicit() {
        return slotToElicit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(intentName());
        hashCode = 31 * hashCode + Objects.hashCode(checkpointLabel());
        hashCode = 31 * hashCode + Objects.hashCode(slots());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dialogActionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(fulfillmentStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(slotToElicit());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IntentSummary)) {
            return false;
        }
        IntentSummary other = (IntentSummary) obj;
        return Objects.equals(intentName(), other.intentName()) && Objects.equals(checkpointLabel(), other.checkpointLabel())
                && Objects.equals(slots(), other.slots())
                && Objects.equals(confirmationStatusAsString(), other.confirmationStatusAsString())
                && Objects.equals(dialogActionTypeAsString(), other.dialogActionTypeAsString())
                && Objects.equals(fulfillmentStateAsString(), other.fulfillmentStateAsString())
                && Objects.equals(slotToElicit(), other.slotToElicit());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("IntentSummary").add("IntentName", intentName()).add("CheckpointLabel", checkpointLabel())
                .add("Slots", slots() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ConfirmationStatus", confirmationStatusAsString()).add("DialogActionType", dialogActionTypeAsString())
                .add("FulfillmentState", fulfillmentStateAsString()).add("SlotToElicit", slotToElicit()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "intentName":
            return Optional.ofNullable(clazz.cast(intentName()));
        case "checkpointLabel":
            return Optional.ofNullable(clazz.cast(checkpointLabel()));
        case "slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "confirmationStatus":
            return Optional.ofNullable(clazz.cast(confirmationStatusAsString()));
        case "dialogActionType":
            return Optional.ofNullable(clazz.cast(dialogActionTypeAsString()));
        case "fulfillmentState":
            return Optional.ofNullable(clazz.cast(fulfillmentStateAsString()));
        case "slotToElicit":
            return Optional.ofNullable(clazz.cast(slotToElicit()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IntentSummary, T> g) {
        return obj -> g.apply((IntentSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IntentSummary> {
        /**
         * <p>
         * The name of the intent.
         * </p>
         * 
         * @param intentName
         *        The name of the intent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intentName(String intentName);

        /**
         * <p>
         * A user-defined label that identifies a particular intent. You can use this label to return to a previous
         * intent.
         * </p>
         * <p>
         * Use the <code>checkpointLabelFilter</code> parameter of the <code>GetSessionRequest</code> operation to
         * filter the intents returned by the operation to those with only the specified label.
         * </p>
         * 
         * @param checkpointLabel
         *        A user-defined label that identifies a particular intent. You can use this label to return to a
         *        previous intent. </p>
         *        <p>
         *        Use the <code>checkpointLabelFilter</code> parameter of the <code>GetSessionRequest</code> operation
         *        to filter the intents returned by the operation to those with only the specified label.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checkpointLabel(String checkpointLabel);

        /**
         * <p>
         * Map of the slots that have been gathered and their values.
         * </p>
         * 
         * @param slots
         *        Map of the slots that have been gathered and their values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(Map<String, String> slots);

        /**
         * <p>
         * The status of the intent after the user responds to the confirmation prompt. If the user confirms the intent,
         * Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon Lex sets this
         * value to <code>Denied</code>. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the intent
         * is complete and that it is ready to be fulfilled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Denied</code> - The user has responded "No" to the confirmation prompt.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but did not
         * confirm or deny the prompt.
         * </p>
         * </li>
         * </ul>
         * 
         * @param confirmationStatus
         *        The status of the intent after the user responds to the confirmation prompt. If the user confirms the
         *        intent, Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon
         *        Lex sets this value to <code>Denied</code>. The possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the
         *        intent is complete and that it is ready to be fulfilled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Denied</code> - The user has responded "No" to the confirmation prompt.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but
         *        did not confirm or deny the prompt.
         *        </p>
         *        </li>
         * @see ConfirmationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationStatus
         */
        Builder confirmationStatus(String confirmationStatus);

        /**
         * <p>
         * The status of the intent after the user responds to the confirmation prompt. If the user confirms the intent,
         * Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon Lex sets this
         * value to <code>Denied</code>. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the intent
         * is complete and that it is ready to be fulfilled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Denied</code> - The user has responded "No" to the confirmation prompt.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but did not
         * confirm or deny the prompt.
         * </p>
         * </li>
         * </ul>
         * 
         * @param confirmationStatus
         *        The status of the intent after the user responds to the confirmation prompt. If the user confirms the
         *        intent, Amazon Lex sets this field to <code>Confirmed</code>. If the user denies the intent, Amazon
         *        Lex sets this value to <code>Denied</code>. The possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Confirmed</code> - The user has responded "Yes" to the confirmation prompt, confirming that the
         *        intent is complete and that it is ready to be fulfilled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Denied</code> - The user has responded "No" to the confirmation prompt.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>None</code> - The user has never been prompted for confirmation; or, the user was prompted but
         *        did not confirm or deny the prompt.
         *        </p>
         *        </li>
         * @see ConfirmationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationStatus
         */
        Builder confirmationStatus(ConfirmationStatus confirmationStatus);

        /**
         * <p>
         * The next action that the bot should take in its interaction with the user. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
         * fulfilled. This is a yes/no question such as "Place the order?"
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         * statement "Your order has been placed" does not require a response.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dialogActionType
         *        The next action that the bot should take in its interaction with the user. The possible values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to
         *        be fulfilled. This is a yes/no question such as "Place the order?"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         *        statement "Your order has been placed" does not require a response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         *        </p>
         *        </li>
         * @see DialogActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogActionType
         */
        Builder dialogActionType(String dialogActionType);

        /**
         * <p>
         * The next action that the bot should take in its interaction with the user. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to be
         * fulfilled. This is a yes/no question such as "Place the order?"
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         * statement "Your order has been placed" does not require a response.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dialogActionType
         *        The next action that the bot should take in its interaction with the user. The possible values
         *        are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - The next action is asking the user if the intent is complete and ready to
         *        be fulfilled. This is a yes/no question such as "Place the order?"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Close</code> - Indicates that the there will not be a response from the user. For example, the
         *        statement "Your order has been placed" does not require a response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - The next action is to determine the intent that the user wants to fulfill.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - The next action is to elicit a slot value from the user.
         *        </p>
         *        </li>
         * @see DialogActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogActionType
         */
        Builder dialogActionType(DialogActionType dialogActionType);

        /**
         * <p>
         * The fulfillment state of the intent. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
         * ready to be fulfilled by the client application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param fulfillmentState
         *        The fulfillment state of the intent. The possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
         *        intent ready to be fulfilled by the client application.
         *        </p>
         *        </li>
         * @see FulfillmentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FulfillmentState
         */
        Builder fulfillmentState(String fulfillmentState);

        /**
         * <p>
         * The fulfillment state of the intent. The possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the intent
         * ready to be fulfilled by the client application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param fulfillmentState
         *        The fulfillment state of the intent. The possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The Lambda function associated with the intent failed to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - The intent has fulfilled by the Lambda function associated with the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - All of the information necessary for the intent is present and the
         *        intent ready to be fulfilled by the client application.
         *        </p>
         *        </li>
         * @see FulfillmentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FulfillmentState
         */
        Builder fulfillmentState(FulfillmentState fulfillmentState);

        /**
         * <p>
         * The next slot to elicit from the user. If there is not slot to elicit, the field is blank.
         * </p>
         * 
         * @param slotToElicit
         *        The next slot to elicit from the user. If there is not slot to elicit, the field is blank.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotToElicit(String slotToElicit);
    }

    static final class BuilderImpl implements Builder {
        private String intentName;

        private String checkpointLabel;

        private Map<String, String> slots = DefaultSdkAutoConstructMap.getInstance();

        private String confirmationStatus;

        private String dialogActionType;

        private String fulfillmentState;

        private String slotToElicit;

        private BuilderImpl() {
        }

        private BuilderImpl(IntentSummary model) {
            intentName(model.intentName);
            checkpointLabel(model.checkpointLabel);
            slots(model.slots);
            confirmationStatus(model.confirmationStatus);
            dialogActionType(model.dialogActionType);
            fulfillmentState(model.fulfillmentState);
            slotToElicit(model.slotToElicit);
        }

        public final String getIntentName() {
            return intentName;
        }

        @Override
        public final Builder intentName(String intentName) {
            this.intentName = intentName;
            return this;
        }

        public final void setIntentName(String intentName) {
            this.intentName = intentName;
        }

        public final String getCheckpointLabel() {
            return checkpointLabel;
        }

        @Override
        public final Builder checkpointLabel(String checkpointLabel) {
            this.checkpointLabel = checkpointLabel;
            return this;
        }

        public final void setCheckpointLabel(String checkpointLabel) {
            this.checkpointLabel = checkpointLabel;
        }

        public final Map<String, String> getSlots() {
            return slots;
        }

        @Override
        public final Builder slots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
            return this;
        }

        public final void setSlots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
        }

        public final String getConfirmationStatusAsString() {
            return confirmationStatus;
        }

        @Override
        public final Builder confirmationStatus(String confirmationStatus) {
            this.confirmationStatus = confirmationStatus;
            return this;
        }

        @Override
        public final Builder confirmationStatus(ConfirmationStatus confirmationStatus) {
            this.confirmationStatus(confirmationStatus == null ? null : confirmationStatus.toString());
            return this;
        }

        public final void setConfirmationStatus(String confirmationStatus) {
            this.confirmationStatus = confirmationStatus;
        }

        public final String getDialogActionTypeAsString() {
            return dialogActionType;
        }

        @Override
        public final Builder dialogActionType(String dialogActionType) {
            this.dialogActionType = dialogActionType;
            return this;
        }

        @Override
        public final Builder dialogActionType(DialogActionType dialogActionType) {
            this.dialogActionType(dialogActionType == null ? null : dialogActionType.toString());
            return this;
        }

        public final void setDialogActionType(String dialogActionType) {
            this.dialogActionType = dialogActionType;
        }

        public final String getFulfillmentStateAsString() {
            return fulfillmentState;
        }

        @Override
        public final Builder fulfillmentState(String fulfillmentState) {
            this.fulfillmentState = fulfillmentState;
            return this;
        }

        @Override
        public final Builder fulfillmentState(FulfillmentState fulfillmentState) {
            this.fulfillmentState(fulfillmentState == null ? null : fulfillmentState.toString());
            return this;
        }

        public final void setFulfillmentState(String fulfillmentState) {
            this.fulfillmentState = fulfillmentState;
        }

        public final String getSlotToElicit() {
            return slotToElicit;
        }

        @Override
        public final Builder slotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
            return this;
        }

        public final void setSlotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
        }

        @Override
        public IntentSummary build() {
            return new IntentSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
