/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PostTextResponse extends LexRuntimeResponse implements
        ToCopyableBuilder<PostTextResponse.Builder, PostTextResponse> {
    private static final SdkField<String> INTENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PostTextResponse::intentName)).setter(setter(Builder::intentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("intentName").build()).build();

    private static final SdkField<Map<String, String>> SLOTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(PostTextResponse::slots))
            .setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slots").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> SESSION_ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(PostTextResponse::sessionAttributes))
            .setter(setter(Builder::sessionAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PostTextResponse::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<String> MESSAGE_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PostTextResponse::messageFormatAsString)).setter(setter(Builder::messageFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messageFormat").build()).build();

    private static final SdkField<String> DIALOG_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PostTextResponse::dialogStateAsString)).setter(setter(Builder::dialogState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dialogState").build()).build();

    private static final SdkField<String> SLOT_TO_ELICIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PostTextResponse::slotToElicit)).setter(setter(Builder::slotToElicit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotToElicit").build()).build();

    private static final SdkField<ResponseCard> RESPONSE_CARD_FIELD = SdkField.<ResponseCard> builder(MarshallingType.SDK_POJO)
            .getter(getter(PostTextResponse::responseCard)).setter(setter(Builder::responseCard))
            .constructor(ResponseCard::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseCard").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INTENT_NAME_FIELD,
            SLOTS_FIELD, SESSION_ATTRIBUTES_FIELD, MESSAGE_FIELD, MESSAGE_FORMAT_FIELD, DIALOG_STATE_FIELD, SLOT_TO_ELICIT_FIELD,
            RESPONSE_CARD_FIELD));

    private final String intentName;

    private final Map<String, String> slots;

    private final Map<String, String> sessionAttributes;

    private final String message;

    private final String messageFormat;

    private final String dialogState;

    private final String slotToElicit;

    private final ResponseCard responseCard;

    private PostTextResponse(BuilderImpl builder) {
        super(builder);
        this.intentName = builder.intentName;
        this.slots = builder.slots;
        this.sessionAttributes = builder.sessionAttributes;
        this.message = builder.message;
        this.messageFormat = builder.messageFormat;
        this.dialogState = builder.dialogState;
        this.slotToElicit = builder.slotToElicit;
        this.responseCard = builder.responseCard;
    }

    /**
     * <p>
     * The current user intent that Amazon Lex is aware of.
     * </p>
     * 
     * @return The current user intent that Amazon Lex is aware of.
     */
    public String intentName() {
        return intentName;
    }

    /**
     * <p>
     * The intent slots that Amazon Lex detected from the user input in the conversation.
     * </p>
     * <p>
     * Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is determined
     * by the <code>valueSelectionStrategy</code> selected when the slot type was created or updated. If
     * <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value provided by the user is
     * returned, if the user value is similar to the slot values. If <code>valueSelectionStrategy</code> is set to
     * <code>TOP_RESOLUTION</code> Amazon Lex returns the first value in the resolution list or, if there is no
     * resolution list, null. If you don't specify a <code>valueSelectionStrategy</code>, the default is
     * <code>ORIGINAL_VALUE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The intent slots that Amazon Lex detected from the user input in the conversation. </p>
     *         <p>
     *         Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is
     *         determined by the <code>valueSelectionStrategy</code> selected when the slot type was created or updated.
     *         If <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value provided by the
     *         user is returned, if the user value is similar to the slot values. If <code>valueSelectionStrategy</code>
     *         is set to <code>TOP_RESOLUTION</code> Amazon Lex returns the first value in the resolution list or, if
     *         there is no resolution list, null. If you don't specify a <code>valueSelectionStrategy</code>, the
     *         default is <code>ORIGINAL_VALUE</code>.
     */
    public Map<String, String> slots() {
        return slots;
    }

    /**
     * <p>
     * A map of key-value pairs representing the session-specific context information.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A map of key-value pairs representing the session-specific context information.
     */
    public Map<String, String> sessionAttributes() {
        return sessionAttributes;
    }

    /**
     * <p>
     * The message to convey to the user. The message can come from the bot's configuration or from a Lambda function.
     * </p>
     * <p>
     * If the intent is not configured with a Lambda function, or if the Lambda function returned <code>Delegate</code>
     * as the <code>dialogAction.type</code> its response, Amazon Lex decides on the next course of action and selects
     * an appropriate message from the bot's configuration based on the current interaction context. For example, if
     * Amazon Lex isn't able to understand user input, it uses a clarification prompt message.
     * </p>
     * <p>
     * When you create an intent you can assign messages to groups. When messages are assigned to groups Amazon Lex
     * returns one message from each group in the response. The message field is an escaped JSON string containing the
     * messages. For more information about the structure of the JSON string returned, see <a>msg-prompts-formats</a>.
     * </p>
     * <p>
     * If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
     * </p>
     * 
     * @return The message to convey to the user. The message can come from the bot's configuration or from a Lambda
     *         function.</p>
     *         <p>
     *         If the intent is not configured with a Lambda function, or if the Lambda function returned
     *         <code>Delegate</code> as the <code>dialogAction.type</code> its response, Amazon Lex decides on the next
     *         course of action and selects an appropriate message from the bot's configuration based on the current
     *         interaction context. For example, if Amazon Lex isn't able to understand user input, it uses a
     *         clarification prompt message.
     *         </p>
     *         <p>
     *         When you create an intent you can assign messages to groups. When messages are assigned to groups Amazon
     *         Lex returns one message from each group in the response. The message field is an escaped JSON string
     *         containing the messages. For more information about the structure of the JSON string returned, see
     *         <a>msg-prompts-formats</a>.
     *         </p>
     *         <p>
     *         If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
     */
    public String message() {
        return message;
    }

    /**
     * <p>
     * The format of the response message. One of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PlainText</code> - The message contains plain UTF-8 text.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSML</code> - The message contains text formatted for voice output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
     * groups that messages were assigned to when the intent was created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormatType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageFormatAsString}.
     * </p>
     * 
     * @return The format of the response message. One of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PlainText</code> - The message contains plain UTF-8 text.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SSML</code> - The message contains text formatted for voice output.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from
     *         the groups that messages were assigned to when the intent was created.
     *         </p>
     *         </li>
     * @see MessageFormatType
     */
    public MessageFormatType messageFormat() {
        return MessageFormatType.fromValue(messageFormat);
    }

    /**
     * <p>
     * The format of the response message. One of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PlainText</code> - The message contains plain UTF-8 text.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSML</code> - The message contains text formatted for voice output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
     * groups that messages were assigned to when the intent was created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormatType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageFormatAsString}.
     * </p>
     * 
     * @return The format of the response message. One of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PlainText</code> - The message contains plain UTF-8 text.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SSML</code> - The message contains text formatted for voice output.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from
     *         the groups that messages were assigned to when the intent was created.
     *         </p>
     *         </li>
     * @see MessageFormatType
     */
    public String messageFormatAsString() {
        return messageFormat;
    }

    /**
     * <p>
     * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     * </p>
     * <p>
     * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user intent
     * from this utterance, it will return this dialogState.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     * </p>
     * <p>
     * For example, Amazon Lex wants user confirmation before fulfilling an intent.
     * </p>
     * <p>
     * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
     * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
     * OrderDrink).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     * </p>
     * <p>
     * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
     * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in the
     * response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information appropriately.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully fulfilled
     * the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - Conveys that the conversation with the user failed.
     * </p>
     * <p>
     * This can happen for various reasons including that the user did not provide an appropriate response to prompts
     * from the service (you can configure how many times Amazon Lex can prompt a user for specific information), or the
     * Lambda function failed to fulfill the intent.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dialogState} will
     * return {@link DialogState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dialogStateAsString}.
     * </p>
     * 
     * @return Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     *         <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     *         </p>
     *         <p>
     *         For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
     *         user intent from this utterance, it will return this dialogState.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     *         </p>
     *         <p>
     *         For example, Amazon Lex wants user confirmation before fulfilling an intent.
     *         </p>
     *         <p>
     *         Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     *         "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
     *         additional information (in these examples, update the crust type slot value, or change intent from
     *         OrderPizza to OrderDrink).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     *         </p>
     *         <p>
     *         For example, suppose that in the response Amazon Lex sends this message:
     *         "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
     *         might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon Lex
     *         can process such additional information appropriately.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
     *         fulfilled the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - Conveys that the conversation with the user failed.
     *         </p>
     *         <p>
     *         This can happen for various reasons including that the user did not provide an appropriate response to
     *         prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
     *         information), or the Lambda function failed to fulfill the intent.
     *         </p>
     *         </li>
     * @see DialogState
     */
    public DialogState dialogState() {
        return DialogState.fromValue(dialogState);
    }

    /**
     * <p>
     * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     * </p>
     * <p>
     * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user intent
     * from this utterance, it will return this dialogState.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     * </p>
     * <p>
     * For example, Amazon Lex wants user confirmation before fulfilling an intent.
     * </p>
     * <p>
     * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
     * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
     * OrderDrink).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     * </p>
     * <p>
     * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
     * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in the
     * response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information appropriately.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully fulfilled
     * the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - Conveys that the conversation with the user failed.
     * </p>
     * <p>
     * This can happen for various reasons including that the user did not provide an appropriate response to prompts
     * from the service (you can configure how many times Amazon Lex can prompt a user for specific information), or the
     * Lambda function failed to fulfill the intent.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dialogState} will
     * return {@link DialogState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dialogStateAsString}.
     * </p>
     * 
     * @return Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
     *         <code>dialogState</code>. The client can optionally use this information to customize the user interface.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
     *         </p>
     *         <p>
     *         For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
     *         user intent from this utterance, it will return this dialogState.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
     *         </p>
     *         <p>
     *         For example, Amazon Lex wants user confirmation before fulfilling an intent.
     *         </p>
     *         <p>
     *         Instead of a simple "yes" or "no," a user might respond with additional information. For example,
     *         "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
     *         additional information (in these examples, update the crust type slot value, or change intent from
     *         OrderPizza to OrderDrink).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
     *         </p>
     *         <p>
     *         For example, suppose that in the response Amazon Lex sends this message:
     *         "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
     *         might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon Lex
     *         can process such additional information appropriately.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
     *         fulfilled the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - Conveys that the conversation with the user failed.
     *         </p>
     *         <p>
     *         This can happen for various reasons including that the user did not provide an appropriate response to
     *         prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
     *         information), or the Lambda function failed to fulfill the intent.
     *         </p>
     *         </li>
     * @see DialogState
     */
    public String dialogStateAsString() {
        return dialogState;
    }

    /**
     * <p>
     * If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for which Amazon
     * Lex is eliciting a value.
     * </p>
     * 
     * @return If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for which
     *         Amazon Lex is eliciting a value.
     */
    public String slotToElicit() {
        return slotToElicit;
    }

    /**
     * <p>
     * Represents the options that the user has to respond to the current prompt. Response Card can come from the bot
     * configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a code hook (Lambda
     * function).
     * </p>
     * 
     * @return Represents the options that the user has to respond to the current prompt. Response Card can come from
     *         the bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a
     *         code hook (Lambda function).
     */
    public ResponseCard responseCard() {
        return responseCard;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(intentName());
        hashCode = 31 * hashCode + Objects.hashCode(slots());
        hashCode = 31 * hashCode + Objects.hashCode(sessionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(messageFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dialogStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(slotToElicit());
        hashCode = 31 * hashCode + Objects.hashCode(responseCard());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PostTextResponse)) {
            return false;
        }
        PostTextResponse other = (PostTextResponse) obj;
        return Objects.equals(intentName(), other.intentName()) && Objects.equals(slots(), other.slots())
                && Objects.equals(sessionAttributes(), other.sessionAttributes()) && Objects.equals(message(), other.message())
                && Objects.equals(messageFormatAsString(), other.messageFormatAsString())
                && Objects.equals(dialogStateAsString(), other.dialogStateAsString())
                && Objects.equals(slotToElicit(), other.slotToElicit()) && Objects.equals(responseCard(), other.responseCard());
    }

    @Override
    public String toString() {
        return ToString.builder("PostTextResponse").add("IntentName", intentName()).add("Slots", slots())
                .add("SessionAttributes", sessionAttributes()).add("Message", message())
                .add("MessageFormat", messageFormatAsString()).add("DialogState", dialogStateAsString())
                .add("SlotToElicit", slotToElicit()).add("ResponseCard", responseCard()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "intentName":
            return Optional.ofNullable(clazz.cast(intentName()));
        case "slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "sessionAttributes":
            return Optional.ofNullable(clazz.cast(sessionAttributes()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "messageFormat":
            return Optional.ofNullable(clazz.cast(messageFormatAsString()));
        case "dialogState":
            return Optional.ofNullable(clazz.cast(dialogStateAsString()));
        case "slotToElicit":
            return Optional.ofNullable(clazz.cast(slotToElicit()));
        case "responseCard":
            return Optional.ofNullable(clazz.cast(responseCard()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PostTextResponse, T> g) {
        return obj -> g.apply((PostTextResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LexRuntimeResponse.Builder, SdkPojo, CopyableBuilder<Builder, PostTextResponse> {
        /**
         * <p>
         * The current user intent that Amazon Lex is aware of.
         * </p>
         * 
         * @param intentName
         *        The current user intent that Amazon Lex is aware of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intentName(String intentName);

        /**
         * <p>
         * The intent slots that Amazon Lex detected from the user input in the conversation.
         * </p>
         * <p>
         * Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is
         * determined by the <code>valueSelectionStrategy</code> selected when the slot type was created or updated. If
         * <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value provided by the user is
         * returned, if the user value is similar to the slot values. If <code>valueSelectionStrategy</code> is set to
         * <code>TOP_RESOLUTION</code> Amazon Lex returns the first value in the resolution list or, if there is no
         * resolution list, null. If you don't specify a <code>valueSelectionStrategy</code>, the default is
         * <code>ORIGINAL_VALUE</code>.
         * </p>
         * 
         * @param slots
         *        The intent slots that Amazon Lex detected from the user input in the conversation. </p>
         *        <p>
         *        Amazon Lex creates a resolution list containing likely values for a slot. The value that it returns is
         *        determined by the <code>valueSelectionStrategy</code> selected when the slot type was created or
         *        updated. If <code>valueSelectionStrategy</code> is set to <code>ORIGINAL_VALUE</code>, the value
         *        provided by the user is returned, if the user value is similar to the slot values. If
         *        <code>valueSelectionStrategy</code> is set to <code>TOP_RESOLUTION</code> Amazon Lex returns the first
         *        value in the resolution list or, if there is no resolution list, null. If you don't specify a
         *        <code>valueSelectionStrategy</code>, the default is <code>ORIGINAL_VALUE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(Map<String, String> slots);

        /**
         * <p>
         * A map of key-value pairs representing the session-specific context information.
         * </p>
         * 
         * @param sessionAttributes
         *        A map of key-value pairs representing the session-specific context information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionAttributes(Map<String, String> sessionAttributes);

        /**
         * <p>
         * The message to convey to the user. The message can come from the bot's configuration or from a Lambda
         * function.
         * </p>
         * <p>
         * If the intent is not configured with a Lambda function, or if the Lambda function returned
         * <code>Delegate</code> as the <code>dialogAction.type</code> its response, Amazon Lex decides on the next
         * course of action and selects an appropriate message from the bot's configuration based on the current
         * interaction context. For example, if Amazon Lex isn't able to understand user input, it uses a clarification
         * prompt message.
         * </p>
         * <p>
         * When you create an intent you can assign messages to groups. When messages are assigned to groups Amazon Lex
         * returns one message from each group in the response. The message field is an escaped JSON string containing
         * the messages. For more information about the structure of the JSON string returned, see
         * <a>msg-prompts-formats</a>.
         * </p>
         * <p>
         * If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
         * </p>
         * 
         * @param message
         *        The message to convey to the user. The message can come from the bot's configuration or from a Lambda
         *        function.</p>
         *        <p>
         *        If the intent is not configured with a Lambda function, or if the Lambda function returned
         *        <code>Delegate</code> as the <code>dialogAction.type</code> its response, Amazon Lex decides on the
         *        next course of action and selects an appropriate message from the bot's configuration based on the
         *        current interaction context. For example, if Amazon Lex isn't able to understand user input, it uses a
         *        clarification prompt message.
         *        </p>
         *        <p>
         *        When you create an intent you can assign messages to groups. When messages are assigned to groups
         *        Amazon Lex returns one message from each group in the response. The message field is an escaped JSON
         *        string containing the messages. For more information about the structure of the JSON string returned,
         *        see <a>msg-prompts-formats</a>.
         *        </p>
         *        <p>
         *        If the Lambda function returns a message, Amazon Lex passes it to the client in its response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The format of the response message. One of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PlainText</code> - The message contains plain UTF-8 text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSML</code> - The message contains text formatted for voice output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
         * groups that messages were assigned to when the intent was created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param messageFormat
         *        The format of the response message. One of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PlainText</code> - The message contains plain UTF-8 text.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SSML</code> - The message contains text formatted for voice output.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Composite</code> - The message contains an escaped JSON object containing one or more messages
         *        from the groups that messages were assigned to when the intent was created.
         *        </p>
         *        </li>
         * @see MessageFormatType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormatType
         */
        Builder messageFormat(String messageFormat);

        /**
         * <p>
         * The format of the response message. One of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PlainText</code> - The message contains plain UTF-8 text.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSML</code> - The message contains text formatted for voice output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Composite</code> - The message contains an escaped JSON object containing one or more messages from the
         * groups that messages were assigned to when the intent was created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param messageFormat
         *        The format of the response message. One of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PlainText</code> - The message contains plain UTF-8 text.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CustomPayload</code> - The message is a custom format defined by the Lambda function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SSML</code> - The message contains text formatted for voice output.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Composite</code> - The message contains an escaped JSON object containing one or more messages
         *        from the groups that messages were assigned to when the intent was created.
         *        </p>
         *        </li>
         * @see MessageFormatType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormatType
         */
        Builder messageFormat(MessageFormatType messageFormat);

        /**
         * <p>
         * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
         * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         * </p>
         * <p>
         * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user
         * intent from this utterance, it will return this dialogState.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         * </p>
         * <p>
         * For example, Amazon Lex wants user confirmation before fulfilling an intent.
         * </p>
         * <p>
         * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
         * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
         * OrderDrink).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         * </p>
         * <p>
         * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
         * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in
         * the response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information
         * appropriately.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         * fulfilled the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - Conveys that the conversation with the user failed.
         * </p>
         * <p>
         * This can happen for various reasons including that the user did not provide an appropriate response to
         * prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         * information), or the Lambda function failed to fulfill the intent.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dialogState
         *        Identifies the current state of the user interaction. Amazon Lex returns one of the following values
         *        as <code>dialogState</code>. The client can optionally use this information to customize the user
         *        interface. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         *        </p>
         *        <p>
         *        For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
         *        user intent from this utterance, it will return this dialogState.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         *        </p>
         *        <p>
         *        For example, Amazon Lex wants user confirmation before fulfilling an intent.
         *        </p>
         *        <p>
         *        Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         *        "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
         *        additional information (in these examples, update the crust type slot value, or change intent from
         *        OrderPizza to OrderDrink).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         *        </p>
         *        <p>
         *        For example, suppose that in the response Amazon Lex sends this message:
         *        "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
         *        might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon
         *        Lex can process such additional information appropriately.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         *        fulfilled the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - Conveys that the conversation with the user failed.
         *        </p>
         *        <p>
         *        This can happen for various reasons including that the user did not provide an appropriate response to
         *        prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         *        information), or the Lambda function failed to fulfill the intent.
         *        </p>
         *        </li>
         * @see DialogState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogState
         */
        Builder dialogState(String dialogState);

        /**
         * <p>
         * Identifies the current state of the user interaction. Amazon Lex returns one of the following values as
         * <code>dialogState</code>. The client can optionally use this information to customize the user interface.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         * </p>
         * <p>
         * For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the user
         * intent from this utterance, it will return this dialogState.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         * </p>
         * <p>
         * For example, Amazon Lex wants user confirmation before fulfilling an intent.
         * </p>
         * <p>
         * Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         * "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such additional
         * information (in these examples, update the crust type slot value, or change intent from OrderPizza to
         * OrderDrink).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         * </p>
         * <p>
         * For example, suppose that in the response Amazon Lex sends this message: "What size pizza would you like?". A
         * user might reply with the slot value (e.g., "medium"). The user might also provide additional information in
         * the response (e.g., "medium thick crust pizza"). Amazon Lex can process such additional information
         * appropriately.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         * fulfilled the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - Conveys that the conversation with the user failed.
         * </p>
         * <p>
         * This can happen for various reasons including that the user did not provide an appropriate response to
         * prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         * information), or the Lambda function failed to fulfill the intent.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dialogState
         *        Identifies the current state of the user interaction. Amazon Lex returns one of the following values
         *        as <code>dialogState</code>. The client can optionally use this information to customize the user
         *        interface. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ElicitIntent</code> - Amazon Lex wants to elicit user intent.
         *        </p>
         *        <p>
         *        For example, a user might utter an intent ("I want to order a pizza"). If Amazon Lex cannot infer the
         *        user intent from this utterance, it will return this dialogState.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfirmIntent</code> - Amazon Lex is expecting a "yes" or "no" response.
         *        </p>
         *        <p>
         *        For example, Amazon Lex wants user confirmation before fulfilling an intent.
         *        </p>
         *        <p>
         *        Instead of a simple "yes" or "no," a user might respond with additional information. For example,
         *        "yes, but make it thick crust pizza" or "no, I want to order a drink". Amazon Lex can process such
         *        additional information (in these examples, update the crust type slot value, or change intent from
         *        OrderPizza to OrderDrink).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ElicitSlot</code> - Amazon Lex is expecting a slot value for the current intent.
         *        </p>
         *        <p>
         *        For example, suppose that in the response Amazon Lex sends this message:
         *        "What size pizza would you like?". A user might reply with the slot value (e.g., "medium"). The user
         *        might also provide additional information in the response (e.g., "medium thick crust pizza"). Amazon
         *        Lex can process such additional information appropriately.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Fulfilled</code> - Conveys that the Lambda function configured for the intent has successfully
         *        fulfilled the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReadyForFulfillment</code> - Conveys that the client has to fulfill the intent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - Conveys that the conversation with the user failed.
         *        </p>
         *        <p>
         *        This can happen for various reasons including that the user did not provide an appropriate response to
         *        prompts from the service (you can configure how many times Amazon Lex can prompt a user for specific
         *        information), or the Lambda function failed to fulfill the intent.
         *        </p>
         *        </li>
         * @see DialogState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DialogState
         */
        Builder dialogState(DialogState dialogState);

        /**
         * <p>
         * If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for which
         * Amazon Lex is eliciting a value.
         * </p>
         * 
         * @param slotToElicit
         *        If the <code>dialogState</code> value is <code>ElicitSlot</code>, returns the name of the slot for
         *        which Amazon Lex is eliciting a value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotToElicit(String slotToElicit);

        /**
         * <p>
         * Represents the options that the user has to respond to the current prompt. Response Card can come from the
         * bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a code hook
         * (Lambda function).
         * </p>
         * 
         * @param responseCard
         *        Represents the options that the user has to respond to the current prompt. Response Card can come from
         *        the bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a
         *        code hook (Lambda function).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseCard(ResponseCard responseCard);

        /**
         * <p>
         * Represents the options that the user has to respond to the current prompt. Response Card can come from the
         * bot configuration (in the Amazon Lex console, choose the settings button next to a slot) or from a code hook
         * (Lambda function).
         * </p>
         * This is a convenience that creates an instance of the {@link ResponseCard.Builder} avoiding the need to
         * create one manually via {@link ResponseCard#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResponseCard.Builder#build()} is called immediately and its
         * result is passed to {@link #responseCard(ResponseCard)}.
         * 
         * @param responseCard
         *        a consumer that will call methods on {@link ResponseCard.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #responseCard(ResponseCard)
         */
        default Builder responseCard(Consumer<ResponseCard.Builder> responseCard) {
            return responseCard(ResponseCard.builder().applyMutation(responseCard).build());
        }
    }

    static final class BuilderImpl extends LexRuntimeResponse.BuilderImpl implements Builder {
        private String intentName;

        private Map<String, String> slots = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> sessionAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String message;

        private String messageFormat;

        private String dialogState;

        private String slotToElicit;

        private ResponseCard responseCard;

        private BuilderImpl() {
        }

        private BuilderImpl(PostTextResponse model) {
            super(model);
            intentName(model.intentName);
            slots(model.slots);
            sessionAttributes(model.sessionAttributes);
            message(model.message);
            messageFormat(model.messageFormat);
            dialogState(model.dialogState);
            slotToElicit(model.slotToElicit);
            responseCard(model.responseCard);
        }

        public final String getIntentName() {
            return intentName;
        }

        @Override
        public final Builder intentName(String intentName) {
            this.intentName = intentName;
            return this;
        }

        public final void setIntentName(String intentName) {
            this.intentName = intentName;
        }

        public final Map<String, String> getSlots() {
            return slots;
        }

        @Override
        public final Builder slots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
            return this;
        }

        public final void setSlots(Map<String, String> slots) {
            this.slots = StringMapCopier.copy(slots);
        }

        public final Map<String, String> getSessionAttributes() {
            return sessionAttributes;
        }

        @Override
        public final Builder sessionAttributes(Map<String, String> sessionAttributes) {
            this.sessionAttributes = StringMapCopier.copy(sessionAttributes);
            return this;
        }

        public final void setSessionAttributes(Map<String, String> sessionAttributes) {
            this.sessionAttributes = StringMapCopier.copy(sessionAttributes);
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final String getMessageFormat() {
            return messageFormat;
        }

        @Override
        public final Builder messageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
            return this;
        }

        @Override
        public final Builder messageFormat(MessageFormatType messageFormat) {
            this.messageFormat(messageFormat.toString());
            return this;
        }

        public final void setMessageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
        }

        public final String getDialogState() {
            return dialogState;
        }

        @Override
        public final Builder dialogState(String dialogState) {
            this.dialogState = dialogState;
            return this;
        }

        @Override
        public final Builder dialogState(DialogState dialogState) {
            this.dialogState(dialogState.toString());
            return this;
        }

        public final void setDialogState(String dialogState) {
            this.dialogState = dialogState;
        }

        public final String getSlotToElicit() {
            return slotToElicit;
        }

        @Override
        public final Builder slotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
            return this;
        }

        public final void setSlotToElicit(String slotToElicit) {
            this.slotToElicit = slotToElicit;
        }

        public final ResponseCard.Builder getResponseCard() {
            return responseCard != null ? responseCard.toBuilder() : null;
        }

        @Override
        public final Builder responseCard(ResponseCard responseCard) {
            this.responseCard = responseCard;
            return this;
        }

        public final void setResponseCard(ResponseCard.BuilderImpl responseCard) {
            this.responseCard = responseCard != null ? responseCard.build() : null;
        }

        @Override
        public PostTextResponse build() {
            return new PostTextResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
