/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.lexruntime.transform.ResponseCardMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * If you configure a response card when creating your bots, Amazon Lex substitutes the session attributes and slot
 * values that are available, and then returns it. The response card can also come from a Lambda function (
 * <code>dialogCodeHook</code> and <code>fulfillmentActivity</code> on an intent).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseCard implements StructuredPojo, ToCopyableBuilder<ResponseCard.Builder, ResponseCard> {
    private final String version;

    private final String contentType;

    private final List<GenericAttachment> genericAttachments;

    private ResponseCard(BuilderImpl builder) {
        this.version = builder.version;
        this.contentType = builder.contentType;
        this.genericAttachments = builder.genericAttachments;
    }

    /**
     * <p>
     * The version of the response card format.
     * </p>
     * 
     * @return The version of the response card format.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * The content type of the response.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentType} will
     * return {@link ContentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contentTypeAsString}.
     * </p>
     * 
     * @return The content type of the response.
     * @see ContentType
     */
    public ContentType contentType() {
        return ContentType.fromValue(contentType);
    }

    /**
     * <p>
     * The content type of the response.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentType} will
     * return {@link ContentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contentTypeAsString}.
     * </p>
     * 
     * @return The content type of the response.
     * @see ContentType
     */
    public String contentTypeAsString() {
        return contentType;
    }

    /**
     * <p>
     * An array of attachment objects representing options.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of attachment objects representing options.
     */
    public List<GenericAttachment> genericAttachments() {
        return genericAttachments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(contentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(genericAttachments());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseCard)) {
            return false;
        }
        ResponseCard other = (ResponseCard) obj;
        return Objects.equals(version(), other.version()) && Objects.equals(contentTypeAsString(), other.contentTypeAsString())
                && Objects.equals(genericAttachments(), other.genericAttachments());
    }

    @Override
    public String toString() {
        return ToString.builder("ResponseCard").add("Version", version()).add("ContentType", contentTypeAsString())
                .add("GenericAttachments", genericAttachments()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "contentType":
            return Optional.ofNullable(clazz.cast(contentTypeAsString()));
        case "genericAttachments":
            return Optional.ofNullable(clazz.cast(genericAttachments()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ResponseCardMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ResponseCard> {
        /**
         * <p>
         * The version of the response card format.
         * </p>
         * 
         * @param version
         *        The version of the response card format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The content type of the response.
         * </p>
         * 
         * @param contentType
         *        The content type of the response.
         * @see ContentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentType
         */
        Builder contentType(String contentType);

        /**
         * <p>
         * The content type of the response.
         * </p>
         * 
         * @param contentType
         *        The content type of the response.
         * @see ContentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentType
         */
        Builder contentType(ContentType contentType);

        /**
         * <p>
         * An array of attachment objects representing options.
         * </p>
         * 
         * @param genericAttachments
         *        An array of attachment objects representing options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder genericAttachments(Collection<GenericAttachment> genericAttachments);

        /**
         * <p>
         * An array of attachment objects representing options.
         * </p>
         * 
         * @param genericAttachments
         *        An array of attachment objects representing options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder genericAttachments(GenericAttachment... genericAttachments);

        /**
         * <p>
         * An array of attachment objects representing options.
         * </p>
         * This is a convenience that creates an instance of the {@link List<GenericAttachment>.Builder} avoiding the
         * need to create one manually via {@link List<GenericAttachment>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GenericAttachment>.Builder#build()} is called immediately
         * and its result is passed to {@link #genericAttachments(List<GenericAttachment>)}.
         * 
         * @param genericAttachments
         *        a consumer that will call methods on {@link List<GenericAttachment>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #genericAttachments(List<GenericAttachment>)
         */
        Builder genericAttachments(Consumer<GenericAttachment.Builder>... genericAttachments);
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private String contentType;

        private List<GenericAttachment> genericAttachments = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseCard model) {
            version(model.version);
            contentType(model.contentType);
            genericAttachments(model.genericAttachments);
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final String getContentType() {
            return contentType;
        }

        @Override
        public final Builder contentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        @Override
        public final Builder contentType(ContentType contentType) {
            this.contentType(contentType.toString());
            return this;
        }

        public final void setContentType(String contentType) {
            this.contentType = contentType;
        }

        public final Collection<GenericAttachment.Builder> getGenericAttachments() {
            return genericAttachments != null ? genericAttachments.stream().map(GenericAttachment::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder genericAttachments(Collection<GenericAttachment> genericAttachments) {
            this.genericAttachments = _genericAttachmentListCopier.copy(genericAttachments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder genericAttachments(GenericAttachment... genericAttachments) {
            genericAttachments(Arrays.asList(genericAttachments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder genericAttachments(Consumer<GenericAttachment.Builder>... genericAttachments) {
            genericAttachments(Stream.of(genericAttachments).map(c -> GenericAttachment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setGenericAttachments(Collection<GenericAttachment.BuilderImpl> genericAttachments) {
            this.genericAttachments = _genericAttachmentListCopier.copyFromBuilder(genericAttachments);
        }

        @Override
        public ResponseCard build() {
            return new ResponseCard(this);
        }
    }
}
