/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.endpoints.internal;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.EndpointAuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4aAuthScheme;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SelectedAuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4HttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4aHttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.RegionSet;
import software.amazon.awssdk.http.auth.spi.scheme.AuthSchemeOption;
import software.amazon.awssdk.identity.spi.Identity;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.services.kinesis.endpoints.KinesisEndpointParams;
import software.amazon.awssdk.services.kinesis.endpoints.KinesisEndpointProvider;
import software.amazon.awssdk.services.kinesis.model.AddTagsToStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DecreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.kinesis.model.DeleteStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DeregisterStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamRequest;
import software.amazon.awssdk.services.kinesis.model.DescribeStreamSummaryRequest;
import software.amazon.awssdk.services.kinesis.model.DisableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.EnableEnhancedMonitoringRequest;
import software.amazon.awssdk.services.kinesis.model.GetRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.kinesis.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.kinesis.model.IncreaseStreamRetentionPeriodRequest;
import software.amazon.awssdk.services.kinesis.model.ListShardsRequest;
import software.amazon.awssdk.services.kinesis.model.ListStreamConsumersRequest;
import software.amazon.awssdk.services.kinesis.model.ListTagsForStreamRequest;
import software.amazon.awssdk.services.kinesis.model.MergeShardsRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordRequest;
import software.amazon.awssdk.services.kinesis.model.PutRecordsRequest;
import software.amazon.awssdk.services.kinesis.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.kinesis.model.RegisterStreamConsumerRequest;
import software.amazon.awssdk.services.kinesis.model.RemoveTagsFromStreamRequest;
import software.amazon.awssdk.services.kinesis.model.SplitShardRequest;
import software.amazon.awssdk.services.kinesis.model.StartStreamEncryptionRequest;
import software.amazon.awssdk.services.kinesis.model.StopStreamEncryptionRequest;
import software.amazon.awssdk.services.kinesis.model.SubscribeToShardRequest;
import software.amazon.awssdk.services.kinesis.model.UpdateShardCountRequest;
import software.amazon.awssdk.services.kinesis.model.UpdateStreamModeRequest;
import software.amazon.awssdk.utils.CollectionUtils;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class KinesisResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        SdkRequest result = context.request();
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return result;
        }
        KinesisEndpointProvider provider = (KinesisEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            long resolveEndpointStart = System.nanoTime();
            KinesisEndpointParams endpointParams = ruleParams(result, executionAttributes);
            Endpoint endpoint = provider.resolveEndpoint(endpointParams).join();
            Duration resolveEndpointDuration = Duration.ofNanos(System.nanoTime() - resolveEndpointStart);
            Optional<MetricCollector> metricCollector = executionAttributes
                    .getOptionalAttribute(SdkExecutionAttribute.API_CALL_METRIC_COLLECTOR);
            metricCollector.ifPresent(mc -> mc.reportMetric(CoreMetric.ENDPOINT_RESOLVE_DURATION, resolveEndpointDuration));
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        result);
                if (hostPrefix.isPresent()) {
                    endpoint = AwsEndpointProviderUtils.addHostPrefix(endpoint, hostPrefix.get());
                }
            }
            List<EndpointAuthScheme> endpointAuthSchemes = endpoint.attribute(AwsEndpointAttribute.AUTH_SCHEMES);
            SelectedAuthScheme<?> selectedAuthScheme = executionAttributes
                    .getAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME);
            if (endpointAuthSchemes != null && selectedAuthScheme != null) {
                selectedAuthScheme = authSchemeWithEndpointSignerProperties(endpointAuthSchemes, selectedAuthScheme);
                executionAttributes.putAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME, selectedAuthScheme);
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, endpoint);
            return result;
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    @Override
    public SdkHttpRequest modifyHttpRequest(Context.ModifyHttpRequest context, ExecutionAttributes executionAttributes) {
        Endpoint resolvedEndpoint = executionAttributes.getAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT);
        if (resolvedEndpoint.headers().isEmpty()) {
            return context.httpRequest();
        }
        SdkHttpRequest.Builder httpRequestBuilder = context.httpRequest().toBuilder();
        resolvedEndpoint.headers().forEach((name, values) -> {
            values.forEach(v -> httpRequestBuilder.appendHeader(name, v));
        });
        return httpRequestBuilder.build();
    }

    public static KinesisEndpointParams ruleParams(SdkRequest request, ExecutionAttributes executionAttributes) {
        KinesisEndpointParams.Builder builder = KinesisEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), request);
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        setOperationContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), request);
        return builder.build();
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, String operationName, SdkRequest request) {
        switch (operationName) {
        case "AddTagsToStream":
            setContextParams(params, (AddTagsToStreamRequest) request);
            break;
        case "DecreaseStreamRetentionPeriod":
            setContextParams(params, (DecreaseStreamRetentionPeriodRequest) request);
            break;
        case "DeleteResourcePolicy":
            setContextParams(params, (DeleteResourcePolicyRequest) request);
            break;
        case "DeleteStream":
            setContextParams(params, (DeleteStreamRequest) request);
            break;
        case "DeregisterStreamConsumer":
            setContextParams(params, (DeregisterStreamConsumerRequest) request);
            break;
        case "DescribeStream":
            setContextParams(params, (DescribeStreamRequest) request);
            break;
        case "DescribeStreamConsumer":
            setContextParams(params, (DescribeStreamConsumerRequest) request);
            break;
        case "DescribeStreamSummary":
            setContextParams(params, (DescribeStreamSummaryRequest) request);
            break;
        case "DisableEnhancedMonitoring":
            setContextParams(params, (DisableEnhancedMonitoringRequest) request);
            break;
        case "EnableEnhancedMonitoring":
            setContextParams(params, (EnableEnhancedMonitoringRequest) request);
            break;
        case "GetRecords":
            setContextParams(params, (GetRecordsRequest) request);
            break;
        case "GetResourcePolicy":
            setContextParams(params, (GetResourcePolicyRequest) request);
            break;
        case "GetShardIterator":
            setContextParams(params, (GetShardIteratorRequest) request);
            break;
        case "IncreaseStreamRetentionPeriod":
            setContextParams(params, (IncreaseStreamRetentionPeriodRequest) request);
            break;
        case "ListShards":
            setContextParams(params, (ListShardsRequest) request);
            break;
        case "ListStreamConsumers":
            setContextParams(params, (ListStreamConsumersRequest) request);
            break;
        case "ListTagsForStream":
            setContextParams(params, (ListTagsForStreamRequest) request);
            break;
        case "MergeShards":
            setContextParams(params, (MergeShardsRequest) request);
            break;
        case "PutRecord":
            setContextParams(params, (PutRecordRequest) request);
            break;
        case "PutRecords":
            setContextParams(params, (PutRecordsRequest) request);
            break;
        case "PutResourcePolicy":
            setContextParams(params, (PutResourcePolicyRequest) request);
            break;
        case "RegisterStreamConsumer":
            setContextParams(params, (RegisterStreamConsumerRequest) request);
            break;
        case "RemoveTagsFromStream":
            setContextParams(params, (RemoveTagsFromStreamRequest) request);
            break;
        case "SplitShard":
            setContextParams(params, (SplitShardRequest) request);
            break;
        case "StartStreamEncryption":
            setContextParams(params, (StartStreamEncryptionRequest) request);
            break;
        case "StopStreamEncryption":
            setContextParams(params, (StopStreamEncryptionRequest) request);
            break;
        case "SubscribeToShard":
            setContextParams(params, (SubscribeToShardRequest) request);
            break;
        case "UpdateShardCount":
            setContextParams(params, (UpdateShardCountRequest) request);
            break;
        case "UpdateStreamMode":
            setContextParams(params, (UpdateStreamModeRequest) request);
            break;
        default:
            break;
        }
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, AddTagsToStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DecreaseStreamRetentionPeriodRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DeleteResourcePolicyRequest request) {
        params.resourceArn(request.resourceARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DeleteStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DeregisterStreamConsumerRequest request) {
        params.streamArn(request.streamARN());
        params.consumerArn(request.consumerARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DescribeStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DescribeStreamConsumerRequest request) {
        params.streamArn(request.streamARN());
        params.consumerArn(request.consumerARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DescribeStreamSummaryRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, DisableEnhancedMonitoringRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, EnableEnhancedMonitoringRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, GetRecordsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, GetResourcePolicyRequest request) {
        params.resourceArn(request.resourceARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, GetShardIteratorRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, IncreaseStreamRetentionPeriodRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, ListShardsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, ListStreamConsumersRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, ListTagsForStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, MergeShardsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, PutRecordRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, PutRecordsRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, PutResourcePolicyRequest request) {
        params.resourceArn(request.resourceARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, RegisterStreamConsumerRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, RemoveTagsFromStreamRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, SplitShardRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, StartStreamEncryptionRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, StopStreamEncryptionRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, SubscribeToShardRequest request) {
        params.consumerArn(request.consumerARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, UpdateShardCountRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setContextParams(KinesisEndpointParams.Builder params, UpdateStreamModeRequest request) {
        params.streamArn(request.streamARN());
    }

    private static void setStaticContextParams(KinesisEndpointParams.Builder params, String operationName) {
        switch (operationName) {
        case "AddTagsToStream":
            addTagsToStreamStaticContextParams(params);
            break;
        case "DecreaseStreamRetentionPeriod":
            decreaseStreamRetentionPeriodStaticContextParams(params);
            break;
        case "DeleteResourcePolicy":
            deleteResourcePolicyStaticContextParams(params);
            break;
        case "DeleteStream":
            deleteStreamStaticContextParams(params);
            break;
        case "DeregisterStreamConsumer":
            deregisterStreamConsumerStaticContextParams(params);
            break;
        case "DescribeStream":
            describeStreamStaticContextParams(params);
            break;
        case "DescribeStreamConsumer":
            describeStreamConsumerStaticContextParams(params);
            break;
        case "DescribeStreamSummary":
            describeStreamSummaryStaticContextParams(params);
            break;
        case "DisableEnhancedMonitoring":
            disableEnhancedMonitoringStaticContextParams(params);
            break;
        case "EnableEnhancedMonitoring":
            enableEnhancedMonitoringStaticContextParams(params);
            break;
        case "GetRecords":
            getRecordsStaticContextParams(params);
            break;
        case "GetResourcePolicy":
            getResourcePolicyStaticContextParams(params);
            break;
        case "GetShardIterator":
            getShardIteratorStaticContextParams(params);
            break;
        case "IncreaseStreamRetentionPeriod":
            increaseStreamRetentionPeriodStaticContextParams(params);
            break;
        case "ListShards":
            listShardsStaticContextParams(params);
            break;
        case "ListStreamConsumers":
            listStreamConsumersStaticContextParams(params);
            break;
        case "ListTagsForStream":
            listTagsForStreamStaticContextParams(params);
            break;
        case "MergeShards":
            mergeShardsStaticContextParams(params);
            break;
        case "PutRecord":
            putRecordStaticContextParams(params);
            break;
        case "PutRecords":
            putRecordsStaticContextParams(params);
            break;
        case "PutResourcePolicy":
            putResourcePolicyStaticContextParams(params);
            break;
        case "RegisterStreamConsumer":
            registerStreamConsumerStaticContextParams(params);
            break;
        case "RemoveTagsFromStream":
            removeTagsFromStreamStaticContextParams(params);
            break;
        case "SplitShard":
            splitShardStaticContextParams(params);
            break;
        case "StartStreamEncryption":
            startStreamEncryptionStaticContextParams(params);
            break;
        case "StopStreamEncryption":
            stopStreamEncryptionStaticContextParams(params);
            break;
        case "SubscribeToShard":
            subscribeToShardStaticContextParams(params);
            break;
        case "UpdateShardCount":
            updateShardCountStaticContextParams(params);
            break;
        case "UpdateStreamMode":
            updateStreamModeStaticContextParams(params);
            break;
        default:
            break;
        }
    }

    private static void addTagsToStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void decreaseStreamRetentionPeriodStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void deleteResourcePolicyStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void deleteStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void deregisterStreamConsumerStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void describeStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void describeStreamConsumerStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void describeStreamSummaryStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void disableEnhancedMonitoringStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void enableEnhancedMonitoringStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void getRecordsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void getResourcePolicyStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void getShardIteratorStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void increaseStreamRetentionPeriodStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void listShardsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void listStreamConsumersStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void listTagsForStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void mergeShardsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void putRecordStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void putRecordsStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void putResourcePolicyStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void registerStreamConsumerStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void removeTagsFromStreamStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void splitShardStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void startStreamEncryptionStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void stopStreamEncryptionStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void subscribeToShardStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("data");
    }

    private static void updateShardCountStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private static void updateStreamModeStaticContextParams(KinesisEndpointParams.Builder params) {
        params.operationType("control");
    }

    private <T extends Identity> SelectedAuthScheme<T> authSchemeWithEndpointSignerProperties(
            List<EndpointAuthScheme> endpointAuthSchemes, SelectedAuthScheme<T> selectedAuthScheme) {
        for (EndpointAuthScheme endpointAuthScheme : endpointAuthSchemes) {
            if (!endpointAuthScheme.schemeId().equals(selectedAuthScheme.authSchemeOption().schemeId())) {
                continue;
            }
            AuthSchemeOption.Builder option = selectedAuthScheme.authSchemeOption().toBuilder();
            if (endpointAuthScheme instanceof SigV4AuthScheme) {
                SigV4AuthScheme v4AuthScheme = (SigV4AuthScheme) endpointAuthScheme;
                if (v4AuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4HttpSigner.DOUBLE_URL_ENCODE, !v4AuthScheme.disableDoubleEncoding());
                }
                if (v4AuthScheme.signingRegion() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.REGION_NAME, v4AuthScheme.signingRegion());
                }
                if (v4AuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.SERVICE_SIGNING_NAME, v4AuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            if (endpointAuthScheme instanceof SigV4aAuthScheme) {
                SigV4aAuthScheme v4aAuthScheme = (SigV4aAuthScheme) endpointAuthScheme;
                if (v4aAuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4aHttpSigner.DOUBLE_URL_ENCODE, !v4aAuthScheme.disableDoubleEncoding());
                }
                if (!CollectionUtils.isNullOrEmpty(v4aAuthScheme.signingRegionSet())) {
                    RegionSet regionSet = RegionSet.create(v4aAuthScheme.signingRegionSet());
                    option.putSignerProperty(AwsV4aHttpSigner.REGION_SET, regionSet);
                }
                if (v4aAuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4aHttpSigner.SERVICE_SIGNING_NAME, v4aAuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            throw new IllegalArgumentException("Endpoint auth scheme '" + endpointAuthScheme.name()
                    + "' cannot be mapped to the SDK auth scheme. Was it declared in the service's model?");
        }
        return selectedAuthScheme;
    }

    private static void setOperationContextParams(KinesisEndpointParams.Builder params, String operationName, SdkRequest request) {
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        return Optional.empty();
    }
}
