/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output for <a>DescribeStream</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StreamDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<StreamDescription.Builder, StreamDescription> {
    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(StreamDescription::streamName)).setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<String> STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamARN").getter(getter(StreamDescription::streamARN)).setter(setter(Builder::streamARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamARN").build()).build();

    private static final SdkField<String> STREAM_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamStatus").getter(getter(StreamDescription::streamStatusAsString))
            .setter(setter(Builder::streamStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamStatus").build()).build();

    private static final SdkField<StreamModeDetails> STREAM_MODE_DETAILS_FIELD = SdkField
            .<StreamModeDetails> builder(MarshallingType.SDK_POJO).memberName("StreamModeDetails")
            .getter(getter(StreamDescription::streamModeDetails)).setter(setter(Builder::streamModeDetails))
            .constructor(StreamModeDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamModeDetails").build()).build();

    private static final SdkField<List<Shard>> SHARDS_FIELD = SdkField
            .<List<Shard>> builder(MarshallingType.LIST)
            .memberName("Shards")
            .getter(getter(StreamDescription::shards))
            .setter(setter(Builder::shards))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Shards").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Shard> builder(MarshallingType.SDK_POJO)
                                            .constructor(Shard::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> HAS_MORE_SHARDS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("HasMoreShards").getter(getter(StreamDescription::hasMoreShards)).setter(setter(Builder::hasMoreShards))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HasMoreShards").build()).build();

    private static final SdkField<Integer> RETENTION_PERIOD_HOURS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RetentionPeriodHours").getter(getter(StreamDescription::retentionPeriodHours))
            .setter(setter(Builder::retentionPeriodHours))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetentionPeriodHours").build())
            .build();

    private static final SdkField<Instant> STREAM_CREATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StreamCreationTimestamp").getter(getter(StreamDescription::streamCreationTimestamp))
            .setter(setter(Builder::streamCreationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamCreationTimestamp").build())
            .build();

    private static final SdkField<List<EnhancedMetrics>> ENHANCED_MONITORING_FIELD = SdkField
            .<List<EnhancedMetrics>> builder(MarshallingType.LIST)
            .memberName("EnhancedMonitoring")
            .getter(getter(StreamDescription::enhancedMonitoring))
            .setter(setter(Builder::enhancedMonitoring))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnhancedMonitoring").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EnhancedMetrics> builder(MarshallingType.SDK_POJO)
                                            .constructor(EnhancedMetrics::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ENCRYPTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionType").getter(getter(StreamDescription::encryptionTypeAsString))
            .setter(setter(Builder::encryptionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionType").build()).build();

    private static final SdkField<String> KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyId")
            .getter(getter(StreamDescription::keyId)).setter(setter(Builder::keyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_NAME_FIELD,
            STREAM_ARN_FIELD, STREAM_STATUS_FIELD, STREAM_MODE_DETAILS_FIELD, SHARDS_FIELD, HAS_MORE_SHARDS_FIELD,
            RETENTION_PERIOD_HOURS_FIELD, STREAM_CREATION_TIMESTAMP_FIELD, ENHANCED_MONITORING_FIELD, ENCRYPTION_TYPE_FIELD,
            KEY_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String streamName;

    private final String streamARN;

    private final String streamStatus;

    private final StreamModeDetails streamModeDetails;

    private final List<Shard> shards;

    private final Boolean hasMoreShards;

    private final Integer retentionPeriodHours;

    private final Instant streamCreationTimestamp;

    private final List<EnhancedMetrics> enhancedMonitoring;

    private final String encryptionType;

    private final String keyId;

    private StreamDescription(BuilderImpl builder) {
        this.streamName = builder.streamName;
        this.streamARN = builder.streamARN;
        this.streamStatus = builder.streamStatus;
        this.streamModeDetails = builder.streamModeDetails;
        this.shards = builder.shards;
        this.hasMoreShards = builder.hasMoreShards;
        this.retentionPeriodHours = builder.retentionPeriodHours;
        this.streamCreationTimestamp = builder.streamCreationTimestamp;
        this.enhancedMonitoring = builder.enhancedMonitoring;
        this.encryptionType = builder.encryptionType;
        this.keyId = builder.keyId;
    }

    /**
     * <p>
     * The name of the stream being described.
     * </p>
     * 
     * @return The name of the stream being described.
     */
    public final String streamName() {
        return streamName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the stream being described.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the stream being described.
     */
    public final String streamARN() {
        return streamARN;
    }

    /**
     * <p>
     * The current status of the stream being described. The stream status is one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     * <code>StreamStatus</code> to <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code> state
     * until Kinesis Data Streams completes the deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
     * perform read and write operations only on an <code>ACTIVE</code> stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
     * work while the stream is in the <code>UPDATING</code> state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamStatus} will
     * return {@link StreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #streamStatusAsString}.
     * </p>
     * 
     * @return The current status of the stream being described. The stream status is one of the following states:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     *         <code>StreamStatus</code> to <code>CREATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
     *         state until Kinesis Data Streams completes the deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
     *         should perform read and write operations only on an <code>ACTIVE</code> stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
     *         continue to work while the stream is in the <code>UPDATING</code> state.
     *         </p>
     *         </li>
     * @see StreamStatus
     */
    public final StreamStatus streamStatus() {
        return StreamStatus.fromValue(streamStatus);
    }

    /**
     * <p>
     * The current status of the stream being described. The stream status is one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     * <code>StreamStatus</code> to <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code> state
     * until Kinesis Data Streams completes the deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
     * perform read and write operations only on an <code>ACTIVE</code> stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
     * work while the stream is in the <code>UPDATING</code> state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamStatus} will
     * return {@link StreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #streamStatusAsString}.
     * </p>
     * 
     * @return The current status of the stream being described. The stream status is one of the following states:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
     *         <code>StreamStatus</code> to <code>CREATING</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
     *         state until Kinesis Data Streams completes the deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
     *         should perform read and write operations only on an <code>ACTIVE</code> stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
     *         continue to work while the stream is in the <code>UPDATING</code> state.
     *         </p>
     *         </li>
     * @see StreamStatus
     */
    public final String streamStatusAsString() {
        return streamStatus;
    }

    /**
     * <p>
     * Specifies the capacity mode to which you want to set your data stream. Currently, in Kinesis Data Streams, you
     * can choose between an <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data
     * streams.
     * </p>
     * 
     * @return Specifies the capacity mode to which you want to set your data stream. Currently, in Kinesis Data
     *         Streams, you can choose between an <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode
     *         for your data streams.
     */
    public final StreamModeDetails streamModeDetails() {
        return streamModeDetails;
    }

    /**
     * For responses, this returns true if the service returned a value for the Shards property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasShards() {
        return shards != null && !(shards instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The shards that comprise the stream.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasShards} method.
     * </p>
     * 
     * @return The shards that comprise the stream.
     */
    public final List<Shard> shards() {
        return shards;
    }

    /**
     * <p>
     * If set to <code>true</code>, more shards in the stream are available to describe.
     * </p>
     * 
     * @return If set to <code>true</code>, more shards in the stream are available to describe.
     */
    public final Boolean hasMoreShards() {
        return hasMoreShards;
    }

    /**
     * <p>
     * The current retention period, in hours. Minimum value of 24. Maximum value of 168.
     * </p>
     * 
     * @return The current retention period, in hours. Minimum value of 24. Maximum value of 168.
     */
    public final Integer retentionPeriodHours() {
        return retentionPeriodHours;
    }

    /**
     * <p>
     * The approximate time that the stream was created.
     * </p>
     * 
     * @return The approximate time that the stream was created.
     */
    public final Instant streamCreationTimestamp() {
        return streamCreationTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnhancedMonitoring property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnhancedMonitoring() {
        return enhancedMonitoring != null && !(enhancedMonitoring instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Represents the current enhanced monitoring settings of the stream.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnhancedMonitoring} method.
     * </p>
     * 
     * @return Represents the current enhanced monitoring settings of the stream.
     */
    public final List<EnhancedMetrics> enhancedMonitoring() {
        return enhancedMonitoring;
    }

    /**
     * <p>
     * The server-side encryption type used on the stream. This parameter can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code>: Do not encrypt the records in the stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed Amazon Web
     * Services KMS key.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The server-side encryption type used on the stream. This parameter can be one of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Do not encrypt the records in the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed Amazon
     *         Web Services KMS key.
     *         </p>
     *         </li>
     * @see EncryptionType
     */
    public final EncryptionType encryptionType() {
        return EncryptionType.fromValue(encryptionType);
    }

    /**
     * <p>
     * The server-side encryption type used on the stream. This parameter can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code>: Do not encrypt the records in the stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed Amazon Web
     * Services KMS key.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link EncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The server-side encryption type used on the stream. This parameter can be one of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Do not encrypt the records in the stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed Amazon
     *         Web Services KMS key.
     *         </p>
     *         </li>
     * @see EncryptionType
     */
    public final String encryptionTypeAsString() {
        return encryptionType;
    }

    /**
     * <p>
     * The GUID for the customer-managed Amazon Web Services KMS key to use for encryption. This value can be a globally
     * unique identifier, a fully specified ARN to either an alias or a key, or an alias name prefixed by "alias/".You
     * can also use a master key owned by Kinesis Data Streams by specifying the alias <code>aws/kinesis</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ARN example: <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN example: <code>arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name example: <code>alias/MyAliasName</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The GUID for the customer-managed Amazon Web Services KMS key to use for encryption. This value can be a
     *         globally unique identifier, a fully specified ARN to either an alias or a key, or an alias name prefixed
     *         by "alias/".You can also use a master key owned by Kinesis Data Streams by specifying the alias
     *         <code>aws/kinesis</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ARN example: <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN example: <code>arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name example: <code>alias/MyAliasName</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
     *         </p>
     *         </li>
     */
    public final String keyId() {
        return keyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(streamARN());
        hashCode = 31 * hashCode + Objects.hashCode(streamStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(streamModeDetails());
        hashCode = 31 * hashCode + Objects.hashCode(hasShards() ? shards() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMoreShards());
        hashCode = 31 * hashCode + Objects.hashCode(retentionPeriodHours());
        hashCode = 31 * hashCode + Objects.hashCode(streamCreationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnhancedMonitoring() ? enhancedMonitoring() : null);
        hashCode = 31 * hashCode + Objects.hashCode(encryptionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StreamDescription)) {
            return false;
        }
        StreamDescription other = (StreamDescription) obj;
        return Objects.equals(streamName(), other.streamName()) && Objects.equals(streamARN(), other.streamARN())
                && Objects.equals(streamStatusAsString(), other.streamStatusAsString())
                && Objects.equals(streamModeDetails(), other.streamModeDetails()) && hasShards() == other.hasShards()
                && Objects.equals(shards(), other.shards()) && Objects.equals(hasMoreShards(), other.hasMoreShards())
                && Objects.equals(retentionPeriodHours(), other.retentionPeriodHours())
                && Objects.equals(streamCreationTimestamp(), other.streamCreationTimestamp())
                && hasEnhancedMonitoring() == other.hasEnhancedMonitoring()
                && Objects.equals(enhancedMonitoring(), other.enhancedMonitoring())
                && Objects.equals(encryptionTypeAsString(), other.encryptionTypeAsString())
                && Objects.equals(keyId(), other.keyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StreamDescription").add("StreamName", streamName()).add("StreamARN", streamARN())
                .add("StreamStatus", streamStatusAsString()).add("StreamModeDetails", streamModeDetails())
                .add("Shards", hasShards() ? shards() : null).add("HasMoreShards", hasMoreShards())
                .add("RetentionPeriodHours", retentionPeriodHours()).add("StreamCreationTimestamp", streamCreationTimestamp())
                .add("EnhancedMonitoring", hasEnhancedMonitoring() ? enhancedMonitoring() : null)
                .add("EncryptionType", encryptionTypeAsString()).add("KeyId", keyId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "StreamARN":
            return Optional.ofNullable(clazz.cast(streamARN()));
        case "StreamStatus":
            return Optional.ofNullable(clazz.cast(streamStatusAsString()));
        case "StreamModeDetails":
            return Optional.ofNullable(clazz.cast(streamModeDetails()));
        case "Shards":
            return Optional.ofNullable(clazz.cast(shards()));
        case "HasMoreShards":
            return Optional.ofNullable(clazz.cast(hasMoreShards()));
        case "RetentionPeriodHours":
            return Optional.ofNullable(clazz.cast(retentionPeriodHours()));
        case "StreamCreationTimestamp":
            return Optional.ofNullable(clazz.cast(streamCreationTimestamp()));
        case "EnhancedMonitoring":
            return Optional.ofNullable(clazz.cast(enhancedMonitoring()));
        case "EncryptionType":
            return Optional.ofNullable(clazz.cast(encryptionTypeAsString()));
        case "KeyId":
            return Optional.ofNullable(clazz.cast(keyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StreamName", STREAM_NAME_FIELD);
        map.put("StreamARN", STREAM_ARN_FIELD);
        map.put("StreamStatus", STREAM_STATUS_FIELD);
        map.put("StreamModeDetails", STREAM_MODE_DETAILS_FIELD);
        map.put("Shards", SHARDS_FIELD);
        map.put("HasMoreShards", HAS_MORE_SHARDS_FIELD);
        map.put("RetentionPeriodHours", RETENTION_PERIOD_HOURS_FIELD);
        map.put("StreamCreationTimestamp", STREAM_CREATION_TIMESTAMP_FIELD);
        map.put("EnhancedMonitoring", ENHANCED_MONITORING_FIELD);
        map.put("EncryptionType", ENCRYPTION_TYPE_FIELD);
        map.put("KeyId", KEY_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StreamDescription, T> g) {
        return obj -> g.apply((StreamDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StreamDescription> {
        /**
         * <p>
         * The name of the stream being described.
         * </p>
         * 
         * @param streamName
         *        The name of the stream being described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the stream being described.
         * </p>
         * 
         * @param streamARN
         *        The Amazon Resource Name (ARN) for the stream being described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamARN(String streamARN);

        /**
         * <p>
         * The current status of the stream being described. The stream status is one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         * <code>StreamStatus</code> to <code>CREATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
         * state until Kinesis Data Streams completes the deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
         * perform read and write operations only on an <code>ACTIVE</code> stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
         * work while the stream is in the <code>UPDATING</code> state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamStatus
         *        The current status of the stream being described. The stream status is one of the following
         *        states:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         *        <code>StreamStatus</code> to <code>CREATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The stream is being deleted. The specified stream is in the
         *        <code>DELETING</code> state until Kinesis Data Streams completes the deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
         *        should perform read and write operations only on an <code>ACTIVE</code> stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
         *        continue to work while the stream is in the <code>UPDATING</code> state.
         *        </p>
         *        </li>
         * @see StreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamStatus
         */
        Builder streamStatus(String streamStatus);

        /**
         * <p>
         * The current status of the stream being described. The stream status is one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         * <code>StreamStatus</code> to <code>CREATING</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The stream is being deleted. The specified stream is in the <code>DELETING</code>
         * state until Kinesis Data Streams completes the deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You should
         * perform read and write operations only on an <code>ACTIVE</code> stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations continue to
         * work while the stream is in the <code>UPDATING</code> state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamStatus
         *        The current status of the stream being described. The stream status is one of the following
         *        states:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The stream is being created. Kinesis Data Streams immediately returns and sets
         *        <code>StreamStatus</code> to <code>CREATING</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The stream is being deleted. The specified stream is in the
         *        <code>DELETING</code> state until Kinesis Data Streams completes the deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The stream exists and is ready for read and write operations or deletion. You
         *        should perform read and write operations only on an <code>ACTIVE</code> stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - Shards in the stream are being merged or split. Read and write operations
         *        continue to work while the stream is in the <code>UPDATING</code> state.
         *        </p>
         *        </li>
         * @see StreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamStatus
         */
        Builder streamStatus(StreamStatus streamStatus);

        /**
         * <p>
         * Specifies the capacity mode to which you want to set your data stream. Currently, in Kinesis Data Streams,
         * you can choose between an <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data
         * streams.
         * </p>
         * 
         * @param streamModeDetails
         *        Specifies the capacity mode to which you want to set your data stream. Currently, in Kinesis Data
         *        Streams, you can choose between an <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity
         *        mode for your data streams.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamModeDetails(StreamModeDetails streamModeDetails);

        /**
         * <p>
         * Specifies the capacity mode to which you want to set your data stream. Currently, in Kinesis Data Streams,
         * you can choose between an <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data
         * streams.
         * </p>
         * This is a convenience method that creates an instance of the {@link StreamModeDetails.Builder} avoiding the
         * need to create one manually via {@link StreamModeDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StreamModeDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #streamModeDetails(StreamModeDetails)}.
         * 
         * @param streamModeDetails
         *        a consumer that will call methods on {@link StreamModeDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #streamModeDetails(StreamModeDetails)
         */
        default Builder streamModeDetails(Consumer<StreamModeDetails.Builder> streamModeDetails) {
            return streamModeDetails(StreamModeDetails.builder().applyMutation(streamModeDetails).build());
        }

        /**
         * <p>
         * The shards that comprise the stream.
         * </p>
         * 
         * @param shards
         *        The shards that comprise the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shards(Collection<Shard> shards);

        /**
         * <p>
         * The shards that comprise the stream.
         * </p>
         * 
         * @param shards
         *        The shards that comprise the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shards(Shard... shards);

        /**
         * <p>
         * The shards that comprise the stream.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kinesis.model.Shard.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.kinesis.model.Shard#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kinesis.model.Shard.Builder#build()} is called immediately and its
         * result is passed to {@link #shards(List<Shard>)}.
         * 
         * @param shards
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kinesis.model.Shard.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #shards(java.util.Collection<Shard>)
         */
        Builder shards(Consumer<Shard.Builder>... shards);

        /**
         * <p>
         * If set to <code>true</code>, more shards in the stream are available to describe.
         * </p>
         * 
         * @param hasMoreShards
         *        If set to <code>true</code>, more shards in the stream are available to describe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hasMoreShards(Boolean hasMoreShards);

        /**
         * <p>
         * The current retention period, in hours. Minimum value of 24. Maximum value of 168.
         * </p>
         * 
         * @param retentionPeriodHours
         *        The current retention period, in hours. Minimum value of 24. Maximum value of 168.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retentionPeriodHours(Integer retentionPeriodHours);

        /**
         * <p>
         * The approximate time that the stream was created.
         * </p>
         * 
         * @param streamCreationTimestamp
         *        The approximate time that the stream was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamCreationTimestamp(Instant streamCreationTimestamp);

        /**
         * <p>
         * Represents the current enhanced monitoring settings of the stream.
         * </p>
         * 
         * @param enhancedMonitoring
         *        Represents the current enhanced monitoring settings of the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enhancedMonitoring(Collection<EnhancedMetrics> enhancedMonitoring);

        /**
         * <p>
         * Represents the current enhanced monitoring settings of the stream.
         * </p>
         * 
         * @param enhancedMonitoring
         *        Represents the current enhanced monitoring settings of the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enhancedMonitoring(EnhancedMetrics... enhancedMonitoring);

        /**
         * <p>
         * Represents the current enhanced monitoring settings of the stream.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kinesis.model.EnhancedMetrics.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.kinesis.model.EnhancedMetrics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kinesis.model.EnhancedMetrics.Builder#build()} is called immediately
         * and its result is passed to {@link #enhancedMonitoring(List<EnhancedMetrics>)}.
         * 
         * @param enhancedMonitoring
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kinesis.model.EnhancedMetrics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #enhancedMonitoring(java.util.Collection<EnhancedMetrics>)
         */
        Builder enhancedMonitoring(Consumer<EnhancedMetrics.Builder>... enhancedMonitoring);

        /**
         * <p>
         * The server-side encryption type used on the stream. This parameter can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code>: Do not encrypt the records in the stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed Amazon Web
         * Services KMS key.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionType
         *        The server-side encryption type used on the stream. This parameter can be one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Do not encrypt the records in the stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed
         *        Amazon Web Services KMS key.
         *        </p>
         *        </li>
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(String encryptionType);

        /**
         * <p>
         * The server-side encryption type used on the stream. This parameter can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code>: Do not encrypt the records in the stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed Amazon Web
         * Services KMS key.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionType
         *        The server-side encryption type used on the stream. This parameter can be one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Do not encrypt the records in the stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS</code>: Use server-side encryption on the records in the stream using a customer-managed
         *        Amazon Web Services KMS key.
         *        </p>
         *        </li>
         * @see EncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionType
         */
        Builder encryptionType(EncryptionType encryptionType);

        /**
         * <p>
         * The GUID for the customer-managed Amazon Web Services KMS key to use for encryption. This value can be a
         * globally unique identifier, a fully specified ARN to either an alias or a key, or an alias name prefixed by
         * "alias/".You can also use a master key owned by Kinesis Data Streams by specifying the alias
         * <code>aws/kinesis</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ARN example: <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN example: <code>arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name example: <code>alias/MyAliasName</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyId
         *        The GUID for the customer-managed Amazon Web Services KMS key to use for encryption. This value can be
         *        a globally unique identifier, a fully specified ARN to either an alias or a key, or an alias name
         *        prefixed by "alias/".You can also use a master key owned by Kinesis Data Streams by specifying the
         *        alias <code>aws/kinesis</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ARN example:
         *        <code>arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN example: <code>arn:aws:kms:us-east-1:123456789012:alias/MyAliasName</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Globally unique key ID example: <code>12345678-1234-1234-1234-123456789012</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name example: <code>alias/MyAliasName</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Master key owned by Kinesis Data Streams: <code>alias/aws/kinesis</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyId(String keyId);
    }

    static final class BuilderImpl implements Builder {
        private String streamName;

        private String streamARN;

        private String streamStatus;

        private StreamModeDetails streamModeDetails;

        private List<Shard> shards = DefaultSdkAutoConstructList.getInstance();

        private Boolean hasMoreShards;

        private Integer retentionPeriodHours;

        private Instant streamCreationTimestamp;

        private List<EnhancedMetrics> enhancedMonitoring = DefaultSdkAutoConstructList.getInstance();

        private String encryptionType;

        private String keyId;

        private BuilderImpl() {
        }

        private BuilderImpl(StreamDescription model) {
            streamName(model.streamName);
            streamARN(model.streamARN);
            streamStatus(model.streamStatus);
            streamModeDetails(model.streamModeDetails);
            shards(model.shards);
            hasMoreShards(model.hasMoreShards);
            retentionPeriodHours(model.retentionPeriodHours);
            streamCreationTimestamp(model.streamCreationTimestamp);
            enhancedMonitoring(model.enhancedMonitoring);
            encryptionType(model.encryptionType);
            keyId(model.keyId);
        }

        public final String getStreamName() {
            return streamName;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final String getStreamARN() {
            return streamARN;
        }

        public final void setStreamARN(String streamARN) {
            this.streamARN = streamARN;
        }

        @Override
        public final Builder streamARN(String streamARN) {
            this.streamARN = streamARN;
            return this;
        }

        public final String getStreamStatus() {
            return streamStatus;
        }

        public final void setStreamStatus(String streamStatus) {
            this.streamStatus = streamStatus;
        }

        @Override
        public final Builder streamStatus(String streamStatus) {
            this.streamStatus = streamStatus;
            return this;
        }

        @Override
        public final Builder streamStatus(StreamStatus streamStatus) {
            this.streamStatus(streamStatus == null ? null : streamStatus.toString());
            return this;
        }

        public final StreamModeDetails.Builder getStreamModeDetails() {
            return streamModeDetails != null ? streamModeDetails.toBuilder() : null;
        }

        public final void setStreamModeDetails(StreamModeDetails.BuilderImpl streamModeDetails) {
            this.streamModeDetails = streamModeDetails != null ? streamModeDetails.build() : null;
        }

        @Override
        public final Builder streamModeDetails(StreamModeDetails streamModeDetails) {
            this.streamModeDetails = streamModeDetails;
            return this;
        }

        public final List<Shard.Builder> getShards() {
            List<Shard.Builder> result = ShardListCopier.copyToBuilder(this.shards);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setShards(Collection<Shard.BuilderImpl> shards) {
            this.shards = ShardListCopier.copyFromBuilder(shards);
        }

        @Override
        public final Builder shards(Collection<Shard> shards) {
            this.shards = ShardListCopier.copy(shards);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder shards(Shard... shards) {
            shards(Arrays.asList(shards));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder shards(Consumer<Shard.Builder>... shards) {
            shards(Stream.of(shards).map(c -> Shard.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getHasMoreShards() {
            return hasMoreShards;
        }

        public final void setHasMoreShards(Boolean hasMoreShards) {
            this.hasMoreShards = hasMoreShards;
        }

        @Override
        public final Builder hasMoreShards(Boolean hasMoreShards) {
            this.hasMoreShards = hasMoreShards;
            return this;
        }

        public final Integer getRetentionPeriodHours() {
            return retentionPeriodHours;
        }

        public final void setRetentionPeriodHours(Integer retentionPeriodHours) {
            this.retentionPeriodHours = retentionPeriodHours;
        }

        @Override
        public final Builder retentionPeriodHours(Integer retentionPeriodHours) {
            this.retentionPeriodHours = retentionPeriodHours;
            return this;
        }

        public final Instant getStreamCreationTimestamp() {
            return streamCreationTimestamp;
        }

        public final void setStreamCreationTimestamp(Instant streamCreationTimestamp) {
            this.streamCreationTimestamp = streamCreationTimestamp;
        }

        @Override
        public final Builder streamCreationTimestamp(Instant streamCreationTimestamp) {
            this.streamCreationTimestamp = streamCreationTimestamp;
            return this;
        }

        public final List<EnhancedMetrics.Builder> getEnhancedMonitoring() {
            List<EnhancedMetrics.Builder> result = EnhancedMonitoringListCopier.copyToBuilder(this.enhancedMonitoring);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnhancedMonitoring(Collection<EnhancedMetrics.BuilderImpl> enhancedMonitoring) {
            this.enhancedMonitoring = EnhancedMonitoringListCopier.copyFromBuilder(enhancedMonitoring);
        }

        @Override
        public final Builder enhancedMonitoring(Collection<EnhancedMetrics> enhancedMonitoring) {
            this.enhancedMonitoring = EnhancedMonitoringListCopier.copy(enhancedMonitoring);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enhancedMonitoring(EnhancedMetrics... enhancedMonitoring) {
            enhancedMonitoring(Arrays.asList(enhancedMonitoring));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enhancedMonitoring(Consumer<EnhancedMetrics.Builder>... enhancedMonitoring) {
            enhancedMonitoring(Stream.of(enhancedMonitoring).map(c -> EnhancedMetrics.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getEncryptionType() {
            return encryptionType;
        }

        public final void setEncryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
        }

        @Override
        public final Builder encryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
            return this;
        }

        @Override
        public final Builder encryptionType(EncryptionType encryptionType) {
            this.encryptionType(encryptionType == null ? null : encryptionType.toString());
            return this;
        }

        public final String getKeyId() {
            return keyId;
        }

        public final void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        @Override
        public final Builder keyId(String keyId) {
            this.keyId = keyId;
            return this;
        }

        @Override
        public StreamDescription build() {
            return new StreamDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
