/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an action to publish to an Amazon SNS topic.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SnsAction implements SdkPojo, Serializable, ToCopyableBuilder<SnsAction.Builder, SnsAction> {
    private static final SdkField<String> TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SnsAction::targetArn)).setter(setter(Builder::targetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetArn").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SnsAction::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> MESSAGE_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SnsAction::messageFormatAsString)).setter(setter(Builder::messageFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("messageFormat").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_ARN_FIELD,
            ROLE_ARN_FIELD, MESSAGE_FORMAT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String targetArn;

    private final String roleArn;

    private final String messageFormat;

    private SnsAction(BuilderImpl builder) {
        this.targetArn = builder.targetArn;
        this.roleArn = builder.roleArn;
        this.messageFormat = builder.messageFormat;
    }

    /**
     * <p>
     * The ARN of the SNS topic.
     * </p>
     * 
     * @return The ARN of the SNS topic.
     */
    public String targetArn() {
        return targetArn;
    }

    /**
     * <p>
     * The ARN of the IAM role that grants access.
     * </p>
     * 
     * @return The ARN of the IAM role that grants access.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The default value
     * of the attribute is "RAW". SNS uses this setting to determine if the payload should be parsed and relevant
     * platform-specific bits of the payload should be extracted. To read more about SNS message formats, see <a
     * href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html"
     * >http://docs.aws.amazon.com/sns/latest/dg/json-formats.html</a> refer to their official documentation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #messageFormatAsString}.
     * </p>
     * 
     * @return (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The
     *         default value of the attribute is "RAW". SNS uses this setting to determine if the payload should be
     *         parsed and relevant platform-specific bits of the payload should be extracted. To read more about SNS
     *         message formats, see <a
     *         href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html">http://docs.aws.
     *         amazon.com/sns/latest/dg/json-formats.html</a> refer to their official documentation.
     * @see MessageFormat
     */
    public MessageFormat messageFormat() {
        return MessageFormat.fromValue(messageFormat);
    }

    /**
     * <p>
     * (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The default value
     * of the attribute is "RAW". SNS uses this setting to determine if the payload should be parsed and relevant
     * platform-specific bits of the payload should be extracted. To read more about SNS message formats, see <a
     * href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html"
     * >http://docs.aws.amazon.com/sns/latest/dg/json-formats.html</a> refer to their official documentation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageFormat}
     * will return {@link MessageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #messageFormatAsString}.
     * </p>
     * 
     * @return (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The
     *         default value of the attribute is "RAW". SNS uses this setting to determine if the payload should be
     *         parsed and relevant platform-specific bits of the payload should be extracted. To read more about SNS
     *         message formats, see <a
     *         href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html">http://docs.aws.
     *         amazon.com/sns/latest/dg/json-formats.html</a> refer to their official documentation.
     * @see MessageFormat
     */
    public String messageFormatAsString() {
        return messageFormat;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(targetArn());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(messageFormatAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SnsAction)) {
            return false;
        }
        SnsAction other = (SnsAction) obj;
        return Objects.equals(targetArn(), other.targetArn()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(messageFormatAsString(), other.messageFormatAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("SnsAction").add("TargetArn", targetArn()).add("RoleArn", roleArn())
                .add("MessageFormat", messageFormatAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "targetArn":
            return Optional.ofNullable(clazz.cast(targetArn()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "messageFormat":
            return Optional.ofNullable(clazz.cast(messageFormatAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SnsAction, T> g) {
        return obj -> g.apply((SnsAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SnsAction> {
        /**
         * <p>
         * The ARN of the SNS topic.
         * </p>
         * 
         * @param targetArn
         *        The ARN of the SNS topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArn(String targetArn);

        /**
         * <p>
         * The ARN of the IAM role that grants access.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the IAM role that grants access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The default
         * value of the attribute is "RAW". SNS uses this setting to determine if the payload should be parsed and
         * relevant platform-specific bits of the payload should be extracted. To read more about SNS message formats,
         * see <a
         * href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html">http://docs.aws.amazon.com/sns/latest
         * /dg/json-formats.html</a> refer to their official documentation.
         * </p>
         * 
         * @param messageFormat
         *        (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The
         *        default value of the attribute is "RAW". SNS uses this setting to determine if the payload should be
         *        parsed and relevant platform-specific bits of the payload should be extracted. To read more about SNS
         *        message formats, see <a
         *        href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html">http://docs.aws
         *        .amazon.com/sns/latest/dg/json-formats.html</a> refer to their official documentation.
         * @see MessageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormat
         */
        Builder messageFormat(String messageFormat);

        /**
         * <p>
         * (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The default
         * value of the attribute is "RAW". SNS uses this setting to determine if the payload should be parsed and
         * relevant platform-specific bits of the payload should be extracted. To read more about SNS message formats,
         * see <a
         * href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html">http://docs.aws.amazon.com/sns/latest
         * /dg/json-formats.html</a> refer to their official documentation.
         * </p>
         * 
         * @param messageFormat
         *        (Optional) The message format of the message to publish. Accepted values are "JSON" and "RAW". The
         *        default value of the attribute is "RAW". SNS uses this setting to determine if the payload should be
         *        parsed and relevant platform-specific bits of the payload should be extracted. To read more about SNS
         *        message formats, see <a
         *        href="http://docs.aws.amazon.com/sns/latest/dg/json-formats.html">http://docs.aws
         *        .amazon.com/sns/latest/dg/json-formats.html</a> refer to their official documentation.
         * @see MessageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageFormat
         */
        Builder messageFormat(MessageFormat messageFormat);
    }

    static final class BuilderImpl implements Builder {
        private String targetArn;

        private String roleArn;

        private String messageFormat;

        private BuilderImpl() {
        }

        private BuilderImpl(SnsAction model) {
            targetArn(model.targetArn);
            roleArn(model.roleArn);
            messageFormat(model.messageFormat);
        }

        public final String getTargetArn() {
            return targetArn;
        }

        @Override
        public final Builder targetArn(String targetArn) {
            this.targetArn = targetArn;
            return this;
        }

        public final void setTargetArn(String targetArn) {
            this.targetArn = targetArn;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final String getMessageFormat() {
            return messageFormat;
        }

        @Override
        public final Builder messageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
            return this;
        }

        @Override
        public final Builder messageFormat(MessageFormat messageFormat) {
            this.messageFormat(messageFormat.toString());
            return this;
        }

        public final void setMessageFormat(String messageFormat) {
            this.messageFormat = messageFormat;
        }

        @Override
        public SnsAction build() {
            return new SnsAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
