/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Amazon CloudWatch Internet Monitor calculates measurements about the performance for your application's internet
 * traffic between client locations and Amazon Web Services. Amazon Web Services has substantial historical data about
 * internet performance and availability between Amazon Web Services services and different network providers and
 * geographies. By applying statistical analysis to the data, Internet Monitor can detect when the performance and
 * availability for your application has dropped, compared to an estimated baseline that's already calculated. To make
 * it easier to see those drops, we report that information to you in the form of health scores: a performance score and
 * an availability score.
 * </p>
 * <p>
 * Performance in Internet Monitor represents the estimated percentage of traffic that is not seeing a performance drop.
 * For example, a performance score of 99% for an end user and service location pair is equivalent to 1% of the traffic
 * experiencing a performance drop for that pair.
 * </p>
 * <p>
 * For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMExperienceScores"
 * >How Internet Monitor calculates performance and availability scores</a> in the Amazon CloudWatch Internet Monitor
 * section of the <i>CloudWatch User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PerformanceMeasurement implements SdkPojo, Serializable,
        ToCopyableBuilder<PerformanceMeasurement.Builder, PerformanceMeasurement> {
    private static final SdkField<Double> EXPERIENCE_SCORE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("ExperienceScore").getter(getter(PerformanceMeasurement::experienceScore))
            .setter(setter(Builder::experienceScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExperienceScore").build()).build();

    private static final SdkField<Double> PERCENT_OF_TOTAL_TRAFFIC_IMPACTED_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("PercentOfTotalTrafficImpacted")
            .getter(getter(PerformanceMeasurement::percentOfTotalTrafficImpacted))
            .setter(setter(Builder::percentOfTotalTrafficImpacted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentOfTotalTrafficImpacted")
                    .build()).build();

    private static final SdkField<Double> PERCENT_OF_CLIENT_LOCATION_IMPACTED_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("PercentOfClientLocationImpacted")
            .getter(getter(PerformanceMeasurement::percentOfClientLocationImpacted))
            .setter(setter(Builder::percentOfClientLocationImpacted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentOfClientLocationImpacted")
                    .build()).build();

    private static final SdkField<RoundTripTime> ROUND_TRIP_TIME_FIELD = SdkField
            .<RoundTripTime> builder(MarshallingType.SDK_POJO).memberName("RoundTripTime")
            .getter(getter(PerformanceMeasurement::roundTripTime)).setter(setter(Builder::roundTripTime))
            .constructor(RoundTripTime::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoundTripTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPERIENCE_SCORE_FIELD,
            PERCENT_OF_TOTAL_TRAFFIC_IMPACTED_FIELD, PERCENT_OF_CLIENT_LOCATION_IMPACTED_FIELD, ROUND_TRIP_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Double experienceScore;

    private final Double percentOfTotalTrafficImpacted;

    private final Double percentOfClientLocationImpacted;

    private final RoundTripTime roundTripTime;

    private PerformanceMeasurement(BuilderImpl builder) {
        this.experienceScore = builder.experienceScore;
        this.percentOfTotalTrafficImpacted = builder.percentOfTotalTrafficImpacted;
        this.percentOfClientLocationImpacted = builder.percentOfClientLocationImpacted;
        this.roundTripTime = builder.roundTripTime;
    }

    /**
     * <p>
     * Experience scores, or health scores, are calculated for different geographic and network provider combinations
     * (that is, different granularities) and also totaled into global scores. If you view performance or availability
     * scores without filtering for any specific geography or service provider, Amazon CloudWatch Internet Monitor
     * provides global health scores.
     * </p>
     * <p>
     * The Amazon CloudWatch Internet Monitor chapter in the CloudWatch User Guide includes detailed information about
     * how Internet Monitor calculates health scores, including performance and availability scores, and when it creates
     * and resolves health events. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMExperienceScores"
     * >How Amazon Web Services calculates performance and availability scores</a> in the Amazon CloudWatch Internet
     * Monitor section of the <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return Experience scores, or health scores, are calculated for different geographic and network provider
     *         combinations (that is, different granularities) and also totaled into global scores. If you view
     *         performance or availability scores without filtering for any specific geography or service provider,
     *         Amazon CloudWatch Internet Monitor provides global health scores.</p>
     *         <p>
     *         The Amazon CloudWatch Internet Monitor chapter in the CloudWatch User Guide includes detailed information
     *         about how Internet Monitor calculates health scores, including performance and availability scores, and
     *         when it creates and resolves health events. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMExperienceScores"
     *         >How Amazon Web Services calculates performance and availability scores</a> in the Amazon CloudWatch
     *         Internet Monitor section of the <i>CloudWatch User Guide</i>.
     */
    public final Double experienceScore() {
        return experienceScore;
    }

    /**
     * <p>
     * The impact on total traffic that a health event has, in increased latency or reduced availability. This is the
     * percentage of how much latency has increased or availability has decreased during the event, compared to what is
     * typical for traffic from this client location to the Amazon Web Services location using this client network.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
     * >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet Monitor
     * section of the <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return The impact on total traffic that a health event has, in increased latency or reduced availability. This
     *         is the percentage of how much latency has increased or availability has decreased during the event,
     *         compared to what is typical for traffic from this client location to the Amazon Web Services location
     *         using this client network.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
     *         >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet
     *         Monitor section of the <i>CloudWatch User Guide</i>.
     */
    public final Double percentOfTotalTrafficImpacted() {
        return percentOfTotalTrafficImpacted;
    }

    /**
     * <p>
     * How much performance impact was caused by a health event at a client location. For performance, this is the
     * percentage of how much latency increased during the event compared to typical performance for traffic, from this
     * client location to an Amazon Web Services location, using a specific client network.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
     * >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet Monitor
     * section of the <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return How much performance impact was caused by a health event at a client location. For performance, this is
     *         the percentage of how much latency increased during the event compared to typical performance for
     *         traffic, from this client location to an Amazon Web Services location, using a specific client network.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
     *         >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet
     *         Monitor section of the <i>CloudWatch User Guide</i>.
     */
    public final Double percentOfClientLocationImpacted() {
        return percentOfClientLocationImpacted;
    }

    /**
     * <p>
     * This is the percentage of how much round-trip time increased during the event compared to typical round-trip time
     * for your application for traffic.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
     * >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet Monitor
     * section of the <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return This is the percentage of how much round-trip time increased during the event compared to typical
     *         round-trip time for your application for traffic. </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
     *         >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet
     *         Monitor section of the <i>CloudWatch User Guide</i>.
     */
    public final RoundTripTime roundTripTime() {
        return roundTripTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(experienceScore());
        hashCode = 31 * hashCode + Objects.hashCode(percentOfTotalTrafficImpacted());
        hashCode = 31 * hashCode + Objects.hashCode(percentOfClientLocationImpacted());
        hashCode = 31 * hashCode + Objects.hashCode(roundTripTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PerformanceMeasurement)) {
            return false;
        }
        PerformanceMeasurement other = (PerformanceMeasurement) obj;
        return Objects.equals(experienceScore(), other.experienceScore())
                && Objects.equals(percentOfTotalTrafficImpacted(), other.percentOfTotalTrafficImpacted())
                && Objects.equals(percentOfClientLocationImpacted(), other.percentOfClientLocationImpacted())
                && Objects.equals(roundTripTime(), other.roundTripTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PerformanceMeasurement").add("ExperienceScore", experienceScore())
                .add("PercentOfTotalTrafficImpacted", percentOfTotalTrafficImpacted())
                .add("PercentOfClientLocationImpacted", percentOfClientLocationImpacted()).add("RoundTripTime", roundTripTime())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExperienceScore":
            return Optional.ofNullable(clazz.cast(experienceScore()));
        case "PercentOfTotalTrafficImpacted":
            return Optional.ofNullable(clazz.cast(percentOfTotalTrafficImpacted()));
        case "PercentOfClientLocationImpacted":
            return Optional.ofNullable(clazz.cast(percentOfClientLocationImpacted()));
        case "RoundTripTime":
            return Optional.ofNullable(clazz.cast(roundTripTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ExperienceScore", EXPERIENCE_SCORE_FIELD);
        map.put("PercentOfTotalTrafficImpacted", PERCENT_OF_TOTAL_TRAFFIC_IMPACTED_FIELD);
        map.put("PercentOfClientLocationImpacted", PERCENT_OF_CLIENT_LOCATION_IMPACTED_FIELD);
        map.put("RoundTripTime", ROUND_TRIP_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PerformanceMeasurement, T> g) {
        return obj -> g.apply((PerformanceMeasurement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PerformanceMeasurement> {
        /**
         * <p>
         * Experience scores, or health scores, are calculated for different geographic and network provider
         * combinations (that is, different granularities) and also totaled into global scores. If you view performance
         * or availability scores without filtering for any specific geography or service provider, Amazon CloudWatch
         * Internet Monitor provides global health scores.
         * </p>
         * <p>
         * The Amazon CloudWatch Internet Monitor chapter in the CloudWatch User Guide includes detailed information
         * about how Internet Monitor calculates health scores, including performance and availability scores, and when
         * it creates and resolves health events. For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMExperienceScores"
         * >How Amazon Web Services calculates performance and availability scores</a> in the Amazon CloudWatch Internet
         * Monitor section of the <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param experienceScore
         *        Experience scores, or health scores, are calculated for different geographic and network provider
         *        combinations (that is, different granularities) and also totaled into global scores. If you view
         *        performance or availability scores without filtering for any specific geography or service provider,
         *        Amazon CloudWatch Internet Monitor provides global health scores.</p>
         *        <p>
         *        The Amazon CloudWatch Internet Monitor chapter in the CloudWatch User Guide includes detailed
         *        information about how Internet Monitor calculates health scores, including performance and
         *        availability scores, and when it creates and resolves health events. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMExperienceScores"
         *        >How Amazon Web Services calculates performance and availability scores</a> in the Amazon CloudWatch
         *        Internet Monitor section of the <i>CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder experienceScore(Double experienceScore);

        /**
         * <p>
         * The impact on total traffic that a health event has, in increased latency or reduced availability. This is
         * the percentage of how much latency has increased or availability has decreased during the event, compared to
         * what is typical for traffic from this client location to the Amazon Web Services location using this client
         * network.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
         * >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet Monitor
         * section of the <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param percentOfTotalTrafficImpacted
         *        The impact on total traffic that a health event has, in increased latency or reduced availability.
         *        This is the percentage of how much latency has increased or availability has decreased during the
         *        event, compared to what is typical for traffic from this client location to the Amazon Web Services
         *        location using this client network.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
         *        >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet
         *        Monitor section of the <i>CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentOfTotalTrafficImpacted(Double percentOfTotalTrafficImpacted);

        /**
         * <p>
         * How much performance impact was caused by a health event at a client location. For performance, this is the
         * percentage of how much latency increased during the event compared to typical performance for traffic, from
         * this client location to an Amazon Web Services location, using a specific client network.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
         * >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet Monitor
         * section of the <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param percentOfClientLocationImpacted
         *        How much performance impact was caused by a health event at a client location. For performance, this
         *        is the percentage of how much latency increased during the event compared to typical performance for
         *        traffic, from this client location to an Amazon Web Services location, using a specific client
         *        network. </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
         *        >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet
         *        Monitor section of the <i>CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentOfClientLocationImpacted(Double percentOfClientLocationImpacted);

        /**
         * <p>
         * This is the percentage of how much round-trip time increased during the event compared to typical round-trip
         * time for your application for traffic.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
         * >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet Monitor
         * section of the <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param roundTripTime
         *        This is the percentage of how much round-trip time increased during the event compared to typical
         *        round-trip time for your application for traffic. </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
         *        >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet
         *        Monitor section of the <i>CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roundTripTime(RoundTripTime roundTripTime);

        /**
         * <p>
         * This is the percentage of how much round-trip time increased during the event compared to typical round-trip
         * time for your application for traffic.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-inside-internet-monitor.html#IMHealthEventStartStop"
         * >When Amazon Web Services creates and resolves health events</a> in the Amazon CloudWatch Internet Monitor
         * section of the <i>CloudWatch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link RoundTripTime.Builder} avoiding the need
         * to create one manually via {@link RoundTripTime#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RoundTripTime.Builder#build()} is called immediately and its
         * result is passed to {@link #roundTripTime(RoundTripTime)}.
         * 
         * @param roundTripTime
         *        a consumer that will call methods on {@link RoundTripTime.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #roundTripTime(RoundTripTime)
         */
        default Builder roundTripTime(Consumer<RoundTripTime.Builder> roundTripTime) {
            return roundTripTime(RoundTripTime.builder().applyMutation(roundTripTime).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Double experienceScore;

        private Double percentOfTotalTrafficImpacted;

        private Double percentOfClientLocationImpacted;

        private RoundTripTime roundTripTime;

        private BuilderImpl() {
        }

        private BuilderImpl(PerformanceMeasurement model) {
            experienceScore(model.experienceScore);
            percentOfTotalTrafficImpacted(model.percentOfTotalTrafficImpacted);
            percentOfClientLocationImpacted(model.percentOfClientLocationImpacted);
            roundTripTime(model.roundTripTime);
        }

        public final Double getExperienceScore() {
            return experienceScore;
        }

        public final void setExperienceScore(Double experienceScore) {
            this.experienceScore = experienceScore;
        }

        @Override
        public final Builder experienceScore(Double experienceScore) {
            this.experienceScore = experienceScore;
            return this;
        }

        public final Double getPercentOfTotalTrafficImpacted() {
            return percentOfTotalTrafficImpacted;
        }

        public final void setPercentOfTotalTrafficImpacted(Double percentOfTotalTrafficImpacted) {
            this.percentOfTotalTrafficImpacted = percentOfTotalTrafficImpacted;
        }

        @Override
        public final Builder percentOfTotalTrafficImpacted(Double percentOfTotalTrafficImpacted) {
            this.percentOfTotalTrafficImpacted = percentOfTotalTrafficImpacted;
            return this;
        }

        public final Double getPercentOfClientLocationImpacted() {
            return percentOfClientLocationImpacted;
        }

        public final void setPercentOfClientLocationImpacted(Double percentOfClientLocationImpacted) {
            this.percentOfClientLocationImpacted = percentOfClientLocationImpacted;
        }

        @Override
        public final Builder percentOfClientLocationImpacted(Double percentOfClientLocationImpacted) {
            this.percentOfClientLocationImpacted = percentOfClientLocationImpacted;
            return this;
        }

        public final RoundTripTime.Builder getRoundTripTime() {
            return roundTripTime != null ? roundTripTime.toBuilder() : null;
        }

        public final void setRoundTripTime(RoundTripTime.BuilderImpl roundTripTime) {
            this.roundTripTime = roundTripTime != null ? roundTripTime.build() : null;
        }

        @Override
        public final Builder roundTripTime(RoundTripTime roundTripTime) {
            this.roundTripTime = roundTripTime;
            return this;
        }

        @Override
        public PerformanceMeasurement build() {
            return new PerformanceMeasurement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
