/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A summary of information about an internet event in Amazon CloudWatch Internet Monitor. Internet events are issues
 * that cause performance degradation or availability problems for impacted Amazon Web Services client locations.
 * Internet Monitor displays information about recent global health events, called internet events, on a global outages
 * map that is available to all Amazon Web Services customers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InternetEventSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<InternetEventSummary.Builder, InternetEventSummary> {
    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventId").getter(getter(InternetEventSummary::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventId").build()).build();

    private static final SdkField<String> EVENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventArn").getter(getter(InternetEventSummary::eventArn)).setter(setter(Builder::eventArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventArn").build()).build();

    private static final SdkField<Instant> STARTED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedAt")
            .getter(getter(InternetEventSummary::startedAt))
            .setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> ENDED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndedAt")
            .getter(getter(InternetEventSummary::endedAt))
            .setter(setter(Builder::endedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<ClientLocation> CLIENT_LOCATION_FIELD = SdkField
            .<ClientLocation> builder(MarshallingType.SDK_POJO).memberName("ClientLocation")
            .getter(getter(InternetEventSummary::clientLocation)).setter(setter(Builder::clientLocation))
            .constructor(ClientLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientLocation").build()).build();

    private static final SdkField<String> EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventType").getter(getter(InternetEventSummary::eventTypeAsString)).setter(setter(Builder::eventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventType").build()).build();

    private static final SdkField<String> EVENT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventStatus").getter(getter(InternetEventSummary::eventStatusAsString))
            .setter(setter(Builder::eventStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_ID_FIELD,
            EVENT_ARN_FIELD, STARTED_AT_FIELD, ENDED_AT_FIELD, CLIENT_LOCATION_FIELD, EVENT_TYPE_FIELD, EVENT_STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String eventId;

    private final String eventArn;

    private final Instant startedAt;

    private final Instant endedAt;

    private final ClientLocation clientLocation;

    private final String eventType;

    private final String eventStatus;

    private InternetEventSummary(BuilderImpl builder) {
        this.eventId = builder.eventId;
        this.eventArn = builder.eventArn;
        this.startedAt = builder.startedAt;
        this.endedAt = builder.endedAt;
        this.clientLocation = builder.clientLocation;
        this.eventType = builder.eventType;
        this.eventStatus = builder.eventStatus;
    }

    /**
     * <p>
     * The internally-generated identifier of an internet event.
     * </p>
     * 
     * @return The internally-generated identifier of an internet event.
     */
    public final String eventId() {
        return eventId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the internet event.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the internet event.
     */
    public final String eventArn() {
        return eventArn;
    }

    /**
     * <p>
     * The time when an internet event started.
     * </p>
     * 
     * @return The time when an internet event started.
     */
    public final Instant startedAt() {
        return startedAt;
    }

    /**
     * <p>
     * The time when an internet event ended. If the event hasn't ended yet, this value is empty.
     * </p>
     * 
     * @return The time when an internet event ended. If the event hasn't ended yet, this value is empty.
     */
    public final Instant endedAt() {
        return endedAt;
    }

    /**
     * <p>
     * The impacted location, such as a city, that Amazon Web Services clients access application resources from.
     * </p>
     * 
     * @return The impacted location, such as a city, that Amazon Web Services clients access application resources
     *         from.
     */
    public final ClientLocation clientLocation() {
        return clientLocation;
    }

    /**
     * <p>
     * The type of network impairment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link InternetEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The type of network impairment.
     * @see InternetEventType
     */
    public final InternetEventType eventType() {
        return InternetEventType.fromValue(eventType);
    }

    /**
     * <p>
     * The type of network impairment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link InternetEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The type of network impairment.
     * @see InternetEventType
     */
    public final String eventTypeAsString() {
        return eventType;
    }

    /**
     * <p>
     * The status of an internet event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventStatus} will
     * return {@link InternetEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventStatusAsString}.
     * </p>
     * 
     * @return The status of an internet event.
     * @see InternetEventStatus
     */
    public final InternetEventStatus eventStatus() {
        return InternetEventStatus.fromValue(eventStatus);
    }

    /**
     * <p>
     * The status of an internet event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventStatus} will
     * return {@link InternetEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventStatusAsString}.
     * </p>
     * 
     * @return The status of an internet event.
     * @see InternetEventStatus
     */
    public final String eventStatusAsString() {
        return eventStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        hashCode = 31 * hashCode + Objects.hashCode(eventArn());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        hashCode = 31 * hashCode + Objects.hashCode(endedAt());
        hashCode = 31 * hashCode + Objects.hashCode(clientLocation());
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InternetEventSummary)) {
            return false;
        }
        InternetEventSummary other = (InternetEventSummary) obj;
        return Objects.equals(eventId(), other.eventId()) && Objects.equals(eventArn(), other.eventArn())
                && Objects.equals(startedAt(), other.startedAt()) && Objects.equals(endedAt(), other.endedAt())
                && Objects.equals(clientLocation(), other.clientLocation())
                && Objects.equals(eventTypeAsString(), other.eventTypeAsString())
                && Objects.equals(eventStatusAsString(), other.eventStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InternetEventSummary").add("EventId", eventId()).add("EventArn", eventArn())
                .add("StartedAt", startedAt()).add("EndedAt", endedAt()).add("ClientLocation", clientLocation())
                .add("EventType", eventTypeAsString()).add("EventStatus", eventStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        case "EventArn":
            return Optional.ofNullable(clazz.cast(eventArn()));
        case "StartedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        case "EndedAt":
            return Optional.ofNullable(clazz.cast(endedAt()));
        case "ClientLocation":
            return Optional.ofNullable(clazz.cast(clientLocation()));
        case "EventType":
            return Optional.ofNullable(clazz.cast(eventTypeAsString()));
        case "EventStatus":
            return Optional.ofNullable(clazz.cast(eventStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EventId", EVENT_ID_FIELD);
        map.put("EventArn", EVENT_ARN_FIELD);
        map.put("StartedAt", STARTED_AT_FIELD);
        map.put("EndedAt", ENDED_AT_FIELD);
        map.put("ClientLocation", CLIENT_LOCATION_FIELD);
        map.put("EventType", EVENT_TYPE_FIELD);
        map.put("EventStatus", EVENT_STATUS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InternetEventSummary, T> g) {
        return obj -> g.apply((InternetEventSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InternetEventSummary> {
        /**
         * <p>
         * The internally-generated identifier of an internet event.
         * </p>
         * 
         * @param eventId
         *        The internally-generated identifier of an internet event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the internet event.
         * </p>
         * 
         * @param eventArn
         *        The Amazon Resource Name (ARN) of the internet event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventArn(String eventArn);

        /**
         * <p>
         * The time when an internet event started.
         * </p>
         * 
         * @param startedAt
         *        The time when an internet event started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);

        /**
         * <p>
         * The time when an internet event ended. If the event hasn't ended yet, this value is empty.
         * </p>
         * 
         * @param endedAt
         *        The time when an internet event ended. If the event hasn't ended yet, this value is empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endedAt(Instant endedAt);

        /**
         * <p>
         * The impacted location, such as a city, that Amazon Web Services clients access application resources from.
         * </p>
         * 
         * @param clientLocation
         *        The impacted location, such as a city, that Amazon Web Services clients access application resources
         *        from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientLocation(ClientLocation clientLocation);

        /**
         * <p>
         * The impacted location, such as a city, that Amazon Web Services clients access application resources from.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClientLocation.Builder} avoiding the need
         * to create one manually via {@link ClientLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClientLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #clientLocation(ClientLocation)}.
         * 
         * @param clientLocation
         *        a consumer that will call methods on {@link ClientLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clientLocation(ClientLocation)
         */
        default Builder clientLocation(Consumer<ClientLocation.Builder> clientLocation) {
            return clientLocation(ClientLocation.builder().applyMutation(clientLocation).build());
        }

        /**
         * <p>
         * The type of network impairment.
         * </p>
         * 
         * @param eventType
         *        The type of network impairment.
         * @see InternetEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InternetEventType
         */
        Builder eventType(String eventType);

        /**
         * <p>
         * The type of network impairment.
         * </p>
         * 
         * @param eventType
         *        The type of network impairment.
         * @see InternetEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InternetEventType
         */
        Builder eventType(InternetEventType eventType);

        /**
         * <p>
         * The status of an internet event.
         * </p>
         * 
         * @param eventStatus
         *        The status of an internet event.
         * @see InternetEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InternetEventStatus
         */
        Builder eventStatus(String eventStatus);

        /**
         * <p>
         * The status of an internet event.
         * </p>
         * 
         * @param eventStatus
         *        The status of an internet event.
         * @see InternetEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InternetEventStatus
         */
        Builder eventStatus(InternetEventStatus eventStatus);
    }

    static final class BuilderImpl implements Builder {
        private String eventId;

        private String eventArn;

        private Instant startedAt;

        private Instant endedAt;

        private ClientLocation clientLocation;

        private String eventType;

        private String eventStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(InternetEventSummary model) {
            eventId(model.eventId);
            eventArn(model.eventArn);
            startedAt(model.startedAt);
            endedAt(model.endedAt);
            clientLocation(model.clientLocation);
            eventType(model.eventType);
            eventStatus(model.eventStatus);
        }

        public final String getEventId() {
            return eventId;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public final String getEventArn() {
            return eventArn;
        }

        public final void setEventArn(String eventArn) {
            this.eventArn = eventArn;
        }

        @Override
        public final Builder eventArn(String eventArn) {
            this.eventArn = eventArn;
            return this;
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final Instant getEndedAt() {
            return endedAt;
        }

        public final void setEndedAt(Instant endedAt) {
            this.endedAt = endedAt;
        }

        @Override
        public final Builder endedAt(Instant endedAt) {
            this.endedAt = endedAt;
            return this;
        }

        public final ClientLocation.Builder getClientLocation() {
            return clientLocation != null ? clientLocation.toBuilder() : null;
        }

        public final void setClientLocation(ClientLocation.BuilderImpl clientLocation) {
            this.clientLocation = clientLocation != null ? clientLocation.build() : null;
        }

        @Override
        public final Builder clientLocation(ClientLocation clientLocation) {
            this.clientLocation = clientLocation;
            return this;
        }

        public final String getEventType() {
            return eventType;
        }

        public final void setEventType(String eventType) {
            this.eventType = eventType;
        }

        @Override
        public final Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        @Override
        public final Builder eventType(InternetEventType eventType) {
            this.eventType(eventType == null ? null : eventType.toString());
            return this;
        }

        public final String getEventStatus() {
            return eventStatus;
        }

        public final void setEventStatus(String eventStatus) {
            this.eventStatus = eventStatus;
        }

        @Override
        public final Builder eventStatus(String eventStatus) {
            this.eventStatus = eventStatus;
            return this;
        }

        @Override
        public final Builder eventStatus(InternetEventStatus eventStatus) {
            this.eventStatus(eventStatus == null ? null : eventStatus.toString());
            return this;
        }

        @Override
        public InternetEventSummary build() {
            return new InternetEventSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
