/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateMonitorRequest extends InternetMonitorRequest implements
        ToCopyableBuilder<UpdateMonitorRequest.Builder, UpdateMonitorRequest> {
    private static final SdkField<String> MONITOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MonitorName").getter(getter(UpdateMonitorRequest::monitorName)).setter(setter(Builder::monitorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("MonitorName").build()).build();

    private static final SdkField<List<String>> RESOURCES_TO_ADD_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourcesToAdd")
            .getter(getter(UpdateMonitorRequest::resourcesToAdd))
            .setter(setter(Builder::resourcesToAdd))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourcesToAdd").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCES_TO_REMOVE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourcesToRemove")
            .getter(getter(UpdateMonitorRequest::resourcesToRemove))
            .setter(setter(Builder::resourcesToRemove))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourcesToRemove").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(UpdateMonitorRequest::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(UpdateMonitorRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Integer> MAX_CITY_NETWORKS_TO_MONITOR_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaxCityNetworksToMonitor")
            .getter(getter(UpdateMonitorRequest::maxCityNetworksToMonitor)).setter(setter(Builder::maxCityNetworksToMonitor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCityNetworksToMonitor").build())
            .build();

    private static final SdkField<InternetMeasurementsLogDelivery> INTERNET_MEASUREMENTS_LOG_DELIVERY_FIELD = SdkField
            .<InternetMeasurementsLogDelivery> builder(MarshallingType.SDK_POJO)
            .memberName("InternetMeasurementsLogDelivery")
            .getter(getter(UpdateMonitorRequest::internetMeasurementsLogDelivery))
            .setter(setter(Builder::internetMeasurementsLogDelivery))
            .constructor(InternetMeasurementsLogDelivery::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InternetMeasurementsLogDelivery")
                    .build()).build();

    private static final SdkField<Integer> TRAFFIC_PERCENTAGE_TO_MONITOR_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TrafficPercentageToMonitor")
            .getter(getter(UpdateMonitorRequest::trafficPercentageToMonitor))
            .setter(setter(Builder::trafficPercentageToMonitor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrafficPercentageToMonitor").build())
            .build();

    private static final SdkField<HealthEventsConfig> HEALTH_EVENTS_CONFIG_FIELD = SdkField
            .<HealthEventsConfig> builder(MarshallingType.SDK_POJO).memberName("HealthEventsConfig")
            .getter(getter(UpdateMonitorRequest::healthEventsConfig)).setter(setter(Builder::healthEventsConfig))
            .constructor(HealthEventsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthEventsConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MONITOR_NAME_FIELD,
            RESOURCES_TO_ADD_FIELD, RESOURCES_TO_REMOVE_FIELD, STATUS_FIELD, CLIENT_TOKEN_FIELD,
            MAX_CITY_NETWORKS_TO_MONITOR_FIELD, INTERNET_MEASUREMENTS_LOG_DELIVERY_FIELD, TRAFFIC_PERCENTAGE_TO_MONITOR_FIELD,
            HEALTH_EVENTS_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("MonitorName", MONITOR_NAME_FIELD);
                    put("ResourcesToAdd", RESOURCES_TO_ADD_FIELD);
                    put("ResourcesToRemove", RESOURCES_TO_REMOVE_FIELD);
                    put("Status", STATUS_FIELD);
                    put("ClientToken", CLIENT_TOKEN_FIELD);
                    put("MaxCityNetworksToMonitor", MAX_CITY_NETWORKS_TO_MONITOR_FIELD);
                    put("InternetMeasurementsLogDelivery", INTERNET_MEASUREMENTS_LOG_DELIVERY_FIELD);
                    put("TrafficPercentageToMonitor", TRAFFIC_PERCENTAGE_TO_MONITOR_FIELD);
                    put("HealthEventsConfig", HEALTH_EVENTS_CONFIG_FIELD);
                }
            });

    private final String monitorName;

    private final List<String> resourcesToAdd;

    private final List<String> resourcesToRemove;

    private final String status;

    private final String clientToken;

    private final Integer maxCityNetworksToMonitor;

    private final InternetMeasurementsLogDelivery internetMeasurementsLogDelivery;

    private final Integer trafficPercentageToMonitor;

    private final HealthEventsConfig healthEventsConfig;

    private UpdateMonitorRequest(BuilderImpl builder) {
        super(builder);
        this.monitorName = builder.monitorName;
        this.resourcesToAdd = builder.resourcesToAdd;
        this.resourcesToRemove = builder.resourcesToRemove;
        this.status = builder.status;
        this.clientToken = builder.clientToken;
        this.maxCityNetworksToMonitor = builder.maxCityNetworksToMonitor;
        this.internetMeasurementsLogDelivery = builder.internetMeasurementsLogDelivery;
        this.trafficPercentageToMonitor = builder.trafficPercentageToMonitor;
        this.healthEventsConfig = builder.healthEventsConfig;
    }

    /**
     * <p>
     * The name of the monitor.
     * </p>
     * 
     * @return The name of the monitor.
     */
    public final String monitorName() {
        return monitorName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourcesToAdd property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourcesToAdd() {
        return resourcesToAdd != null && !(resourcesToAdd instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs). Resources can
     * be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.
     * </p>
     * <p>
     * You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or you can
     * add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
     * </p>
     * <note>
     * <p>
     * If you add only Amazon Virtual Private Clouds resources, at least one VPC must have an Internet Gateway attached
     * to it, to make sure that it has internet connectivity.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourcesToAdd} method.
     * </p>
     * 
     * @return The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs).
     *         Resources can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.</p>
     *         <p>
     *         You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or
     *         you can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
     *         </p>
     *         <note>
     *         <p>
     *         If you add only Amazon Virtual Private Clouds resources, at least one VPC must have an Internet Gateway
     *         attached to it, to make sure that it has internet connectivity.
     *         </p>
     */
    public final List<String> resourcesToAdd() {
        return resourcesToAdd;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourcesToRemove property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourcesToRemove() {
        return resourcesToRemove != null && !(resourcesToRemove instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resources to remove from a monitor, which you provide as a set of Amazon Resource Names (ARNs).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourcesToRemove} method.
     * </p>
     * 
     * @return The resources to remove from a monitor, which you provide as a set of Amazon Resource Names (ARNs).
     */
    public final List<String> resourcesToRemove() {
        return resourcesToRemove;
    }

    /**
     * <p>
     * The status for a monitor. The accepted values for <code>Status</code> with the <code>UpdateMonitor</code> API
     * call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>. The following values are <i>not</i>
     * accepted: <code>PENDING</code>, and <code>ERROR</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MonitorConfigState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status for a monitor. The accepted values for <code>Status</code> with the <code>UpdateMonitor</code>
     *         API call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>. The following values are
     *         <i>not</i> accepted: <code>PENDING</code>, and <code>ERROR</code>.
     * @see MonitorConfigState
     */
    public final MonitorConfigState status() {
        return MonitorConfigState.fromValue(status);
    }

    /**
     * <p>
     * The status for a monitor. The accepted values for <code>Status</code> with the <code>UpdateMonitor</code> API
     * call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>. The following values are <i>not</i>
     * accepted: <code>PENDING</code>, and <code>ERROR</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MonitorConfigState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status for a monitor. The accepted values for <code>Status</code> with the <code>UpdateMonitor</code>
     *         API call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>. The following values are
     *         <i>not</i> accepted: <code>PENDING</code>, and <code>ERROR</code>.
     * @see MonitorConfigState
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent API request.
     * You should not reuse the same client token for other API requests.
     * </p>
     * 
     * @return A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent API
     *         request. You should not reuse the same client token for other API requests.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The maximum number of city-networks to monitor for your application. A city-network is the location (city) where
     * clients access your application resources from and the ASN or network provider, such as an internet service
     * provider (ISP), that clients access the resources through. Setting this limit can help control billing costs.
     * </p>
     * 
     * @return The maximum number of city-networks to monitor for your application. A city-network is the location
     *         (city) where clients access your application resources from and the ASN or network provider, such as an
     *         internet service provider (ISP), that clients access the resources through. Setting this limit can help
     *         control billing costs.
     */
    public final Integer maxCityNetworksToMonitor() {
        return maxCityNetworksToMonitor;
    }

    /**
     * <p>
     * Publish internet measurements for Internet Monitor to another location, such as an Amazon S3 bucket. The
     * measurements are also published to Amazon CloudWatch Logs.
     * </p>
     * 
     * @return Publish internet measurements for Internet Monitor to another location, such as an Amazon S3 bucket. The
     *         measurements are also published to Amazon CloudWatch Logs.
     */
    public final InternetMeasurementsLogDelivery internetMeasurementsLogDelivery() {
        return internetMeasurementsLogDelivery;
    }

    /**
     * <p>
     * The percentage of the internet-facing traffic for your application that you want to monitor with this monitor. If
     * you set a city-networks maximum, that limit overrides the traffic percentage that you set.
     * </p>
     * <p>
     * To learn more, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html">Choosing an
     * application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor section of the
     * <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return The percentage of the internet-facing traffic for your application that you want to monitor with this
     *         monitor. If you set a city-networks maximum, that limit overrides the traffic percentage that you
     *         set.</p>
     *         <p>
     *         To learn more, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html">Choosing
     *         an application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor section of
     *         the <i>CloudWatch User Guide</i>.
     */
    public final Integer trafficPercentageToMonitor() {
        return trafficPercentageToMonitor;
    }

    /**
     * <p>
     * The list of health score thresholds. A threshold percentage for health scores, along with other configuration
     * information, determines when Internet Monitor creates a health event when there's an internet issue that affects
     * your application end users.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
     * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return The list of health score thresholds. A threshold percentage for health scores, along with other
     *         configuration information, determines when Internet Monitor creates a health event when there's an
     *         internet issue that affects your application end users.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
     *         > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
     */
    public final HealthEventsConfig healthEventsConfig() {
        return healthEventsConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(monitorName());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourcesToAdd() ? resourcesToAdd() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourcesToRemove() ? resourcesToRemove() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(maxCityNetworksToMonitor());
        hashCode = 31 * hashCode + Objects.hashCode(internetMeasurementsLogDelivery());
        hashCode = 31 * hashCode + Objects.hashCode(trafficPercentageToMonitor());
        hashCode = 31 * hashCode + Objects.hashCode(healthEventsConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateMonitorRequest)) {
            return false;
        }
        UpdateMonitorRequest other = (UpdateMonitorRequest) obj;
        return Objects.equals(monitorName(), other.monitorName()) && hasResourcesToAdd() == other.hasResourcesToAdd()
                && Objects.equals(resourcesToAdd(), other.resourcesToAdd())
                && hasResourcesToRemove() == other.hasResourcesToRemove()
                && Objects.equals(resourcesToRemove(), other.resourcesToRemove())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(clientToken(), other.clientToken())
                && Objects.equals(maxCityNetworksToMonitor(), other.maxCityNetworksToMonitor())
                && Objects.equals(internetMeasurementsLogDelivery(), other.internetMeasurementsLogDelivery())
                && Objects.equals(trafficPercentageToMonitor(), other.trafficPercentageToMonitor())
                && Objects.equals(healthEventsConfig(), other.healthEventsConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateMonitorRequest").add("MonitorName", monitorName())
                .add("ResourcesToAdd", hasResourcesToAdd() ? resourcesToAdd() : null)
                .add("ResourcesToRemove", hasResourcesToRemove() ? resourcesToRemove() : null).add("Status", statusAsString())
                .add("ClientToken", clientToken()).add("MaxCityNetworksToMonitor", maxCityNetworksToMonitor())
                .add("InternetMeasurementsLogDelivery", internetMeasurementsLogDelivery())
                .add("TrafficPercentageToMonitor", trafficPercentageToMonitor()).add("HealthEventsConfig", healthEventsConfig())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MonitorName":
            return Optional.ofNullable(clazz.cast(monitorName()));
        case "ResourcesToAdd":
            return Optional.ofNullable(clazz.cast(resourcesToAdd()));
        case "ResourcesToRemove":
            return Optional.ofNullable(clazz.cast(resourcesToRemove()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "MaxCityNetworksToMonitor":
            return Optional.ofNullable(clazz.cast(maxCityNetworksToMonitor()));
        case "InternetMeasurementsLogDelivery":
            return Optional.ofNullable(clazz.cast(internetMeasurementsLogDelivery()));
        case "TrafficPercentageToMonitor":
            return Optional.ofNullable(clazz.cast(trafficPercentageToMonitor()));
        case "HealthEventsConfig":
            return Optional.ofNullable(clazz.cast(healthEventsConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<UpdateMonitorRequest, T> g) {
        return obj -> g.apply((UpdateMonitorRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends InternetMonitorRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateMonitorRequest> {
        /**
         * <p>
         * The name of the monitor.
         * </p>
         * 
         * @param monitorName
         *        The name of the monitor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitorName(String monitorName);

        /**
         * <p>
         * The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs). Resources
         * can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.
         * </p>
         * <p>
         * You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or you
         * can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         * </p>
         * <note>
         * <p>
         * If you add only Amazon Virtual Private Clouds resources, at least one VPC must have an Internet Gateway
         * attached to it, to make sure that it has internet connectivity.
         * </p>
         * </note>
         * 
         * @param resourcesToAdd
         *        The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         *        Resources can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.</p>
         *        <p>
         *        You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories,
         *        or you can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         *        </p>
         *        <note>
         *        <p>
         *        If you add only Amazon Virtual Private Clouds resources, at least one VPC must have an Internet
         *        Gateway attached to it, to make sure that it has internet connectivity.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesToAdd(Collection<String> resourcesToAdd);

        /**
         * <p>
         * The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs). Resources
         * can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.
         * </p>
         * <p>
         * You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or you
         * can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         * </p>
         * <note>
         * <p>
         * If you add only Amazon Virtual Private Clouds resources, at least one VPC must have an Internet Gateway
         * attached to it, to make sure that it has internet connectivity.
         * </p>
         * </note>
         * 
         * @param resourcesToAdd
         *        The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         *        Resources can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.</p>
         *        <p>
         *        You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories,
         *        or you can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         *        </p>
         *        <note>
         *        <p>
         *        If you add only Amazon Virtual Private Clouds resources, at least one VPC must have an Internet
         *        Gateway attached to it, to make sure that it has internet connectivity.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesToAdd(String... resourcesToAdd);

        /**
         * <p>
         * The resources to remove from a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         * </p>
         * 
         * @param resourcesToRemove
         *        The resources to remove from a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesToRemove(Collection<String> resourcesToRemove);

        /**
         * <p>
         * The resources to remove from a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         * </p>
         * 
         * @param resourcesToRemove
         *        The resources to remove from a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesToRemove(String... resourcesToRemove);

        /**
         * <p>
         * The status for a monitor. The accepted values for <code>Status</code> with the <code>UpdateMonitor</code> API
         * call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>. The following values are <i>not</i>
         * accepted: <code>PENDING</code>, and <code>ERROR</code>.
         * </p>
         * 
         * @param status
         *        The status for a monitor. The accepted values for <code>Status</code> with the
         *        <code>UpdateMonitor</code> API call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>.
         *        The following values are <i>not</i> accepted: <code>PENDING</code>, and <code>ERROR</code>.
         * @see MonitorConfigState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MonitorConfigState
         */
        Builder status(String status);

        /**
         * <p>
         * The status for a monitor. The accepted values for <code>Status</code> with the <code>UpdateMonitor</code> API
         * call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>. The following values are <i>not</i>
         * accepted: <code>PENDING</code>, and <code>ERROR</code>.
         * </p>
         * 
         * @param status
         *        The status for a monitor. The accepted values for <code>Status</code> with the
         *        <code>UpdateMonitor</code> API call are the following: <code>ACTIVE</code> and <code>INACTIVE</code>.
         *        The following values are <i>not</i> accepted: <code>PENDING</code>, and <code>ERROR</code>.
         * @see MonitorConfigState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MonitorConfigState
         */
        Builder status(MonitorConfigState status);

        /**
         * <p>
         * A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent API
         * request. You should not reuse the same client token for other API requests.
         * </p>
         * 
         * @param clientToken
         *        A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent
         *        API request. You should not reuse the same client token for other API requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The maximum number of city-networks to monitor for your application. A city-network is the location (city)
         * where clients access your application resources from and the ASN or network provider, such as an internet
         * service provider (ISP), that clients access the resources through. Setting this limit can help control
         * billing costs.
         * </p>
         * 
         * @param maxCityNetworksToMonitor
         *        The maximum number of city-networks to monitor for your application. A city-network is the location
         *        (city) where clients access your application resources from and the ASN or network provider, such as
         *        an internet service provider (ISP), that clients access the resources through. Setting this limit can
         *        help control billing costs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCityNetworksToMonitor(Integer maxCityNetworksToMonitor);

        /**
         * <p>
         * Publish internet measurements for Internet Monitor to another location, such as an Amazon S3 bucket. The
         * measurements are also published to Amazon CloudWatch Logs.
         * </p>
         * 
         * @param internetMeasurementsLogDelivery
         *        Publish internet measurements for Internet Monitor to another location, such as an Amazon S3 bucket.
         *        The measurements are also published to Amazon CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery internetMeasurementsLogDelivery);

        /**
         * <p>
         * Publish internet measurements for Internet Monitor to another location, such as an Amazon S3 bucket. The
         * measurements are also published to Amazon CloudWatch Logs.
         * </p>
         * This is a convenience method that creates an instance of the {@link InternetMeasurementsLogDelivery.Builder}
         * avoiding the need to create one manually via {@link InternetMeasurementsLogDelivery#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InternetMeasurementsLogDelivery.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery)}.
         * 
         * @param internetMeasurementsLogDelivery
         *        a consumer that will call methods on {@link InternetMeasurementsLogDelivery.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery)
         */
        default Builder internetMeasurementsLogDelivery(
                Consumer<InternetMeasurementsLogDelivery.Builder> internetMeasurementsLogDelivery) {
            return internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery.builder()
                    .applyMutation(internetMeasurementsLogDelivery).build());
        }

        /**
         * <p>
         * The percentage of the internet-facing traffic for your application that you want to monitor with this
         * monitor. If you set a city-networks maximum, that limit overrides the traffic percentage that you set.
         * </p>
         * <p>
         * To learn more, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html">Choosing an
         * application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor section of the
         * <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param trafficPercentageToMonitor
         *        The percentage of the internet-facing traffic for your application that you want to monitor with this
         *        monitor. If you set a city-networks maximum, that limit overrides the traffic percentage that you
         *        set.</p>
         *        <p>
         *        To learn more, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html"
         *        >Choosing an application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor
         *        section of the <i>CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trafficPercentageToMonitor(Integer trafficPercentageToMonitor);

        /**
         * <p>
         * The list of health score thresholds. A threshold percentage for health scores, along with other configuration
         * information, determines when Internet Monitor creates a health event when there's an internet issue that
         * affects your application end users.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
         * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param healthEventsConfig
         *        The list of health score thresholds. A threshold percentage for health scores, along with other
         *        configuration information, determines when Internet Monitor creates a health event when there's an
         *        internet issue that affects your application end users.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
         *        > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthEventsConfig(HealthEventsConfig healthEventsConfig);

        /**
         * <p>
         * The list of health score thresholds. A threshold percentage for health scores, along with other configuration
         * information, determines when Internet Monitor creates a health event when there's an internet issue that
         * affects your application end users.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
         * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link HealthEventsConfig.Builder} avoiding the
         * need to create one manually via {@link HealthEventsConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HealthEventsConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #healthEventsConfig(HealthEventsConfig)}.
         * 
         * @param healthEventsConfig
         *        a consumer that will call methods on {@link HealthEventsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #healthEventsConfig(HealthEventsConfig)
         */
        default Builder healthEventsConfig(Consumer<HealthEventsConfig.Builder> healthEventsConfig) {
            return healthEventsConfig(HealthEventsConfig.builder().applyMutation(healthEventsConfig).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends InternetMonitorRequest.BuilderImpl implements Builder {
        private String monitorName;

        private List<String> resourcesToAdd = DefaultSdkAutoConstructList.getInstance();

        private List<String> resourcesToRemove = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String clientToken;

        private Integer maxCityNetworksToMonitor;

        private InternetMeasurementsLogDelivery internetMeasurementsLogDelivery;

        private Integer trafficPercentageToMonitor;

        private HealthEventsConfig healthEventsConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateMonitorRequest model) {
            super(model);
            monitorName(model.monitorName);
            resourcesToAdd(model.resourcesToAdd);
            resourcesToRemove(model.resourcesToRemove);
            status(model.status);
            clientToken(model.clientToken);
            maxCityNetworksToMonitor(model.maxCityNetworksToMonitor);
            internetMeasurementsLogDelivery(model.internetMeasurementsLogDelivery);
            trafficPercentageToMonitor(model.trafficPercentageToMonitor);
            healthEventsConfig(model.healthEventsConfig);
        }

        public final String getMonitorName() {
            return monitorName;
        }

        public final void setMonitorName(String monitorName) {
            this.monitorName = monitorName;
        }

        @Override
        public final Builder monitorName(String monitorName) {
            this.monitorName = monitorName;
            return this;
        }

        public final Collection<String> getResourcesToAdd() {
            if (resourcesToAdd instanceof SdkAutoConstructList) {
                return null;
            }
            return resourcesToAdd;
        }

        public final void setResourcesToAdd(Collection<String> resourcesToAdd) {
            this.resourcesToAdd = SetOfARNsCopier.copy(resourcesToAdd);
        }

        @Override
        public final Builder resourcesToAdd(Collection<String> resourcesToAdd) {
            this.resourcesToAdd = SetOfARNsCopier.copy(resourcesToAdd);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourcesToAdd(String... resourcesToAdd) {
            resourcesToAdd(Arrays.asList(resourcesToAdd));
            return this;
        }

        public final Collection<String> getResourcesToRemove() {
            if (resourcesToRemove instanceof SdkAutoConstructList) {
                return null;
            }
            return resourcesToRemove;
        }

        public final void setResourcesToRemove(Collection<String> resourcesToRemove) {
            this.resourcesToRemove = SetOfARNsCopier.copy(resourcesToRemove);
        }

        @Override
        public final Builder resourcesToRemove(Collection<String> resourcesToRemove) {
            this.resourcesToRemove = SetOfARNsCopier.copy(resourcesToRemove);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourcesToRemove(String... resourcesToRemove) {
            resourcesToRemove(Arrays.asList(resourcesToRemove));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MonitorConfigState status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final Integer getMaxCityNetworksToMonitor() {
            return maxCityNetworksToMonitor;
        }

        public final void setMaxCityNetworksToMonitor(Integer maxCityNetworksToMonitor) {
            this.maxCityNetworksToMonitor = maxCityNetworksToMonitor;
        }

        @Override
        public final Builder maxCityNetworksToMonitor(Integer maxCityNetworksToMonitor) {
            this.maxCityNetworksToMonitor = maxCityNetworksToMonitor;
            return this;
        }

        public final InternetMeasurementsLogDelivery.Builder getInternetMeasurementsLogDelivery() {
            return internetMeasurementsLogDelivery != null ? internetMeasurementsLogDelivery.toBuilder() : null;
        }

        public final void setInternetMeasurementsLogDelivery(
                InternetMeasurementsLogDelivery.BuilderImpl internetMeasurementsLogDelivery) {
            this.internetMeasurementsLogDelivery = internetMeasurementsLogDelivery != null ? internetMeasurementsLogDelivery
                    .build() : null;
        }

        @Override
        public final Builder internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery internetMeasurementsLogDelivery) {
            this.internetMeasurementsLogDelivery = internetMeasurementsLogDelivery;
            return this;
        }

        public final Integer getTrafficPercentageToMonitor() {
            return trafficPercentageToMonitor;
        }

        public final void setTrafficPercentageToMonitor(Integer trafficPercentageToMonitor) {
            this.trafficPercentageToMonitor = trafficPercentageToMonitor;
        }

        @Override
        public final Builder trafficPercentageToMonitor(Integer trafficPercentageToMonitor) {
            this.trafficPercentageToMonitor = trafficPercentageToMonitor;
            return this;
        }

        public final HealthEventsConfig.Builder getHealthEventsConfig() {
            return healthEventsConfig != null ? healthEventsConfig.toBuilder() : null;
        }

        public final void setHealthEventsConfig(HealthEventsConfig.BuilderImpl healthEventsConfig) {
            this.healthEventsConfig = healthEventsConfig != null ? healthEventsConfig.build() : null;
        }

        @Override
        public final Builder healthEventsConfig(HealthEventsConfig healthEventsConfig) {
            this.healthEventsConfig = healthEventsConfig;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateMonitorRequest build() {
            return new UpdateMonitorRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
