/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type with the configuration information that determines the threshold and other conditions for when
 * Internet Monitor creates a health event for a local performance or availability issue, when scores cross a threshold
 * for one or more city-networks.
 * </p>
 * <p>
 * Defines the percentages, for performance scores or availability scores, that are the local thresholds for when Amazon
 * CloudWatch Internet Monitor creates a health event. Also defines whether a local threshold is enabled or disabled,
 * and the minimum percentage of overall traffic that must be impacted by an issue before Internet Monitor creates an
 * event when a threshold is crossed for a local health score.
 * </p>
 * <p>
 * For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
 * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LocalHealthEventsConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<LocalHealthEventsConfig.Builder, LocalHealthEventsConfig> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(LocalHealthEventsConfig::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Double> HEALTH_SCORE_THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("HealthScoreThreshold").getter(getter(LocalHealthEventsConfig::healthScoreThreshold))
            .setter(setter(Builder::healthScoreThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthScoreThreshold").build())
            .build();

    private static final SdkField<Double> MIN_TRAFFIC_IMPACT_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MinTrafficImpact").getter(getter(LocalHealthEventsConfig::minTrafficImpact))
            .setter(setter(Builder::minTrafficImpact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinTrafficImpact").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            HEALTH_SCORE_THRESHOLD_FIELD, MIN_TRAFFIC_IMPACT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String status;

    private final Double healthScoreThreshold;

    private final Double minTrafficImpact;

    private LocalHealthEventsConfig(BuilderImpl builder) {
        this.status = builder.status;
        this.healthScoreThreshold = builder.healthScoreThreshold;
        this.minTrafficImpact = builder.minTrafficImpact;
    }

    /**
     * <p>
     * The status of whether Internet Monitor creates a health event based on a threshold percentage set for a local
     * health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link LocalHealthEventsConfigStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of whether Internet Monitor creates a health event based on a threshold percentage set for a
     *         local health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
     * @see LocalHealthEventsConfigStatus
     */
    public final LocalHealthEventsConfigStatus status() {
        return LocalHealthEventsConfigStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of whether Internet Monitor creates a health event based on a threshold percentage set for a local
     * health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link LocalHealthEventsConfigStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of whether Internet Monitor creates a health event based on a threshold percentage set for a
     *         local health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
     * @see LocalHealthEventsConfigStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The health event threshold percentage set for a local health score.
     * </p>
     * 
     * @return The health event threshold percentage set for a local health score.
     */
    public final Double healthScoreThreshold() {
        return healthScoreThreshold;
    }

    /**
     * <p>
     * The minimum percentage of overall traffic for an application that must be impacted by an issue before Internet
     * Monitor creates an event when a threshold is crossed for a local health score.
     * </p>
     * 
     * @return The minimum percentage of overall traffic for an application that must be impacted by an issue before
     *         Internet Monitor creates an event when a threshold is crossed for a local health score.
     */
    public final Double minTrafficImpact() {
        return minTrafficImpact;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(healthScoreThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(minTrafficImpact());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LocalHealthEventsConfig)) {
            return false;
        }
        LocalHealthEventsConfig other = (LocalHealthEventsConfig) obj;
        return Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(healthScoreThreshold(), other.healthScoreThreshold())
                && Objects.equals(minTrafficImpact(), other.minTrafficImpact());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LocalHealthEventsConfig").add("Status", statusAsString())
                .add("HealthScoreThreshold", healthScoreThreshold()).add("MinTrafficImpact", minTrafficImpact()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "HealthScoreThreshold":
            return Optional.ofNullable(clazz.cast(healthScoreThreshold()));
        case "MinTrafficImpact":
            return Optional.ofNullable(clazz.cast(minTrafficImpact()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LocalHealthEventsConfig, T> g) {
        return obj -> g.apply((LocalHealthEventsConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LocalHealthEventsConfig> {
        /**
         * <p>
         * The status of whether Internet Monitor creates a health event based on a threshold percentage set for a local
         * health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
         * </p>
         * 
         * @param status
         *        The status of whether Internet Monitor creates a health event based on a threshold percentage set for
         *        a local health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
         * @see LocalHealthEventsConfigStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalHealthEventsConfigStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of whether Internet Monitor creates a health event based on a threshold percentage set for a local
         * health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
         * </p>
         * 
         * @param status
         *        The status of whether Internet Monitor creates a health event based on a threshold percentage set for
         *        a local health score. The status can be <code>ENABLED</code> or <code>DISABLED</code>.
         * @see LocalHealthEventsConfigStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalHealthEventsConfigStatus
         */
        Builder status(LocalHealthEventsConfigStatus status);

        /**
         * <p>
         * The health event threshold percentage set for a local health score.
         * </p>
         * 
         * @param healthScoreThreshold
         *        The health event threshold percentage set for a local health score.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthScoreThreshold(Double healthScoreThreshold);

        /**
         * <p>
         * The minimum percentage of overall traffic for an application that must be impacted by an issue before
         * Internet Monitor creates an event when a threshold is crossed for a local health score.
         * </p>
         * 
         * @param minTrafficImpact
         *        The minimum percentage of overall traffic for an application that must be impacted by an issue before
         *        Internet Monitor creates an event when a threshold is crossed for a local health score.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minTrafficImpact(Double minTrafficImpact);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private Double healthScoreThreshold;

        private Double minTrafficImpact;

        private BuilderImpl() {
        }

        private BuilderImpl(LocalHealthEventsConfig model) {
            status(model.status);
            healthScoreThreshold(model.healthScoreThreshold);
            minTrafficImpact(model.minTrafficImpact);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(LocalHealthEventsConfigStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Double getHealthScoreThreshold() {
            return healthScoreThreshold;
        }

        public final void setHealthScoreThreshold(Double healthScoreThreshold) {
            this.healthScoreThreshold = healthScoreThreshold;
        }

        @Override
        public final Builder healthScoreThreshold(Double healthScoreThreshold) {
            this.healthScoreThreshold = healthScoreThreshold;
            return this;
        }

        public final Double getMinTrafficImpact() {
            return minTrafficImpact;
        }

        public final void setMinTrafficImpact(Double minTrafficImpact) {
            this.minTrafficImpact = minTrafficImpact;
        }

        @Override
        public final Builder minTrafficImpact(Double minTrafficImpact) {
            this.minTrafficImpact = minTrafficImpact;
            return this;
        }

        @Override
        public LocalHealthEventsConfig build() {
            return new LocalHealthEventsConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
