/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a location impacted by a health event in Amazon CloudWatch Internet Monitor.
 * </p>
 * <p>
 * Geographic regions are hierarchically categorized into country, subdivision, metro and city geographic granularities.
 * The geographic region is identified based on the IP address used at the client locations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImpactedLocation implements SdkPojo, Serializable,
        ToCopyableBuilder<ImpactedLocation.Builder, ImpactedLocation> {
    private static final SdkField<String> AS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ASName")
            .getter(getter(ImpactedLocation::asName)).setter(setter(Builder::asName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ASName").build()).build();

    private static final SdkField<Long> AS_NUMBER_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("ASNumber")
            .getter(getter(ImpactedLocation::asNumber)).setter(setter(Builder::asNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ASNumber").build()).build();

    private static final SdkField<String> COUNTRY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Country")
            .getter(getter(ImpactedLocation::country)).setter(setter(Builder::country))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Country").build()).build();

    private static final SdkField<String> SUBDIVISION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdivision").getter(getter(ImpactedLocation::subdivision)).setter(setter(Builder::subdivision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdivision").build()).build();

    private static final SdkField<String> METRO_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Metro")
            .getter(getter(ImpactedLocation::metro)).setter(setter(Builder::metro))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metro").build()).build();

    private static final SdkField<String> CITY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("City")
            .getter(getter(ImpactedLocation::city)).setter(setter(Builder::city))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("City").build()).build();

    private static final SdkField<Double> LATITUDE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Latitude").getter(getter(ImpactedLocation::latitude)).setter(setter(Builder::latitude))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Latitude").build()).build();

    private static final SdkField<Double> LONGITUDE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Longitude").getter(getter(ImpactedLocation::longitude)).setter(setter(Builder::longitude))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Longitude").build()).build();

    private static final SdkField<String> COUNTRY_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CountryCode").getter(getter(ImpactedLocation::countryCode)).setter(setter(Builder::countryCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CountryCode").build()).build();

    private static final SdkField<String> SUBDIVISION_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubdivisionCode").getter(getter(ImpactedLocation::subdivisionCode))
            .setter(setter(Builder::subdivisionCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubdivisionCode").build()).build();

    private static final SdkField<String> SERVICE_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceLocation").getter(getter(ImpactedLocation::serviceLocation))
            .setter(setter(Builder::serviceLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceLocation").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ImpactedLocation::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<NetworkImpairment> CAUSED_BY_FIELD = SdkField
            .<NetworkImpairment> builder(MarshallingType.SDK_POJO).memberName("CausedBy")
            .getter(getter(ImpactedLocation::causedBy)).setter(setter(Builder::causedBy)).constructor(NetworkImpairment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CausedBy").build()).build();

    private static final SdkField<InternetHealth> INTERNET_HEALTH_FIELD = SdkField
            .<InternetHealth> builder(MarshallingType.SDK_POJO).memberName("InternetHealth")
            .getter(getter(ImpactedLocation::internetHealth)).setter(setter(Builder::internetHealth))
            .constructor(InternetHealth::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InternetHealth").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AS_NAME_FIELD,
            AS_NUMBER_FIELD, COUNTRY_FIELD, SUBDIVISION_FIELD, METRO_FIELD, CITY_FIELD, LATITUDE_FIELD, LONGITUDE_FIELD,
            COUNTRY_CODE_FIELD, SUBDIVISION_CODE_FIELD, SERVICE_LOCATION_FIELD, STATUS_FIELD, CAUSED_BY_FIELD,
            INTERNET_HEALTH_FIELD));

    private static final long serialVersionUID = 1L;

    private final String asName;

    private final Long asNumber;

    private final String country;

    private final String subdivision;

    private final String metro;

    private final String city;

    private final Double latitude;

    private final Double longitude;

    private final String countryCode;

    private final String subdivisionCode;

    private final String serviceLocation;

    private final String status;

    private final NetworkImpairment causedBy;

    private final InternetHealth internetHealth;

    private ImpactedLocation(BuilderImpl builder) {
        this.asName = builder.asName;
        this.asNumber = builder.asNumber;
        this.country = builder.country;
        this.subdivision = builder.subdivision;
        this.metro = builder.metro;
        this.city = builder.city;
        this.latitude = builder.latitude;
        this.longitude = builder.longitude;
        this.countryCode = builder.countryCode;
        this.subdivisionCode = builder.subdivisionCode;
        this.serviceLocation = builder.serviceLocation;
        this.status = builder.status;
        this.causedBy = builder.causedBy;
        this.internetHealth = builder.internetHealth;
    }

    /**
     * <p>
     * The name of the network at an impacted location.
     * </p>
     * 
     * @return The name of the network at an impacted location.
     */
    public final String asName() {
        return asName;
    }

    /**
     * <p>
     * The Autonomous System Number (ASN) of the network at an impacted location.
     * </p>
     * 
     * @return The Autonomous System Number (ASN) of the network at an impacted location.
     */
    public final Long asNumber() {
        return asNumber;
    }

    /**
     * <p>
     * The name of the country where the health event is located.
     * </p>
     * 
     * @return The name of the country where the health event is located.
     */
    public final String country() {
        return country;
    }

    /**
     * <p>
     * The subdivision location where the health event is located. The subdivision usually maps to states in most
     * countries (including the United States). For United Kingdom, it maps to a country (England, Scotland, Wales) or
     * province (Northern Ireland).
     * </p>
     * 
     * @return The subdivision location where the health event is located. The subdivision usually maps to states in
     *         most countries (including the United States). For United Kingdom, it maps to a country (England,
     *         Scotland, Wales) or province (Northern Ireland).
     */
    public final String subdivision() {
        return subdivision;
    }

    /**
     * <p>
     * The metro area where the health event is located.
     * </p>
     * <p>
     * Metro indicates a metropolitan region in the United States, such as the region around New York City. In non-US
     * countries, this is a second-level subdivision. For example, in the United Kingdom, it could be a county, a London
     * borough, a unitary authority, council area, and so on.
     * </p>
     * 
     * @return The metro area where the health event is located.</p>
     *         <p>
     *         Metro indicates a metropolitan region in the United States, such as the region around New York City. In
     *         non-US countries, this is a second-level subdivision. For example, in the United Kingdom, it could be a
     *         county, a London borough, a unitary authority, council area, and so on.
     */
    public final String metro() {
        return metro;
    }

    /**
     * <p>
     * The name of the city where the health event is located.
     * </p>
     * 
     * @return The name of the city where the health event is located.
     */
    public final String city() {
        return city;
    }

    /**
     * <p>
     * The latitude where the health event is located.
     * </p>
     * 
     * @return The latitude where the health event is located.
     */
    public final Double latitude() {
        return latitude;
    }

    /**
     * <p>
     * The longitude where the health event is located.
     * </p>
     * 
     * @return The longitude where the health event is located.
     */
    public final Double longitude() {
        return longitude;
    }

    /**
     * <p>
     * The country code where the health event is located. The ISO 3166-2 codes for the country is provided, when
     * available.
     * </p>
     * 
     * @return The country code where the health event is located. The ISO 3166-2 codes for the country is provided,
     *         when available.
     */
    public final String countryCode() {
        return countryCode;
    }

    /**
     * <p>
     * The subdivision code where the health event is located. The ISO 3166-2 codes for country subdivisions is
     * provided, when available.
     * </p>
     * 
     * @return The subdivision code where the health event is located. The ISO 3166-2 codes for country subdivisions is
     *         provided, when available.
     */
    public final String subdivisionCode() {
        return subdivisionCode;
    }

    /**
     * <p>
     * The service location where the health event is located.
     * </p>
     * 
     * @return The service location where the health event is located.
     */
    public final String serviceLocation() {
        return serviceLocation;
    }

    /**
     * <p>
     * The status of the health event at an impacted location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link HealthEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the health event at an impacted location.
     * @see HealthEventStatus
     */
    public final HealthEventStatus status() {
        return HealthEventStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the health event at an impacted location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link HealthEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the health event at an impacted location.
     * @see HealthEventStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The cause of the impairment. There are two types of network impairments: Amazon Web Services network issues or
     * internet issues. Internet issues are typically a problem with a network provider, like an internet service
     * provider (ISP).
     * </p>
     * 
     * @return The cause of the impairment. There are two types of network impairments: Amazon Web Services network
     *         issues or internet issues. Internet issues are typically a problem with a network provider, like an
     *         internet service provider (ISP).
     */
    public final NetworkImpairment causedBy() {
        return causedBy;
    }

    /**
     * <p>
     * The calculated health at a specific location.
     * </p>
     * 
     * @return The calculated health at a specific location.
     */
    public final InternetHealth internetHealth() {
        return internetHealth;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(asName());
        hashCode = 31 * hashCode + Objects.hashCode(asNumber());
        hashCode = 31 * hashCode + Objects.hashCode(country());
        hashCode = 31 * hashCode + Objects.hashCode(subdivision());
        hashCode = 31 * hashCode + Objects.hashCode(metro());
        hashCode = 31 * hashCode + Objects.hashCode(city());
        hashCode = 31 * hashCode + Objects.hashCode(latitude());
        hashCode = 31 * hashCode + Objects.hashCode(longitude());
        hashCode = 31 * hashCode + Objects.hashCode(countryCode());
        hashCode = 31 * hashCode + Objects.hashCode(subdivisionCode());
        hashCode = 31 * hashCode + Objects.hashCode(serviceLocation());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(causedBy());
        hashCode = 31 * hashCode + Objects.hashCode(internetHealth());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImpactedLocation)) {
            return false;
        }
        ImpactedLocation other = (ImpactedLocation) obj;
        return Objects.equals(asName(), other.asName()) && Objects.equals(asNumber(), other.asNumber())
                && Objects.equals(country(), other.country()) && Objects.equals(subdivision(), other.subdivision())
                && Objects.equals(metro(), other.metro()) && Objects.equals(city(), other.city())
                && Objects.equals(latitude(), other.latitude()) && Objects.equals(longitude(), other.longitude())
                && Objects.equals(countryCode(), other.countryCode())
                && Objects.equals(subdivisionCode(), other.subdivisionCode())
                && Objects.equals(serviceLocation(), other.serviceLocation())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(causedBy(), other.causedBy())
                && Objects.equals(internetHealth(), other.internetHealth());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImpactedLocation").add("ASName", asName()).add("ASNumber", asNumber()).add("Country", country())
                .add("Subdivision", subdivision()).add("Metro", metro()).add("City", city()).add("Latitude", latitude())
                .add("Longitude", longitude()).add("CountryCode", countryCode()).add("SubdivisionCode", subdivisionCode())
                .add("ServiceLocation", serviceLocation()).add("Status", statusAsString()).add("CausedBy", causedBy())
                .add("InternetHealth", internetHealth()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ASName":
            return Optional.ofNullable(clazz.cast(asName()));
        case "ASNumber":
            return Optional.ofNullable(clazz.cast(asNumber()));
        case "Country":
            return Optional.ofNullable(clazz.cast(country()));
        case "Subdivision":
            return Optional.ofNullable(clazz.cast(subdivision()));
        case "Metro":
            return Optional.ofNullable(clazz.cast(metro()));
        case "City":
            return Optional.ofNullable(clazz.cast(city()));
        case "Latitude":
            return Optional.ofNullable(clazz.cast(latitude()));
        case "Longitude":
            return Optional.ofNullable(clazz.cast(longitude()));
        case "CountryCode":
            return Optional.ofNullable(clazz.cast(countryCode()));
        case "SubdivisionCode":
            return Optional.ofNullable(clazz.cast(subdivisionCode()));
        case "ServiceLocation":
            return Optional.ofNullable(clazz.cast(serviceLocation()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CausedBy":
            return Optional.ofNullable(clazz.cast(causedBy()));
        case "InternetHealth":
            return Optional.ofNullable(clazz.cast(internetHealth()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImpactedLocation, T> g) {
        return obj -> g.apply((ImpactedLocation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImpactedLocation> {
        /**
         * <p>
         * The name of the network at an impacted location.
         * </p>
         * 
         * @param asName
         *        The name of the network at an impacted location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asName(String asName);

        /**
         * <p>
         * The Autonomous System Number (ASN) of the network at an impacted location.
         * </p>
         * 
         * @param asNumber
         *        The Autonomous System Number (ASN) of the network at an impacted location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asNumber(Long asNumber);

        /**
         * <p>
         * The name of the country where the health event is located.
         * </p>
         * 
         * @param country
         *        The name of the country where the health event is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder country(String country);

        /**
         * <p>
         * The subdivision location where the health event is located. The subdivision usually maps to states in most
         * countries (including the United States). For United Kingdom, it maps to a country (England, Scotland, Wales)
         * or province (Northern Ireland).
         * </p>
         * 
         * @param subdivision
         *        The subdivision location where the health event is located. The subdivision usually maps to states in
         *        most countries (including the United States). For United Kingdom, it maps to a country (England,
         *        Scotland, Wales) or province (Northern Ireland).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdivision(String subdivision);

        /**
         * <p>
         * The metro area where the health event is located.
         * </p>
         * <p>
         * Metro indicates a metropolitan region in the United States, such as the region around New York City. In
         * non-US countries, this is a second-level subdivision. For example, in the United Kingdom, it could be a
         * county, a London borough, a unitary authority, council area, and so on.
         * </p>
         * 
         * @param metro
         *        The metro area where the health event is located.</p>
         *        <p>
         *        Metro indicates a metropolitan region in the United States, such as the region around New York City.
         *        In non-US countries, this is a second-level subdivision. For example, in the United Kingdom, it could
         *        be a county, a London borough, a unitary authority, council area, and so on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metro(String metro);

        /**
         * <p>
         * The name of the city where the health event is located.
         * </p>
         * 
         * @param city
         *        The name of the city where the health event is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder city(String city);

        /**
         * <p>
         * The latitude where the health event is located.
         * </p>
         * 
         * @param latitude
         *        The latitude where the health event is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latitude(Double latitude);

        /**
         * <p>
         * The longitude where the health event is located.
         * </p>
         * 
         * @param longitude
         *        The longitude where the health event is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longitude(Double longitude);

        /**
         * <p>
         * The country code where the health event is located. The ISO 3166-2 codes for the country is provided, when
         * available.
         * </p>
         * 
         * @param countryCode
         *        The country code where the health event is located. The ISO 3166-2 codes for the country is provided,
         *        when available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder countryCode(String countryCode);

        /**
         * <p>
         * The subdivision code where the health event is located. The ISO 3166-2 codes for country subdivisions is
         * provided, when available.
         * </p>
         * 
         * @param subdivisionCode
         *        The subdivision code where the health event is located. The ISO 3166-2 codes for country subdivisions
         *        is provided, when available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdivisionCode(String subdivisionCode);

        /**
         * <p>
         * The service location where the health event is located.
         * </p>
         * 
         * @param serviceLocation
         *        The service location where the health event is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceLocation(String serviceLocation);

        /**
         * <p>
         * The status of the health event at an impacted location.
         * </p>
         * 
         * @param status
         *        The status of the health event at an impacted location.
         * @see HealthEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthEventStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the health event at an impacted location.
         * </p>
         * 
         * @param status
         *        The status of the health event at an impacted location.
         * @see HealthEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthEventStatus
         */
        Builder status(HealthEventStatus status);

        /**
         * <p>
         * The cause of the impairment. There are two types of network impairments: Amazon Web Services network issues
         * or internet issues. Internet issues are typically a problem with a network provider, like an internet service
         * provider (ISP).
         * </p>
         * 
         * @param causedBy
         *        The cause of the impairment. There are two types of network impairments: Amazon Web Services network
         *        issues or internet issues. Internet issues are typically a problem with a network provider, like an
         *        internet service provider (ISP).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder causedBy(NetworkImpairment causedBy);

        /**
         * <p>
         * The cause of the impairment. There are two types of network impairments: Amazon Web Services network issues
         * or internet issues. Internet issues are typically a problem with a network provider, like an internet service
         * provider (ISP).
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkImpairment.Builder} avoiding the
         * need to create one manually via {@link NetworkImpairment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkImpairment.Builder#build()} is called immediately and its
         * result is passed to {@link #causedBy(NetworkImpairment)}.
         * 
         * @param causedBy
         *        a consumer that will call methods on {@link NetworkImpairment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #causedBy(NetworkImpairment)
         */
        default Builder causedBy(Consumer<NetworkImpairment.Builder> causedBy) {
            return causedBy(NetworkImpairment.builder().applyMutation(causedBy).build());
        }

        /**
         * <p>
         * The calculated health at a specific location.
         * </p>
         * 
         * @param internetHealth
         *        The calculated health at a specific location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder internetHealth(InternetHealth internetHealth);

        /**
         * <p>
         * The calculated health at a specific location.
         * </p>
         * This is a convenience method that creates an instance of the {@link InternetHealth.Builder} avoiding the need
         * to create one manually via {@link InternetHealth#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InternetHealth.Builder#build()} is called immediately and its
         * result is passed to {@link #internetHealth(InternetHealth)}.
         * 
         * @param internetHealth
         *        a consumer that will call methods on {@link InternetHealth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #internetHealth(InternetHealth)
         */
        default Builder internetHealth(Consumer<InternetHealth.Builder> internetHealth) {
            return internetHealth(InternetHealth.builder().applyMutation(internetHealth).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String asName;

        private Long asNumber;

        private String country;

        private String subdivision;

        private String metro;

        private String city;

        private Double latitude;

        private Double longitude;

        private String countryCode;

        private String subdivisionCode;

        private String serviceLocation;

        private String status;

        private NetworkImpairment causedBy;

        private InternetHealth internetHealth;

        private BuilderImpl() {
        }

        private BuilderImpl(ImpactedLocation model) {
            asName(model.asName);
            asNumber(model.asNumber);
            country(model.country);
            subdivision(model.subdivision);
            metro(model.metro);
            city(model.city);
            latitude(model.latitude);
            longitude(model.longitude);
            countryCode(model.countryCode);
            subdivisionCode(model.subdivisionCode);
            serviceLocation(model.serviceLocation);
            status(model.status);
            causedBy(model.causedBy);
            internetHealth(model.internetHealth);
        }

        public final String getAsName() {
            return asName;
        }

        public final void setAsName(String asName) {
            this.asName = asName;
        }

        @Override
        public final Builder asName(String asName) {
            this.asName = asName;
            return this;
        }

        public final Long getAsNumber() {
            return asNumber;
        }

        public final void setAsNumber(Long asNumber) {
            this.asNumber = asNumber;
        }

        @Override
        public final Builder asNumber(Long asNumber) {
            this.asNumber = asNumber;
            return this;
        }

        public final String getCountry() {
            return country;
        }

        public final void setCountry(String country) {
            this.country = country;
        }

        @Override
        public final Builder country(String country) {
            this.country = country;
            return this;
        }

        public final String getSubdivision() {
            return subdivision;
        }

        public final void setSubdivision(String subdivision) {
            this.subdivision = subdivision;
        }

        @Override
        public final Builder subdivision(String subdivision) {
            this.subdivision = subdivision;
            return this;
        }

        public final String getMetro() {
            return metro;
        }

        public final void setMetro(String metro) {
            this.metro = metro;
        }

        @Override
        public final Builder metro(String metro) {
            this.metro = metro;
            return this;
        }

        public final String getCity() {
            return city;
        }

        public final void setCity(String city) {
            this.city = city;
        }

        @Override
        public final Builder city(String city) {
            this.city = city;
            return this;
        }

        public final Double getLatitude() {
            return latitude;
        }

        public final void setLatitude(Double latitude) {
            this.latitude = latitude;
        }

        @Override
        public final Builder latitude(Double latitude) {
            this.latitude = latitude;
            return this;
        }

        public final Double getLongitude() {
            return longitude;
        }

        public final void setLongitude(Double longitude) {
            this.longitude = longitude;
        }

        @Override
        public final Builder longitude(Double longitude) {
            this.longitude = longitude;
            return this;
        }

        public final String getCountryCode() {
            return countryCode;
        }

        public final void setCountryCode(String countryCode) {
            this.countryCode = countryCode;
        }

        @Override
        public final Builder countryCode(String countryCode) {
            this.countryCode = countryCode;
            return this;
        }

        public final String getSubdivisionCode() {
            return subdivisionCode;
        }

        public final void setSubdivisionCode(String subdivisionCode) {
            this.subdivisionCode = subdivisionCode;
        }

        @Override
        public final Builder subdivisionCode(String subdivisionCode) {
            this.subdivisionCode = subdivisionCode;
            return this;
        }

        public final String getServiceLocation() {
            return serviceLocation;
        }

        public final void setServiceLocation(String serviceLocation) {
            this.serviceLocation = serviceLocation;
        }

        @Override
        public final Builder serviceLocation(String serviceLocation) {
            this.serviceLocation = serviceLocation;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(HealthEventStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final NetworkImpairment.Builder getCausedBy() {
            return causedBy != null ? causedBy.toBuilder() : null;
        }

        public final void setCausedBy(NetworkImpairment.BuilderImpl causedBy) {
            this.causedBy = causedBy != null ? causedBy.build() : null;
        }

        @Override
        public final Builder causedBy(NetworkImpairment causedBy) {
            this.causedBy = causedBy;
            return this;
        }

        public final InternetHealth.Builder getInternetHealth() {
            return internetHealth != null ? internetHealth.toBuilder() : null;
        }

        public final void setInternetHealth(InternetHealth.BuilderImpl internetHealth) {
            this.internetHealth = internetHealth != null ? internetHealth.build() : null;
        }

        @Override
        public final Builder internetHealth(InternetHealth internetHealth) {
            this.internetHealth = internetHealth;
            return this;
        }

        @Override
        public ImpactedLocation build() {
            return new ImpactedLocation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
