/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.identitystore;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.identitystore.model.AccessDeniedException;
import software.amazon.awssdk.services.identitystore.model.ConflictException;
import software.amazon.awssdk.services.identitystore.model.CreateGroupMembershipRequest;
import software.amazon.awssdk.services.identitystore.model.CreateGroupMembershipResponse;
import software.amazon.awssdk.services.identitystore.model.CreateGroupRequest;
import software.amazon.awssdk.services.identitystore.model.CreateGroupResponse;
import software.amazon.awssdk.services.identitystore.model.CreateUserRequest;
import software.amazon.awssdk.services.identitystore.model.CreateUserResponse;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupMembershipRequest;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupMembershipResponse;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupRequest;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupResponse;
import software.amazon.awssdk.services.identitystore.model.DeleteUserRequest;
import software.amazon.awssdk.services.identitystore.model.DeleteUserResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupMembershipRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupMembershipResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeUserRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeUserResponse;
import software.amazon.awssdk.services.identitystore.model.GetGroupIdRequest;
import software.amazon.awssdk.services.identitystore.model.GetGroupIdResponse;
import software.amazon.awssdk.services.identitystore.model.GetGroupMembershipIdRequest;
import software.amazon.awssdk.services.identitystore.model.GetGroupMembershipIdResponse;
import software.amazon.awssdk.services.identitystore.model.GetUserIdRequest;
import software.amazon.awssdk.services.identitystore.model.GetUserIdResponse;
import software.amazon.awssdk.services.identitystore.model.IdentitystoreException;
import software.amazon.awssdk.services.identitystore.model.IdentitystoreRequest;
import software.amazon.awssdk.services.identitystore.model.InternalServerException;
import software.amazon.awssdk.services.identitystore.model.IsMemberInGroupsRequest;
import software.amazon.awssdk.services.identitystore.model.IsMemberInGroupsResponse;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberResponse;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsResponse;
import software.amazon.awssdk.services.identitystore.model.ListGroupsRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupsResponse;
import software.amazon.awssdk.services.identitystore.model.ListUsersRequest;
import software.amazon.awssdk.services.identitystore.model.ListUsersResponse;
import software.amazon.awssdk.services.identitystore.model.ResourceNotFoundException;
import software.amazon.awssdk.services.identitystore.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.identitystore.model.ThrottlingException;
import software.amazon.awssdk.services.identitystore.model.UpdateGroupRequest;
import software.amazon.awssdk.services.identitystore.model.UpdateGroupResponse;
import software.amazon.awssdk.services.identitystore.model.UpdateUserRequest;
import software.amazon.awssdk.services.identitystore.model.UpdateUserResponse;
import software.amazon.awssdk.services.identitystore.model.ValidationException;
import software.amazon.awssdk.services.identitystore.paginators.ListGroupMembershipsForMemberPublisher;
import software.amazon.awssdk.services.identitystore.paginators.ListGroupMembershipsPublisher;
import software.amazon.awssdk.services.identitystore.paginators.ListGroupsPublisher;
import software.amazon.awssdk.services.identitystore.paginators.ListUsersPublisher;
import software.amazon.awssdk.services.identitystore.transform.CreateGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DeleteGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.GetGroupIdRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.GetGroupMembershipIdRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.GetUserIdRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.IsMemberInGroupsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupMembershipsForMemberRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupMembershipsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IdentitystoreAsyncClient}.
 *
 * @see IdentitystoreAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIdentitystoreAsyncClient implements IdentitystoreAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIdentitystoreAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIdentitystoreAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a group within the specified identity store.
     * </p>
     *
     * @param createGroupRequest
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>ServiceQuotaExceededException The request would cause the number of users or groups in the identity
     *         store to exceed the maximum allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupResponse> createGroup(CreateGroupRequest createGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                            .withOperationName("CreateGroup").withMarshaller(new CreateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGroupRequest));
            CompletableFuture<CreateGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a relationship between a member and a group. The following identifiers must be specified:
     * <code>GroupId</code>, <code>IdentityStoreId</code>, and <code>MemberId</code>.
     * </p>
     *
     * @param createGroupMembershipRequest
     * @return A Java Future containing the result of the CreateGroupMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>ServiceQuotaExceededException The request would cause the number of users or groups in the identity
     *         store to exceed the maximum allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.CreateGroupMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/CreateGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupMembershipResponse> createGroupMembership(
            CreateGroupMembershipRequest createGroupMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroupMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGroupMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGroupMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGroupMembershipRequest, CreateGroupMembershipResponse>()
                            .withOperationName("CreateGroupMembership")
                            .withMarshaller(new CreateGroupMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGroupMembershipRequest));
            CompletableFuture<CreateGroupMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new user within the specified identity store.
     * </p>
     *
     * @param createUserRequest
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>ServiceQuotaExceededException The request would cause the number of users or groups in the identity
     *         store to exceed the maximum allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createUserRequest));
            CompletableFuture<CreateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a group within an identity store given <code>GroupId</code>.
     * </p>
     *
     * @param deleteGroupRequest
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupResponse> deleteGroup(DeleteGroupRequest deleteGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                            .withOperationName("DeleteGroup").withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGroupRequest));
            CompletableFuture<DeleteGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a membership within a group given <code>MembershipId</code>.
     * </p>
     *
     * @param deleteGroupMembershipRequest
     * @return A Java Future containing the result of the DeleteGroupMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DeleteGroupMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DeleteGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupMembershipResponse> deleteGroupMembership(
            DeleteGroupMembershipRequest deleteGroupMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroupMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGroupMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGroupMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGroupMembershipRequest, DeleteGroupMembershipResponse>()
                            .withOperationName("DeleteGroupMembership")
                            .withMarshaller(new DeleteGroupMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGroupMembershipRequest));
            CompletableFuture<DeleteGroupMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a user within an identity store given <code>UserId</code>.
     * </p>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteUserRequest));
            CompletableFuture<DeleteUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the group metadata and attributes from <code>GroupId</code> in an identity store.
     * </p>
     *
     * @param describeGroupRequest
     * @return A Java Future containing the result of the DescribeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DescribeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGroupResponse> describeGroup(DescribeGroupRequest describeGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGroupRequest, DescribeGroupResponse>()
                            .withOperationName("DescribeGroup")
                            .withMarshaller(new DescribeGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGroupRequest));
            CompletableFuture<DescribeGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves membership metadata and attributes from <code>MembershipId</code> in an identity store.
     * </p>
     *
     * @param describeGroupMembershipRequest
     * @return A Java Future containing the result of the DescribeGroupMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DescribeGroupMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGroupMembershipResponse> describeGroupMembership(
            DescribeGroupMembershipRequest describeGroupMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroupMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGroupMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGroupMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGroupMembershipRequest, DescribeGroupMembershipResponse>()
                            .withOperationName("DescribeGroupMembership")
                            .withMarshaller(new DescribeGroupMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGroupMembershipRequest));
            CompletableFuture<DescribeGroupMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the user metadata and attributes from the <code>UserId</code> in an identity store.
     * </p>
     *
     * @param describeUserRequest
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeUser" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUserResponse> describeUser(DescribeUserRequest describeUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                            .withOperationName("DescribeUser").withMarshaller(new DescribeUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeUserRequest));
            CompletableFuture<DescribeUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves <code>GroupId</code> in an identity store.
     * </p>
     *
     * @param getGroupIdRequest
     * @return A Java Future containing the result of the GetGroupId operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.GetGroupId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/GetGroupId" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupIdResponse> getGroupId(GetGroupIdRequest getGroupIdRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupId");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGroupIdResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupIdRequest, GetGroupIdResponse>().withOperationName("GetGroupId")
                            .withMarshaller(new GetGroupIdRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGroupIdRequest));
            CompletableFuture<GetGroupIdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <code>MembershipId</code> in an identity store.
     * </p>
     *
     * @param getGroupMembershipIdRequest
     * @return A Java Future containing the result of the GetGroupMembershipId operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.GetGroupMembershipId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/GetGroupMembershipId"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetGroupMembershipIdResponse> getGroupMembershipId(
            GetGroupMembershipIdRequest getGroupMembershipIdRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupMembershipIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupMembershipId");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGroupMembershipIdResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetGroupMembershipIdResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGroupMembershipIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGroupMembershipIdRequest, GetGroupMembershipIdResponse>()
                            .withOperationName("GetGroupMembershipId")
                            .withMarshaller(new GetGroupMembershipIdRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGroupMembershipIdRequest));
            CompletableFuture<GetGroupMembershipIdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <code>UserId</code> in an identity store.
     * </p>
     *
     * @param getUserIdRequest
     * @return A Java Future containing the result of the GetUserId operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.GetUserId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/GetUserId" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetUserIdResponse> getUserId(GetUserIdRequest getUserIdRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUserIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUserId");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetUserIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetUserIdResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetUserIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetUserIdRequest, GetUserIdResponse>().withOperationName("GetUserId")
                            .withMarshaller(new GetUserIdRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getUserIdRequest));
            CompletableFuture<GetUserIdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks the user's membership in all requested groups and returns if the member exists in all queried groups.
     * </p>
     *
     * @param isMemberInGroupsRequest
     * @return A Java Future containing the result of the IsMemberInGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.IsMemberInGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/IsMemberInGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<IsMemberInGroupsResponse> isMemberInGroups(IsMemberInGroupsRequest isMemberInGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, isMemberInGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IsMemberInGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<IsMemberInGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, IsMemberInGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<IsMemberInGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<IsMemberInGroupsRequest, IsMemberInGroupsResponse>()
                            .withOperationName("IsMemberInGroups")
                            .withMarshaller(new IsMemberInGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(isMemberInGroupsRequest));
            CompletableFuture<IsMemberInGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For the specified group in the specified identity store, returns the list of all <code>GroupMembership</code>
     * objects and returns results in paginated form.
     * </p>
     *
     * @param listGroupMembershipsRequest
     * @return A Java Future containing the result of the ListGroupMemberships operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroupMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroupMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupMembershipsResponse> listGroupMemberships(
            ListGroupMembershipsRequest listGroupMembershipsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupMembershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupMemberships");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGroupMembershipsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupMembershipsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupMembershipsRequest, ListGroupMembershipsResponse>()
                            .withOperationName("ListGroupMemberships")
                            .withMarshaller(new ListGroupMembershipsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGroupMembershipsRequest));
            CompletableFuture<ListGroupMembershipsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For the specified member in the specified identity store, returns the list of all <code>GroupMembership</code>
     * objects and returns results in paginated form.
     * </p>
     *
     * @param listGroupMembershipsForMemberRequest
     * @return A Java Future containing the result of the ListGroupMembershipsForMember operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroupMembershipsForMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroupMembershipsForMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupMembershipsForMemberResponse> listGroupMembershipsForMember(
            ListGroupMembershipsForMemberRequest listGroupMembershipsForMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listGroupMembershipsForMemberRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupMembershipsForMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupMembershipsForMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGroupMembershipsForMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupMembershipsForMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupMembershipsForMemberRequest, ListGroupMembershipsForMemberResponse>()
                            .withOperationName("ListGroupMembershipsForMember")
                            .withMarshaller(new ListGroupMembershipsForMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGroupMembershipsForMemberRequest));
            CompletableFuture<ListGroupMembershipsForMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For the specified member in the specified identity store, returns the list of all <code>GroupMembership</code>
     * objects and returns results in paginated form.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGroupMembershipsForMember(software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupMembershipsForMemberPublisher publisher = client.listGroupMembershipsForMemberPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupMembershipsForMemberPublisher publisher = client.listGroupMembershipsForMemberPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroupMembershipsForMember(software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGroupMembershipsForMemberRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroupMembershipsForMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroupMembershipsForMember"
     *      target="_top">AWS API Documentation</a>
     */
    public ListGroupMembershipsForMemberPublisher listGroupMembershipsForMemberPaginator(
            ListGroupMembershipsForMemberRequest listGroupMembershipsForMemberRequest) {
        return new ListGroupMembershipsForMemberPublisher(this, applyPaginatorUserAgent(listGroupMembershipsForMemberRequest));
    }

    /**
     * <p>
     * For the specified group in the specified identity store, returns the list of all <code>GroupMembership</code>
     * objects and returns results in paginated form.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGroupMemberships(software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupMembershipsPublisher publisher = client.listGroupMembershipsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupMembershipsPublisher publisher = client.listGroupMembershipsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroupMemberships(software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGroupMembershipsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroupMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroupMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    public ListGroupMembershipsPublisher listGroupMembershipsPaginator(ListGroupMembershipsRequest listGroupMembershipsRequest) {
        return new ListGroupMembershipsPublisher(this, applyPaginatorUserAgent(listGroupMembershipsRequest));
    }

    /**
     * <p>
     * Lists all groups in the identity store. Returns a paginated list of complete <code>Group</code> objects.
     * Filtering for a <code>Group</code> by the <code>DisplayName</code> attribute is deprecated. Instead, use the
     * <code>GetGroupId</code> API action.
     * </p>
     *
     * @param listGroupsRequest
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupsResponse> listGroups(ListGroupsRequest listGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>().withOperationName("ListGroups")
                            .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGroupsRequest));
            CompletableFuture<ListGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all groups in the identity store. Returns a paginated list of complete <code>Group</code> objects.
     * Filtering for a <code>Group</code> by the <code>DisplayName</code> attribute is deprecated. Instead, use the
     * <code>GetGroupId</code> API action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.identitystore.model.ListGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.identitystore.model.ListGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.identitystore.model.ListGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.identitystore.model.ListGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    public ListGroupsPublisher listGroupsPaginator(ListGroupsRequest listGroupsRequest) {
        return new ListGroupsPublisher(this, applyPaginatorUserAgent(listGroupsRequest));
    }

    /**
     * <p>
     * Lists all users in the identity store. Returns a paginated list of complete <code>User</code> objects. Filtering
     * for a <code>User</code> by the <code>UserName</code> attribute is deprecated. Instead, use the
     * <code>GetUserId</code> API action.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>().withOperationName("ListUsers")
                            .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listUsersRequest));
            CompletableFuture<ListUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all users in the identity store. Returns a paginated list of complete <code>User</code> objects. Filtering
     * for a <code>User</code> by the <code>UserName</code> attribute is deprecated. Instead, use the
     * <code>GetUserId</code> API action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.identitystore.model.ListUsersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.identitystore.model.ListUsersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.identitystore.model.ListUsersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.identitystore.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListUsersPublisher listUsersPaginator(ListUsersRequest listUsersRequest) {
        return new ListUsersPublisher(this, applyPaginatorUserAgent(listUsersRequest));
    }

    /**
     * <p>
     * For the specified group in the specified identity store, updates the group metadata and attributes.
     * </p>
     *
     * @param updateGroupRequest
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>ServiceQuotaExceededException The request would cause the number of users or groups in the identity
     *         store to exceed the maximum allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGroupResponse> updateGroup(UpdateGroupRequest updateGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                            .withOperationName("UpdateGroup").withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGroupRequest));
            CompletableFuture<UpdateGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For the specified user in the specified identity store, updates the user metadata and attributes.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>ServiceQuotaExceededException The request would cause the number of users or groups in the identity
     *         store to exceed the maximum allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateUserRequest));
            CompletableFuture<UpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IdentitystoreException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends IdentitystoreRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
