/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsResponse;
import software.amazon.awssdk.services.health.model.DescribeEventTypesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventTypesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventsResponse;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher;

/**
 * Service client for accessing AWSHealth asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <fullname>AWS Health</fullname>
 * <p>
 * The AWS Health API provides programmatic access to the AWS Health information that is presented in the <a
 * href="https://phd.aws.amazon.com/phd/home#/">AWS Personal Health Dashboard</a>. You can get information about events
 * that affect your AWS resources:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>DescribeEvents</a>: Summary information about events.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeEventDetails</a>: Detailed information about one or more events.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeAffectedEntities</a>: Information about AWS resources that are affected by one or more events.
 * </p>
 * </li>
 * </ul>
 * <p>
 * In addition, these operations provide information about event types and summary counts of events or affected
 * entities:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>DescribeEventTypes</a>: Information about the kinds of events that AWS Health tracks.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeEventAggregates</a>: A count of the number of events that meet specified criteria.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeEntityAggregates</a>: A count of the number of affected entities that meet specified criteria.
 * </p>
 * </li>
 * </ul>
 * <p>
 * The Health API requires a Business or Enterprise support plan from <a
 * href="http://aws.amazon.com/premiumsupport/">AWS Support</a>. Calling the Health API from an account that does not
 * have a Business or Enterprise support plan causes a <code>SubscriptionRequiredException</code>.
 * </p>
 * <p>
 * For authentication of requests, AWS Health uses the <a
 * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4 Signing
 * Process</a>.
 * </p>
 * <p>
 * See the <a href="https://docs.aws.amazon.com/health/latest/ug/what-is-aws-health.html">AWS Health User Guide</a> for
 * information about how to use the API.
 * </p>
 * <p>
 * <b>Service Endpoint</b>
 * </p>
 * <p>
 * The HTTP endpoint for the AWS Health API is:
 * </p>
 * <ul>
 * <li>
 * <p>
 * https://health.us-east-1.amazonaws.com
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public interface HealthAsyncClient extends SdkClient {
    String SERVICE_NAME = "health";

    /**
     * Create a {@link HealthAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static HealthAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link HealthAsyncClient}.
     */
    static HealthAsyncClientBuilder builder() {
        return new DefaultHealthAsyncClientBuilder();
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     * @return A Java Future containing the result of the DescribeAffectedEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAffectedEntitiesResponse> describeAffectedEntities(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAffectedEntitiesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeAffectedEntitiesRequest#builder()}
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeAffectedEntitiesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeAffectedEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAffectedEntitiesResponse> describeAffectedEntities(
            Consumer<DescribeAffectedEntitiesRequest.Builder> describeAffectedEntitiesRequest) {
        return describeAffectedEntities(DescribeAffectedEntitiesRequest.builder().applyMutation(describeAffectedEntitiesRequest)
                .build());
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher publisher = client.describeAffectedEntitiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher publisher = client.describeAffectedEntitiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAffectedEntitiesPublisher describeAffectedEntitiesPaginator(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher publisher = client.describeAffectedEntitiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher publisher = client.describeAffectedEntitiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAffectedEntitiesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeAffectedEntitiesRequest#builder()}
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeAffectedEntitiesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAffectedEntitiesPublisher describeAffectedEntitiesPaginator(
            Consumer<DescribeAffectedEntitiesRequest.Builder> describeAffectedEntitiesRequest) {
        return describeAffectedEntitiesPaginator(DescribeAffectedEntitiesRequest.builder()
                .applyMutation(describeAffectedEntitiesRequest).build());
    }

    /**
     * <p>
     * Returns the number of entities that are affected by each of the specified events. If no events are specified, the
     * counts of all affected entities are returned.
     * </p>
     *
     * @param describeEntityAggregatesRequest
     * @return A Java Future containing the result of the DescribeEntityAggregates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEntityAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEntityAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeEntityAggregatesResponse> describeEntityAggregates(
            DescribeEntityAggregatesRequest describeEntityAggregatesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the number of entities that are affected by each of the specified events. If no events are specified, the
     * counts of all affected entities are returned.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEntityAggregatesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeEntityAggregatesRequest#builder()}
     * </p>
     *
     * @param describeEntityAggregatesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEntityAggregatesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeEntityAggregates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEntityAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEntityAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeEntityAggregatesResponse> describeEntityAggregates(
            Consumer<DescribeEntityAggregatesRequest.Builder> describeEntityAggregatesRequest) {
        return describeEntityAggregates(DescribeEntityAggregatesRequest.builder().applyMutation(describeEntityAggregatesRequest)
                .build());
    }

    /**
     * <p>
     * Returns the number of entities that are affected by each of the specified events. If no events are specified, the
     * counts of all affected entities are returned.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeEntityAggregates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEntityAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEntityAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeEntityAggregatesResponse> describeEntityAggregates() {
        return describeEntityAggregates(DescribeEntityAggregatesRequest.builder().build());
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     *
     * @param describeEventAggregatesRequest
     * @return A Java Future containing the result of the DescribeEventAggregates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeEventAggregatesResponse> describeEventAggregates(
            DescribeEventAggregatesRequest describeEventAggregatesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEventAggregatesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeEventAggregatesRequest#builder()}
     * </p>
     *
     * @param describeEventAggregatesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEventAggregatesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeEventAggregates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeEventAggregatesResponse> describeEventAggregates(
            Consumer<DescribeEventAggregatesRequest.Builder> describeEventAggregatesRequest) {
        return describeEventAggregates(DescribeEventAggregatesRequest.builder().applyMutation(describeEventAggregatesRequest)
                .build());
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher publisher = client.describeEventAggregatesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher publisher = client.describeEventAggregatesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventAggregatesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeEventAggregatesPublisher describeEventAggregatesPaginator(
            DescribeEventAggregatesRequest describeEventAggregatesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher publisher = client.describeEventAggregatesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher publisher = client.describeEventAggregatesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEventAggregatesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeEventAggregatesRequest#builder()}
     * </p>
     *
     * @param describeEventAggregatesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEventAggregatesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeEventAggregatesPublisher describeEventAggregatesPaginator(
            Consumer<DescribeEventAggregatesRequest.Builder> describeEventAggregatesRequest) {
        return describeEventAggregatesPaginator(DescribeEventAggregatesRequest.builder()
                .applyMutation(describeEventAggregatesRequest).build());
    }

    /**
     * <p>
     * Returns detailed information about one or more specified events. Information includes standard event data
     * (region, service, etc., as returned by <a>DescribeEvents</a>), a detailed event description, and possible
     * additional metadata that depends upon the nature of the event. Affected entities are not included; to retrieve
     * those, use the <a>DescribeAffectedEntities</a> operation.
     * </p>
     * <p>
     * If a specified event cannot be retrieved, an error message is returned for that event.
     * </p>
     *
     * @param describeEventDetailsRequest
     * @return A Java Future containing the result of the DescribeEventDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventDetails" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeEventDetailsResponse> describeEventDetails(
            DescribeEventDetailsRequest describeEventDetailsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns detailed information about one or more specified events. Information includes standard event data
     * (region, service, etc., as returned by <a>DescribeEvents</a>), a detailed event description, and possible
     * additional metadata that depends upon the nature of the event. Affected entities are not included; to retrieve
     * those, use the <a>DescribeAffectedEntities</a> operation.
     * </p>
     * <p>
     * If a specified event cannot be retrieved, an error message is returned for that event.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEventDetailsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeEventDetailsRequest#builder()}
     * </p>
     *
     * @param describeEventDetailsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEventDetailsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeEventDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventDetails" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeEventDetailsResponse> describeEventDetails(
            Consumer<DescribeEventDetailsRequest.Builder> describeEventDetailsRequest) {
        return describeEventDetails(DescribeEventDetailsRequest.builder().applyMutation(describeEventDetailsRequest).build());
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     *
     * @param describeEventTypesRequest
     * @return A Java Future containing the result of the DescribeEventTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeEventTypesResponse> describeEventTypes(DescribeEventTypesRequest describeEventTypesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEventTypesRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeEventTypesRequest#builder()}
     * </p>
     *
     * @param describeEventTypesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEventTypesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeEventTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeEventTypesResponse> describeEventTypes(
            Consumer<DescribeEventTypesRequest.Builder> describeEventTypesRequest) {
        return describeEventTypes(DescribeEventTypesRequest.builder().applyMutation(describeEventTypesRequest).build());
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeEventTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeEventTypesResponse> describeEventTypes() {
        return describeEventTypes(DescribeEventTypesRequest.builder().build());
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeEventTypesPublisher describeEventTypesPaginator() {
        return describeEventTypesPaginator(DescribeEventTypesRequest.builder().build());
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventTypesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeEventTypesPublisher describeEventTypesPaginator(DescribeEventTypesRequest describeEventTypesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEventTypesRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeEventTypesRequest#builder()}
     * </p>
     *
     * @param describeEventTypesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEventTypesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeEventTypesPublisher describeEventTypesPaginator(
            Consumer<DescribeEventTypesRequest.Builder> describeEventTypesRequest) {
        return describeEventTypesPaginator(DescribeEventTypesRequest.builder().applyMutation(describeEventTypesRequest).build());
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     *
     * @param describeEventsRequest
     * @return A Java Future containing the result of the DescribeEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeEventsResponse> describeEvents(DescribeEventsRequest describeEventsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEventsRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeEventsRequest#builder()}
     * </p>
     *
     * @param describeEventsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEventsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeEventsResponse> describeEvents(Consumer<DescribeEventsRequest.Builder> describeEventsRequest) {
        return describeEvents(DescribeEventsRequest.builder().applyMutation(describeEventsRequest).build());
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeEventsResponse> describeEvents() {
        return describeEvents(DescribeEventsRequest.builder().build());
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)} operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeEventsPublisher describeEventsPaginator() {
        return describeEventsPaginator(DescribeEventsRequest.builder().build());
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)} operation.</b>
     * </p>
     *
     * @param describeEventsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeEventsPublisher describeEventsPaginator(DescribeEventsRequest describeEventsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeEventsRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeEventsRequest#builder()}
     * </p>
     *
     * @param describeEventsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeEventsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeEventsPublisher describeEventsPaginator(Consumer<DescribeEventsRequest.Builder> describeEventsRequest) {
        return describeEventsPaginator(DescribeEventsRequest.builder().applyMutation(describeEventsRequest).build());
    }
}
