/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Summary information about an AWS Health event.
 * </p>
 * <p>
 * AWS Health events can be public or account-specific:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <i>Public events</i> might be service events that are not specific to an AWS account. For example, if there is an
 * issue with an AWS Region, AWS Health provides information about the event, even if you don't use services or
 * resources in that Region.
 * </p>
 * </li>
 * <li>
 * <p>
 * <i>Account-specific</i> events are specific to either your AWS account or an account in your organization. For
 * example, if there's an issue with Amazon Elastic Compute Cloud in a Region that you use, AWS Health provides
 * information about the event and the affected resources in the account.
 * </p>
 * </li>
 * </ul>
 * <p>
 * You can determine if an event is public or account-specific by using the <code>eventScopeCode</code> parameter. For
 * more information, see <a
 * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html#AWSHealth-Type-Event-eventScopeCode"
 * >eventScopeCode</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::service)).setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("service").build()).build();

    private static final SdkField<String> EVENT_TYPE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::eventTypeCode)).setter(setter(Builder::eventTypeCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventTypeCode").build()).build();

    private static final SdkField<String> EVENT_TYPE_CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::eventTypeCategoryAsString)).setter(setter(Builder::eventTypeCategory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventTypeCategory").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::availabilityZone)).setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("availabilityZone").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Event::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Event::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Event::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<String> EVENT_SCOPE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Event::eventScopeCodeAsString)).setter(setter(Builder::eventScopeCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventScopeCode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, SERVICE_FIELD,
            EVENT_TYPE_CODE_FIELD, EVENT_TYPE_CATEGORY_FIELD, REGION_FIELD, AVAILABILITY_ZONE_FIELD, START_TIME_FIELD,
            END_TIME_FIELD, LAST_UPDATED_TIME_FIELD, STATUS_CODE_FIELD, EVENT_SCOPE_CODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String service;

    private final String eventTypeCode;

    private final String eventTypeCategory;

    private final String region;

    private final String availabilityZone;

    private final Instant startTime;

    private final Instant endTime;

    private final Instant lastUpdatedTime;

    private final String statusCodeValue;

    private final String eventScopeCode;

    private Event(BuilderImpl builder) {
        this.arn = builder.arn;
        this.service = builder.service;
        this.eventTypeCode = builder.eventTypeCode;
        this.eventTypeCategory = builder.eventTypeCategory;
        this.region = builder.region;
        this.availabilityZone = builder.availabilityZone;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.statusCodeValue = builder.statusCodeValue;
        this.eventScopeCode = builder.eventScopeCode;
    }

    /**
     * <p>
     * The unique identifier for the event. Format:
     * <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
     * . Example:
     * <code>Example: arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
     * </p>
     * 
     * @return The unique identifier for the event. Format:
     *         <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
     *         . Example:
     *         <code>Example: arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
     * </p>
     * 
     * @return The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
     */
    public String service() {
        return service;
    }

    /**
     * <p>
     * The unique identifier for the event type. The format is <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for
     * example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     * </p>
     * 
     * @return The unique identifier for the event type. The format is
     *         <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for example,
     *         <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     */
    public String eventTypeCode() {
        return eventTypeCode;
    }

    /**
     * <p>
     * The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
     * <code>accountNotification</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventTypeCategory}
     * will return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventTypeCategoryAsString}.
     * </p>
     * 
     * @return The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
     *         <code>accountNotification</code>.
     * @see EventTypeCategory
     */
    public EventTypeCategory eventTypeCategory() {
        return EventTypeCategory.fromValue(eventTypeCategory);
    }

    /**
     * <p>
     * The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
     * <code>accountNotification</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventTypeCategory}
     * will return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventTypeCategoryAsString}.
     * </p>
     * 
     * @return The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
     *         <code>accountNotification</code>.
     * @see EventTypeCategory
     */
    public String eventTypeCategoryAsString() {
        return eventTypeCategory;
    }

    /**
     * <p>
     * The AWS region name of the event.
     * </p>
     * 
     * @return The AWS region name of the event.
     */
    public String region() {
        return region;
    }

    /**
     * <p>
     * The AWS Availability Zone of the event. For example, us-east-1a.
     * </p>
     * 
     * @return The AWS Availability Zone of the event. For example, us-east-1a.
     */
    public String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The date and time that the event began.
     * </p>
     * 
     * @return The date and time that the event began.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time that the event ended.
     * </p>
     * 
     * @return The date and time that the event ended.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The most recent date and time that the event was updated.
     * </p>
     * 
     * @return The most recent date and time that the event was updated.
     */
    public Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     * <code>upcoming</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link EventStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     *         <code>upcoming</code>.
     * @see EventStatusCode
     */
    public EventStatusCode statusCode() {
        return EventStatusCode.fromValue(statusCodeValue);
    }

    /**
     * <p>
     * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     * <code>upcoming</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link EventStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
     *         <code>upcoming</code>.
     * @see EventStatusCode
     */
    public String statusCodeAsString() {
        return statusCodeValue;
    }

    /**
     * <p>
     * This parameter specifies if the AWS Health event is a public AWS service event or an account-specific event.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value is
     * always empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the <code>affectedAccounts</code>
     * value lists the affected AWS accounts in your organization. For example, if an event affects a service such as
     * Amazon Elastic Compute Cloud and you have AWS accounts that use that service, those account IDs appear in the
     * response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you specified
     * in the request is invalid or doesn't exist.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventScopeCode}
     * will return {@link EventScopeCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventScopeCodeAsString}.
     * </p>
     * 
     * @return This parameter specifies if the AWS Health event is a public AWS service event or an account-specific
     *         event.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code>
     *         value is always empty.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
     *         <code>affectedAccounts</code> value lists the affected AWS accounts in your organization. For example, if
     *         an event affects a service such as Amazon Elastic Compute Cloud and you have AWS accounts that use that
     *         service, those account IDs appear in the response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
     *         specified in the request is invalid or doesn't exist.
     *         </p>
     *         </li>
     * @see EventScopeCode
     */
    public EventScopeCode eventScopeCode() {
        return EventScopeCode.fromValue(eventScopeCode);
    }

    /**
     * <p>
     * This parameter specifies if the AWS Health event is a public AWS service event or an account-specific event.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value is
     * always empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the <code>affectedAccounts</code>
     * value lists the affected AWS accounts in your organization. For example, if an event affects a service such as
     * Amazon Elastic Compute Cloud and you have AWS accounts that use that service, those account IDs appear in the
     * response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you specified
     * in the request is invalid or doesn't exist.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventScopeCode}
     * will return {@link EventScopeCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventScopeCodeAsString}.
     * </p>
     * 
     * @return This parameter specifies if the AWS Health event is a public AWS service event or an account-specific
     *         event.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code>
     *         value is always empty.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
     *         <code>affectedAccounts</code> value lists the affected AWS accounts in your organization. For example, if
     *         an event affects a service such as Amazon Elastic Compute Cloud and you have AWS accounts that use that
     *         service, those account IDs appear in the response.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
     *         specified in the request is invalid or doesn't exist.
     *         </p>
     *         </li>
     * @see EventScopeCode
     */
    public String eventScopeCodeAsString() {
        return eventScopeCode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(service());
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeCode());
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeCategoryAsString());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventScopeCodeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(service(), other.service())
                && Objects.equals(eventTypeCode(), other.eventTypeCode())
                && Objects.equals(eventTypeCategoryAsString(), other.eventTypeCategoryAsString())
                && Objects.equals(region(), other.region()) && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString())
                && Objects.equals(eventScopeCodeAsString(), other.eventScopeCodeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Event").add("Arn", arn()).add("Service", service()).add("EventTypeCode", eventTypeCode())
                .add("EventTypeCategory", eventTypeCategoryAsString()).add("Region", region())
                .add("AvailabilityZone", availabilityZone()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("LastUpdatedTime", lastUpdatedTime()).add("StatusCode", statusCodeAsString())
                .add("EventScopeCode", eventScopeCodeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "service":
            return Optional.ofNullable(clazz.cast(service()));
        case "eventTypeCode":
            return Optional.ofNullable(clazz.cast(eventTypeCode()));
        case "eventTypeCategory":
            return Optional.ofNullable(clazz.cast(eventTypeCategoryAsString()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "availabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "eventScopeCode":
            return Optional.ofNullable(clazz.cast(eventScopeCodeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The unique identifier for the event. Format:
         * <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
         * . Example:
         * <code>Example: arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
         * </p>
         * 
         * @param arn
         *        The unique identifier for the event. Format:
         *        <code>arn:aws:health:<i>event-region</i>::event/<i>SERVICE</i>/<i>EVENT_TYPE_CODE</i>/<i>EVENT_TYPE_PLUS_ID</i> </code>
         *        . Example:
         *        <code>Example: arn:aws:health:us-east-1::event/EC2/EC2_INSTANCE_RETIREMENT_SCHEDULED/EC2_INSTANCE_RETIREMENT_SCHEDULED_ABC123-DEF456</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
         * </p>
         * 
         * @param service
         *        The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder service(String service);

        /**
         * <p>
         * The unique identifier for the event type. The format is <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>;
         * for example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * </p>
         * 
         * @param eventTypeCode
         *        The unique identifier for the event type. The format is
         *        <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for example,
         *        <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTypeCode(String eventTypeCode);

        /**
         * <p>
         * The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
         * <code>accountNotification</code>.
         * </p>
         * 
         * @param eventTypeCategory
         *        The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
         *        <code>accountNotification</code>.
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder eventTypeCategory(String eventTypeCategory);

        /**
         * <p>
         * The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
         * <code>accountNotification</code>.
         * </p>
         * 
         * @param eventTypeCategory
         *        The category of the event. Possible values are <code>issue</code>, <code>scheduledChange</code>, and
         *        <code>accountNotification</code>.
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder eventTypeCategory(EventTypeCategory eventTypeCategory);

        /**
         * <p>
         * The AWS region name of the event.
         * </p>
         * 
         * @param region
         *        The AWS region name of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The AWS Availability Zone of the event. For example, us-east-1a.
         * </p>
         * 
         * @param availabilityZone
         *        The AWS Availability Zone of the event. For example, us-east-1a.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The date and time that the event began.
         * </p>
         * 
         * @param startTime
         *        The date and time that the event began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time that the event ended.
         * </p>
         * 
         * @param endTime
         *        The date and time that the event ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The most recent date and time that the event was updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The most recent date and time that the event was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         * <code>upcoming</code>.
         * </p>
         * 
         * @param statusCodeValue
         *        The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         *        <code>upcoming</code>.
         * @see EventStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventStatusCode
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         * <code>upcoming</code>.
         * </p>
         * 
         * @param statusCodeValue
         *        The most recent status of the event. Possible values are <code>open</code>, <code>closed</code>, and
         *        <code>upcoming</code>.
         * @see EventStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventStatusCode
         */
        Builder statusCode(EventStatusCode statusCodeValue);

        /**
         * <p>
         * This parameter specifies if the AWS Health event is a public AWS service event or an account-specific event.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value
         * is always empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         * <code>affectedAccounts</code> value lists the affected AWS accounts in your organization. For example, if an
         * event affects a service such as Amazon Elastic Compute Cloud and you have AWS accounts that use that service,
         * those account IDs appear in the response.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         * specified in the request is invalid or doesn't exist.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventScopeCode
         *        This parameter specifies if the AWS Health event is a public AWS service event or an account-specific
         *        event.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the
         *        <code>affectedAccounts</code> value is always empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         *        <code>affectedAccounts</code> value lists the affected AWS accounts in your organization. For example,
         *        if an event affects a service such as Amazon Elastic Compute Cloud and you have AWS accounts that use
         *        that service, those account IDs appear in the response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         *        specified in the request is invalid or doesn't exist.
         *        </p>
         *        </li>
         * @see EventScopeCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventScopeCode
         */
        Builder eventScopeCode(String eventScopeCode);

        /**
         * <p>
         * This parameter specifies if the AWS Health event is a public AWS service event or an account-specific event.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the <code>affectedAccounts</code> value
         * is always empty.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         * <code>affectedAccounts</code> value lists the affected AWS accounts in your organization. For example, if an
         * event affects a service such as Amazon Elastic Compute Cloud and you have AWS accounts that use that service,
         * those account IDs appear in the response.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         * specified in the request is invalid or doesn't exist.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventScopeCode
         *        This parameter specifies if the AWS Health event is a public AWS service event or an account-specific
         *        event.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>PUBLIC</code>, then the
         *        <code>affectedAccounts</code> value is always empty.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>ACCOUNT_SPECIFIC</code>, then the
         *        <code>affectedAccounts</code> value lists the affected AWS accounts in your organization. For example,
         *        if an event affects a service such as Amazon Elastic Compute Cloud and you have AWS accounts that use
         *        that service, those account IDs appear in the response.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the <code>eventScopeCode</code> value is <code>NONE</code>, then the <code>eventArn</code> that you
         *        specified in the request is invalid or doesn't exist.
         *        </p>
         *        </li>
         * @see EventScopeCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventScopeCode
         */
        Builder eventScopeCode(EventScopeCode eventScopeCode);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String service;

        private String eventTypeCode;

        private String eventTypeCategory;

        private String region;

        private String availabilityZone;

        private Instant startTime;

        private Instant endTime;

        private Instant lastUpdatedTime;

        private String statusCodeValue;

        private String eventScopeCode;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            arn(model.arn);
            service(model.service);
            eventTypeCode(model.eventTypeCode);
            eventTypeCategory(model.eventTypeCategory);
            region(model.region);
            availabilityZone(model.availabilityZone);
            startTime(model.startTime);
            endTime(model.endTime);
            lastUpdatedTime(model.lastUpdatedTime);
            statusCode(model.statusCodeValue);
            eventScopeCode(model.eventScopeCode);
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getService() {
            return service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        public final void setService(String service) {
            this.service = service;
        }

        public final String getEventTypeCode() {
            return eventTypeCode;
        }

        @Override
        public final Builder eventTypeCode(String eventTypeCode) {
            this.eventTypeCode = eventTypeCode;
            return this;
        }

        public final void setEventTypeCode(String eventTypeCode) {
            this.eventTypeCode = eventTypeCode;
        }

        public final String getEventTypeCategory() {
            return eventTypeCategory;
        }

        @Override
        public final Builder eventTypeCategory(String eventTypeCategory) {
            this.eventTypeCategory = eventTypeCategory;
            return this;
        }

        @Override
        public final Builder eventTypeCategory(EventTypeCategory eventTypeCategory) {
            this.eventTypeCategory(eventTypeCategory == null ? null : eventTypeCategory.toString());
            return this;
        }

        public final void setEventTypeCategory(String eventTypeCategory) {
            this.eventTypeCategory = eventTypeCategory;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        @Override
        public final Builder statusCode(EventStatusCode statusCodeValue) {
            this.statusCode(statusCodeValue == null ? null : statusCodeValue.toString());
            return this;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        public final String getEventScopeCode() {
            return eventScopeCode;
        }

        @Override
        public final Builder eventScopeCode(String eventScopeCode) {
            this.eventScopeCode = eventScopeCode;
            return this;
        }

        @Override
        public final Builder eventScopeCode(EventScopeCode eventScopeCode) {
            this.eventScopeCode(eventScopeCode == null ? null : eventScopeCode.toString());
            return this;
        }

        public final void setEventScopeCode(String eventScopeCode) {
            this.eventScopeCode = eventScopeCode;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
