/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsResponse;
import software.amazon.awssdk.services.health.model.DescribeEventTypesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventTypesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventsResponse;
import software.amazon.awssdk.services.health.model.HealthException;
import software.amazon.awssdk.services.health.model.HealthRequest;
import software.amazon.awssdk.services.health.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.health.model.UnsupportedLocaleException;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable;
import software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable;
import software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable;
import software.amazon.awssdk.services.health.paginators.DescribeEventsIterable;
import software.amazon.awssdk.services.health.transform.DescribeAffectedEntitiesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEntityAggregatesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventAggregatesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventDetailsRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventTypesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventsRequestMarshaller;

/**
 * Internal implementation of {@link HealthClient}.
 *
 * @see HealthClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHealthClient implements HealthClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultHealthClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     * @return Result of the DescribeAffectedEntities operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedEntitiesResponse describeAffectedEntities(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) throws InvalidPaginationTokenException,
            UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAffectedEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAffectedEntitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeAffectedEntitiesRequest, DescribeAffectedEntitiesResponse>()
                        .withOperationName("DescribeAffectedEntities").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeAffectedEntitiesRequest)
                        .withMarshaller(new DescribeAffectedEntitiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable responses = client.describeAffectedEntitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable responses = client
     *             .describeAffectedEntitiesPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesIterable responses = client.describeAffectedEntitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAffectedEntitiesIterable describeAffectedEntitiesPaginator(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) throws InvalidPaginationTokenException,
            UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        return new DescribeAffectedEntitiesIterable(this, applyPaginatorUserAgent(describeAffectedEntitiesRequest));
    }

    /**
     * <p>
     * Returns the number of entities that are affected by each of the specified events. If no events are specified, the
     * counts of all affected entities are returned.
     * </p>
     *
     * @param describeEntityAggregatesRequest
     * @return Result of the DescribeEntityAggregates operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEntityAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEntityAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEntityAggregatesResponse describeEntityAggregates(
            DescribeEntityAggregatesRequest describeEntityAggregatesRequest) throws AwsServiceException, SdkClientException,
            HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEntityAggregatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEntityAggregatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeEntityAggregatesRequest, DescribeEntityAggregatesResponse>()
                        .withOperationName("DescribeEntityAggregates").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeEntityAggregatesRequest)
                        .withMarshaller(new DescribeEntityAggregatesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     *
     * @param describeEventAggregatesRequest
     * @return Result of the DescribeEventAggregates operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeEventAggregatesResponse describeEventAggregates(DescribeEventAggregatesRequest describeEventAggregatesRequest)
            throws InvalidPaginationTokenException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventAggregatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventAggregatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeEventAggregatesRequest, DescribeEventAggregatesResponse>()
                .withOperationName("DescribeEventAggregates").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeEventAggregatesRequest)
                .withMarshaller(new DescribeEventAggregatesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable responses = client.describeEventAggregatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable responses = client
     *             .describeEventAggregatesPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesIterable responses = client.describeEventAggregatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventAggregatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeEventAggregatesIterable describeEventAggregatesPaginator(
            DescribeEventAggregatesRequest describeEventAggregatesRequest) throws InvalidPaginationTokenException,
            AwsServiceException, SdkClientException, HealthException {
        return new DescribeEventAggregatesIterable(this, applyPaginatorUserAgent(describeEventAggregatesRequest));
    }

    /**
     * <p>
     * Returns detailed information about one or more specified events. Information includes standard event data
     * (region, service, etc., as returned by <a>DescribeEvents</a>), a detailed event description, and possible
     * additional metadata that depends upon the nature of the event. Affected entities are not included; to retrieve
     * those, use the <a>DescribeAffectedEntities</a> operation.
     * </p>
     * <p>
     * If a specified event cannot be retrieved, an error message is returned for that event.
     * </p>
     *
     * @param describeEventDetailsRequest
     * @return Result of the DescribeEventDetails operation returned by the service.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventDetails" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeEventDetailsResponse describeEventDetails(DescribeEventDetailsRequest describeEventDetailsRequest)
            throws UnsupportedLocaleException, AwsServiceException, SdkClientException, HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeEventDetailsRequest, DescribeEventDetailsResponse>()
                .withOperationName("DescribeEventDetails").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeEventDetailsRequest)
                .withMarshaller(new DescribeEventDetailsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     *
     * @param describeEventTypesRequest
     * @return Result of the DescribeEventTypes operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventTypesResponse describeEventTypes(DescribeEventTypesRequest describeEventTypesRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeEventTypesRequest, DescribeEventTypesResponse>()
                .withOperationName("DescribeEventTypes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeEventTypesRequest)
                .withMarshaller(new DescribeEventTypesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable responses = client.describeEventTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable responses = client
     *             .describeEventTypesPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeEventTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesIterable responses = client.describeEventTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventTypesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventTypesIterable describeEventTypesPaginator(DescribeEventTypesRequest describeEventTypesRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        return new DescribeEventTypesIterable(this, applyPaginatorUserAgent(describeEventTypesRequest));
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     *
     * @param describeEventsRequest
     * @return Result of the DescribeEvents operation returned by the service.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventsResponse describeEvents(DescribeEventsRequest describeEventsRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeEventsRequest, DescribeEventsResponse>()
                .withOperationName("DescribeEvents").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeEventsRequest)
                .withMarshaller(new DescribeEventsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsIterable responses = client.describeEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.health.paginators.DescribeEventsIterable responses = client.describeEventsPaginator(request);
     *     for (software.amazon.awssdk.services.health.model.DescribeEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsIterable responses = client.describeEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)} operation.</b>
     * </p>
     *
     * @param describeEventsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidPaginationTokenException
     *         The specified pagination token (<code>nextToken</code>) is not valid.
     * @throws UnsupportedLocaleException
     *         The specified locale is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventsIterable describeEventsPaginator(DescribeEventsRequest describeEventsRequest)
            throws InvalidPaginationTokenException, UnsupportedLocaleException, AwsServiceException, SdkClientException,
            HealthException {
        return new DescribeEventsIterable(this, applyPaginatorUserAgent(describeEventsRequest));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(HealthException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedLocale")
                                .exceptionBuilderSupplier(UnsupportedLocaleException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationToken")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends HealthRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
