/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An alternate list of prioritized locations for use with a game session queue. When this property is included in a <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartGameSessionPlacement.html">
 * StartGameSessionPlacement</a> request, this list overrides the queue's default location prioritization, as defined in
 * the queue's <a href="gamelift/latest/apireference/API_PriorityConfiguration.html">PriorityConfiguration</a> setting
 * (<i>LocationOrder</i>). This property overrides the queue's default priority list for individual placement requests
 * only. Use this property only with queues that have a <code>PriorityConfiguration</code> setting that prioritizes
 * first.
 * </p>
 * <note>
 * <p>
 * A priority configuration override list does not override a queue's FilterConfiguration setting, if the queue has one.
 * Filter configurations are used to limit placements to a subset of the locations in a queue's destinations. If the
 * override list includes a location that's not included in the FilterConfiguration allowed list, Amazon GameLift won't
 * attempt to place a game session there.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PriorityConfigurationOverride implements SdkPojo, Serializable,
        ToCopyableBuilder<PriorityConfigurationOverride.Builder, PriorityConfigurationOverride> {
    private static final SdkField<String> PLACEMENT_FALLBACK_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlacementFallbackStrategy")
            .getter(getter(PriorityConfigurationOverride::placementFallbackStrategyAsString))
            .setter(setter(Builder::placementFallbackStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacementFallbackStrategy").build())
            .build();

    private static final SdkField<List<String>> LOCATION_ORDER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LocationOrder")
            .getter(getter(PriorityConfigurationOverride::locationOrder))
            .setter(setter(Builder::locationOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationOrder").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            PLACEMENT_FALLBACK_STRATEGY_FIELD, LOCATION_ORDER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String placementFallbackStrategy;

    private final List<String> locationOrder;

    private PriorityConfigurationOverride(BuilderImpl builder) {
        this.placementFallbackStrategy = builder.placementFallbackStrategy;
        this.locationOrder = builder.locationOrder;
    }

    /**
     * <p>
     * Instructions for how to use the override list if the first round of placement attempts fails. The first round is
     * a failure if Amazon GameLift searches all listed locations, in all of the queue's destinations, without finding
     * an available hosting resource for a new game session. Valid strategies include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the override list
     * and use the queue's default location priority list. Continue to use the queue's default list until the placement
     * request times out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the placement
     * request times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #placementFallbackStrategy} will return {@link PlacementFallbackStrategy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #placementFallbackStrategyAsString}.
     * </p>
     * 
     * @return Instructions for how to use the override list if the first round of placement attempts fails. The first
     *         round is a failure if Amazon GameLift searches all listed locations, in all of the queue's destinations,
     *         without finding an available hosting resource for a new game session. Valid strategies include: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the
     *         override list and use the queue's default location priority list. Continue to use the queue's default
     *         list until the placement request times out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the
     *         placement request times out.
     *         </p>
     *         </li>
     * @see PlacementFallbackStrategy
     */
    public final PlacementFallbackStrategy placementFallbackStrategy() {
        return PlacementFallbackStrategy.fromValue(placementFallbackStrategy);
    }

    /**
     * <p>
     * Instructions for how to use the override list if the first round of placement attempts fails. The first round is
     * a failure if Amazon GameLift searches all listed locations, in all of the queue's destinations, without finding
     * an available hosting resource for a new game session. Valid strategies include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the override list
     * and use the queue's default location priority list. Continue to use the queue's default list until the placement
     * request times out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the placement
     * request times out.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #placementFallbackStrategy} will return {@link PlacementFallbackStrategy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #placementFallbackStrategyAsString}.
     * </p>
     * 
     * @return Instructions for how to use the override list if the first round of placement attempts fails. The first
     *         round is a failure if Amazon GameLift searches all listed locations, in all of the queue's destinations,
     *         without finding an available hosting resource for a new game session. Valid strategies include: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the
     *         override list and use the queue's default location priority list. Continue to use the queue's default
     *         list until the placement request times out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the
     *         placement request times out.
     *         </p>
     *         </li>
     * @see PlacementFallbackStrategy
     */
    public final String placementFallbackStrategyAsString() {
        return placementFallbackStrategy;
    }

    /**
     * For responses, this returns true if the service returned a value for the LocationOrder property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLocationOrder() {
        return locationOrder != null && !(locationOrder instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A prioritized list of hosting locations. The list can include Amazon Web Services Regions (such as
     * <code>us-west-2</code>), local zones, and custom locations (for Anywhere fleets). Each location must be listed
     * only once. For details, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-regions.html">Amazon GameLift service
     * locations.</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocationOrder} method.
     * </p>
     * 
     * @return A prioritized list of hosting locations. The list can include Amazon Web Services Regions (such as
     *         <code>us-west-2</code>), local zones, and custom locations (for Anywhere fleets). Each location must be
     *         listed only once. For details, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-regions.html">Amazon GameLift
     *         service locations.</a>
     */
    public final List<String> locationOrder() {
        return locationOrder;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(placementFallbackStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLocationOrder() ? locationOrder() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PriorityConfigurationOverride)) {
            return false;
        }
        PriorityConfigurationOverride other = (PriorityConfigurationOverride) obj;
        return Objects.equals(placementFallbackStrategyAsString(), other.placementFallbackStrategyAsString())
                && hasLocationOrder() == other.hasLocationOrder() && Objects.equals(locationOrder(), other.locationOrder());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PriorityConfigurationOverride")
                .add("PlacementFallbackStrategy", placementFallbackStrategyAsString())
                .add("LocationOrder", hasLocationOrder() ? locationOrder() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PlacementFallbackStrategy":
            return Optional.ofNullable(clazz.cast(placementFallbackStrategyAsString()));
        case "LocationOrder":
            return Optional.ofNullable(clazz.cast(locationOrder()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PlacementFallbackStrategy", PLACEMENT_FALLBACK_STRATEGY_FIELD);
        map.put("LocationOrder", LOCATION_ORDER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PriorityConfigurationOverride, T> g) {
        return obj -> g.apply((PriorityConfigurationOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PriorityConfigurationOverride> {
        /**
         * <p>
         * Instructions for how to use the override list if the first round of placement attempts fails. The first round
         * is a failure if Amazon GameLift searches all listed locations, in all of the queue's destinations, without
         * finding an available hosting resource for a new game session. Valid strategies include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the override
         * list and use the queue's default location priority list. Continue to use the queue's default list until the
         * placement request times out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the
         * placement request times out.
         * </p>
         * </li>
         * </ul>
         * 
         * @param placementFallbackStrategy
         *        Instructions for how to use the override list if the first round of placement attempts fails. The
         *        first round is a failure if Amazon GameLift searches all listed locations, in all of the queue's
         *        destinations, without finding an available hosting resource for a new game session. Valid strategies
         *        include: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the
         *        override list and use the queue's default location priority list. Continue to use the queue's default
         *        list until the placement request times out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the
         *        placement request times out.
         *        </p>
         *        </li>
         * @see PlacementFallbackStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementFallbackStrategy
         */
        Builder placementFallbackStrategy(String placementFallbackStrategy);

        /**
         * <p>
         * Instructions for how to use the override list if the first round of placement attempts fails. The first round
         * is a failure if Amazon GameLift searches all listed locations, in all of the queue's destinations, without
         * finding an available hosting resource for a new game session. Valid strategies include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the override
         * list and use the queue's default location priority list. Continue to use the queue's default list until the
         * placement request times out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the
         * placement request times out.
         * </p>
         * </li>
         * </ul>
         * 
         * @param placementFallbackStrategy
         *        Instructions for how to use the override list if the first round of placement attempts fails. The
         *        first round is a failure if Amazon GameLift searches all listed locations, in all of the queue's
         *        destinations, without finding an available hosting resource for a new game session. Valid strategies
         *        include: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DEFAULT_AFTER_SINGLE_PASS</code> -- After the first round of placement attempts, discard the
         *        override list and use the queue's default location priority list. Continue to use the queue's default
         *        list until the placement request times out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code> -- Continue to use the override list for all rounds of placement attempts until the
         *        placement request times out.
         *        </p>
         *        </li>
         * @see PlacementFallbackStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementFallbackStrategy
         */
        Builder placementFallbackStrategy(PlacementFallbackStrategy placementFallbackStrategy);

        /**
         * <p>
         * A prioritized list of hosting locations. The list can include Amazon Web Services Regions (such as
         * <code>us-west-2</code>), local zones, and custom locations (for Anywhere fleets). Each location must be
         * listed only once. For details, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-regions.html">Amazon GameLift
         * service locations.</a>
         * </p>
         * 
         * @param locationOrder
         *        A prioritized list of hosting locations. The list can include Amazon Web Services Regions (such as
         *        <code>us-west-2</code>), local zones, and custom locations (for Anywhere fleets). Each location must
         *        be listed only once. For details, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-regions.html">Amazon
         *        GameLift service locations.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationOrder(Collection<String> locationOrder);

        /**
         * <p>
         * A prioritized list of hosting locations. The list can include Amazon Web Services Regions (such as
         * <code>us-west-2</code>), local zones, and custom locations (for Anywhere fleets). Each location must be
         * listed only once. For details, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-regions.html">Amazon GameLift
         * service locations.</a>
         * </p>
         * 
         * @param locationOrder
         *        A prioritized list of hosting locations. The list can include Amazon Web Services Regions (such as
         *        <code>us-west-2</code>), local zones, and custom locations (for Anywhere fleets). Each location must
         *        be listed only once. For details, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-regions.html">Amazon
         *        GameLift service locations.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationOrder(String... locationOrder);
    }

    static final class BuilderImpl implements Builder {
        private String placementFallbackStrategy;

        private List<String> locationOrder = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PriorityConfigurationOverride model) {
            placementFallbackStrategy(model.placementFallbackStrategy);
            locationOrder(model.locationOrder);
        }

        public final String getPlacementFallbackStrategy() {
            return placementFallbackStrategy;
        }

        public final void setPlacementFallbackStrategy(String placementFallbackStrategy) {
            this.placementFallbackStrategy = placementFallbackStrategy;
        }

        @Override
        public final Builder placementFallbackStrategy(String placementFallbackStrategy) {
            this.placementFallbackStrategy = placementFallbackStrategy;
            return this;
        }

        @Override
        public final Builder placementFallbackStrategy(PlacementFallbackStrategy placementFallbackStrategy) {
            this.placementFallbackStrategy(placementFallbackStrategy == null ? null : placementFallbackStrategy.toString());
            return this;
        }

        public final Collection<String> getLocationOrder() {
            if (locationOrder instanceof SdkAutoConstructList) {
                return null;
            }
            return locationOrder;
        }

        public final void setLocationOrder(Collection<String> locationOrder) {
            this.locationOrder = LocationOrderOverrideListCopier.copy(locationOrder);
        }

        @Override
        public final Builder locationOrder(Collection<String> locationOrder) {
            this.locationOrder = LocationOrderOverrideListCopier.copy(locationOrder);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder locationOrder(String... locationOrder) {
            locationOrder(Arrays.asList(locationOrder));
            return this;
        }

        @Override
        public PriorityConfigurationOverride build() {
            return new PriorityConfigurationOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
