/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the Amazon GameLift FleetIQ and game server groups.</b>
 * </p>
 * <p>
 * An allowed instance type for a game server group. All game server groups must have at least two instance types
 * defined for it. Amazon GameLift FleetIQ periodically evaluates each defined instance type for viability. It then
 * updates the Auto Scaling group with the list of viable instance types.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceDefinition.Builder, InstanceDefinition> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(InstanceDefinition::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> WEIGHTED_CAPACITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WeightedCapacity").getter(getter(InstanceDefinition::weightedCapacity))
            .setter(setter(Builder::weightedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeightedCapacity").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            WEIGHTED_CAPACITY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final String weightedCapacity;

    private InstanceDefinition(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.weightedCapacity = builder.weightedCapacity;
    }

    /**
     * <p>
     * An Amazon EC2 instance type designation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link GameServerGroupInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return An Amazon EC2 instance type designation.
     * @see GameServerGroupInstanceType
     */
    public final GameServerGroupInstanceType instanceType() {
        return GameServerGroupInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * An Amazon EC2 instance type designation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link GameServerGroupInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return An Amazon EC2 instance type designation.
     * @see GameServerGroupInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * Instance weighting that indicates how much this instance type contributes to the total capacity of a game server
     * group. Instance weights are used by Amazon GameLift FleetIQ to calculate the instance type's cost per unit hour
     * and better identify the most cost-effective options. For detailed information on weighting instance capacity, see
     * <a href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-weighting.html">Instance
     * Weighting</a> in the <i>Amazon Elastic Compute Cloud Auto Scaling User Guide</i>. Default value is "1".
     * </p>
     * 
     * @return Instance weighting that indicates how much this instance type contributes to the total capacity of a game
     *         server group. Instance weights are used by Amazon GameLift FleetIQ to calculate the instance type's cost
     *         per unit hour and better identify the most cost-effective options. For detailed information on weighting
     *         instance capacity, see <a
     *         href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-weighting.html">Instance
     *         Weighting</a> in the <i>Amazon Elastic Compute Cloud Auto Scaling User Guide</i>. Default value is "1".
     */
    public final String weightedCapacity() {
        return weightedCapacity;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(weightedCapacity());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceDefinition)) {
            return false;
        }
        InstanceDefinition other = (InstanceDefinition) obj;
        return Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(weightedCapacity(), other.weightedCapacity());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceDefinition").add("InstanceType", instanceTypeAsString())
                .add("WeightedCapacity", weightedCapacity()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "WeightedCapacity":
            return Optional.ofNullable(clazz.cast(weightedCapacity()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("WeightedCapacity", WEIGHTED_CAPACITY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceDefinition, T> g) {
        return obj -> g.apply((InstanceDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceDefinition> {
        /**
         * <p>
         * An Amazon EC2 instance type designation.
         * </p>
         * 
         * @param instanceType
         *        An Amazon EC2 instance type designation.
         * @see GameServerGroupInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerGroupInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * An Amazon EC2 instance type designation.
         * </p>
         * 
         * @param instanceType
         *        An Amazon EC2 instance type designation.
         * @see GameServerGroupInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerGroupInstanceType
         */
        Builder instanceType(GameServerGroupInstanceType instanceType);

        /**
         * <p>
         * Instance weighting that indicates how much this instance type contributes to the total capacity of a game
         * server group. Instance weights are used by Amazon GameLift FleetIQ to calculate the instance type's cost per
         * unit hour and better identify the most cost-effective options. For detailed information on weighting instance
         * capacity, see <a
         * href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-weighting.html">Instance
         * Weighting</a> in the <i>Amazon Elastic Compute Cloud Auto Scaling User Guide</i>. Default value is "1".
         * </p>
         * 
         * @param weightedCapacity
         *        Instance weighting that indicates how much this instance type contributes to the total capacity of a
         *        game server group. Instance weights are used by Amazon GameLift FleetIQ to calculate the instance
         *        type's cost per unit hour and better identify the most cost-effective options. For detailed
         *        information on weighting instance capacity, see <a
         *        href="https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-instance-weighting.html">Instance
         *        Weighting</a> in the <i>Amazon Elastic Compute Cloud Auto Scaling User Guide</i>. Default value is
         *        "1".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weightedCapacity(String weightedCapacity);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private String weightedCapacity;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceDefinition model) {
            instanceType(model.instanceType);
            weightedCapacity(model.weightedCapacity);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(GameServerGroupInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final String getWeightedCapacity() {
            return weightedCapacity;
        }

        public final void setWeightedCapacity(String weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
        }

        @Override
        public final Builder weightedCapacity(String weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
            return this;
        }

        @Override
        public InstanceDefinition build() {
            return new InstanceDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
