/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information and credentials that you can use to remotely connect to an instance in an EC2 managed fleet. This data
 * type is returned in response to a call to <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetInstanceAccess"
 * >https://docs.aws.amazon.com/gamelift/latest/apireference/API_GetInstanceAccess</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceAccess implements SdkPojo, Serializable, ToCopyableBuilder<InstanceAccess.Builder, InstanceAccess> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(InstanceAccess::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(InstanceAccess::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(InstanceAccess::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystem").getter(getter(InstanceAccess::operatingSystemAsString))
            .setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<InstanceCredentials> CREDENTIALS_FIELD = SdkField
            .<InstanceCredentials> builder(MarshallingType.SDK_POJO).memberName("Credentials")
            .getter(getter(InstanceAccess::credentials)).setter(setter(Builder::credentials))
            .constructor(InstanceCredentials::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Credentials").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD,
            INSTANCE_ID_FIELD, IP_ADDRESS_FIELD, OPERATING_SYSTEM_FIELD, CREDENTIALS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String instanceId;

    private final String ipAddress;

    private final String operatingSystem;

    private final InstanceCredentials credentials;

    private InstanceAccess(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.instanceId = builder.instanceId;
        this.ipAddress = builder.ipAddress;
        this.operatingSystem = builder.operatingSystem;
        this.credentials = builder.credentials;
    }

    /**
     * <p>
     * A unique identifier for the fleet containing the instance to be accessed.
     * </p>
     * 
     * @return A unique identifier for the fleet containing the instance to be accessed.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * A unique identifier for the instance to be accessed.
     * </p>
     * 
     * @return A unique identifier for the instance to be accessed.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * IP address assigned to the instance.
     * </p>
     * 
     * @return IP address assigned to the instance.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * Operating system that is running on the instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that is running on the instance.
     * @see OperatingSystem
     */
    public final OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * Operating system that is running on the instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that is running on the instance.
     * @see OperatingSystem
     */
    public final String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * Security credentials that are required to access the instance.
     * </p>
     * 
     * @return Security credentials that are required to access the instance.
     */
    public final InstanceCredentials credentials() {
        return credentials;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(credentials());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceAccess)) {
            return false;
        }
        InstanceAccess other = (InstanceAccess) obj;
        return Objects.equals(fleetId(), other.fleetId()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(credentials(), other.credentials());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceAccess").add("FleetId", fleetId()).add("InstanceId", instanceId())
                .add("IpAddress", ipAddress() == null ? null : "*** Sensitive Data Redacted ***")
                .add("OperatingSystem", operatingSystemAsString())
                .add("Credentials", credentials() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "Credentials":
            return Optional.ofNullable(clazz.cast(credentials()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FleetId", FLEET_ID_FIELD);
        map.put("InstanceId", INSTANCE_ID_FIELD);
        map.put("IpAddress", IP_ADDRESS_FIELD);
        map.put("OperatingSystem", OPERATING_SYSTEM_FIELD);
        map.put("Credentials", CREDENTIALS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceAccess, T> g) {
        return obj -> g.apply((InstanceAccess) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceAccess> {
        /**
         * <p>
         * A unique identifier for the fleet containing the instance to be accessed.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for the fleet containing the instance to be accessed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * A unique identifier for the instance to be accessed.
         * </p>
         * 
         * @param instanceId
         *        A unique identifier for the instance to be accessed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * IP address assigned to the instance.
         * </p>
         * 
         * @param ipAddress
         *        IP address assigned to the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * Operating system that is running on the instance.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that is running on the instance.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * Operating system that is running on the instance.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that is running on the instance.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * <p>
         * Security credentials that are required to access the instance.
         * </p>
         * 
         * @param credentials
         *        Security credentials that are required to access the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credentials(InstanceCredentials credentials);

        /**
         * <p>
         * Security credentials that are required to access the instance.
         * </p>
         * This is a convenience method that creates an instance of the {@link InstanceCredentials.Builder} avoiding the
         * need to create one manually via {@link InstanceCredentials#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceCredentials.Builder#build()} is called immediately and
         * its result is passed to {@link #credentials(InstanceCredentials)}.
         * 
         * @param credentials
         *        a consumer that will call methods on {@link InstanceCredentials.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #credentials(InstanceCredentials)
         */
        default Builder credentials(Consumer<InstanceCredentials.Builder> credentials) {
            return credentials(InstanceCredentials.builder().applyMutation(credentials).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String instanceId;

        private String ipAddress;

        private String operatingSystem;

        private InstanceCredentials credentials;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceAccess model) {
            fleetId(model.fleetId);
            instanceId(model.instanceId);
            ipAddress(model.ipAddress);
            operatingSystem(model.operatingSystem);
            credentials(model.credentials);
        }

        public final String getFleetId() {
            return fleetId;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final InstanceCredentials.Builder getCredentials() {
            return credentials != null ? credentials.toBuilder() : null;
        }

        public final void setCredentials(InstanceCredentials.BuilderImpl credentials) {
            this.credentials = credentials != null ? credentials.build() : null;
        }

        @Override
        public final Builder credentials(InstanceCredentials credentials) {
            this.credentials = credentials;
            return this;
        }

        @Override
        public InstanceAccess build() {
            return new InstanceAccess(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
