/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Instructions on when and how to check the health of a support container in a container fleet. These properties
 * override any Docker health checks that are set in the container image. For more information on container health
 * checks, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html#ECS-Type-HealthCheck-command"
 * >HealthCheck command</a> in the <i>Amazon Elastic Container Service API</i>. Game server containers don't have a
 * health check parameter; Amazon GameLift automatically handles health checks for these containers.
 * </p>
 * <p>
 * The following example instructs the container to initiate a health check command every 60 seconds and wait 10 seconds
 * for it to succeed. If it fails, retry the command 3 times before flagging the container as unhealthy. It also tells
 * the container to wait 100 seconds after launch before counting failed health checks.
 * </p>
 * <p>
 * <code>{"Command": [ "CMD-SHELL", "ps cax | grep "processmanager" || exit 1" ], "Interval": 60, "Timeout": 10, "Retries": 3, "StartPeriod": 100 }</code>
 * </p>
 * <p>
 * <b>Part of:</b> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_SupportContainerDefinition.html"
 * >SupportContainerDefinition</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_SupportContainerDefinitionInput.html"
 * >SupportContainerDefinitionInput</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerHealthCheck implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerHealthCheck.Builder, ContainerHealthCheck> {
    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Command")
            .getter(getter(ContainerHealthCheck::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Interval").getter(getter(ContainerHealthCheck::interval)).setter(setter(Builder::interval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Interval").build()).build();

    private static final SdkField<Integer> RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Retries").getter(getter(ContainerHealthCheck::retries)).setter(setter(Builder::retries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Retries").build()).build();

    private static final SdkField<Integer> START_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StartPeriod").getter(getter(ContainerHealthCheck::startPeriod)).setter(setter(Builder::startPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartPeriod").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Timeout").getter(getter(ContainerHealthCheck::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timeout").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMAND_FIELD, INTERVAL_FIELD,
            RETRIES_FIELD, START_PERIOD_FIELD, TIMEOUT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> command;

    private final Integer interval;

    private final Integer retries;

    private final Integer startPeriod;

    private final Integer timeout;

    private ContainerHealthCheck(BuilderImpl builder) {
        this.command = builder.command;
        this.interval = builder.interval;
        this.retries = builder.retries;
        this.startPeriod = builder.startPeriod;
        this.timeout = builder.timeout;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A string array that specifies the command that the container runs to determine if it's healthy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return A string array that specifies the command that the container runs to determine if it's healthy.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * <p>
     * The time period (in seconds) between each health check.
     * </p>
     * 
     * @return The time period (in seconds) between each health check.
     */
    public final Integer interval() {
        return interval;
    }

    /**
     * <p>
     * The number of times to retry a failed health check before flagging the container unhealthy. The first run of the
     * command does not count as a retry.
     * </p>
     * 
     * @return The number of times to retry a failed health check before flagging the container unhealthy. The first run
     *         of the command does not count as a retry.
     */
    public final Integer retries() {
        return retries;
    }

    /**
     * <p>
     * The optional grace period (in seconds) to give a container time to bootstrap before the first failed health check
     * counts toward the number of retries.
     * </p>
     * 
     * @return The optional grace period (in seconds) to give a container time to bootstrap before the first failed
     *         health check counts toward the number of retries.
     */
    public final Integer startPeriod() {
        return startPeriod;
    }

    /**
     * <p>
     * The time period (in seconds) to wait for a health check to succeed before counting a failed health check.
     * </p>
     * 
     * @return The time period (in seconds) to wait for a health check to succeed before counting a failed health check.
     */
    public final Integer timeout() {
        return timeout;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(interval());
        hashCode = 31 * hashCode + Objects.hashCode(retries());
        hashCode = 31 * hashCode + Objects.hashCode(startPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerHealthCheck)) {
            return false;
        }
        ContainerHealthCheck other = (ContainerHealthCheck) obj;
        return hasCommand() == other.hasCommand() && Objects.equals(command(), other.command())
                && Objects.equals(interval(), other.interval()) && Objects.equals(retries(), other.retries())
                && Objects.equals(startPeriod(), other.startPeriod()) && Objects.equals(timeout(), other.timeout());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerHealthCheck").add("Command", hasCommand() ? command() : null)
                .add("Interval", interval()).add("Retries", retries()).add("StartPeriod", startPeriod())
                .add("Timeout", timeout()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Command":
            return Optional.ofNullable(clazz.cast(command()));
        case "Interval":
            return Optional.ofNullable(clazz.cast(interval()));
        case "Retries":
            return Optional.ofNullable(clazz.cast(retries()));
        case "StartPeriod":
            return Optional.ofNullable(clazz.cast(startPeriod()));
        case "Timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Command", COMMAND_FIELD);
        map.put("Interval", INTERVAL_FIELD);
        map.put("Retries", RETRIES_FIELD);
        map.put("StartPeriod", START_PERIOD_FIELD);
        map.put("Timeout", TIMEOUT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ContainerHealthCheck, T> g) {
        return obj -> g.apply((ContainerHealthCheck) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerHealthCheck> {
        /**
         * <p>
         * A string array that specifies the command that the container runs to determine if it's healthy.
         * </p>
         * 
         * @param command
         *        A string array that specifies the command that the container runs to determine if it's healthy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * A string array that specifies the command that the container runs to determine if it's healthy.
         * </p>
         * 
         * @param command
         *        A string array that specifies the command that the container runs to determine if it's healthy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The time period (in seconds) between each health check.
         * </p>
         * 
         * @param interval
         *        The time period (in seconds) between each health check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interval(Integer interval);

        /**
         * <p>
         * The number of times to retry a failed health check before flagging the container unhealthy. The first run of
         * the command does not count as a retry.
         * </p>
         * 
         * @param retries
         *        The number of times to retry a failed health check before flagging the container unhealthy. The first
         *        run of the command does not count as a retry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retries(Integer retries);

        /**
         * <p>
         * The optional grace period (in seconds) to give a container time to bootstrap before the first failed health
         * check counts toward the number of retries.
         * </p>
         * 
         * @param startPeriod
         *        The optional grace period (in seconds) to give a container time to bootstrap before the first failed
         *        health check counts toward the number of retries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startPeriod(Integer startPeriod);

        /**
         * <p>
         * The time period (in seconds) to wait for a health check to succeed before counting a failed health check.
         * </p>
         * 
         * @param timeout
         *        The time period (in seconds) to wait for a health check to succeed before counting a failed health
         *        check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);
    }

    static final class BuilderImpl implements Builder {
        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private Integer interval;

        private Integer retries;

        private Integer startPeriod;

        private Integer timeout;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerHealthCheck model) {
            command(model.command);
            interval(model.interval);
            retries(model.retries);
            startPeriod(model.startPeriod);
            timeout(model.timeout);
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = ContainerCommandStringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = ContainerCommandStringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final Integer getInterval() {
            return interval;
        }

        public final void setInterval(Integer interval) {
            this.interval = interval;
        }

        @Override
        public final Builder interval(Integer interval) {
            this.interval = interval;
            return this;
        }

        public final Integer getRetries() {
            return retries;
        }

        public final void setRetries(Integer retries) {
            this.retries = retries;
        }

        @Override
        public final Builder retries(Integer retries) {
            this.retries = retries;
            return this;
        }

        public final Integer getStartPeriod() {
            return startPeriod;
        }

        public final void setStartPeriod(Integer startPeriod) {
            this.startPeriod = startPeriod;
        }

        @Override
        public final Builder startPeriod(Integer startPeriod) {
            this.startPeriod = startPeriod;
            return this;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        @Override
        public ContainerHealthCheck build() {
            return new ContainerHealthCheck(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
