/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon GameLift managed container fleet.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerFleet implements SdkPojo, Serializable, ToCopyableBuilder<ContainerFleet.Builder, ContainerFleet> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(ContainerFleet::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> FLEET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetArn").getter(getter(ContainerFleet::fleetArn)).setter(setter(Builder::fleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetArn").build()).build();

    private static final SdkField<String> FLEET_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetRoleArn").getter(getter(ContainerFleet::fleetRoleArn)).setter(setter(Builder::fleetRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetRoleArn").build()).build();

    private static final SdkField<String> GAME_SERVER_CONTAINER_GROUP_DEFINITION_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GameServerContainerGroupDefinitionName")
            .getter(getter(ContainerFleet::gameServerContainerGroupDefinitionName))
            .setter(setter(Builder::gameServerContainerGroupDefinitionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("GameServerContainerGroupDefinitionName").build()).build();

    private static final SdkField<String> GAME_SERVER_CONTAINER_GROUP_DEFINITION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GameServerContainerGroupDefinitionArn")
            .getter(getter(ContainerFleet::gameServerContainerGroupDefinitionArn))
            .setter(setter(Builder::gameServerContainerGroupDefinitionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("GameServerContainerGroupDefinitionArn").build()).build();

    private static final SdkField<String> PER_INSTANCE_CONTAINER_GROUP_DEFINITION_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PerInstanceContainerGroupDefinitionName")
            .getter(getter(ContainerFleet::perInstanceContainerGroupDefinitionName))
            .setter(setter(Builder::perInstanceContainerGroupDefinitionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PerInstanceContainerGroupDefinitionName").build()).build();

    private static final SdkField<String> PER_INSTANCE_CONTAINER_GROUP_DEFINITION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PerInstanceContainerGroupDefinitionArn")
            .getter(getter(ContainerFleet::perInstanceContainerGroupDefinitionArn))
            .setter(setter(Builder::perInstanceContainerGroupDefinitionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PerInstanceContainerGroupDefinitionArn").build()).build();

    private static final SdkField<ConnectionPortRange> INSTANCE_CONNECTION_PORT_RANGE_FIELD = SdkField
            .<ConnectionPortRange> builder(MarshallingType.SDK_POJO)
            .memberName("InstanceConnectionPortRange")
            .getter(getter(ContainerFleet::instanceConnectionPortRange))
            .setter(setter(Builder::instanceConnectionPortRange))
            .constructor(ConnectionPortRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceConnectionPortRange")
                    .build()).build();

    private static final SdkField<List<IpPermission>> INSTANCE_INBOUND_PERMISSIONS_FIELD = SdkField
            .<List<IpPermission>> builder(MarshallingType.LIST)
            .memberName("InstanceInboundPermissions")
            .getter(getter(ContainerFleet::instanceInboundPermissions))
            .setter(setter(Builder::instanceInboundPermissions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceInboundPermissions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<IpPermission> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpPermission::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> GAME_SERVER_CONTAINER_GROUPS_PER_INSTANCE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("GameServerContainerGroupsPerInstance")
            .getter(getter(ContainerFleet::gameServerContainerGroupsPerInstance))
            .setter(setter(Builder::gameServerContainerGroupsPerInstance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("GameServerContainerGroupsPerInstance").build()).build();

    private static final SdkField<Integer> MAXIMUM_GAME_SERVER_CONTAINER_GROUPS_PER_INSTANCE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumGameServerContainerGroupsPerInstance")
            .getter(getter(ContainerFleet::maximumGameServerContainerGroupsPerInstance))
            .setter(setter(Builder::maximumGameServerContainerGroupsPerInstance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("MaximumGameServerContainerGroupsPerInstance").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(ContainerFleet::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> BILLING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BillingType").getter(getter(ContainerFleet::billingTypeAsString)).setter(setter(Builder::billingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BillingType").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ContainerFleet::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(ContainerFleet::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<List<String>> METRIC_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MetricGroups")
            .getter(getter(ContainerFleet::metricGroups))
            .setter(setter(Builder::metricGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEW_GAME_SESSION_PROTECTION_POLICY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NewGameSessionProtectionPolicy")
            .getter(getter(ContainerFleet::newGameSessionProtectionPolicyAsString))
            .setter(setter(Builder::newGameSessionProtectionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewGameSessionProtectionPolicy")
                    .build()).build();

    private static final SdkField<GameSessionCreationLimitPolicy> GAME_SESSION_CREATION_LIMIT_POLICY_FIELD = SdkField
            .<GameSessionCreationLimitPolicy> builder(MarshallingType.SDK_POJO)
            .memberName("GameSessionCreationLimitPolicy")
            .getter(getter(ContainerFleet::gameSessionCreationLimitPolicy))
            .setter(setter(Builder::gameSessionCreationLimitPolicy))
            .constructor(GameSessionCreationLimitPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionCreationLimitPolicy")
                    .build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ContainerFleet::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<DeploymentDetails> DEPLOYMENT_DETAILS_FIELD = SdkField
            .<DeploymentDetails> builder(MarshallingType.SDK_POJO).memberName("DeploymentDetails")
            .getter(getter(ContainerFleet::deploymentDetails)).setter(setter(Builder::deploymentDetails))
            .constructor(DeploymentDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentDetails").build()).build();

    private static final SdkField<LogConfiguration> LOG_CONFIGURATION_FIELD = SdkField
            .<LogConfiguration> builder(MarshallingType.SDK_POJO).memberName("LogConfiguration")
            .getter(getter(ContainerFleet::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(LogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogConfiguration").build()).build();

    private static final SdkField<List<ContainerFleetLocationAttributes>> LOCATION_ATTRIBUTES_FIELD = SdkField
            .<List<ContainerFleetLocationAttributes>> builder(MarshallingType.LIST)
            .memberName("LocationAttributes")
            .getter(getter(ContainerFleet::locationAttributes))
            .setter(setter(Builder::locationAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerFleetLocationAttributes> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerFleetLocationAttributes::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD,
            FLEET_ARN_FIELD, FLEET_ROLE_ARN_FIELD, GAME_SERVER_CONTAINER_GROUP_DEFINITION_NAME_FIELD,
            GAME_SERVER_CONTAINER_GROUP_DEFINITION_ARN_FIELD, PER_INSTANCE_CONTAINER_GROUP_DEFINITION_NAME_FIELD,
            PER_INSTANCE_CONTAINER_GROUP_DEFINITION_ARN_FIELD, INSTANCE_CONNECTION_PORT_RANGE_FIELD,
            INSTANCE_INBOUND_PERMISSIONS_FIELD, GAME_SERVER_CONTAINER_GROUPS_PER_INSTANCE_FIELD,
            MAXIMUM_GAME_SERVER_CONTAINER_GROUPS_PER_INSTANCE_FIELD, INSTANCE_TYPE_FIELD, BILLING_TYPE_FIELD, DESCRIPTION_FIELD,
            CREATION_TIME_FIELD, METRIC_GROUPS_FIELD, NEW_GAME_SESSION_PROTECTION_POLICY_FIELD,
            GAME_SESSION_CREATION_LIMIT_POLICY_FIELD, STATUS_FIELD, DEPLOYMENT_DETAILS_FIELD, LOG_CONFIGURATION_FIELD,
            LOCATION_ATTRIBUTES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String fleetArn;

    private final String fleetRoleArn;

    private final String gameServerContainerGroupDefinitionName;

    private final String gameServerContainerGroupDefinitionArn;

    private final String perInstanceContainerGroupDefinitionName;

    private final String perInstanceContainerGroupDefinitionArn;

    private final ConnectionPortRange instanceConnectionPortRange;

    private final List<IpPermission> instanceInboundPermissions;

    private final Integer gameServerContainerGroupsPerInstance;

    private final Integer maximumGameServerContainerGroupsPerInstance;

    private final String instanceType;

    private final String billingType;

    private final String description;

    private final Instant creationTime;

    private final List<String> metricGroups;

    private final String newGameSessionProtectionPolicy;

    private final GameSessionCreationLimitPolicy gameSessionCreationLimitPolicy;

    private final String status;

    private final DeploymentDetails deploymentDetails;

    private final LogConfiguration logConfiguration;

    private final List<ContainerFleetLocationAttributes> locationAttributes;

    private ContainerFleet(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.fleetArn = builder.fleetArn;
        this.fleetRoleArn = builder.fleetRoleArn;
        this.gameServerContainerGroupDefinitionName = builder.gameServerContainerGroupDefinitionName;
        this.gameServerContainerGroupDefinitionArn = builder.gameServerContainerGroupDefinitionArn;
        this.perInstanceContainerGroupDefinitionName = builder.perInstanceContainerGroupDefinitionName;
        this.perInstanceContainerGroupDefinitionArn = builder.perInstanceContainerGroupDefinitionArn;
        this.instanceConnectionPortRange = builder.instanceConnectionPortRange;
        this.instanceInboundPermissions = builder.instanceInboundPermissions;
        this.gameServerContainerGroupsPerInstance = builder.gameServerContainerGroupsPerInstance;
        this.maximumGameServerContainerGroupsPerInstance = builder.maximumGameServerContainerGroupsPerInstance;
        this.instanceType = builder.instanceType;
        this.billingType = builder.billingType;
        this.description = builder.description;
        this.creationTime = builder.creationTime;
        this.metricGroups = builder.metricGroups;
        this.newGameSessionProtectionPolicy = builder.newGameSessionProtectionPolicy;
        this.gameSessionCreationLimitPolicy = builder.gameSessionCreationLimitPolicy;
        this.status = builder.status;
        this.deploymentDetails = builder.deploymentDetails;
        this.logConfiguration = builder.logConfiguration;
        this.locationAttributes = builder.locationAttributes;
    }

    /**
     * <p>
     * A unique identifier for the container fleet to retrieve.
     * </p>
     * 
     * @return A unique identifier for the container fleet to retrieve.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to a Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all
     * Regions. Format is <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>
     * . In a GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
     *         Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
     *         GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
     */
    public final String fleetArn() {
        return fleetArn;
    }

    /**
     * <p>
     * The unique identifier for an Identity and Access Management (IAM) role with permissions to run your containers on
     * resources that are managed by Amazon GameLift. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/setting-up-role.html">Set up an IAM service
     * role</a>. This fleet property can't be changed.
     * </p>
     * 
     * @return The unique identifier for an Identity and Access Management (IAM) role with permissions to run your
     *         containers on resources that are managed by Amazon GameLift. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/setting-up-role.html">Set up an IAM
     *         service role</a>. This fleet property can't be changed.
     */
    public final String fleetRoleArn() {
        return fleetRoleArn;
    }

    /**
     * <p>
     * The name of the fleet's game server container group definition, which describes how to deploy containers with
     * your game server build and support software onto each fleet instance.
     * </p>
     * 
     * @return The name of the fleet's game server container group definition, which describes how to deploy containers
     *         with your game server build and support software onto each fleet instance.
     */
    public final String gameServerContainerGroupDefinitionName() {
        return gameServerContainerGroupDefinitionName;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to the fleet's game server container group. The ARN value also identifies the specific container
     * group definition version in use.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
     *         the fleet's game server container group. The ARN value also identifies the specific container group
     *         definition version in use.
     */
    public final String gameServerContainerGroupDefinitionArn() {
        return gameServerContainerGroupDefinitionArn;
    }

    /**
     * <p>
     * The name of the fleet's per-instance container group definition.
     * </p>
     * 
     * @return The name of the fleet's per-instance container group definition.
     */
    public final String perInstanceContainerGroupDefinitionName() {
        return perInstanceContainerGroupDefinitionName;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to the fleet's per-instance container group. The ARN value also identifies the specific
     * container group definition version in use.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
     *         the fleet's per-instance container group. The ARN value also identifies the specific container group
     *         definition version in use.
     */
    public final String perInstanceContainerGroupDefinitionArn() {
        return perInstanceContainerGroupDefinitionArn;
    }

    /**
     * Returns the value of the InstanceConnectionPortRange property for this object.
     * 
     * @return The value of the InstanceConnectionPortRange property for this object.
     */
    public final ConnectionPortRange instanceConnectionPortRange() {
        return instanceConnectionPortRange;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceInboundPermissions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInstanceInboundPermissions() {
        return instanceInboundPermissions != null && !(instanceInboundPermissions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP address ranges and port settings that allow inbound traffic to access game server processes and other
     * processes on this fleet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceInboundPermissions} method.
     * </p>
     * 
     * @return The IP address ranges and port settings that allow inbound traffic to access game server processes and
     *         other processes on this fleet.
     */
    public final List<IpPermission> instanceInboundPermissions() {
        return instanceInboundPermissions;
    }

    /**
     * <p>
     * The number of times to replicate the game server container group on each fleet instance.
     * </p>
     * 
     * @return The number of times to replicate the game server container group on each fleet instance.
     */
    public final Integer gameServerContainerGroupsPerInstance() {
        return gameServerContainerGroupsPerInstance;
    }

    /**
     * <p>
     * The calculated maximum number of game server container group that can be deployed on each fleet instance. The
     * calculation depends on the resource needs of the container group and the CPU and memory resources of the fleet's
     * instance type.
     * </p>
     * 
     * @return The calculated maximum number of game server container group that can be deployed on each fleet instance.
     *         The calculation depends on the resource needs of the container group and the CPU and memory resources of
     *         the fleet's instance type.
     */
    public final Integer maximumGameServerContainerGroupsPerInstance() {
        return maximumGameServerContainerGroupsPerInstance;
    }

    /**
     * <p>
     * The Amazon EC2 instance type to use for all instances in the fleet. Instance type determines the computing
     * resources and processing power that's available to host your game servers. This includes including CPU, memory,
     * storage, and networking capacity. You can't update this fleet property.
     * </p>
     * 
     * @return The Amazon EC2 instance type to use for all instances in the fleet. Instance type determines the
     *         computing resources and processing power that's available to host your game servers. This includes
     *         including CPU, memory, storage, and networking capacity. You can't update this fleet property.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to use <a
     * href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     * > On-Demand versus Spot Instances</a>. You can't update this fleet property.
     * </p>
     * <p>
     * By default, this property is set to <code>ON_DEMAND</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #billingType} will
     * return {@link ContainerFleetBillingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #billingTypeAsString}.
     * </p>
     * 
     * @return Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to use
     *         <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     *         > On-Demand versus Spot Instances</a>. You can't update this fleet property.</p>
     *         <p>
     *         By default, this property is set to <code>ON_DEMAND</code>.
     * @see ContainerFleetBillingType
     */
    public final ContainerFleetBillingType billingType() {
        return ContainerFleetBillingType.fromValue(billingType);
    }

    /**
     * <p>
     * Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to use <a
     * href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     * > On-Demand versus Spot Instances</a>. You can't update this fleet property.
     * </p>
     * <p>
     * By default, this property is set to <code>ON_DEMAND</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #billingType} will
     * return {@link ContainerFleetBillingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #billingTypeAsString}.
     * </p>
     * 
     * @return Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to use
     *         <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     *         > On-Demand versus Spot Instances</a>. You can't update this fleet property.</p>
     *         <p>
     *         By default, this property is set to <code>ON_DEMAND</code>.
     * @see ContainerFleetBillingType
     */
    public final String billingTypeAsString() {
        return billingType;
    }

    /**
     * <p>
     * A meaningful description of the container fleet.
     * </p>
     * 
     * @return A meaningful description of the container fleet.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricGroups() {
        return metricGroups != null && !(metricGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of an Amazon Web Services CloudWatch metric group to add this fleet to. Metric groups aggregate metrics
     * for multiple fleets.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricGroups} method.
     * </p>
     * 
     * @return The name of an Amazon Web Services CloudWatch metric group to add this fleet to. Metric groups aggregate
     *         metrics for multiple fleets.
     */
    public final List<String> metricGroups() {
        return metricGroups;
    }

    /**
     * <p>
     * Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running and hosting
     * players. Amazon GameLift might prompt an instance shutdown when scaling down fleet capacity or when retiring
     * unhealthy instances. You can also set game session protection for individual game sessions using <a
     * href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #newGameSessionProtectionPolicy} will return {@link ProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #newGameSessionProtectionPolicyAsString}.
     * </p>
     * 
     * @return Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running and
     *         hosting players. Amazon GameLift might prompt an instance shutdown when scaling down fleet capacity or
     *         when retiring unhealthy instances. You can also set game session protection for individual game sessions
     *         using <a href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
     *         </p>
     *         </li>
     * @see ProtectionPolicy
     */
    public final ProtectionPolicy newGameSessionProtectionPolicy() {
        return ProtectionPolicy.fromValue(newGameSessionProtectionPolicy);
    }

    /**
     * <p>
     * Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running and hosting
     * players. Amazon GameLift might prompt an instance shutdown when scaling down fleet capacity or when retiring
     * unhealthy instances. You can also set game session protection for individual game sessions using <a
     * href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #newGameSessionProtectionPolicy} will return {@link ProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #newGameSessionProtectionPolicyAsString}.
     * </p>
     * 
     * @return Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running and
     *         hosting players. Amazon GameLift might prompt an instance shutdown when scaling down fleet capacity or
     *         when retiring unhealthy instances. You can also set game session protection for individual game sessions
     *         using <a href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
     *         </p>
     *         </li>
     * @see ProtectionPolicy
     */
    public final String newGameSessionProtectionPolicyAsString() {
        return newGameSessionProtectionPolicy;
    }

    /**
     * <p>
     * A policy that limits the number of game sessions that each individual player can create on instances in this
     * fleet. The limit applies for a specified span of time.
     * </p>
     * 
     * @return A policy that limits the number of game sessions that each individual player can create on instances in
     *         this fleet. The limit applies for a specified span of time.
     */
    public final GameSessionCreationLimitPolicy gameSessionCreationLimitPolicy() {
        return gameSessionCreationLimitPolicy;
    }

    /**
     * <p>
     * The current status of the container fleet.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> -- A new container fleet has been requested.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> -- A new container fleet resource is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been deployed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVATING</code> -- New container fleet instances are being deployed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContainerFleetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the container fleet.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> -- A new container fleet has been requested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> -- A new container fleet resource is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been
     *         deployed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVATING</code> -- New container fleet instances are being deployed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
     *         </p>
     *         </li>
     * @see ContainerFleetStatus
     */
    public final ContainerFleetStatus status() {
        return ContainerFleetStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the container fleet.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> -- A new container fleet has been requested.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> -- A new container fleet resource is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been deployed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVATING</code> -- New container fleet instances are being deployed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContainerFleetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the container fleet.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> -- A new container fleet has been requested.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> -- A new container fleet resource is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been
     *         deployed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVATING</code> -- New container fleet instances are being deployed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
     *         </p>
     *         </li>
     * @see ContainerFleetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Information about the most recent deployment for the container fleet.
     * </p>
     * 
     * @return Information about the most recent deployment for the container fleet.
     */
    public final DeploymentDetails deploymentDetails() {
        return deploymentDetails;
    }

    /**
     * <p>
     * The method that is used to collect container logs for the fleet. Amazon GameLift saves all standard output for
     * each container in logs, including game session logs.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CLOUDWATCH</code> -- Send logs to an Amazon CloudWatch log group that you define. Each container emits a
     * log stream, which is organized in the log group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code> -- Store logs in an Amazon S3 bucket that you define.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code> -- Don't collect container logs.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The method that is used to collect container logs for the fleet. Amazon GameLift saves all standard
     *         output for each container in logs, including game session logs. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CLOUDWATCH</code> -- Send logs to an Amazon CloudWatch log group that you define. Each container
     *         emits a log stream, which is organized in the log group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code> -- Store logs in an Amazon S3 bucket that you define.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code> -- Don't collect container logs.
     *         </p>
     *         </li>
     */
    public final LogConfiguration logConfiguration() {
        return logConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the LocationAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLocationAttributes() {
        return locationAttributes != null && !(locationAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the container fleet's remote locations where fleet instances are deployed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocationAttributes} method.
     * </p>
     * 
     * @return Information about the container fleet's remote locations where fleet instances are deployed.
     */
    public final List<ContainerFleetLocationAttributes> locationAttributes() {
        return locationAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(fleetRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerContainerGroupDefinitionName());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerContainerGroupDefinitionArn());
        hashCode = 31 * hashCode + Objects.hashCode(perInstanceContainerGroupDefinitionName());
        hashCode = 31 * hashCode + Objects.hashCode(perInstanceContainerGroupDefinitionArn());
        hashCode = 31 * hashCode + Objects.hashCode(instanceConnectionPortRange());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceInboundPermissions() ? instanceInboundPermissions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(gameServerContainerGroupsPerInstance());
        hashCode = 31 * hashCode + Objects.hashCode(maximumGameServerContainerGroupsPerInstance());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(billingTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricGroups() ? metricGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(newGameSessionProtectionPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionCreationLimitPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentDetails());
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasLocationAttributes() ? locationAttributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerFleet)) {
            return false;
        }
        ContainerFleet other = (ContainerFleet) obj;
        return Objects.equals(fleetId(), other.fleetId())
                && Objects.equals(fleetArn(), other.fleetArn())
                && Objects.equals(fleetRoleArn(), other.fleetRoleArn())
                && Objects.equals(gameServerContainerGroupDefinitionName(), other.gameServerContainerGroupDefinitionName())
                && Objects.equals(gameServerContainerGroupDefinitionArn(), other.gameServerContainerGroupDefinitionArn())
                && Objects.equals(perInstanceContainerGroupDefinitionName(), other.perInstanceContainerGroupDefinitionName())
                && Objects.equals(perInstanceContainerGroupDefinitionArn(), other.perInstanceContainerGroupDefinitionArn())
                && Objects.equals(instanceConnectionPortRange(), other.instanceConnectionPortRange())
                && hasInstanceInboundPermissions() == other.hasInstanceInboundPermissions()
                && Objects.equals(instanceInboundPermissions(), other.instanceInboundPermissions())
                && Objects.equals(gameServerContainerGroupsPerInstance(), other.gameServerContainerGroupsPerInstance())
                && Objects.equals(maximumGameServerContainerGroupsPerInstance(),
                        other.maximumGameServerContainerGroupsPerInstance())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(billingTypeAsString(), other.billingTypeAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(creationTime(), other.creationTime())
                && hasMetricGroups() == other.hasMetricGroups() && Objects.equals(metricGroups(), other.metricGroups())
                && Objects.equals(newGameSessionProtectionPolicyAsString(), other.newGameSessionProtectionPolicyAsString())
                && Objects.equals(gameSessionCreationLimitPolicy(), other.gameSessionCreationLimitPolicy())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(deploymentDetails(), other.deploymentDetails())
                && Objects.equals(logConfiguration(), other.logConfiguration())
                && hasLocationAttributes() == other.hasLocationAttributes()
                && Objects.equals(locationAttributes(), other.locationAttributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerFleet").add("FleetId", fleetId()).add("FleetArn", fleetArn())
                .add("FleetRoleArn", fleetRoleArn())
                .add("GameServerContainerGroupDefinitionName", gameServerContainerGroupDefinitionName())
                .add("GameServerContainerGroupDefinitionArn", gameServerContainerGroupDefinitionArn())
                .add("PerInstanceContainerGroupDefinitionName", perInstanceContainerGroupDefinitionName())
                .add("PerInstanceContainerGroupDefinitionArn", perInstanceContainerGroupDefinitionArn())
                .add("InstanceConnectionPortRange", instanceConnectionPortRange())
                .add("InstanceInboundPermissions", hasInstanceInboundPermissions() ? instanceInboundPermissions() : null)
                .add("GameServerContainerGroupsPerInstance", gameServerContainerGroupsPerInstance())
                .add("MaximumGameServerContainerGroupsPerInstance", maximumGameServerContainerGroupsPerInstance())
                .add("InstanceType", instanceType()).add("BillingType", billingTypeAsString()).add("Description", description())
                .add("CreationTime", creationTime()).add("MetricGroups", hasMetricGroups() ? metricGroups() : null)
                .add("NewGameSessionProtectionPolicy", newGameSessionProtectionPolicyAsString())
                .add("GameSessionCreationLimitPolicy", gameSessionCreationLimitPolicy()).add("Status", statusAsString())
                .add("DeploymentDetails", deploymentDetails()).add("LogConfiguration", logConfiguration())
                .add("LocationAttributes", hasLocationAttributes() ? locationAttributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "FleetArn":
            return Optional.ofNullable(clazz.cast(fleetArn()));
        case "FleetRoleArn":
            return Optional.ofNullable(clazz.cast(fleetRoleArn()));
        case "GameServerContainerGroupDefinitionName":
            return Optional.ofNullable(clazz.cast(gameServerContainerGroupDefinitionName()));
        case "GameServerContainerGroupDefinitionArn":
            return Optional.ofNullable(clazz.cast(gameServerContainerGroupDefinitionArn()));
        case "PerInstanceContainerGroupDefinitionName":
            return Optional.ofNullable(clazz.cast(perInstanceContainerGroupDefinitionName()));
        case "PerInstanceContainerGroupDefinitionArn":
            return Optional.ofNullable(clazz.cast(perInstanceContainerGroupDefinitionArn()));
        case "InstanceConnectionPortRange":
            return Optional.ofNullable(clazz.cast(instanceConnectionPortRange()));
        case "InstanceInboundPermissions":
            return Optional.ofNullable(clazz.cast(instanceInboundPermissions()));
        case "GameServerContainerGroupsPerInstance":
            return Optional.ofNullable(clazz.cast(gameServerContainerGroupsPerInstance()));
        case "MaximumGameServerContainerGroupsPerInstance":
            return Optional.ofNullable(clazz.cast(maximumGameServerContainerGroupsPerInstance()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "BillingType":
            return Optional.ofNullable(clazz.cast(billingTypeAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "MetricGroups":
            return Optional.ofNullable(clazz.cast(metricGroups()));
        case "NewGameSessionProtectionPolicy":
            return Optional.ofNullable(clazz.cast(newGameSessionProtectionPolicyAsString()));
        case "GameSessionCreationLimitPolicy":
            return Optional.ofNullable(clazz.cast(gameSessionCreationLimitPolicy()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "DeploymentDetails":
            return Optional.ofNullable(clazz.cast(deploymentDetails()));
        case "LogConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        case "LocationAttributes":
            return Optional.ofNullable(clazz.cast(locationAttributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FleetId", FLEET_ID_FIELD);
        map.put("FleetArn", FLEET_ARN_FIELD);
        map.put("FleetRoleArn", FLEET_ROLE_ARN_FIELD);
        map.put("GameServerContainerGroupDefinitionName", GAME_SERVER_CONTAINER_GROUP_DEFINITION_NAME_FIELD);
        map.put("GameServerContainerGroupDefinitionArn", GAME_SERVER_CONTAINER_GROUP_DEFINITION_ARN_FIELD);
        map.put("PerInstanceContainerGroupDefinitionName", PER_INSTANCE_CONTAINER_GROUP_DEFINITION_NAME_FIELD);
        map.put("PerInstanceContainerGroupDefinitionArn", PER_INSTANCE_CONTAINER_GROUP_DEFINITION_ARN_FIELD);
        map.put("InstanceConnectionPortRange", INSTANCE_CONNECTION_PORT_RANGE_FIELD);
        map.put("InstanceInboundPermissions", INSTANCE_INBOUND_PERMISSIONS_FIELD);
        map.put("GameServerContainerGroupsPerInstance", GAME_SERVER_CONTAINER_GROUPS_PER_INSTANCE_FIELD);
        map.put("MaximumGameServerContainerGroupsPerInstance", MAXIMUM_GAME_SERVER_CONTAINER_GROUPS_PER_INSTANCE_FIELD);
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("BillingType", BILLING_TYPE_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("MetricGroups", METRIC_GROUPS_FIELD);
        map.put("NewGameSessionProtectionPolicy", NEW_GAME_SESSION_PROTECTION_POLICY_FIELD);
        map.put("GameSessionCreationLimitPolicy", GAME_SESSION_CREATION_LIMIT_POLICY_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("DeploymentDetails", DEPLOYMENT_DETAILS_FIELD);
        map.put("LogConfiguration", LOG_CONFIGURATION_FIELD);
        map.put("LocationAttributes", LOCATION_ATTRIBUTES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ContainerFleet, T> g) {
        return obj -> g.apply((ContainerFleet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerFleet> {
        /**
         * <p>
         * A unique identifier for the container fleet to retrieve.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for the container fleet to retrieve.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a GameLift
         * fleet ARN, the resource ID matches the <code>FleetId</code> value.
         * </p>
         * 
         * @param fleetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions.
         *        Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
         *        GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetArn(String fleetArn);

        /**
         * <p>
         * The unique identifier for an Identity and Access Management (IAM) role with permissions to run your
         * containers on resources that are managed by Amazon GameLift. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/setting-up-role.html">Set up an IAM service
         * role</a>. This fleet property can't be changed.
         * </p>
         * 
         * @param fleetRoleArn
         *        The unique identifier for an Identity and Access Management (IAM) role with permissions to run your
         *        containers on resources that are managed by Amazon GameLift. See <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/setting-up-role.html">Set up an IAM
         *        service role</a>. This fleet property can't be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetRoleArn(String fleetRoleArn);

        /**
         * <p>
         * The name of the fleet's game server container group definition, which describes how to deploy containers with
         * your game server build and support software onto each fleet instance.
         * </p>
         * 
         * @param gameServerContainerGroupDefinitionName
         *        The name of the fleet's game server container group definition, which describes how to deploy
         *        containers with your game server build and support software onto each fleet instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerContainerGroupDefinitionName(String gameServerContainerGroupDefinitionName);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to the
         * fleet's game server container group. The ARN value also identifies the specific container group definition
         * version in use.
         * </p>
         * 
         * @param gameServerContainerGroupDefinitionArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        the fleet's game server container group. The ARN value also identifies the specific container group
         *        definition version in use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerContainerGroupDefinitionArn(String gameServerContainerGroupDefinitionArn);

        /**
         * <p>
         * The name of the fleet's per-instance container group definition.
         * </p>
         * 
         * @param perInstanceContainerGroupDefinitionName
         *        The name of the fleet's per-instance container group definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder perInstanceContainerGroupDefinitionName(String perInstanceContainerGroupDefinitionName);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to the
         * fleet's per-instance container group. The ARN value also identifies the specific container group definition
         * version in use.
         * </p>
         * 
         * @param perInstanceContainerGroupDefinitionArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        the fleet's per-instance container group. The ARN value also identifies the specific container group
         *        definition version in use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder perInstanceContainerGroupDefinitionArn(String perInstanceContainerGroupDefinitionArn);

        /**
         * Sets the value of the InstanceConnectionPortRange property for this object.
         *
         * @param instanceConnectionPortRange
         *        The new value for the InstanceConnectionPortRange property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceConnectionPortRange(ConnectionPortRange instanceConnectionPortRange);

        /**
         * Sets the value of the InstanceConnectionPortRange property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ConnectionPortRange.Builder} avoiding the
         * need to create one manually via {@link ConnectionPortRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConnectionPortRange.Builder#build()} is called immediately and
         * its result is passed to {@link #instanceConnectionPortRange(ConnectionPortRange)}.
         * 
         * @param instanceConnectionPortRange
         *        a consumer that will call methods on {@link ConnectionPortRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceConnectionPortRange(ConnectionPortRange)
         */
        default Builder instanceConnectionPortRange(Consumer<ConnectionPortRange.Builder> instanceConnectionPortRange) {
            return instanceConnectionPortRange(ConnectionPortRange.builder().applyMutation(instanceConnectionPortRange).build());
        }

        /**
         * <p>
         * The IP address ranges and port settings that allow inbound traffic to access game server processes and other
         * processes on this fleet.
         * </p>
         * 
         * @param instanceInboundPermissions
         *        The IP address ranges and port settings that allow inbound traffic to access game server processes and
         *        other processes on this fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceInboundPermissions(Collection<IpPermission> instanceInboundPermissions);

        /**
         * <p>
         * The IP address ranges and port settings that allow inbound traffic to access game server processes and other
         * processes on this fleet.
         * </p>
         * 
         * @param instanceInboundPermissions
         *        The IP address ranges and port settings that allow inbound traffic to access game server processes and
         *        other processes on this fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceInboundPermissions(IpPermission... instanceInboundPermissions);

        /**
         * <p>
         * The IP address ranges and port settings that allow inbound traffic to access game server processes and other
         * processes on this fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.IpPermission.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.gamelift.model.IpPermission#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.IpPermission.Builder#build()} is called immediately and
         * its result is passed to {@link #instanceInboundPermissions(List<IpPermission>)}.
         * 
         * @param instanceInboundPermissions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.IpPermission.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceInboundPermissions(java.util.Collection<IpPermission>)
         */
        Builder instanceInboundPermissions(Consumer<IpPermission.Builder>... instanceInboundPermissions);

        /**
         * <p>
         * The number of times to replicate the game server container group on each fleet instance.
         * </p>
         * 
         * @param gameServerContainerGroupsPerInstance
         *        The number of times to replicate the game server container group on each fleet instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerContainerGroupsPerInstance(Integer gameServerContainerGroupsPerInstance);

        /**
         * <p>
         * The calculated maximum number of game server container group that can be deployed on each fleet instance. The
         * calculation depends on the resource needs of the container group and the CPU and memory resources of the
         * fleet's instance type.
         * </p>
         * 
         * @param maximumGameServerContainerGroupsPerInstance
         *        The calculated maximum number of game server container group that can be deployed on each fleet
         *        instance. The calculation depends on the resource needs of the container group and the CPU and memory
         *        resources of the fleet's instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumGameServerContainerGroupsPerInstance(Integer maximumGameServerContainerGroupsPerInstance);

        /**
         * <p>
         * The Amazon EC2 instance type to use for all instances in the fleet. Instance type determines the computing
         * resources and processing power that's available to host your game servers. This includes including CPU,
         * memory, storage, and networking capacity. You can't update this fleet property.
         * </p>
         * 
         * @param instanceType
         *        The Amazon EC2 instance type to use for all instances in the fleet. Instance type determines the
         *        computing resources and processing power that's available to host your game servers. This includes
         *        including CPU, memory, storage, and networking capacity. You can't update this fleet property.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to use <a
         * href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         * > On-Demand versus Spot Instances</a>. You can't update this fleet property.
         * </p>
         * <p>
         * By default, this property is set to <code>ON_DEMAND</code>.
         * </p>
         * 
         * @param billingType
         *        Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to
         *        use <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         *        > On-Demand versus Spot Instances</a>. You can't update this fleet property.</p>
         *        <p>
         *        By default, this property is set to <code>ON_DEMAND</code>.
         * @see ContainerFleetBillingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerFleetBillingType
         */
        Builder billingType(String billingType);

        /**
         * <p>
         * Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to use <a
         * href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         * > On-Demand versus Spot Instances</a>. You can't update this fleet property.
         * </p>
         * <p>
         * By default, this property is set to <code>ON_DEMAND</code>.
         * </p>
         * 
         * @param billingType
         *        Indicates whether the fleet uses On-Demand or Spot instances for this fleet. Learn more about when to
         *        use <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         *        > On-Demand versus Spot Instances</a>. You can't update this fleet property.</p>
         *        <p>
         *        By default, this property is set to <code>ON_DEMAND</code>.
         * @see ContainerFleetBillingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerFleetBillingType
         */
        Builder billingType(ContainerFleetBillingType billingType);

        /**
         * <p>
         * A meaningful description of the container fleet.
         * </p>
         * 
         * @param description
         *        A meaningful description of the container fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The name of an Amazon Web Services CloudWatch metric group to add this fleet to. Metric groups aggregate
         * metrics for multiple fleets.
         * </p>
         * 
         * @param metricGroups
         *        The name of an Amazon Web Services CloudWatch metric group to add this fleet to. Metric groups
         *        aggregate metrics for multiple fleets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricGroups(Collection<String> metricGroups);

        /**
         * <p>
         * The name of an Amazon Web Services CloudWatch metric group to add this fleet to. Metric groups aggregate
         * metrics for multiple fleets.
         * </p>
         * 
         * @param metricGroups
         *        The name of an Amazon Web Services CloudWatch metric group to add this fleet to. Metric groups
         *        aggregate metrics for multiple fleets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricGroups(String... metricGroups);

        /**
         * <p>
         * Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running and
         * hosting players. Amazon GameLift might prompt an instance shutdown when scaling down fleet capacity or when
         * retiring unhealthy instances. You can also set game session protection for individual game sessions using <a
         * href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param newGameSessionProtectionPolicy
         *        Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running
         *        and hosting players. Amazon GameLift might prompt an instance shutdown when scaling down fleet
         *        capacity or when retiring unhealthy instances. You can also set game session protection for individual
         *        game sessions using <a
         *        href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
         *        </p>
         *        </li>
         * @see ProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionPolicy
         */
        Builder newGameSessionProtectionPolicy(String newGameSessionProtectionPolicy);

        /**
         * <p>
         * Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running and
         * hosting players. Amazon GameLift might prompt an instance shutdown when scaling down fleet capacity or when
         * retiring unhealthy instances. You can also set game session protection for individual game sessions using <a
         * href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param newGameSessionProtectionPolicy
         *        Determines whether Amazon GameLift can shut down game sessions on the fleet that are actively running
         *        and hosting players. Amazon GameLift might prompt an instance shutdown when scaling down fleet
         *        capacity or when retiring unhealthy instances. You can also set game session protection for individual
         *        game sessions using <a
         *        href="gamelift/latest/apireference/API_UpdateGameSession.html">UpdateGameSession</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NoProtection</b> -- Game sessions can be shut down during active gameplay.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FullProtection</b> -- Game sessions in <code>ACTIVE</code> status can't be shut down.
         *        </p>
         *        </li>
         * @see ProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionPolicy
         */
        Builder newGameSessionProtectionPolicy(ProtectionPolicy newGameSessionProtectionPolicy);

        /**
         * <p>
         * A policy that limits the number of game sessions that each individual player can create on instances in this
         * fleet. The limit applies for a specified span of time.
         * </p>
         * 
         * @param gameSessionCreationLimitPolicy
         *        A policy that limits the number of game sessions that each individual player can create on instances
         *        in this fleet. The limit applies for a specified span of time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionCreationLimitPolicy(GameSessionCreationLimitPolicy gameSessionCreationLimitPolicy);

        /**
         * <p>
         * A policy that limits the number of game sessions that each individual player can create on instances in this
         * fleet. The limit applies for a specified span of time.
         * </p>
         * This is a convenience method that creates an instance of the {@link GameSessionCreationLimitPolicy.Builder}
         * avoiding the need to create one manually via {@link GameSessionCreationLimitPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GameSessionCreationLimitPolicy.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #gameSessionCreationLimitPolicy(GameSessionCreationLimitPolicy)}.
         * 
         * @param gameSessionCreationLimitPolicy
         *        a consumer that will call methods on {@link GameSessionCreationLimitPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameSessionCreationLimitPolicy(GameSessionCreationLimitPolicy)
         */
        default Builder gameSessionCreationLimitPolicy(
                Consumer<GameSessionCreationLimitPolicy.Builder> gameSessionCreationLimitPolicy) {
            return gameSessionCreationLimitPolicy(GameSessionCreationLimitPolicy.builder()
                    .applyMutation(gameSessionCreationLimitPolicy).build());
        }

        /**
         * <p>
         * The current status of the container fleet.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> -- A new container fleet has been requested.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> -- A new container fleet resource is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been
         * deployed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVATING</code> -- New container fleet instances are being deployed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the container fleet.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> -- A new container fleet has been requested.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> -- A new container fleet resource is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been
         *        deployed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVATING</code> -- New container fleet instances are being deployed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
         *        </p>
         *        </li>
         * @see ContainerFleetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerFleetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the container fleet.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> -- A new container fleet has been requested.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> -- A new container fleet resource is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been
         * deployed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVATING</code> -- New container fleet instances are being deployed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the container fleet.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> -- A new container fleet has been requested.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> -- A new container fleet resource is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATED</code> -- A new container fleet resource has been created. No fleet instances have been
         *        deployed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVATING</code> -- New container fleet instances are being deployed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> -- The container fleet has been deployed and is ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> -- Updates to the container fleet is being updated. A deployment is in progress.
         *        </p>
         *        </li>
         * @see ContainerFleetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerFleetStatus
         */
        Builder status(ContainerFleetStatus status);

        /**
         * <p>
         * Information about the most recent deployment for the container fleet.
         * </p>
         * 
         * @param deploymentDetails
         *        Information about the most recent deployment for the container fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentDetails(DeploymentDetails deploymentDetails);

        /**
         * <p>
         * Information about the most recent deployment for the container fleet.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeploymentDetails.Builder} avoiding the
         * need to create one manually via {@link DeploymentDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #deploymentDetails(DeploymentDetails)}.
         * 
         * @param deploymentDetails
         *        a consumer that will call methods on {@link DeploymentDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentDetails(DeploymentDetails)
         */
        default Builder deploymentDetails(Consumer<DeploymentDetails.Builder> deploymentDetails) {
            return deploymentDetails(DeploymentDetails.builder().applyMutation(deploymentDetails).build());
        }

        /**
         * <p>
         * The method that is used to collect container logs for the fleet. Amazon GameLift saves all standard output
         * for each container in logs, including game session logs.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLOUDWATCH</code> -- Send logs to an Amazon CloudWatch log group that you define. Each container emits
         * a log stream, which is organized in the log group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code> -- Store logs in an Amazon S3 bucket that you define.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> -- Don't collect container logs.
         * </p>
         * </li>
         * </ul>
         * 
         * @param logConfiguration
         *        The method that is used to collect container logs for the fleet. Amazon GameLift saves all standard
         *        output for each container in logs, including game session logs. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CLOUDWATCH</code> -- Send logs to an Amazon CloudWatch log group that you define. Each container
         *        emits a log stream, which is organized in the log group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code> -- Store logs in an Amazon S3 bucket that you define.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code> -- Don't collect container logs.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(LogConfiguration logConfiguration);

        /**
         * <p>
         * The method that is used to collect container logs for the fleet. Amazon GameLift saves all standard output
         * for each container in logs, including game session logs.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLOUDWATCH</code> -- Send logs to an Amazon CloudWatch log group that you define. Each container emits
         * a log stream, which is organized in the log group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code> -- Store logs in an Amazon S3 bucket that you define.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> -- Don't collect container logs.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link LogConfiguration.Builder} avoiding the
         * need to create one manually via {@link LogConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LogConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #logConfiguration(LogConfiguration)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link LogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(LogConfiguration)
         */
        default Builder logConfiguration(Consumer<LogConfiguration.Builder> logConfiguration) {
            return logConfiguration(LogConfiguration.builder().applyMutation(logConfiguration).build());
        }

        /**
         * <p>
         * Information about the container fleet's remote locations where fleet instances are deployed.
         * </p>
         * 
         * @param locationAttributes
         *        Information about the container fleet's remote locations where fleet instances are deployed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationAttributes(Collection<ContainerFleetLocationAttributes> locationAttributes);

        /**
         * <p>
         * Information about the container fleet's remote locations where fleet instances are deployed.
         * </p>
         * 
         * @param locationAttributes
         *        Information about the container fleet's remote locations where fleet instances are deployed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationAttributes(ContainerFleetLocationAttributes... locationAttributes);

        /**
         * <p>
         * Information about the container fleet's remote locations where fleet instances are deployed.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerFleetLocationAttributes.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerFleetLocationAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerFleetLocationAttributes.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #locationAttributes(List<ContainerFleetLocationAttributes>)}.
         * 
         * @param locationAttributes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerFleetLocationAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #locationAttributes(java.util.Collection<ContainerFleetLocationAttributes>)
         */
        Builder locationAttributes(Consumer<ContainerFleetLocationAttributes.Builder>... locationAttributes);
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String fleetArn;

        private String fleetRoleArn;

        private String gameServerContainerGroupDefinitionName;

        private String gameServerContainerGroupDefinitionArn;

        private String perInstanceContainerGroupDefinitionName;

        private String perInstanceContainerGroupDefinitionArn;

        private ConnectionPortRange instanceConnectionPortRange;

        private List<IpPermission> instanceInboundPermissions = DefaultSdkAutoConstructList.getInstance();

        private Integer gameServerContainerGroupsPerInstance;

        private Integer maximumGameServerContainerGroupsPerInstance;

        private String instanceType;

        private String billingType;

        private String description;

        private Instant creationTime;

        private List<String> metricGroups = DefaultSdkAutoConstructList.getInstance();

        private String newGameSessionProtectionPolicy;

        private GameSessionCreationLimitPolicy gameSessionCreationLimitPolicy;

        private String status;

        private DeploymentDetails deploymentDetails;

        private LogConfiguration logConfiguration;

        private List<ContainerFleetLocationAttributes> locationAttributes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerFleet model) {
            fleetId(model.fleetId);
            fleetArn(model.fleetArn);
            fleetRoleArn(model.fleetRoleArn);
            gameServerContainerGroupDefinitionName(model.gameServerContainerGroupDefinitionName);
            gameServerContainerGroupDefinitionArn(model.gameServerContainerGroupDefinitionArn);
            perInstanceContainerGroupDefinitionName(model.perInstanceContainerGroupDefinitionName);
            perInstanceContainerGroupDefinitionArn(model.perInstanceContainerGroupDefinitionArn);
            instanceConnectionPortRange(model.instanceConnectionPortRange);
            instanceInboundPermissions(model.instanceInboundPermissions);
            gameServerContainerGroupsPerInstance(model.gameServerContainerGroupsPerInstance);
            maximumGameServerContainerGroupsPerInstance(model.maximumGameServerContainerGroupsPerInstance);
            instanceType(model.instanceType);
            billingType(model.billingType);
            description(model.description);
            creationTime(model.creationTime);
            metricGroups(model.metricGroups);
            newGameSessionProtectionPolicy(model.newGameSessionProtectionPolicy);
            gameSessionCreationLimitPolicy(model.gameSessionCreationLimitPolicy);
            status(model.status);
            deploymentDetails(model.deploymentDetails);
            logConfiguration(model.logConfiguration);
            locationAttributes(model.locationAttributes);
        }

        public final String getFleetId() {
            return fleetId;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final String getFleetArn() {
            return fleetArn;
        }

        public final void setFleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
        }

        @Override
        public final Builder fleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
            return this;
        }

        public final String getFleetRoleArn() {
            return fleetRoleArn;
        }

        public final void setFleetRoleArn(String fleetRoleArn) {
            this.fleetRoleArn = fleetRoleArn;
        }

        @Override
        public final Builder fleetRoleArn(String fleetRoleArn) {
            this.fleetRoleArn = fleetRoleArn;
            return this;
        }

        public final String getGameServerContainerGroupDefinitionName() {
            return gameServerContainerGroupDefinitionName;
        }

        public final void setGameServerContainerGroupDefinitionName(String gameServerContainerGroupDefinitionName) {
            this.gameServerContainerGroupDefinitionName = gameServerContainerGroupDefinitionName;
        }

        @Override
        public final Builder gameServerContainerGroupDefinitionName(String gameServerContainerGroupDefinitionName) {
            this.gameServerContainerGroupDefinitionName = gameServerContainerGroupDefinitionName;
            return this;
        }

        public final String getGameServerContainerGroupDefinitionArn() {
            return gameServerContainerGroupDefinitionArn;
        }

        public final void setGameServerContainerGroupDefinitionArn(String gameServerContainerGroupDefinitionArn) {
            this.gameServerContainerGroupDefinitionArn = gameServerContainerGroupDefinitionArn;
        }

        @Override
        public final Builder gameServerContainerGroupDefinitionArn(String gameServerContainerGroupDefinitionArn) {
            this.gameServerContainerGroupDefinitionArn = gameServerContainerGroupDefinitionArn;
            return this;
        }

        public final String getPerInstanceContainerGroupDefinitionName() {
            return perInstanceContainerGroupDefinitionName;
        }

        public final void setPerInstanceContainerGroupDefinitionName(String perInstanceContainerGroupDefinitionName) {
            this.perInstanceContainerGroupDefinitionName = perInstanceContainerGroupDefinitionName;
        }

        @Override
        public final Builder perInstanceContainerGroupDefinitionName(String perInstanceContainerGroupDefinitionName) {
            this.perInstanceContainerGroupDefinitionName = perInstanceContainerGroupDefinitionName;
            return this;
        }

        public final String getPerInstanceContainerGroupDefinitionArn() {
            return perInstanceContainerGroupDefinitionArn;
        }

        public final void setPerInstanceContainerGroupDefinitionArn(String perInstanceContainerGroupDefinitionArn) {
            this.perInstanceContainerGroupDefinitionArn = perInstanceContainerGroupDefinitionArn;
        }

        @Override
        public final Builder perInstanceContainerGroupDefinitionArn(String perInstanceContainerGroupDefinitionArn) {
            this.perInstanceContainerGroupDefinitionArn = perInstanceContainerGroupDefinitionArn;
            return this;
        }

        public final ConnectionPortRange.Builder getInstanceConnectionPortRange() {
            return instanceConnectionPortRange != null ? instanceConnectionPortRange.toBuilder() : null;
        }

        public final void setInstanceConnectionPortRange(ConnectionPortRange.BuilderImpl instanceConnectionPortRange) {
            this.instanceConnectionPortRange = instanceConnectionPortRange != null ? instanceConnectionPortRange.build() : null;
        }

        @Override
        public final Builder instanceConnectionPortRange(ConnectionPortRange instanceConnectionPortRange) {
            this.instanceConnectionPortRange = instanceConnectionPortRange;
            return this;
        }

        public final List<IpPermission.Builder> getInstanceInboundPermissions() {
            List<IpPermission.Builder> result = IpPermissionsListCopier.copyToBuilder(this.instanceInboundPermissions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceInboundPermissions(Collection<IpPermission.BuilderImpl> instanceInboundPermissions) {
            this.instanceInboundPermissions = IpPermissionsListCopier.copyFromBuilder(instanceInboundPermissions);
        }

        @Override
        public final Builder instanceInboundPermissions(Collection<IpPermission> instanceInboundPermissions) {
            this.instanceInboundPermissions = IpPermissionsListCopier.copy(instanceInboundPermissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceInboundPermissions(IpPermission... instanceInboundPermissions) {
            instanceInboundPermissions(Arrays.asList(instanceInboundPermissions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceInboundPermissions(Consumer<IpPermission.Builder>... instanceInboundPermissions) {
            instanceInboundPermissions(Stream.of(instanceInboundPermissions)
                    .map(c -> IpPermission.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getGameServerContainerGroupsPerInstance() {
            return gameServerContainerGroupsPerInstance;
        }

        public final void setGameServerContainerGroupsPerInstance(Integer gameServerContainerGroupsPerInstance) {
            this.gameServerContainerGroupsPerInstance = gameServerContainerGroupsPerInstance;
        }

        @Override
        public final Builder gameServerContainerGroupsPerInstance(Integer gameServerContainerGroupsPerInstance) {
            this.gameServerContainerGroupsPerInstance = gameServerContainerGroupsPerInstance;
            return this;
        }

        public final Integer getMaximumGameServerContainerGroupsPerInstance() {
            return maximumGameServerContainerGroupsPerInstance;
        }

        public final void setMaximumGameServerContainerGroupsPerInstance(Integer maximumGameServerContainerGroupsPerInstance) {
            this.maximumGameServerContainerGroupsPerInstance = maximumGameServerContainerGroupsPerInstance;
        }

        @Override
        public final Builder maximumGameServerContainerGroupsPerInstance(Integer maximumGameServerContainerGroupsPerInstance) {
            this.maximumGameServerContainerGroupsPerInstance = maximumGameServerContainerGroupsPerInstance;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final String getBillingType() {
            return billingType;
        }

        public final void setBillingType(String billingType) {
            this.billingType = billingType;
        }

        @Override
        public final Builder billingType(String billingType) {
            this.billingType = billingType;
            return this;
        }

        @Override
        public final Builder billingType(ContainerFleetBillingType billingType) {
            this.billingType(billingType == null ? null : billingType.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Collection<String> getMetricGroups() {
            if (metricGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return metricGroups;
        }

        public final void setMetricGroups(Collection<String> metricGroups) {
            this.metricGroups = MetricGroupListCopier.copy(metricGroups);
        }

        @Override
        public final Builder metricGroups(Collection<String> metricGroups) {
            this.metricGroups = MetricGroupListCopier.copy(metricGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricGroups(String... metricGroups) {
            metricGroups(Arrays.asList(metricGroups));
            return this;
        }

        public final String getNewGameSessionProtectionPolicy() {
            return newGameSessionProtectionPolicy;
        }

        public final void setNewGameSessionProtectionPolicy(String newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy = newGameSessionProtectionPolicy;
        }

        @Override
        public final Builder newGameSessionProtectionPolicy(String newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy = newGameSessionProtectionPolicy;
            return this;
        }

        @Override
        public final Builder newGameSessionProtectionPolicy(ProtectionPolicy newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy(newGameSessionProtectionPolicy == null ? null : newGameSessionProtectionPolicy
                    .toString());
            return this;
        }

        public final GameSessionCreationLimitPolicy.Builder getGameSessionCreationLimitPolicy() {
            return gameSessionCreationLimitPolicy != null ? gameSessionCreationLimitPolicy.toBuilder() : null;
        }

        public final void setGameSessionCreationLimitPolicy(
                GameSessionCreationLimitPolicy.BuilderImpl gameSessionCreationLimitPolicy) {
            this.gameSessionCreationLimitPolicy = gameSessionCreationLimitPolicy != null ? gameSessionCreationLimitPolicy.build()
                    : null;
        }

        @Override
        public final Builder gameSessionCreationLimitPolicy(GameSessionCreationLimitPolicy gameSessionCreationLimitPolicy) {
            this.gameSessionCreationLimitPolicy = gameSessionCreationLimitPolicy;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ContainerFleetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final DeploymentDetails.Builder getDeploymentDetails() {
            return deploymentDetails != null ? deploymentDetails.toBuilder() : null;
        }

        public final void setDeploymentDetails(DeploymentDetails.BuilderImpl deploymentDetails) {
            this.deploymentDetails = deploymentDetails != null ? deploymentDetails.build() : null;
        }

        @Override
        public final Builder deploymentDetails(DeploymentDetails deploymentDetails) {
            this.deploymentDetails = deploymentDetails;
            return this;
        }

        public final LogConfiguration.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        public final void setLogConfiguration(LogConfiguration.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        @Override
        public final Builder logConfiguration(LogConfiguration logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        public final List<ContainerFleetLocationAttributes.Builder> getLocationAttributes() {
            List<ContainerFleetLocationAttributes.Builder> result = ContainerFleetLocationAttributesListCopier
                    .copyToBuilder(this.locationAttributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLocationAttributes(Collection<ContainerFleetLocationAttributes.BuilderImpl> locationAttributes) {
            this.locationAttributes = ContainerFleetLocationAttributesListCopier.copyFromBuilder(locationAttributes);
        }

        @Override
        public final Builder locationAttributes(Collection<ContainerFleetLocationAttributes> locationAttributes) {
            this.locationAttributes = ContainerFleetLocationAttributesListCopier.copy(locationAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder locationAttributes(ContainerFleetLocationAttributes... locationAttributes) {
            locationAttributes(Arrays.asList(locationAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder locationAttributes(Consumer<ContainerFleetLocationAttributes.Builder>... locationAttributes) {
            locationAttributes(Stream.of(locationAttributes)
                    .map(c -> ContainerFleetLocationAttributes.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ContainerFleet build() {
            return new ContainerFleet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
