/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the Amazon GameLift containers feature, which is currently in public preview.</b>
 * </p>
 * <p>
 * Describes a container's configuration, resources, and start instructions. Use this data type to create a container
 * group definition. For the properties of a container that's been deployed to a fleet, see <a>ContainerDefinition</a>.
 * You can't change these properties after you've created the container group definition. If you need a container group
 * with different properties, then you must create a new one.
 * </p>
 * <p>
 * <b>Used with: </b> <a>CreateContainerGroupDefinition</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDefinitionInput implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerDefinitionInput.Builder, ContainerDefinitionInput> {
    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerName").getter(getter(ContainerDefinitionInput::containerName))
            .setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerName").build()).build();

    private static final SdkField<String> IMAGE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageUri").getter(getter(ContainerDefinitionInput::imageUri)).setter(setter(Builder::imageUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUri").build()).build();

    private static final SdkField<ContainerMemoryLimits> MEMORY_LIMITS_FIELD = SdkField
            .<ContainerMemoryLimits> builder(MarshallingType.SDK_POJO).memberName("MemoryLimits")
            .getter(getter(ContainerDefinitionInput::memoryLimits)).setter(setter(Builder::memoryLimits))
            .constructor(ContainerMemoryLimits::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemoryLimits").build()).build();

    private static final SdkField<ContainerPortConfiguration> PORT_CONFIGURATION_FIELD = SdkField
            .<ContainerPortConfiguration> builder(MarshallingType.SDK_POJO).memberName("PortConfiguration")
            .getter(getter(ContainerDefinitionInput::portConfiguration)).setter(setter(Builder::portConfiguration))
            .constructor(ContainerPortConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PortConfiguration").build()).build();

    private static final SdkField<Integer> CPU_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Cpu")
            .getter(getter(ContainerDefinitionInput::cpu)).setter(setter(Builder::cpu))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cpu").build()).build();

    private static final SdkField<ContainerHealthCheck> HEALTH_CHECK_FIELD = SdkField
            .<ContainerHealthCheck> builder(MarshallingType.SDK_POJO).memberName("HealthCheck")
            .getter(getter(ContainerDefinitionInput::healthCheck)).setter(setter(Builder::healthCheck))
            .constructor(ContainerHealthCheck::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheck").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Command")
            .getter(getter(ContainerDefinitionInput::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ESSENTIAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Essential").getter(getter(ContainerDefinitionInput::essential)).setter(setter(Builder::essential))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Essential").build()).build();

    private static final SdkField<List<String>> ENTRY_POINT_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EntryPoint")
            .getter(getter(ContainerDefinitionInput::entryPoint))
            .setter(setter(Builder::entryPoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntryPoint").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> WORKING_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkingDirectory").getter(getter(ContainerDefinitionInput::workingDirectory))
            .setter(setter(Builder::workingDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkingDirectory").build()).build();

    private static final SdkField<List<ContainerEnvironment>> ENVIRONMENT_FIELD = SdkField
            .<List<ContainerEnvironment>> builder(MarshallingType.LIST)
            .memberName("Environment")
            .getter(getter(ContainerDefinitionInput::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerEnvironment> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerEnvironment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ContainerDependency>> DEPENDS_ON_FIELD = SdkField
            .<List<ContainerDependency>> builder(MarshallingType.LIST)
            .memberName("DependsOn")
            .getter(getter(ContainerDefinitionInput::dependsOn))
            .setter(setter(Builder::dependsOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DependsOn").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerDependency> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerDependency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_NAME_FIELD,
            IMAGE_URI_FIELD, MEMORY_LIMITS_FIELD, PORT_CONFIGURATION_FIELD, CPU_FIELD, HEALTH_CHECK_FIELD, COMMAND_FIELD,
            ESSENTIAL_FIELD, ENTRY_POINT_FIELD, WORKING_DIRECTORY_FIELD, ENVIRONMENT_FIELD, DEPENDS_ON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerName;

    private final String imageUri;

    private final ContainerMemoryLimits memoryLimits;

    private final ContainerPortConfiguration portConfiguration;

    private final Integer cpu;

    private final ContainerHealthCheck healthCheck;

    private final List<String> command;

    private final Boolean essential;

    private final List<String> entryPoint;

    private final String workingDirectory;

    private final List<ContainerEnvironment> environment;

    private final List<ContainerDependency> dependsOn;

    private ContainerDefinitionInput(BuilderImpl builder) {
        this.containerName = builder.containerName;
        this.imageUri = builder.imageUri;
        this.memoryLimits = builder.memoryLimits;
        this.portConfiguration = builder.portConfiguration;
        this.cpu = builder.cpu;
        this.healthCheck = builder.healthCheck;
        this.command = builder.command;
        this.essential = builder.essential;
        this.entryPoint = builder.entryPoint;
        this.workingDirectory = builder.workingDirectory;
        this.environment = builder.environment;
        this.dependsOn = builder.dependsOn;
    }

    /**
     * <p>
     * A string that uniquely identifies the container definition within a container group.
     * </p>
     * 
     * @return A string that uniquely identifies the container definition within a container group.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * <p>
     * The location of a container image that $short; will copy and deploy to a container fleet. Images in Amazon
     * Elastic Container Registry private repositories are supported. The repository must be in the same Amazon Web
     * Services account and Amazon Web Services Region where you're creating the container group definition. For limits
     * on image size, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon GameLift
     * endpoints and quotas</a>. You can use any of the following image URI formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Image ID and digest: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The location of a container image that $short; will copy and deploy to a container fleet. Images in
     *         Amazon Elastic Container Registry private repositories are supported. The repository must be in the same
     *         Amazon Web Services account and Amazon Web Services Region where you're creating the container group
     *         definition. For limits on image size, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon GameLift endpoints and
     *         quotas</a>. You can use any of the following image URI formats: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Image ID and digest:
     *         <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
     *         </p>
     *         </li>
     */
    public final String imageUri() {
        return imageUri;
    }

    /**
     * <p>
     * The amount of memory to make available to the container. If you don't specify memory limits for this container,
     * then it shares the container group's total memory allocation.
     * </p>
     * <p>
     * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
     * </p>
     * 
     * @return The amount of memory to make available to the container. If you don't specify memory limits for this
     *         container, then it shares the container group's total memory allocation. </p>
     *         <p>
     *         <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
     */
    public final ContainerMemoryLimits memoryLimits() {
        return memoryLimits;
    }

    /**
     * <p>
     * A set of ports that Amazon GameLift can assign to processes in the container. All processes that accept inbound
     * traffic connections, including game server processes, must be assigned a port from this set. The set of ports
     * must be large enough to assign one to each process in the container that needs one. If the container includes
     * your game server, include enough ports to assign one port to each concurrent server process (as defined in a
     * container fleet's <a>RuntimeConfiguration</a>). For more details, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-network">Networking for container
     * fleets</a>.
     * </p>
     * <p>
     * Container ports aren't directly accessed by inbound traffic. Amazon GameLift maps these container ports to
     * externally accessible connection ports, which are assigned as needed from the container fleet's
     * <code>ConnectionPortRange</code>.
     * </p>
     * 
     * @return A set of ports that Amazon GameLift can assign to processes in the container. All processes that accept
     *         inbound traffic connections, including game server processes, must be assigned a port from this set. The
     *         set of ports must be large enough to assign one to each process in the container that needs one. If the
     *         container includes your game server, include enough ports to assign one port to each concurrent server
     *         process (as defined in a container fleet's <a>RuntimeConfiguration</a>). For more details, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-network">Networking for
     *         container fleets</a>. </p>
     *         <p>
     *         Container ports aren't directly accessed by inbound traffic. Amazon GameLift maps these container ports
     *         to externally accessible connection ports, which are assigned as needed from the container fleet's
     *         <code>ConnectionPortRange</code>.
     */
    public final ContainerPortConfiguration portConfiguration() {
        return portConfiguration;
    }

    /**
     * <p>
     * The number of CPU units to reserve for this container. The container can use more resources when needed, if
     * available. Note: 1 vCPU unit equals 1024 CPU units. If you don't reserve CPU units for this container, then it
     * shares the total CPU limit for the container group. This property is similar to the Amazon ECS container
     * definition parameter <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#container_definition_environment"
     * >environment</a> (<i>Amazon Elastic Container Service Developer Guide).</i>
     * </p>
     * <p>
     * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
     * </p>
     * 
     * @return The number of CPU units to reserve for this container. The container can use more resources when needed,
     *         if available. Note: 1 vCPU unit equals 1024 CPU units. If you don't reserve CPU units for this container,
     *         then it shares the total CPU limit for the container group. This property is similar to the Amazon ECS
     *         container definition parameter <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#container_definition_environment"
     *         >environment</a> (<i>Amazon Elastic Container Service Developer Guide).</i> </p>
     *         <p>
     *         <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
     */
    public final Integer cpu() {
        return cpu;
    }

    /**
     * <p>
     * Configuration for a non-terminal health check. A container automatically restarts if it stops functioning. This
     * parameter lets you define additional reasons to consider a container unhealthy and restart it. You can set a
     * health check for any container except for the essential container in the replica container group. If an essential
     * container in the daemon group fails a health check, the entire container group is restarted.
     * </p>
     * 
     * @return Configuration for a non-terminal health check. A container automatically restarts if it stops
     *         functioning. This parameter lets you define additional reasons to consider a container unhealthy and
     *         restart it. You can set a health check for any container except for the essential container in the
     *         replica container group. If an essential container in the daemon group fails a health check, the entire
     *         container group is restarted.
     */
    public final ContainerHealthCheck healthCheck() {
        return healthCheck;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A command to pass to the container on startup. Add multiple arguments as additional strings in the array. See the
     * <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
     * >ContainerDefinition command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return A command to pass to the container on startup. Add multiple arguments as additional strings in the array.
     *         See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
     *         >ContainerDefinition command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
     */
    public final List<String> command() {
        return command;
    }

    /**
     * <p>
     * Specifies whether the container is vital for the container group to function properly. If an essential container
     * fails, it causes the entire container group to restart. Each container group must have an essential container.
     * </p>
     * <p>
     * <b>Replica container groups</b> - A replica group must have exactly one essential container. Use the following to
     * configure an essential replica container:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Choose a container is running your game server and the Amazon GameLift Agent.
     * </p>
     * </li>
     * <li>
     * <p>
     * Include a port configuration. This container runs your game server processes, and each process requires a
     * container port to allow access to game clients.
     * </p>
     * </li>
     * <li>
     * <p>
     * Don't configure a health check. The Agent handles this task for the essential replica container.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Daemon container groups</b> - A daemon group must have at least one essential container.
     * </p>
     * <p>
     * </p>
     * 
     * @return Specifies whether the container is vital for the container group to function properly. If an essential
     *         container fails, it causes the entire container group to restart. Each container group must have an
     *         essential container.</p>
     *         <p>
     *         <b>Replica container groups</b> - A replica group must have exactly one essential container. Use the
     *         following to configure an essential replica container:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Choose a container is running your game server and the Amazon GameLift Agent.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Include a port configuration. This container runs your game server processes, and each process requires a
     *         container port to allow access to game clients.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Don't configure a health check. The Agent handles this task for the essential replica container.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Daemon container groups</b> - A daemon group must have at least one essential container.
     *         </p>
     *         <p>
     */
    public final Boolean essential() {
        return essential;
    }

    /**
     * For responses, this returns true if the service returned a value for the EntryPoint property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEntryPoint() {
        return entryPoint != null && !(entryPoint instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An entry point to pass to the container on startup. Add multiple arguments as additional strings in the array.
     * See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
     * >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntryPoint} method.
     * </p>
     * 
     * @return An entry point to pass to the container on startup. Add multiple arguments as additional strings in the
     *         array. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
     *         >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final List<String> entryPoint() {
        return entryPoint;
    }

    /**
     * <p>
     * The directory in the container where commands are run. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
     * >ContainerDefinition::workingDirectory parameter</a> in the <i>Amazon Elastic Container Service API
     * Reference</i>.
     * </p>
     * 
     * @return The directory in the container where commands are run. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
     *         >ContainerDefinition::workingDirectory parameter</a> in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final String workingDirectory() {
        return workingDirectory;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of environment variables to pass to the container on startup. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return A set of environment variables to pass to the container on startup. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     *         >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final List<ContainerEnvironment> environment() {
        return environment;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependsOn property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDependsOn() {
        return dependsOn != null && !(dependsOn instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Sets up dependencies between this container and the status of other containers in the same container group. A
     * container can have dependencies on multiple different containers.
     * </p>
     * <p>
     * You can use dependencies to establish a startup/shutdown sequence across the container group. A container startup
     * dependency is reversed on shutdown.
     * </p>
     * <p>
     * For example, you might specify that SideCarContainerB has a <code>START</code> dependency on SideCarContainerA.
     * This dependency means that SideCarContainerB can't start until after SideCarContainerA has started. This
     * dependency is reversed on shutdown, which means that SideCarContainerB must shut down before SideCarContainerA
     * can shut down.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependsOn} method.
     * </p>
     * 
     * @return Sets up dependencies between this container and the status of other containers in the same container
     *         group. A container can have dependencies on multiple different containers. </p>
     *         <p>
     *         You can use dependencies to establish a startup/shutdown sequence across the container group. A container
     *         startup dependency is reversed on shutdown.
     *         </p>
     *         <p>
     *         For example, you might specify that SideCarContainerB has a <code>START</code> dependency on
     *         SideCarContainerA. This dependency means that SideCarContainerB can't start until after SideCarContainerA
     *         has started. This dependency is reversed on shutdown, which means that SideCarContainerB must shut down
     *         before SideCarContainerA can shut down.
     */
    public final List<ContainerDependency> dependsOn() {
        return dependsOn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(imageUri());
        hashCode = 31 * hashCode + Objects.hashCode(memoryLimits());
        hashCode = 31 * hashCode + Objects.hashCode(portConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(cpu());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheck());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(essential());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntryPoint() ? entryPoint() : null);
        hashCode = 31 * hashCode + Objects.hashCode(workingDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDependsOn() ? dependsOn() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDefinitionInput)) {
            return false;
        }
        ContainerDefinitionInput other = (ContainerDefinitionInput) obj;
        return Objects.equals(containerName(), other.containerName()) && Objects.equals(imageUri(), other.imageUri())
                && Objects.equals(memoryLimits(), other.memoryLimits())
                && Objects.equals(portConfiguration(), other.portConfiguration()) && Objects.equals(cpu(), other.cpu())
                && Objects.equals(healthCheck(), other.healthCheck()) && hasCommand() == other.hasCommand()
                && Objects.equals(command(), other.command()) && Objects.equals(essential(), other.essential())
                && hasEntryPoint() == other.hasEntryPoint() && Objects.equals(entryPoint(), other.entryPoint())
                && Objects.equals(workingDirectory(), other.workingDirectory()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment()) && hasDependsOn() == other.hasDependsOn()
                && Objects.equals(dependsOn(), other.dependsOn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerDefinitionInput").add("ContainerName", containerName()).add("ImageUri", imageUri())
                .add("MemoryLimits", memoryLimits()).add("PortConfiguration", portConfiguration()).add("Cpu", cpu())
                .add("HealthCheck", healthCheck()).add("Command", hasCommand() ? command() : null).add("Essential", essential())
                .add("EntryPoint", hasEntryPoint() ? entryPoint() : null).add("WorkingDirectory", workingDirectory())
                .add("Environment", hasEnvironment() ? environment() : null)
                .add("DependsOn", hasDependsOn() ? dependsOn() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "ImageUri":
            return Optional.ofNullable(clazz.cast(imageUri()));
        case "MemoryLimits":
            return Optional.ofNullable(clazz.cast(memoryLimits()));
        case "PortConfiguration":
            return Optional.ofNullable(clazz.cast(portConfiguration()));
        case "Cpu":
            return Optional.ofNullable(clazz.cast(cpu()));
        case "HealthCheck":
            return Optional.ofNullable(clazz.cast(healthCheck()));
        case "Command":
            return Optional.ofNullable(clazz.cast(command()));
        case "Essential":
            return Optional.ofNullable(clazz.cast(essential()));
        case "EntryPoint":
            return Optional.ofNullable(clazz.cast(entryPoint()));
        case "WorkingDirectory":
            return Optional.ofNullable(clazz.cast(workingDirectory()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "DependsOn":
            return Optional.ofNullable(clazz.cast(dependsOn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerDefinitionInput, T> g) {
        return obj -> g.apply((ContainerDefinitionInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDefinitionInput> {
        /**
         * <p>
         * A string that uniquely identifies the container definition within a container group.
         * </p>
         * 
         * @param containerName
         *        A string that uniquely identifies the container definition within a container group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * The location of a container image that $short; will copy and deploy to a container fleet. Images in Amazon
         * Elastic Container Registry private repositories are supported. The repository must be in the same Amazon Web
         * Services account and Amazon Web Services Region where you're creating the container group definition. For
         * limits on image size, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon
         * GameLift endpoints and quotas</a>. You can use any of the following image URI formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Image ID and digest: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param imageUri
         *        The location of a container image that $short; will copy and deploy to a container fleet. Images in
         *        Amazon Elastic Container Registry private repositories are supported. The repository must be in the
         *        same Amazon Web Services account and Amazon Web Services Region where you're creating the container
         *        group definition. For limits on image size, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon GameLift endpoints and
         *        quotas</a>. You can use any of the following image URI formats: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Image ID and digest:
         *        <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUri(String imageUri);

        /**
         * <p>
         * The amount of memory to make available to the container. If you don't specify memory limits for this
         * container, then it shares the container group's total memory allocation.
         * </p>
         * <p>
         * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
         * </p>
         * 
         * @param memoryLimits
         *        The amount of memory to make available to the container. If you don't specify memory limits for this
         *        container, then it shares the container group's total memory allocation. </p>
         *        <p>
         *        <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryLimits(ContainerMemoryLimits memoryLimits);

        /**
         * <p>
         * The amount of memory to make available to the container. If you don't specify memory limits for this
         * container, then it shares the container group's total memory allocation.
         * </p>
         * <p>
         * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerMemoryLimits.Builder} avoiding
         * the need to create one manually via {@link ContainerMemoryLimits#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerMemoryLimits.Builder#build()} is called immediately and
         * its result is passed to {@link #memoryLimits(ContainerMemoryLimits)}.
         * 
         * @param memoryLimits
         *        a consumer that will call methods on {@link ContainerMemoryLimits.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryLimits(ContainerMemoryLimits)
         */
        default Builder memoryLimits(Consumer<ContainerMemoryLimits.Builder> memoryLimits) {
            return memoryLimits(ContainerMemoryLimits.builder().applyMutation(memoryLimits).build());
        }

        /**
         * <p>
         * A set of ports that Amazon GameLift can assign to processes in the container. All processes that accept
         * inbound traffic connections, including game server processes, must be assigned a port from this set. The set
         * of ports must be large enough to assign one to each process in the container that needs one. If the container
         * includes your game server, include enough ports to assign one port to each concurrent server process (as
         * defined in a container fleet's <a>RuntimeConfiguration</a>). For more details, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-network">Networking for container
         * fleets</a>.
         * </p>
         * <p>
         * Container ports aren't directly accessed by inbound traffic. Amazon GameLift maps these container ports to
         * externally accessible connection ports, which are assigned as needed from the container fleet's
         * <code>ConnectionPortRange</code>.
         * </p>
         * 
         * @param portConfiguration
         *        A set of ports that Amazon GameLift can assign to processes in the container. All processes that
         *        accept inbound traffic connections, including game server processes, must be assigned a port from this
         *        set. The set of ports must be large enough to assign one to each process in the container that needs
         *        one. If the container includes your game server, include enough ports to assign one port to each
         *        concurrent server process (as defined in a container fleet's <a>RuntimeConfiguration</a>). For more
         *        details, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-network">Networking for
         *        container fleets</a>. </p>
         *        <p>
         *        Container ports aren't directly accessed by inbound traffic. Amazon GameLift maps these container
         *        ports to externally accessible connection ports, which are assigned as needed from the container
         *        fleet's <code>ConnectionPortRange</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portConfiguration(ContainerPortConfiguration portConfiguration);

        /**
         * <p>
         * A set of ports that Amazon GameLift can assign to processes in the container. All processes that accept
         * inbound traffic connections, including game server processes, must be assigned a port from this set. The set
         * of ports must be large enough to assign one to each process in the container that needs one. If the container
         * includes your game server, include enough ports to assign one port to each concurrent server process (as
         * defined in a container fleet's <a>RuntimeConfiguration</a>). For more details, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-network">Networking for container
         * fleets</a>.
         * </p>
         * <p>
         * Container ports aren't directly accessed by inbound traffic. Amazon GameLift maps these container ports to
         * externally accessible connection ports, which are assigned as needed from the container fleet's
         * <code>ConnectionPortRange</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerPortConfiguration.Builder}
         * avoiding the need to create one manually via {@link ContainerPortConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerPortConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #portConfiguration(ContainerPortConfiguration)}.
         * 
         * @param portConfiguration
         *        a consumer that will call methods on {@link ContainerPortConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #portConfiguration(ContainerPortConfiguration)
         */
        default Builder portConfiguration(Consumer<ContainerPortConfiguration.Builder> portConfiguration) {
            return portConfiguration(ContainerPortConfiguration.builder().applyMutation(portConfiguration).build());
        }

        /**
         * <p>
         * The number of CPU units to reserve for this container. The container can use more resources when needed, if
         * available. Note: 1 vCPU unit equals 1024 CPU units. If you don't reserve CPU units for this container, then
         * it shares the total CPU limit for the container group. This property is similar to the Amazon ECS container
         * definition parameter <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#container_definition_environment"
         * >environment</a> (<i>Amazon Elastic Container Service Developer Guide).</i>
         * </p>
         * <p>
         * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
         * </p>
         * 
         * @param cpu
         *        The number of CPU units to reserve for this container. The container can use more resources when
         *        needed, if available. Note: 1 vCPU unit equals 1024 CPU units. If you don't reserve CPU units for this
         *        container, then it shares the total CPU limit for the container group. This property is similar to the
         *        Amazon ECS container definition parameter <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#container_definition_environment"
         *        >environment</a> (<i>Amazon Elastic Container Service Developer Guide).</i> </p>
         *        <p>
         *        <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpu(Integer cpu);

        /**
         * <p>
         * Configuration for a non-terminal health check. A container automatically restarts if it stops functioning.
         * This parameter lets you define additional reasons to consider a container unhealthy and restart it. You can
         * set a health check for any container except for the essential container in the replica container group. If an
         * essential container in the daemon group fails a health check, the entire container group is restarted.
         * </p>
         * 
         * @param healthCheck
         *        Configuration for a non-terminal health check. A container automatically restarts if it stops
         *        functioning. This parameter lets you define additional reasons to consider a container unhealthy and
         *        restart it. You can set a health check for any container except for the essential container in the
         *        replica container group. If an essential container in the daemon group fails a health check, the
         *        entire container group is restarted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheck(ContainerHealthCheck healthCheck);

        /**
         * <p>
         * Configuration for a non-terminal health check. A container automatically restarts if it stops functioning.
         * This parameter lets you define additional reasons to consider a container unhealthy and restart it. You can
         * set a health check for any container except for the essential container in the replica container group. If an
         * essential container in the daemon group fails a health check, the entire container group is restarted.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerHealthCheck.Builder} avoiding
         * the need to create one manually via {@link ContainerHealthCheck#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerHealthCheck.Builder#build()} is called immediately and
         * its result is passed to {@link #healthCheck(ContainerHealthCheck)}.
         * 
         * @param healthCheck
         *        a consumer that will call methods on {@link ContainerHealthCheck.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #healthCheck(ContainerHealthCheck)
         */
        default Builder healthCheck(Consumer<ContainerHealthCheck.Builder> healthCheck) {
            return healthCheck(ContainerHealthCheck.builder().applyMutation(healthCheck).build());
        }

        /**
         * <p>
         * A command to pass to the container on startup. Add multiple arguments as additional strings in the array. See
         * the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         * >ContainerDefinition command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
         * </p>
         * 
         * @param command
         *        A command to pass to the container on startup. Add multiple arguments as additional strings in the
         *        array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         *        >ContainerDefinition command</a> parameter in the <i>Amazon Elastic Container Service API
         *        reference.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * A command to pass to the container on startup. Add multiple arguments as additional strings in the array. See
         * the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         * >ContainerDefinition command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
         * </p>
         * 
         * @param command
         *        A command to pass to the container on startup. Add multiple arguments as additional strings in the
         *        array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         *        >ContainerDefinition command</a> parameter in the <i>Amazon Elastic Container Service API
         *        reference.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * Specifies whether the container is vital for the container group to function properly. If an essential
         * container fails, it causes the entire container group to restart. Each container group must have an essential
         * container.
         * </p>
         * <p>
         * <b>Replica container groups</b> - A replica group must have exactly one essential container. Use the
         * following to configure an essential replica container:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Choose a container is running your game server and the Amazon GameLift Agent.
         * </p>
         * </li>
         * <li>
         * <p>
         * Include a port configuration. This container runs your game server processes, and each process requires a
         * container port to allow access to game clients.
         * </p>
         * </li>
         * <li>
         * <p>
         * Don't configure a health check. The Agent handles this task for the essential replica container.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Daemon container groups</b> - A daemon group must have at least one essential container.
         * </p>
         * <p>
         * </p>
         * 
         * @param essential
         *        Specifies whether the container is vital for the container group to function properly. If an essential
         *        container fails, it causes the entire container group to restart. Each container group must have an
         *        essential container.</p>
         *        <p>
         *        <b>Replica container groups</b> - A replica group must have exactly one essential container. Use the
         *        following to configure an essential replica container:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Choose a container is running your game server and the Amazon GameLift Agent.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Include a port configuration. This container runs your game server processes, and each process
         *        requires a container port to allow access to game clients.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Don't configure a health check. The Agent handles this task for the essential replica container.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Daemon container groups</b> - A daemon group must have at least one essential container.
         *        </p>
         *        <p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder essential(Boolean essential);

        /**
         * <p>
         * An entry point to pass to the container on startup. Add multiple arguments as additional strings in the
         * array. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         * >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param entryPoint
         *        An entry point to pass to the container on startup. Add multiple arguments as additional strings in
         *        the array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         *        >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryPoint(Collection<String> entryPoint);

        /**
         * <p>
         * An entry point to pass to the container on startup. Add multiple arguments as additional strings in the
         * array. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         * >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param entryPoint
         *        An entry point to pass to the container on startup. Add multiple arguments as additional strings in
         *        the array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         *        >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryPoint(String... entryPoint);

        /**
         * <p>
         * The directory in the container where commands are run. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
         * >ContainerDefinition::workingDirectory parameter</a> in the <i>Amazon Elastic Container Service API
         * Reference</i>.
         * </p>
         * 
         * @param workingDirectory
         *        The directory in the container where commands are run. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
         *        >ContainerDefinition::workingDirectory parameter</a> in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workingDirectory(String workingDirectory);

        /**
         * <p>
         * A set of environment variables to pass to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environment
         *        A set of environment variables to pass to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<ContainerEnvironment> environment);

        /**
         * <p>
         * A set of environment variables to pass to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environment
         *        A set of environment variables to pass to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(ContainerEnvironment... environment);

        /**
         * <p>
         * A set of environment variables to pass to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder#build()} is called
         * immediately and its result is passed to {@link #environment(List<ContainerEnvironment>)}.
         * 
         * @param environment
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(java.util.Collection<ContainerEnvironment>)
         */
        Builder environment(Consumer<ContainerEnvironment.Builder>... environment);

        /**
         * <p>
         * Sets up dependencies between this container and the status of other containers in the same container group. A
         * container can have dependencies on multiple different containers.
         * </p>
         * <p>
         * You can use dependencies to establish a startup/shutdown sequence across the container group. A container
         * startup dependency is reversed on shutdown.
         * </p>
         * <p>
         * For example, you might specify that SideCarContainerB has a <code>START</code> dependency on
         * SideCarContainerA. This dependency means that SideCarContainerB can't start until after SideCarContainerA has
         * started. This dependency is reversed on shutdown, which means that SideCarContainerB must shut down before
         * SideCarContainerA can shut down.
         * </p>
         * 
         * @param dependsOn
         *        Sets up dependencies between this container and the status of other containers in the same container
         *        group. A container can have dependencies on multiple different containers. </p>
         *        <p>
         *        You can use dependencies to establish a startup/shutdown sequence across the container group. A
         *        container startup dependency is reversed on shutdown.
         *        </p>
         *        <p>
         *        For example, you might specify that SideCarContainerB has a <code>START</code> dependency on
         *        SideCarContainerA. This dependency means that SideCarContainerB can't start until after
         *        SideCarContainerA has started. This dependency is reversed on shutdown, which means that
         *        SideCarContainerB must shut down before SideCarContainerA can shut down.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(Collection<ContainerDependency> dependsOn);

        /**
         * <p>
         * Sets up dependencies between this container and the status of other containers in the same container group. A
         * container can have dependencies on multiple different containers.
         * </p>
         * <p>
         * You can use dependencies to establish a startup/shutdown sequence across the container group. A container
         * startup dependency is reversed on shutdown.
         * </p>
         * <p>
         * For example, you might specify that SideCarContainerB has a <code>START</code> dependency on
         * SideCarContainerA. This dependency means that SideCarContainerB can't start until after SideCarContainerA has
         * started. This dependency is reversed on shutdown, which means that SideCarContainerB must shut down before
         * SideCarContainerA can shut down.
         * </p>
         * 
         * @param dependsOn
         *        Sets up dependencies between this container and the status of other containers in the same container
         *        group. A container can have dependencies on multiple different containers. </p>
         *        <p>
         *        You can use dependencies to establish a startup/shutdown sequence across the container group. A
         *        container startup dependency is reversed on shutdown.
         *        </p>
         *        <p>
         *        For example, you might specify that SideCarContainerB has a <code>START</code> dependency on
         *        SideCarContainerA. This dependency means that SideCarContainerB can't start until after
         *        SideCarContainerA has started. This dependency is reversed on shutdown, which means that
         *        SideCarContainerB must shut down before SideCarContainerA can shut down.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(ContainerDependency... dependsOn);

        /**
         * <p>
         * Sets up dependencies between this container and the status of other containers in the same container group. A
         * container can have dependencies on multiple different containers.
         * </p>
         * <p>
         * You can use dependencies to establish a startup/shutdown sequence across the container group. A container
         * startup dependency is reversed on shutdown.
         * </p>
         * <p>
         * For example, you might specify that SideCarContainerB has a <code>START</code> dependency on
         * SideCarContainerA. This dependency means that SideCarContainerB can't start until after SideCarContainerA has
         * started. This dependency is reversed on shutdown, which means that SideCarContainerB must shut down before
         * SideCarContainerA can shut down.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder#build()} is called
         * immediately and its result is passed to {@link #dependsOn(List<ContainerDependency>)}.
         * 
         * @param dependsOn
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dependsOn(java.util.Collection<ContainerDependency>)
         */
        Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn);
    }

    static final class BuilderImpl implements Builder {
        private String containerName;

        private String imageUri;

        private ContainerMemoryLimits memoryLimits;

        private ContainerPortConfiguration portConfiguration;

        private Integer cpu;

        private ContainerHealthCheck healthCheck;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private Boolean essential;

        private List<String> entryPoint = DefaultSdkAutoConstructList.getInstance();

        private String workingDirectory;

        private List<ContainerEnvironment> environment = DefaultSdkAutoConstructList.getInstance();

        private List<ContainerDependency> dependsOn = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDefinitionInput model) {
            containerName(model.containerName);
            imageUri(model.imageUri);
            memoryLimits(model.memoryLimits);
            portConfiguration(model.portConfiguration);
            cpu(model.cpu);
            healthCheck(model.healthCheck);
            command(model.command);
            essential(model.essential);
            entryPoint(model.entryPoint);
            workingDirectory(model.workingDirectory);
            environment(model.environment);
            dependsOn(model.dependsOn);
        }

        public final String getContainerName() {
            return containerName;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final String getImageUri() {
            return imageUri;
        }

        public final void setImageUri(String imageUri) {
            this.imageUri = imageUri;
        }

        @Override
        public final Builder imageUri(String imageUri) {
            this.imageUri = imageUri;
            return this;
        }

        public final ContainerMemoryLimits.Builder getMemoryLimits() {
            return memoryLimits != null ? memoryLimits.toBuilder() : null;
        }

        public final void setMemoryLimits(ContainerMemoryLimits.BuilderImpl memoryLimits) {
            this.memoryLimits = memoryLimits != null ? memoryLimits.build() : null;
        }

        @Override
        public final Builder memoryLimits(ContainerMemoryLimits memoryLimits) {
            this.memoryLimits = memoryLimits;
            return this;
        }

        public final ContainerPortConfiguration.Builder getPortConfiguration() {
            return portConfiguration != null ? portConfiguration.toBuilder() : null;
        }

        public final void setPortConfiguration(ContainerPortConfiguration.BuilderImpl portConfiguration) {
            this.portConfiguration = portConfiguration != null ? portConfiguration.build() : null;
        }

        @Override
        public final Builder portConfiguration(ContainerPortConfiguration portConfiguration) {
            this.portConfiguration = portConfiguration;
            return this;
        }

        public final Integer getCpu() {
            return cpu;
        }

        public final void setCpu(Integer cpu) {
            this.cpu = cpu;
        }

        @Override
        public final Builder cpu(Integer cpu) {
            this.cpu = cpu;
            return this;
        }

        public final ContainerHealthCheck.Builder getHealthCheck() {
            return healthCheck != null ? healthCheck.toBuilder() : null;
        }

        public final void setHealthCheck(ContainerHealthCheck.BuilderImpl healthCheck) {
            this.healthCheck = healthCheck != null ? healthCheck.build() : null;
        }

        @Override
        public final Builder healthCheck(ContainerHealthCheck healthCheck) {
            this.healthCheck = healthCheck;
            return this;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = ContainerCommandStringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = ContainerCommandStringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final Boolean getEssential() {
            return essential;
        }

        public final void setEssential(Boolean essential) {
            this.essential = essential;
        }

        @Override
        public final Builder essential(Boolean essential) {
            this.essential = essential;
            return this;
        }

        public final Collection<String> getEntryPoint() {
            if (entryPoint instanceof SdkAutoConstructList) {
                return null;
            }
            return entryPoint;
        }

        public final void setEntryPoint(Collection<String> entryPoint) {
            this.entryPoint = ContainerEntryPointListCopier.copy(entryPoint);
        }

        @Override
        public final Builder entryPoint(Collection<String> entryPoint) {
            this.entryPoint = ContainerEntryPointListCopier.copy(entryPoint);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entryPoint(String... entryPoint) {
            entryPoint(Arrays.asList(entryPoint));
            return this;
        }

        public final String getWorkingDirectory() {
            return workingDirectory;
        }

        public final void setWorkingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
        }

        @Override
        public final Builder workingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
            return this;
        }

        public final List<ContainerEnvironment.Builder> getEnvironment() {
            List<ContainerEnvironment.Builder> result = ContainerEnvironmentListCopier.copyToBuilder(this.environment);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironment(Collection<ContainerEnvironment.BuilderImpl> environment) {
            this.environment = ContainerEnvironmentListCopier.copyFromBuilder(environment);
        }

        @Override
        public final Builder environment(Collection<ContainerEnvironment> environment) {
            this.environment = ContainerEnvironmentListCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(ContainerEnvironment... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<ContainerEnvironment.Builder>... environment) {
            environment(Stream.of(environment).map(c -> ContainerEnvironment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ContainerDependency.Builder> getDependsOn() {
            List<ContainerDependency.Builder> result = ContainerDependencyListCopier.copyToBuilder(this.dependsOn);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDependsOn(Collection<ContainerDependency.BuilderImpl> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copyFromBuilder(dependsOn);
        }

        @Override
        public final Builder dependsOn(Collection<ContainerDependency> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copy(dependsOn);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(ContainerDependency... dependsOn) {
            dependsOn(Arrays.asList(dependsOn));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn) {
            dependsOn(Stream.of(dependsOn).map(c -> ContainerDependency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ContainerDefinitionInput build() {
            return new ContainerDefinitionInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
