/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Rule that controls how a fleet is scaled. Scaling policies are uniquely identified by the combination of name and
 * fleet ID.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScalingPolicy implements SdkPojo, Serializable, ToCopyableBuilder<ScalingPolicy.Builder, ScalingPolicy> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(ScalingPolicy::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> FLEET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetArn").getter(getter(ScalingPolicy::fleetArn)).setter(setter(Builder::fleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ScalingPolicy::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ScalingPolicy::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> SCALING_ADJUSTMENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ScalingAdjustment").getter(getter(ScalingPolicy::scalingAdjustment))
            .setter(setter(Builder::scalingAdjustment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingAdjustment").build()).build();

    private static final SdkField<String> SCALING_ADJUSTMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalingAdjustmentType").getter(getter(ScalingPolicy::scalingAdjustmentTypeAsString))
            .setter(setter(Builder::scalingAdjustmentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingAdjustmentType").build())
            .build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComparisonOperator").getter(getter(ScalingPolicy::comparisonOperatorAsString))
            .setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final SdkField<Double> THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Threshold").getter(getter(ScalingPolicy::threshold)).setter(setter(Builder::threshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Threshold").build()).build();

    private static final SdkField<Integer> EVALUATION_PERIODS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("EvaluationPeriods").getter(getter(ScalingPolicy::evaluationPeriods))
            .setter(setter(Builder::evaluationPeriods))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationPeriods").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(ScalingPolicy::metricNameAsString)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<String> POLICY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyType").getter(getter(ScalingPolicy::policyTypeAsString)).setter(setter(Builder::policyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyType").build()).build();

    private static final SdkField<TargetConfiguration> TARGET_CONFIGURATION_FIELD = SdkField
            .<TargetConfiguration> builder(MarshallingType.SDK_POJO).memberName("TargetConfiguration")
            .getter(getter(ScalingPolicy::targetConfiguration)).setter(setter(Builder::targetConfiguration))
            .constructor(TargetConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetConfiguration").build())
            .build();

    private static final SdkField<String> UPDATE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpdateStatus").getter(getter(ScalingPolicy::updateStatusAsString)).setter(setter(Builder::updateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateStatus").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Location").getter(getter(ScalingPolicy::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Location").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD,
            FLEET_ARN_FIELD, NAME_FIELD, STATUS_FIELD, SCALING_ADJUSTMENT_FIELD, SCALING_ADJUSTMENT_TYPE_FIELD,
            COMPARISON_OPERATOR_FIELD, THRESHOLD_FIELD, EVALUATION_PERIODS_FIELD, METRIC_NAME_FIELD, POLICY_TYPE_FIELD,
            TARGET_CONFIGURATION_FIELD, UPDATE_STATUS_FIELD, LOCATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String fleetArn;

    private final String name;

    private final String status;

    private final Integer scalingAdjustment;

    private final String scalingAdjustmentType;

    private final String comparisonOperator;

    private final Double threshold;

    private final Integer evaluationPeriods;

    private final String metricName;

    private final String policyType;

    private final TargetConfiguration targetConfiguration;

    private final String updateStatus;

    private final String location;

    private ScalingPolicy(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.fleetArn = builder.fleetArn;
        this.name = builder.name;
        this.status = builder.status;
        this.scalingAdjustment = builder.scalingAdjustment;
        this.scalingAdjustmentType = builder.scalingAdjustmentType;
        this.comparisonOperator = builder.comparisonOperator;
        this.threshold = builder.threshold;
        this.evaluationPeriods = builder.evaluationPeriods;
        this.metricName = builder.metricName;
        this.policyType = builder.policyType;
        this.targetConfiguration = builder.targetConfiguration;
        this.updateStatus = builder.updateStatus;
        this.location = builder.location;
    }

    /**
     * <p>
     * A unique identifier for the fleet that is associated with this scaling policy.
     * </p>
     * 
     * @return A unique identifier for the fleet that is associated with this scaling policy.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to a Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all
     * Regions. Format is <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>
     * .
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
     *         Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>.
     */
    public final String fleetArn() {
        return fleetArn;
    }

    /**
     * <p>
     * A descriptive label that is associated with a fleet's scaling policy. Policy names do not need to be unique.
     * </p>
     * 
     * @return A descriptive label that is associated with a fleet's scaling policy. Policy names do not need to be
     *         unique.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
     * status. Scaling policies can be suspended for individual fleets. If the policy is suspended for a fleet, the
     * policy status does not change.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATING</b> -- A change is being made to the scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETING</b> -- The scaling policy is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETED</b> -- The scaling policy has been deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ScalingStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the scaling policy. The scaling policy can be in force only when in an
     *         <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets. If the policy is
     *         suspended for a fleet, the policy status does not change.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATING</b> -- A change is being made to the scaling policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETING</b> -- The scaling policy is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETED</b> -- The scaling policy has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     *         </p>
     *         </li>
     * @see ScalingStatusType
     */
    public final ScalingStatusType status() {
        return ScalingStatusType.fromValue(status);
    }

    /**
     * <p>
     * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
     * status. Scaling policies can be suspended for individual fleets. If the policy is suspended for a fleet, the
     * policy status does not change.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATING</b> -- A change is being made to the scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETING</b> -- The scaling policy is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETED</b> -- The scaling policy has been deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ScalingStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the scaling policy. The scaling policy can be in force only when in an
     *         <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets. If the policy is
     *         suspended for a fleet, the policy status does not change.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATING</b> -- A change is being made to the scaling policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETING</b> -- The scaling policy is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETED</b> -- The scaling policy has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     *         </p>
     *         </li>
     * @see ScalingStatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Amount of adjustment to make, based on the scaling adjustment type.
     * </p>
     * 
     * @return Amount of adjustment to make, based on the scaling adjustment type.
     */
    public final Integer scalingAdjustment() {
        return scalingAdjustment;
    }

    /**
     * <p>
     * The type of adjustment to make to a fleet's instance count.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
     * Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment, read
     * as a percentage. Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scalingAdjustmentType} will return {@link ScalingAdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #scalingAdjustmentTypeAsString}.
     * </p>
     * 
     * @return The type of adjustment to make to a fleet's instance count.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
     *         count. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
     *         adjustment, read as a percentage. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     * @see ScalingAdjustmentType
     */
    public final ScalingAdjustmentType scalingAdjustmentType() {
        return ScalingAdjustmentType.fromValue(scalingAdjustmentType);
    }

    /**
     * <p>
     * The type of adjustment to make to a fleet's instance count.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
     * Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment, read
     * as a percentage. Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scalingAdjustmentType} will return {@link ScalingAdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #scalingAdjustmentTypeAsString}.
     * </p>
     * 
     * @return The type of adjustment to make to a fleet's instance count.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
     *         count. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
     *         adjustment, read as a percentage. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     * @see ScalingAdjustmentType
     */
    public final String scalingAdjustmentTypeAsString() {
        return scalingAdjustmentType;
    }

    /**
     * <p>
     * Comparison operator to use when measuring a metric against the threshold value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperatorType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return Comparison operator to use when measuring a metric against the threshold value.
     * @see ComparisonOperatorType
     */
    public final ComparisonOperatorType comparisonOperator() {
        return ComparisonOperatorType.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * Comparison operator to use when measuring a metric against the threshold value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperatorType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return Comparison operator to use when measuring a metric against the threshold value.
     * @see ComparisonOperatorType
     */
    public final String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * Metric value used to trigger a scaling event.
     * </p>
     * 
     * @return Metric value used to trigger a scaling event.
     */
    public final Double threshold() {
        return threshold;
    }

    /**
     * <p>
     * Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is triggered.
     * </p>
     * 
     * @return Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is
     *         triggered.
     */
    public final Integer evaluationPeriods() {
        return evaluationPeriods;
    }

    /**
     * <p>
     * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     * descriptions of fleet metrics, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
     * GameLift with Amazon CloudWatch</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
     * capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
     * sessions that are not currently accepting players. Reserved player slots are not included.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
     * reserved for a player.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet could
     * host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
     * sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
     * top-priority destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the current
     * fleet is the top-priority destination.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     *         descriptions of fleet metrics, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
     *         Amazon GameLift with Amazon CloudWatch</a>. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
     *         current capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
     *         game sessions that are not currently accepting players. Reserved player slots are not included.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or
     *         are reserved for a player.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
     *         fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
     *         policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero
     *         game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
     *         the top-priority destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
     *         current fleet is the top-priority destination.
     *         </p>
     *         </li>
     * @see MetricName
     */
    public final MetricName metricName() {
        return MetricName.fromValue(metricName);
    }

    /**
     * <p>
     * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     * descriptions of fleet metrics, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
     * GameLift with Amazon CloudWatch</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
     * capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
     * sessions that are not currently accepting players. Reserved player slots are not included.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
     * reserved for a player.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet could
     * host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
     * sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
     * top-priority destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the current
     * fleet is the top-priority destination.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     *         descriptions of fleet metrics, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
     *         Amazon GameLift with Amazon CloudWatch</a>. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
     *         current capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
     *         game sessions that are not currently accepting players. Reserved player slots are not included.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or
     *         are reserved for a player.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
     *         fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
     *         policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero
     *         game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
     *         the top-priority destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
     *         current fleet is the top-priority destination.
     *         </p>
     *         </li>
     * @see MetricName
     */
    public final String metricNameAsString() {
        return metricName;
    }

    /**
     * <p>
     * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>, <i>EvaluationPeriods</i>,
     * <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     *         'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     *         following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
     *         <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * @see PolicyType
     */
    public final PolicyType policyType() {
        return PolicyType.fromValue(policyType);
    }

    /**
     * <p>
     * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>, <i>EvaluationPeriods</i>,
     * <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     *         'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     *         following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
     *         <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * @see PolicyType
     */
    public final String policyTypeAsString() {
        return policyType;
    }

    /**
     * <p>
     * An object that contains settings for a target-based scaling policy.
     * </p>
     * 
     * @return An object that contains settings for a target-based scaling policy.
     */
    public final TargetConfiguration targetConfiguration() {
        return targetConfiguration;
    }

    /**
     * <p>
     * The current status of the fleet's scaling policies in a requested fleet location. The status
     * <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been completed
     * for the location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link LocationUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #updateStatusAsString}.
     * </p>
     * 
     * @return The current status of the fleet's scaling policies in a requested fleet location. The status
     *         <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been
     *         completed for the location.
     * @see LocationUpdateStatus
     */
    public final LocationUpdateStatus updateStatus() {
        return LocationUpdateStatus.fromValue(updateStatus);
    }

    /**
     * <p>
     * The current status of the fleet's scaling policies in a requested fleet location. The status
     * <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been completed
     * for the location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link LocationUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #updateStatusAsString}.
     * </p>
     * 
     * @return The current status of the fleet's scaling policies in a requested fleet location. The status
     *         <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been
     *         completed for the location.
     * @see LocationUpdateStatus
     */
    public final String updateStatusAsString() {
        return updateStatus;
    }

    /**
     * <p>
     * The fleet location.
     * </p>
     * 
     * @return The fleet location.
     */
    public final String location() {
        return location;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scalingAdjustment());
        hashCode = 31 * hashCode + Objects.hashCode(scalingAdjustmentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationPeriods());
        hashCode = 31 * hashCode + Objects.hashCode(metricNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(updateStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScalingPolicy)) {
            return false;
        }
        ScalingPolicy other = (ScalingPolicy) obj;
        return Objects.equals(fleetId(), other.fleetId()) && Objects.equals(fleetArn(), other.fleetArn())
                && Objects.equals(name(), other.name()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(scalingAdjustment(), other.scalingAdjustment())
                && Objects.equals(scalingAdjustmentTypeAsString(), other.scalingAdjustmentTypeAsString())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(threshold(), other.threshold())
                && Objects.equals(evaluationPeriods(), other.evaluationPeriods())
                && Objects.equals(metricNameAsString(), other.metricNameAsString())
                && Objects.equals(policyTypeAsString(), other.policyTypeAsString())
                && Objects.equals(targetConfiguration(), other.targetConfiguration())
                && Objects.equals(updateStatusAsString(), other.updateStatusAsString())
                && Objects.equals(location(), other.location());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScalingPolicy").add("FleetId", fleetId()).add("FleetArn", fleetArn()).add("Name", name())
                .add("Status", statusAsString()).add("ScalingAdjustment", scalingAdjustment())
                .add("ScalingAdjustmentType", scalingAdjustmentTypeAsString())
                .add("ComparisonOperator", comparisonOperatorAsString()).add("Threshold", threshold())
                .add("EvaluationPeriods", evaluationPeriods()).add("MetricName", metricNameAsString())
                .add("PolicyType", policyTypeAsString()).add("TargetConfiguration", targetConfiguration())
                .add("UpdateStatus", updateStatusAsString()).add("Location", location()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "FleetArn":
            return Optional.ofNullable(clazz.cast(fleetArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ScalingAdjustment":
            return Optional.ofNullable(clazz.cast(scalingAdjustment()));
        case "ScalingAdjustmentType":
            return Optional.ofNullable(clazz.cast(scalingAdjustmentTypeAsString()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "Threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        case "EvaluationPeriods":
            return Optional.ofNullable(clazz.cast(evaluationPeriods()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricNameAsString()));
        case "PolicyType":
            return Optional.ofNullable(clazz.cast(policyTypeAsString()));
        case "TargetConfiguration":
            return Optional.ofNullable(clazz.cast(targetConfiguration()));
        case "UpdateStatus":
            return Optional.ofNullable(clazz.cast(updateStatusAsString()));
        case "Location":
            return Optional.ofNullable(clazz.cast(location()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ScalingPolicy, T> g) {
        return obj -> g.apply((ScalingPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScalingPolicy> {
        /**
         * <p>
         * A unique identifier for the fleet that is associated with this scaling policy.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for the fleet that is associated with this scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>.
         * </p>
         * 
         * @param fleetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions.
         *        Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetArn(String fleetArn);

        /**
         * <p>
         * A descriptive label that is associated with a fleet's scaling policy. Policy names do not need to be unique.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with a fleet's scaling policy. Policy names do not need to be
         *        unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
         * status. Scaling policies can be suspended for individual fleets. If the policy is suspended for a fleet, the
         * policy status does not change.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATING</b> -- A change is being made to the scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETING</b> -- The scaling policy is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETED</b> -- The scaling policy has been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the scaling policy. The scaling policy can be in force only when in an
         *        <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets. If the policy is
         *        suspended for a fleet, the policy status does not change.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATING</b> -- A change is being made to the scaling policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETING</b> -- The scaling policy is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETED</b> -- The scaling policy has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         *        </p>
         *        </li>
         * @see ScalingStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStatusType
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
         * status. Scaling policies can be suspended for individual fleets. If the policy is suspended for a fleet, the
         * policy status does not change.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATING</b> -- A change is being made to the scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETING</b> -- The scaling policy is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETED</b> -- The scaling policy has been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the scaling policy. The scaling policy can be in force only when in an
         *        <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets. If the policy is
         *        suspended for a fleet, the policy status does not change.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATING</b> -- A change is being made to the scaling policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETING</b> -- The scaling policy is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETED</b> -- The scaling policy has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         *        </p>
         *        </li>
         * @see ScalingStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStatusType
         */
        Builder status(ScalingStatusType status);

        /**
         * <p>
         * Amount of adjustment to make, based on the scaling adjustment type.
         * </p>
         * 
         * @param scalingAdjustment
         *        Amount of adjustment to make, based on the scaling adjustment type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingAdjustment(Integer scalingAdjustment);

        /**
         * <p>
         * The type of adjustment to make to a fleet's instance count.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
         * Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment,
         * read as a percentage. Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalingAdjustmentType
         *        The type of adjustment to make to a fleet's instance count.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
         *        count. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
         *        adjustment, read as a percentage. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         * @see ScalingAdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingAdjustmentType
         */
        Builder scalingAdjustmentType(String scalingAdjustmentType);

        /**
         * <p>
         * The type of adjustment to make to a fleet's instance count.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
         * Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment,
         * read as a percentage. Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalingAdjustmentType
         *        The type of adjustment to make to a fleet's instance count.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
         *        count. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
         *        adjustment, read as a percentage. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         * @see ScalingAdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingAdjustmentType
         */
        Builder scalingAdjustmentType(ScalingAdjustmentType scalingAdjustmentType);

        /**
         * <p>
         * Comparison operator to use when measuring a metric against the threshold value.
         * </p>
         * 
         * @param comparisonOperator
         *        Comparison operator to use when measuring a metric against the threshold value.
         * @see ComparisonOperatorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperatorType
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * Comparison operator to use when measuring a metric against the threshold value.
         * </p>
         * 
         * @param comparisonOperator
         *        Comparison operator to use when measuring a metric against the threshold value.
         * @see ComparisonOperatorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperatorType
         */
        Builder comparisonOperator(ComparisonOperatorType comparisonOperator);

        /**
         * <p>
         * Metric value used to trigger a scaling event.
         * </p>
         * 
         * @param threshold
         *        Metric value used to trigger a scaling event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Double threshold);

        /**
         * <p>
         * Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is
         * triggered.
         * </p>
         * 
         * @param evaluationPeriods
         *        Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is
         *        triggered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationPeriods(Integer evaluationPeriods);

        /**
         * <p>
         * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         * descriptions of fleet metrics, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
         * GameLift with Amazon CloudWatch</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
         * capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
         * sessions that are not currently accepting players. Reserved player slots are not included.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
         * reserved for a player.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet
         * could host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
         * sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
         * top-priority destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
         * current fleet is the top-priority destination.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         *        descriptions of fleet metrics, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
         *        Amazon GameLift with Amazon CloudWatch</a>. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
         *        current capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
         *        game sessions that are not currently accepting players. Reserved player slots are not included.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player
         *        or are reserved for a player.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
         *        fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
         *        policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting
         *        zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
         *        the top-priority destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where
         *        the current fleet is the top-priority destination.
         *        </p>
         *        </li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         * descriptions of fleet metrics, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
         * GameLift with Amazon CloudWatch</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
         * capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
         * sessions that are not currently accepting players. Reserved player slots are not included.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
         * reserved for a player.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet
         * could host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
         * sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
         * top-priority destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
         * current fleet is the top-priority destination.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         *        descriptions of fleet metrics, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
         *        Amazon GameLift with Amazon CloudWatch</a>. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
         *        current capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
         *        game sessions that are not currently accepting players. Reserved player slots are not included.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player
         *        or are reserved for a player.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
         *        fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
         *        policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting
         *        zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
         *        the top-priority destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where
         *        the current fleet is the top-priority destination.
         *        </p>
         *        </li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder metricName(MetricName metricName);

        /**
         * <p>
         * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
         * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
         * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         * <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * </p>
         * 
         * @param policyType
         *        The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i>
         *        to 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy
         *        set the following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         *        <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(String policyType);

        /**
         * <p>
         * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
         * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
         * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         * <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * </p>
         * 
         * @param policyType
         *        The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i>
         *        to 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy
         *        set the following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         *        <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(PolicyType policyType);

        /**
         * <p>
         * An object that contains settings for a target-based scaling policy.
         * </p>
         * 
         * @param targetConfiguration
         *        An object that contains settings for a target-based scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetConfiguration(TargetConfiguration targetConfiguration);

        /**
         * <p>
         * An object that contains settings for a target-based scaling policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link TargetConfiguration.Builder} avoiding the
         * need to create one manually via {@link TargetConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TargetConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #targetConfiguration(TargetConfiguration)}.
         * 
         * @param targetConfiguration
         *        a consumer that will call methods on {@link TargetConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetConfiguration(TargetConfiguration)
         */
        default Builder targetConfiguration(Consumer<TargetConfiguration.Builder> targetConfiguration) {
            return targetConfiguration(TargetConfiguration.builder().applyMutation(targetConfiguration).build());
        }

        /**
         * <p>
         * The current status of the fleet's scaling policies in a requested fleet location. The status
         * <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been
         * completed for the location.
         * </p>
         * 
         * @param updateStatus
         *        The current status of the fleet's scaling policies in a requested fleet location. The status
         *        <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been
         *        completed for the location.
         * @see LocationUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocationUpdateStatus
         */
        Builder updateStatus(String updateStatus);

        /**
         * <p>
         * The current status of the fleet's scaling policies in a requested fleet location. The status
         * <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been
         * completed for the location.
         * </p>
         * 
         * @param updateStatus
         *        The current status of the fleet's scaling policies in a requested fleet location. The status
         *        <code>PENDING_UPDATE</code> indicates that an update was requested for the fleet but has not yet been
         *        completed for the location.
         * @see LocationUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocationUpdateStatus
         */
        Builder updateStatus(LocationUpdateStatus updateStatus);

        /**
         * <p>
         * The fleet location.
         * </p>
         * 
         * @param location
         *        The fleet location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String fleetArn;

        private String name;

        private String status;

        private Integer scalingAdjustment;

        private String scalingAdjustmentType;

        private String comparisonOperator;

        private Double threshold;

        private Integer evaluationPeriods;

        private String metricName;

        private String policyType;

        private TargetConfiguration targetConfiguration;

        private String updateStatus;

        private String location;

        private BuilderImpl() {
        }

        private BuilderImpl(ScalingPolicy model) {
            fleetId(model.fleetId);
            fleetArn(model.fleetArn);
            name(model.name);
            status(model.status);
            scalingAdjustment(model.scalingAdjustment);
            scalingAdjustmentType(model.scalingAdjustmentType);
            comparisonOperator(model.comparisonOperator);
            threshold(model.threshold);
            evaluationPeriods(model.evaluationPeriods);
            metricName(model.metricName);
            policyType(model.policyType);
            targetConfiguration(model.targetConfiguration);
            updateStatus(model.updateStatus);
            location(model.location);
        }

        public final String getFleetId() {
            return fleetId;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final String getFleetArn() {
            return fleetArn;
        }

        public final void setFleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
        }

        @Override
        public final Builder fleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ScalingStatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Integer getScalingAdjustment() {
            return scalingAdjustment;
        }

        public final void setScalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
        }

        @Override
        public final Builder scalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
            return this;
        }

        public final String getScalingAdjustmentType() {
            return scalingAdjustmentType;
        }

        public final void setScalingAdjustmentType(String scalingAdjustmentType) {
            this.scalingAdjustmentType = scalingAdjustmentType;
        }

        @Override
        public final Builder scalingAdjustmentType(String scalingAdjustmentType) {
            this.scalingAdjustmentType = scalingAdjustmentType;
            return this;
        }

        @Override
        public final Builder scalingAdjustmentType(ScalingAdjustmentType scalingAdjustmentType) {
            this.scalingAdjustmentType(scalingAdjustmentType == null ? null : scalingAdjustmentType.toString());
            return this;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperatorType comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final Double getThreshold() {
            return threshold;
        }

        public final void setThreshold(Double threshold) {
            this.threshold = threshold;
        }

        @Override
        public final Builder threshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        public final Integer getEvaluationPeriods() {
            return evaluationPeriods;
        }

        public final void setEvaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
        }

        @Override
        public final Builder evaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
            return this;
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        public final Builder metricName(MetricName metricName) {
            this.metricName(metricName == null ? null : metricName.toString());
            return this;
        }

        public final String getPolicyType() {
            return policyType;
        }

        public final void setPolicyType(String policyType) {
            this.policyType = policyType;
        }

        @Override
        public final Builder policyType(String policyType) {
            this.policyType = policyType;
            return this;
        }

        @Override
        public final Builder policyType(PolicyType policyType) {
            this.policyType(policyType == null ? null : policyType.toString());
            return this;
        }

        public final TargetConfiguration.Builder getTargetConfiguration() {
            return targetConfiguration != null ? targetConfiguration.toBuilder() : null;
        }

        public final void setTargetConfiguration(TargetConfiguration.BuilderImpl targetConfiguration) {
            this.targetConfiguration = targetConfiguration != null ? targetConfiguration.build() : null;
        }

        @Override
        public final Builder targetConfiguration(TargetConfiguration targetConfiguration) {
            this.targetConfiguration = targetConfiguration;
            return this;
        }

        public final String getUpdateStatus() {
            return updateStatus;
        }

        public final void setUpdateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
        }

        @Override
        public final Builder updateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
            return this;
        }

        @Override
        public final Builder updateStatus(LocationUpdateStatus updateStatus) {
            this.updateStatus(updateStatus == null ? null : updateStatus.toString());
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        @Override
        public ScalingPolicy build() {
            return new ScalingPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
