/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the Amazon GameLift FleetIQ and game server groups.</b>
 * </p>
 * <p>
 * Properties describing a game server that is running on an instance in a game server group.
 * </p>
 * <p>
 * A game server is created by a successful call to <code>RegisterGameServer</code> and deleted by calling
 * <code>DeregisterGameServer</code>. A game server is claimed to host a game session by calling
 * <code>ClaimGameServer</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameServer implements SdkPojo, Serializable, ToCopyableBuilder<GameServer.Builder, GameServer> {
    private static final SdkField<String> GAME_SERVER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerGroupName").getter(getter(GameServer::gameServerGroupName))
            .setter(setter(Builder::gameServerGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupName").build())
            .build();

    private static final SdkField<String> GAME_SERVER_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerGroupArn").getter(getter(GameServer::gameServerGroupArn))
            .setter(setter(Builder::gameServerGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupArn").build())
            .build();

    private static final SdkField<String> GAME_SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerId").getter(getter(GameServer::gameServerId)).setter(setter(Builder::gameServerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerId").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(GameServer::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> CONNECTION_INFO_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionInfo").getter(getter(GameServer::connectionInfo)).setter(setter(Builder::connectionInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionInfo").build()).build();

    private static final SdkField<String> GAME_SERVER_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerData").getter(getter(GameServer::gameServerData)).setter(setter(Builder::gameServerData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerData").build()).build();

    private static final SdkField<String> CLAIM_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClaimStatus").getter(getter(GameServer::claimStatusAsString)).setter(setter(Builder::claimStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClaimStatus").build()).build();

    private static final SdkField<String> UTILIZATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UtilizationStatus").getter(getter(GameServer::utilizationStatusAsString))
            .setter(setter(Builder::utilizationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UtilizationStatus").build()).build();

    private static final SdkField<Instant> REGISTRATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RegistrationTime").getter(getter(GameServer::registrationTime))
            .setter(setter(Builder::registrationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegistrationTime").build()).build();

    private static final SdkField<Instant> LAST_CLAIM_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastClaimTime").getter(getter(GameServer::lastClaimTime)).setter(setter(Builder::lastClaimTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastClaimTime").build()).build();

    private static final SdkField<Instant> LAST_HEALTH_CHECK_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastHealthCheckTime").getter(getter(GameServer::lastHealthCheckTime))
            .setter(setter(Builder::lastHealthCheckTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastHealthCheckTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GAME_SERVER_GROUP_NAME_FIELD,
            GAME_SERVER_GROUP_ARN_FIELD, GAME_SERVER_ID_FIELD, INSTANCE_ID_FIELD, CONNECTION_INFO_FIELD, GAME_SERVER_DATA_FIELD,
            CLAIM_STATUS_FIELD, UTILIZATION_STATUS_FIELD, REGISTRATION_TIME_FIELD, LAST_CLAIM_TIME_FIELD,
            LAST_HEALTH_CHECK_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String gameServerGroupName;

    private final String gameServerGroupArn;

    private final String gameServerId;

    private final String instanceId;

    private final String connectionInfo;

    private final String gameServerData;

    private final String claimStatus;

    private final String utilizationStatus;

    private final Instant registrationTime;

    private final Instant lastClaimTime;

    private final Instant lastHealthCheckTime;

    private GameServer(BuilderImpl builder) {
        this.gameServerGroupName = builder.gameServerGroupName;
        this.gameServerGroupArn = builder.gameServerGroupArn;
        this.gameServerId = builder.gameServerId;
        this.instanceId = builder.instanceId;
        this.connectionInfo = builder.connectionInfo;
        this.gameServerData = builder.gameServerData;
        this.claimStatus = builder.claimStatus;
        this.utilizationStatus = builder.utilizationStatus;
        this.registrationTime = builder.registrationTime;
        this.lastClaimTime = builder.lastClaimTime;
        this.lastHealthCheckTime = builder.lastHealthCheckTime;
    }

    /**
     * <p>
     * A unique identifier for the game server group where the game server is running.
     * </p>
     * 
     * @return A unique identifier for the game server group where the game server is running.
     */
    public final String gameServerGroupName() {
        return gameServerGroupName;
    }

    /**
     * <p>
     * The ARN identifier for the game server group where the game server is located.
     * </p>
     * 
     * @return The ARN identifier for the game server group where the game server is located.
     */
    public final String gameServerGroupArn() {
        return gameServerGroupArn;
    }

    /**
     * <p>
     * A custom string that uniquely identifies the game server. Game server IDs are developer-defined and are unique
     * across all game server groups in an Amazon Web Services account.
     * </p>
     * 
     * @return A custom string that uniquely identifies the game server. Game server IDs are developer-defined and are
     *         unique across all game server groups in an Amazon Web Services account.
     */
    public final String gameServerId() {
        return gameServerId;
    }

    /**
     * <p>
     * The unique identifier for the instance where the game server is running. This ID is available in the instance
     * metadata. EC2 instance IDs use a 17-character format, for example: <code>i-1234567890abcdef0</code>.
     * </p>
     * 
     * @return The unique identifier for the instance where the game server is running. This ID is available in the
     *         instance metadata. EC2 instance IDs use a 17-character format, for example:
     *         <code>i-1234567890abcdef0</code>.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The port and IP address that must be used to establish a client connection to the game server.
     * </p>
     * 
     * @return The port and IP address that must be used to establish a client connection to the game server.
     */
    public final String connectionInfo() {
        return connectionInfo;
    }

    /**
     * <p>
     * A set of custom game server properties, formatted as a single string value. This data is passed to a game client
     * or service when it requests information on game servers.
     * </p>
     * 
     * @return A set of custom game server properties, formatted as a single string value. This data is passed to a game
     *         client or service when it requests information on game servers.
     */
    public final String gameServerData() {
        return gameServerData;
    }

    /**
     * <p>
     * Indicates when an available game server has been reserved for gameplay but has not yet started hosting a game.
     * Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of one minute. During
     * this time, game clients connect to the game server to start the game and trigger the game server to update its
     * utilization status. After one minute, the game server claim status reverts to null.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #claimStatus} will
     * return {@link GameServerClaimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #claimStatusAsString}.
     * </p>
     * 
     * @return Indicates when an available game server has been reserved for gameplay but has not yet started hosting a
     *         game. Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of one
     *         minute. During this time, game clients connect to the game server to start the game and trigger the game
     *         server to update its utilization status. After one minute, the game server claim status reverts to null.
     * @see GameServerClaimStatus
     */
    public final GameServerClaimStatus claimStatus() {
        return GameServerClaimStatus.fromValue(claimStatus);
    }

    /**
     * <p>
     * Indicates when an available game server has been reserved for gameplay but has not yet started hosting a game.
     * Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of one minute. During
     * this time, game clients connect to the game server to start the game and trigger the game server to update its
     * utilization status. After one minute, the game server claim status reverts to null.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #claimStatus} will
     * return {@link GameServerClaimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #claimStatusAsString}.
     * </p>
     * 
     * @return Indicates when an available game server has been reserved for gameplay but has not yet started hosting a
     *         game. Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of one
     *         minute. During this time, game clients connect to the game server to start the game and trigger the game
     *         server to update its utilization status. After one minute, the game server claim status reverts to null.
     * @see GameServerClaimStatus
     */
    public final String claimStatusAsString() {
        return claimStatus;
    }

    /**
     * <p>
     * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been claimed remains
     * in this status until it reports game hosting activity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UTILIZED</code> - The game server is currently hosting a game session with players.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #utilizationStatus}
     * will return {@link GameServerUtilizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #utilizationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether the game server is currently available for new games or is busy. Possible statuses
     *         include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been claimed
     *         remains in this status until it reports game hosting activity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UTILIZED</code> - The game server is currently hosting a game session with players.
     *         </p>
     *         </li>
     * @see GameServerUtilizationStatus
     */
    public final GameServerUtilizationStatus utilizationStatus() {
        return GameServerUtilizationStatus.fromValue(utilizationStatus);
    }

    /**
     * <p>
     * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been claimed remains
     * in this status until it reports game hosting activity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UTILIZED</code> - The game server is currently hosting a game session with players.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #utilizationStatus}
     * will return {@link GameServerUtilizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #utilizationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether the game server is currently available for new games or is busy. Possible statuses
     *         include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been claimed
     *         remains in this status until it reports game hosting activity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UTILIZED</code> - The game server is currently hosting a game session with players.
     *         </p>
     *         </li>
     * @see GameServerUtilizationStatus
     */
    public final String utilizationStatusAsString() {
        return utilizationStatus;
    }

    /**
     * <p>
     * Timestamp that indicates when the game server registered. The format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return Timestamp that indicates when the game server registered. The format is a number expressed in Unix time
     *         as milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant registrationTime() {
        return registrationTime;
    }

    /**
     * <p>
     * Timestamp that indicates the last time the game server was claimed. The format is a number expressed in Unix time
     * as milliseconds (for example <code>"1469498468.057"</code>). This value is used to calculate when a claimed game
     * server's status should revert to null.
     * </p>
     * 
     * @return Timestamp that indicates the last time the game server was claimed. The format is a number expressed in
     *         Unix time as milliseconds (for example <code>"1469498468.057"</code>). This value is used to calculate
     *         when a claimed game server's status should revert to null.
     */
    public final Instant lastClaimTime() {
        return lastClaimTime;
    }

    /**
     * <p>
     * Timestamp that indicates the last time the game server was updated with health status. The format is a number
     * expressed in Unix time as milliseconds (for example <code>"1469498468.057"</code>). After game server
     * registration, this property is only changed when a game server update specifies a health check value.
     * </p>
     * 
     * @return Timestamp that indicates the last time the game server was updated with health status. The format is a
     *         number expressed in Unix time as milliseconds (for example <code>"1469498468.057"</code>). After game
     *         server registration, this property is only changed when a game server update specifies a health check
     *         value.
     */
    public final Instant lastHealthCheckTime() {
        return lastHealthCheckTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(connectionInfo());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerData());
        hashCode = 31 * hashCode + Objects.hashCode(claimStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(utilizationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(registrationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastClaimTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastHealthCheckTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameServer)) {
            return false;
        }
        GameServer other = (GameServer) obj;
        return Objects.equals(gameServerGroupName(), other.gameServerGroupName())
                && Objects.equals(gameServerGroupArn(), other.gameServerGroupArn())
                && Objects.equals(gameServerId(), other.gameServerId()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(connectionInfo(), other.connectionInfo())
                && Objects.equals(gameServerData(), other.gameServerData())
                && Objects.equals(claimStatusAsString(), other.claimStatusAsString())
                && Objects.equals(utilizationStatusAsString(), other.utilizationStatusAsString())
                && Objects.equals(registrationTime(), other.registrationTime())
                && Objects.equals(lastClaimTime(), other.lastClaimTime())
                && Objects.equals(lastHealthCheckTime(), other.lastHealthCheckTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GameServer").add("GameServerGroupName", gameServerGroupName())
                .add("GameServerGroupArn", gameServerGroupArn()).add("GameServerId", gameServerId())
                .add("InstanceId", instanceId()).add("ConnectionInfo", connectionInfo()).add("GameServerData", gameServerData())
                .add("ClaimStatus", claimStatusAsString()).add("UtilizationStatus", utilizationStatusAsString())
                .add("RegistrationTime", registrationTime()).add("LastClaimTime", lastClaimTime())
                .add("LastHealthCheckTime", lastHealthCheckTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GameServerGroupName":
            return Optional.ofNullable(clazz.cast(gameServerGroupName()));
        case "GameServerGroupArn":
            return Optional.ofNullable(clazz.cast(gameServerGroupArn()));
        case "GameServerId":
            return Optional.ofNullable(clazz.cast(gameServerId()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "ConnectionInfo":
            return Optional.ofNullable(clazz.cast(connectionInfo()));
        case "GameServerData":
            return Optional.ofNullable(clazz.cast(gameServerData()));
        case "ClaimStatus":
            return Optional.ofNullable(clazz.cast(claimStatusAsString()));
        case "UtilizationStatus":
            return Optional.ofNullable(clazz.cast(utilizationStatusAsString()));
        case "RegistrationTime":
            return Optional.ofNullable(clazz.cast(registrationTime()));
        case "LastClaimTime":
            return Optional.ofNullable(clazz.cast(lastClaimTime()));
        case "LastHealthCheckTime":
            return Optional.ofNullable(clazz.cast(lastHealthCheckTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameServer, T> g) {
        return obj -> g.apply((GameServer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameServer> {
        /**
         * <p>
         * A unique identifier for the game server group where the game server is running.
         * </p>
         * 
         * @param gameServerGroupName
         *        A unique identifier for the game server group where the game server is running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupName(String gameServerGroupName);

        /**
         * <p>
         * The ARN identifier for the game server group where the game server is located.
         * </p>
         * 
         * @param gameServerGroupArn
         *        The ARN identifier for the game server group where the game server is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupArn(String gameServerGroupArn);

        /**
         * <p>
         * A custom string that uniquely identifies the game server. Game server IDs are developer-defined and are
         * unique across all game server groups in an Amazon Web Services account.
         * </p>
         * 
         * @param gameServerId
         *        A custom string that uniquely identifies the game server. Game server IDs are developer-defined and
         *        are unique across all game server groups in an Amazon Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerId(String gameServerId);

        /**
         * <p>
         * The unique identifier for the instance where the game server is running. This ID is available in the instance
         * metadata. EC2 instance IDs use a 17-character format, for example: <code>i-1234567890abcdef0</code>.
         * </p>
         * 
         * @param instanceId
         *        The unique identifier for the instance where the game server is running. This ID is available in the
         *        instance metadata. EC2 instance IDs use a 17-character format, for example:
         *        <code>i-1234567890abcdef0</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The port and IP address that must be used to establish a client connection to the game server.
         * </p>
         * 
         * @param connectionInfo
         *        The port and IP address that must be used to establish a client connection to the game server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionInfo(String connectionInfo);

        /**
         * <p>
         * A set of custom game server properties, formatted as a single string value. This data is passed to a game
         * client or service when it requests information on game servers.
         * </p>
         * 
         * @param gameServerData
         *        A set of custom game server properties, formatted as a single string value. This data is passed to a
         *        game client or service when it requests information on game servers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerData(String gameServerData);

        /**
         * <p>
         * Indicates when an available game server has been reserved for gameplay but has not yet started hosting a
         * game. Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of one minute.
         * During this time, game clients connect to the game server to start the game and trigger the game server to
         * update its utilization status. After one minute, the game server claim status reverts to null.
         * </p>
         * 
         * @param claimStatus
         *        Indicates when an available game server has been reserved for gameplay but has not yet started hosting
         *        a game. Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of
         *        one minute. During this time, game clients connect to the game server to start the game and trigger
         *        the game server to update its utilization status. After one minute, the game server claim status
         *        reverts to null.
         * @see GameServerClaimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerClaimStatus
         */
        Builder claimStatus(String claimStatus);

        /**
         * <p>
         * Indicates when an available game server has been reserved for gameplay but has not yet started hosting a
         * game. Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of one minute.
         * During this time, game clients connect to the game server to start the game and trigger the game server to
         * update its utilization status. After one minute, the game server claim status reverts to null.
         * </p>
         * 
         * @param claimStatus
         *        Indicates when an available game server has been reserved for gameplay but has not yet started hosting
         *        a game. Once it is claimed, the game server remains in <code>CLAIMED</code> status for a maximum of
         *        one minute. During this time, game clients connect to the game server to start the game and trigger
         *        the game server to update its utilization status. After one minute, the game server claim status
         *        reverts to null.
         * @see GameServerClaimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerClaimStatus
         */
        Builder claimStatus(GameServerClaimStatus claimStatus);

        /**
         * <p>
         * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been claimed
         * remains in this status until it reports game hosting activity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UTILIZED</code> - The game server is currently hosting a game session with players.
         * </p>
         * </li>
         * </ul>
         * 
         * @param utilizationStatus
         *        Indicates whether the game server is currently available for new games or is busy. Possible statuses
         *        include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been
         *        claimed remains in this status until it reports game hosting activity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UTILIZED</code> - The game server is currently hosting a game session with players.
         *        </p>
         *        </li>
         * @see GameServerUtilizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerUtilizationStatus
         */
        Builder utilizationStatus(String utilizationStatus);

        /**
         * <p>
         * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been claimed
         * remains in this status until it reports game hosting activity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UTILIZED</code> - The game server is currently hosting a game session with players.
         * </p>
         * </li>
         * </ul>
         * 
         * @param utilizationStatus
         *        Indicates whether the game server is currently available for new games or is busy. Possible statuses
         *        include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The game server is available to be claimed. A game server that has been
         *        claimed remains in this status until it reports game hosting activity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UTILIZED</code> - The game server is currently hosting a game session with players.
         *        </p>
         *        </li>
         * @see GameServerUtilizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerUtilizationStatus
         */
        Builder utilizationStatus(GameServerUtilizationStatus utilizationStatus);

        /**
         * <p>
         * Timestamp that indicates when the game server registered. The format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param registrationTime
         *        Timestamp that indicates when the game server registered. The format is a number expressed in Unix
         *        time as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registrationTime(Instant registrationTime);

        /**
         * <p>
         * Timestamp that indicates the last time the game server was claimed. The format is a number expressed in Unix
         * time as milliseconds (for example <code>"1469498468.057"</code>). This value is used to calculate when a
         * claimed game server's status should revert to null.
         * </p>
         * 
         * @param lastClaimTime
         *        Timestamp that indicates the last time the game server was claimed. The format is a number expressed
         *        in Unix time as milliseconds (for example <code>"1469498468.057"</code>). This value is used to
         *        calculate when a claimed game server's status should revert to null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastClaimTime(Instant lastClaimTime);

        /**
         * <p>
         * Timestamp that indicates the last time the game server was updated with health status. The format is a number
         * expressed in Unix time as milliseconds (for example <code>"1469498468.057"</code>). After game server
         * registration, this property is only changed when a game server update specifies a health check value.
         * </p>
         * 
         * @param lastHealthCheckTime
         *        Timestamp that indicates the last time the game server was updated with health status. The format is a
         *        number expressed in Unix time as milliseconds (for example <code>"1469498468.057"</code>). After game
         *        server registration, this property is only changed when a game server update specifies a health check
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastHealthCheckTime(Instant lastHealthCheckTime);
    }

    static final class BuilderImpl implements Builder {
        private String gameServerGroupName;

        private String gameServerGroupArn;

        private String gameServerId;

        private String instanceId;

        private String connectionInfo;

        private String gameServerData;

        private String claimStatus;

        private String utilizationStatus;

        private Instant registrationTime;

        private Instant lastClaimTime;

        private Instant lastHealthCheckTime;

        private BuilderImpl() {
        }

        private BuilderImpl(GameServer model) {
            gameServerGroupName(model.gameServerGroupName);
            gameServerGroupArn(model.gameServerGroupArn);
            gameServerId(model.gameServerId);
            instanceId(model.instanceId);
            connectionInfo(model.connectionInfo);
            gameServerData(model.gameServerData);
            claimStatus(model.claimStatus);
            utilizationStatus(model.utilizationStatus);
            registrationTime(model.registrationTime);
            lastClaimTime(model.lastClaimTime);
            lastHealthCheckTime(model.lastHealthCheckTime);
        }

        public final String getGameServerGroupName() {
            return gameServerGroupName;
        }

        public final void setGameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
        }

        @Override
        public final Builder gameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
            return this;
        }

        public final String getGameServerGroupArn() {
            return gameServerGroupArn;
        }

        public final void setGameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
        }

        @Override
        public final Builder gameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
            return this;
        }

        public final String getGameServerId() {
            return gameServerId;
        }

        public final void setGameServerId(String gameServerId) {
            this.gameServerId = gameServerId;
        }

        @Override
        public final Builder gameServerId(String gameServerId) {
            this.gameServerId = gameServerId;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getConnectionInfo() {
            return connectionInfo;
        }

        public final void setConnectionInfo(String connectionInfo) {
            this.connectionInfo = connectionInfo;
        }

        @Override
        public final Builder connectionInfo(String connectionInfo) {
            this.connectionInfo = connectionInfo;
            return this;
        }

        public final String getGameServerData() {
            return gameServerData;
        }

        public final void setGameServerData(String gameServerData) {
            this.gameServerData = gameServerData;
        }

        @Override
        public final Builder gameServerData(String gameServerData) {
            this.gameServerData = gameServerData;
            return this;
        }

        public final String getClaimStatus() {
            return claimStatus;
        }

        public final void setClaimStatus(String claimStatus) {
            this.claimStatus = claimStatus;
        }

        @Override
        public final Builder claimStatus(String claimStatus) {
            this.claimStatus = claimStatus;
            return this;
        }

        @Override
        public final Builder claimStatus(GameServerClaimStatus claimStatus) {
            this.claimStatus(claimStatus == null ? null : claimStatus.toString());
            return this;
        }

        public final String getUtilizationStatus() {
            return utilizationStatus;
        }

        public final void setUtilizationStatus(String utilizationStatus) {
            this.utilizationStatus = utilizationStatus;
        }

        @Override
        public final Builder utilizationStatus(String utilizationStatus) {
            this.utilizationStatus = utilizationStatus;
            return this;
        }

        @Override
        public final Builder utilizationStatus(GameServerUtilizationStatus utilizationStatus) {
            this.utilizationStatus(utilizationStatus == null ? null : utilizationStatus.toString());
            return this;
        }

        public final Instant getRegistrationTime() {
            return registrationTime;
        }

        public final void setRegistrationTime(Instant registrationTime) {
            this.registrationTime = registrationTime;
        }

        @Override
        public final Builder registrationTime(Instant registrationTime) {
            this.registrationTime = registrationTime;
            return this;
        }

        public final Instant getLastClaimTime() {
            return lastClaimTime;
        }

        public final void setLastClaimTime(Instant lastClaimTime) {
            this.lastClaimTime = lastClaimTime;
        }

        @Override
        public final Builder lastClaimTime(Instant lastClaimTime) {
            this.lastClaimTime = lastClaimTime;
            return this;
        }

        public final Instant getLastHealthCheckTime() {
            return lastHealthCheckTime;
        }

        public final void setLastHealthCheckTime(Instant lastHealthCheckTime) {
            this.lastHealthCheckTime = lastHealthCheckTime;
        }

        @Override
        public final Builder lastHealthCheckTime(Instant lastHealthCheckTime) {
            this.lastHealthCheckTime = lastHealthCheckTime;
            return this;
        }

        @Override
        public GameServer build() {
            return new GameServer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
