/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Properties describing a custom game build.
 * </p>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-awssdk.html#reference-awssdk-resources-fleets"
 * >All APIs by task</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Build implements SdkPojo, Serializable, ToCopyableBuilder<Build.Builder, Build> {
    private static final SdkField<String> BUILD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BuildId").getter(getter(Build::buildId)).setter(setter(Builder::buildId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BuildId").build()).build();

    private static final SdkField<String> BUILD_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BuildArn").getter(getter(Build::buildArn)).setter(setter(Builder::buildArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BuildArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Build::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(Build::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Build::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Long> SIZE_ON_DISK_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("SizeOnDisk").getter(getter(Build::sizeOnDisk)).setter(setter(Builder::sizeOnDisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SizeOnDisk").build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystem").getter(getter(Build::operatingSystemAsString))
            .setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Build::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> SERVER_SDK_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerSdkVersion").getter(getter(Build::serverSdkVersion)).setter(setter(Builder::serverSdkVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerSdkVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUILD_ID_FIELD,
            BUILD_ARN_FIELD, NAME_FIELD, VERSION_FIELD, STATUS_FIELD, SIZE_ON_DISK_FIELD, OPERATING_SYSTEM_FIELD,
            CREATION_TIME_FIELD, SERVER_SDK_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String buildId;

    private final String buildArn;

    private final String name;

    private final String version;

    private final String status;

    private final Long sizeOnDisk;

    private final String operatingSystem;

    private final Instant creationTime;

    private final String serverSdkVersion;

    private Build(BuilderImpl builder) {
        this.buildId = builder.buildId;
        this.buildArn = builder.buildArn;
        this.name = builder.name;
        this.version = builder.version;
        this.status = builder.status;
        this.sizeOnDisk = builder.sizeOnDisk;
        this.operatingSystem = builder.operatingSystem;
        this.creationTime = builder.creationTime;
        this.serverSdkVersion = builder.serverSdkVersion;
    }

    /**
     * <p>
     * A unique identifier for the build.
     * </p>
     * 
     * @return A unique identifier for the build.
     */
    public final String buildId() {
        return buildId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * assigned to a GameLift build resource and uniquely identifies it. ARNs are unique across all Regions. Format is
     * <code>arn:aws:gamelift:&lt;region&gt;::build/build-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a GameLift
     * build ARN, the resource ID matches the <i>BuildId</i> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) assigned to a GameLift
     *         build resource and uniquely identifies it. ARNs are unique across all Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::build/build-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
     *         GameLift build ARN, the resource ID matches the <i>BuildId</i> value.
     */
    public final String buildArn() {
        return buildArn;
    }

    /**
     * <p>
     * A descriptive label associated with a build. Build names do not need to be unique. It can be set using <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html">CreateBuild</a> or <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/UpdateBuild">UpdateBuild</a>.
     * </p>
     * 
     * @return A descriptive label associated with a build. Build names do not need to be unique. It can be set using <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html">CreateBuild</a> or
     *         <a href="https://docs.aws.amazon.com/gamelift/latest/apireference/UpdateBuild">UpdateBuild</a>.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Version information associated with a build or script. Version strings do not need to be unique.
     * </p>
     * 
     * @return Version information associated with a build or script. Version strings do not need to be unique.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * Current status of the build.
     * </p>
     * <p>
     * Possible build statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create fleets for
     * builds that are in this status. When a build is successfully created, the build status is set to this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this build.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BuildStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the build.</p>
     *         <p>
     *         Possible build statuses include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create
     *         fleets for builds that are in this status. When a build is successfully created, the build status is set
     *         to this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this
     *         build.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
     *         </p>
     *         </li>
     * @see BuildStatus
     */
    public final BuildStatus status() {
        return BuildStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the build.
     * </p>
     * <p>
     * Possible build statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create fleets for
     * builds that are in this status. When a build is successfully created, the build status is set to this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this build.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link BuildStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the build.</p>
     *         <p>
     *         Possible build statuses include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create
     *         fleets for builds that are in this status. When a build is successfully created, the build status is set
     *         to this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this
     *         build.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
     *         </p>
     *         </li>
     * @see BuildStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * File size of the uploaded game build, expressed in bytes. When the build status is <code>INITIALIZED</code> or
     * when using a custom Amazon S3 storage location, this value is 0.
     * </p>
     * 
     * @return File size of the uploaded game build, expressed in bytes. When the build status is
     *         <code>INITIALIZED</code> or when using a custom Amazon S3 storage location, this value is 0.
     */
    public final Long sizeOnDisk() {
        return sizeOnDisk;
    }

    /**
     * <p>
     * Operating system that the game server binaries are built to run on. This value determines the type of fleet
     * resources that you can use for this build.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that the game server binaries are built to run on. This value determines the type of
     *         fleet resources that you can use for this build.
     * @see OperatingSystem
     */
    public final OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * Operating system that the game server binaries are built to run on. This value determines the type of fleet
     * resources that you can use for this build.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that the game server binaries are built to run on. This value determines the type of
     *         fleet resources that you can use for this build.
     * @see OperatingSystem
     */
    public final String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The GameLift Server SDK version used to develop your game server.
     * </p>
     * 
     * @return The GameLift Server SDK version used to develop your game server.
     */
    public final String serverSdkVersion() {
        return serverSdkVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(buildId());
        hashCode = 31 * hashCode + Objects.hashCode(buildArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sizeOnDisk());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(serverSdkVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Build)) {
            return false;
        }
        Build other = (Build) obj;
        return Objects.equals(buildId(), other.buildId()) && Objects.equals(buildArn(), other.buildArn())
                && Objects.equals(name(), other.name()) && Objects.equals(version(), other.version())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(sizeOnDisk(), other.sizeOnDisk())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(serverSdkVersion(), other.serverSdkVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Build").add("BuildId", buildId()).add("BuildArn", buildArn()).add("Name", name())
                .add("Version", version()).add("Status", statusAsString()).add("SizeOnDisk", sizeOnDisk())
                .add("OperatingSystem", operatingSystemAsString()).add("CreationTime", creationTime())
                .add("ServerSdkVersion", serverSdkVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BuildId":
            return Optional.ofNullable(clazz.cast(buildId()));
        case "BuildArn":
            return Optional.ofNullable(clazz.cast(buildArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "SizeOnDisk":
            return Optional.ofNullable(clazz.cast(sizeOnDisk()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "ServerSdkVersion":
            return Optional.ofNullable(clazz.cast(serverSdkVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Build, T> g) {
        return obj -> g.apply((Build) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Build> {
        /**
         * <p>
         * A unique identifier for the build.
         * </p>
         * 
         * @param buildId
         *        A unique identifier for the build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildId(String buildId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) assigned to a GameLift
         * build resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::build/build-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a GameLift
         * build ARN, the resource ID matches the <i>BuildId</i> value.
         * </p>
         * 
         * @param buildArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) assigned to a
         *        GameLift build resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::build/build-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
         *        GameLift build ARN, the resource ID matches the <i>BuildId</i> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildArn(String buildArn);

        /**
         * <p>
         * A descriptive label associated with a build. Build names do not need to be unique. It can be set using <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html">CreateBuild</a> or <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/UpdateBuild">UpdateBuild</a>.
         * </p>
         * 
         * @param name
         *        A descriptive label associated with a build. Build names do not need to be unique. It can be set using
         *        <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateBuild.html">CreateBuild</a>
         *        or <a href="https://docs.aws.amazon.com/gamelift/latest/apireference/UpdateBuild">UpdateBuild</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Version information associated with a build or script. Version strings do not need to be unique.
         * </p>
         * 
         * @param version
         *        Version information associated with a build or script. Version strings do not need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * Current status of the build.
         * </p>
         * <p>
         * Possible build statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create fleets
         * for builds that are in this status. When a build is successfully created, the build status is set to this
         * value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this build.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the build.</p>
         *        <p>
         *        Possible build statuses include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create
         *        fleets for builds that are in this status. When a build is successfully created, the build status is
         *        set to this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this
         *        build.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
         *        </p>
         *        </li>
         * @see BuildStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BuildStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the build.
         * </p>
         * <p>
         * Possible build statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create fleets
         * for builds that are in this status. When a build is successfully created, the build status is set to this
         * value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this build.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the build.</p>
         *        <p>
         *        Possible build statuses include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>INITIALIZED</b> -- A new build has been defined, but no files have been uploaded. You cannot create
         *        fleets for builds that are in this status. When a build is successfully created, the build status is
         *        set to this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>READY</b> -- The game build has been successfully uploaded. You can now create new fleets for this
         *        build.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED</b> -- The game build upload failed. You cannot create new fleets for this build.
         *        </p>
         *        </li>
         * @see BuildStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BuildStatus
         */
        Builder status(BuildStatus status);

        /**
         * <p>
         * File size of the uploaded game build, expressed in bytes. When the build status is <code>INITIALIZED</code>
         * or when using a custom Amazon S3 storage location, this value is 0.
         * </p>
         * 
         * @param sizeOnDisk
         *        File size of the uploaded game build, expressed in bytes. When the build status is
         *        <code>INITIALIZED</code> or when using a custom Amazon S3 storage location, this value is 0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeOnDisk(Long sizeOnDisk);

        /**
         * <p>
         * Operating system that the game server binaries are built to run on. This value determines the type of fleet
         * resources that you can use for this build.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that the game server binaries are built to run on. This value determines the type of
         *        fleet resources that you can use for this build.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * Operating system that the game server binaries are built to run on. This value determines the type of fleet
         * resources that you can use for this build.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that the game server binaries are built to run on. This value determines the type of
         *        fleet resources that you can use for this build.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The GameLift Server SDK version used to develop your game server.
         * </p>
         * 
         * @param serverSdkVersion
         *        The GameLift Server SDK version used to develop your game server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverSdkVersion(String serverSdkVersion);
    }

    static final class BuilderImpl implements Builder {
        private String buildId;

        private String buildArn;

        private String name;

        private String version;

        private String status;

        private Long sizeOnDisk;

        private String operatingSystem;

        private Instant creationTime;

        private String serverSdkVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(Build model) {
            buildId(model.buildId);
            buildArn(model.buildArn);
            name(model.name);
            version(model.version);
            status(model.status);
            sizeOnDisk(model.sizeOnDisk);
            operatingSystem(model.operatingSystem);
            creationTime(model.creationTime);
            serverSdkVersion(model.serverSdkVersion);
        }

        public final String getBuildId() {
            return buildId;
        }

        public final void setBuildId(String buildId) {
            this.buildId = buildId;
        }

        @Override
        public final Builder buildId(String buildId) {
            this.buildId = buildId;
            return this;
        }

        public final String getBuildArn() {
            return buildArn;
        }

        public final void setBuildArn(String buildArn) {
            this.buildArn = buildArn;
        }

        @Override
        public final Builder buildArn(String buildArn) {
            this.buildArn = buildArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(BuildStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Long getSizeOnDisk() {
            return sizeOnDisk;
        }

        public final void setSizeOnDisk(Long sizeOnDisk) {
            this.sizeOnDisk = sizeOnDisk;
        }

        @Override
        public final Builder sizeOnDisk(Long sizeOnDisk) {
            this.sizeOnDisk = sizeOnDisk;
            return this;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getServerSdkVersion() {
            return serverSdkVersion;
        }

        public final void setServerSdkVersion(String serverSdkVersion) {
            this.serverSdkVersion = serverSdkVersion;
        }

        @Override
        public final Builder serverSdkVersion(String serverSdkVersion) {
            this.serverSdkVersion = serverSdkVersion;
            return this;
        }

        @Override
        public Build build() {
            return new Build(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
