/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a location in a multi-location fleet.
 * </p>
 * <p>
 * <b>Related actions</b>
 * </p>
 * <p>
 * <a>DescribeFleetLocationAttributes</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LocationAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<LocationAttributes.Builder, LocationAttributes> {
    private static final SdkField<LocationState> LOCATION_STATE_FIELD = SdkField
            .<LocationState> builder(MarshallingType.SDK_POJO).memberName("LocationState")
            .getter(getter(LocationAttributes::locationState)).setter(setter(Builder::locationState))
            .constructor(LocationState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationState").build()).build();

    private static final SdkField<List<String>> STOPPED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("StoppedActions")
            .getter(getter(LocationAttributes::stoppedActionsAsStrings))
            .setter(setter(Builder::stoppedActionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StoppedActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> UPDATE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpdateStatus").getter(getter(LocationAttributes::updateStatusAsString))
            .setter(setter(Builder::updateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCATION_STATE_FIELD,
            STOPPED_ACTIONS_FIELD, UPDATE_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final LocationState locationState;

    private final List<String> stoppedActions;

    private final String updateStatus;

    private LocationAttributes(BuilderImpl builder) {
        this.locationState = builder.locationState;
        this.stoppedActions = builder.stoppedActions;
        this.updateStatus = builder.updateStatus;
    }

    /**
     * <p>
     * A fleet location and its current life-cycle state.
     * </p>
     * 
     * @return A fleet location and its current life-cycle state.
     */
    public final LocationState locationState() {
        return locationState;
    }

    /**
     * <p>
     * A list of fleet actions that have been suspended in the fleet location.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStoppedActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of fleet actions that have been suspended in the fleet location.
     */
    public final List<FleetAction> stoppedActions() {
        return FleetActionListCopier.copyStringToEnum(stoppedActions);
    }

    /**
     * Returns true if the StoppedActions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasStoppedActions() {
        return stoppedActions != null && !(stoppedActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of fleet actions that have been suspended in the fleet location.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStoppedActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of fleet actions that have been suspended in the fleet location.
     */
    public final List<String> stoppedActionsAsStrings() {
        return stoppedActions;
    }

    /**
     * <p>
     * The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates that
     * <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet been completed
     * for the location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link LocationUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #updateStatusAsString}.
     * </p>
     * 
     * @return The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates
     *         that <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet
     *         been completed for the location.
     * @see LocationUpdateStatus
     */
    public final LocationUpdateStatus updateStatus() {
        return LocationUpdateStatus.fromValue(updateStatus);
    }

    /**
     * <p>
     * The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates that
     * <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet been completed
     * for the location.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link LocationUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #updateStatusAsString}.
     * </p>
     * 
     * @return The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates
     *         that <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet
     *         been completed for the location.
     * @see LocationUpdateStatus
     */
    public final String updateStatusAsString() {
        return updateStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(locationState());
        hashCode = 31 * hashCode + Objects.hashCode(hasStoppedActions() ? stoppedActionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(updateStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LocationAttributes)) {
            return false;
        }
        LocationAttributes other = (LocationAttributes) obj;
        return Objects.equals(locationState(), other.locationState()) && hasStoppedActions() == other.hasStoppedActions()
                && Objects.equals(stoppedActionsAsStrings(), other.stoppedActionsAsStrings())
                && Objects.equals(updateStatusAsString(), other.updateStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LocationAttributes").add("LocationState", locationState())
                .add("StoppedActions", hasStoppedActions() ? stoppedActionsAsStrings() : null)
                .add("UpdateStatus", updateStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LocationState":
            return Optional.ofNullable(clazz.cast(locationState()));
        case "StoppedActions":
            return Optional.ofNullable(clazz.cast(stoppedActionsAsStrings()));
        case "UpdateStatus":
            return Optional.ofNullable(clazz.cast(updateStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LocationAttributes, T> g) {
        return obj -> g.apply((LocationAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LocationAttributes> {
        /**
         * <p>
         * A fleet location and its current life-cycle state.
         * </p>
         * 
         * @param locationState
         *        A fleet location and its current life-cycle state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationState(LocationState locationState);

        /**
         * <p>
         * A fleet location and its current life-cycle state.
         * </p>
         * This is a convenience that creates an instance of the {@link LocationState.Builder} avoiding the need to
         * create one manually via {@link LocationState#builder()}.
         *
         * When the {@link Consumer} completes, {@link LocationState.Builder#build()} is called immediately and its
         * result is passed to {@link #locationState(LocationState)}.
         * 
         * @param locationState
         *        a consumer that will call methods on {@link LocationState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #locationState(LocationState)
         */
        default Builder locationState(Consumer<LocationState.Builder> locationState) {
            return locationState(LocationState.builder().applyMutation(locationState).build());
        }

        /**
         * <p>
         * A list of fleet actions that have been suspended in the fleet location.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet actions that have been suspended in the fleet location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActionsWithStrings(Collection<String> stoppedActions);

        /**
         * <p>
         * A list of fleet actions that have been suspended in the fleet location.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet actions that have been suspended in the fleet location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActionsWithStrings(String... stoppedActions);

        /**
         * <p>
         * A list of fleet actions that have been suspended in the fleet location.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet actions that have been suspended in the fleet location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActions(Collection<FleetAction> stoppedActions);

        /**
         * <p>
         * A list of fleet actions that have been suspended in the fleet location.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet actions that have been suspended in the fleet location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActions(FleetAction... stoppedActions);

        /**
         * <p>
         * The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates that
         * <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet been
         * completed for the location.
         * </p>
         * 
         * @param updateStatus
         *        The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates
         *        that <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet
         *        been completed for the location.
         * @see LocationUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocationUpdateStatus
         */
        Builder updateStatus(String updateStatus);

        /**
         * <p>
         * The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates that
         * <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet been
         * completed for the location.
         * </p>
         * 
         * @param updateStatus
         *        The status of fleet activity updates to the location. The status <code>PENDING_UPDATE</code> indicates
         *        that <a>StopFleetActions</a> or <a>StartFleetActions</a> has been requested but the update has not yet
         *        been completed for the location.
         * @see LocationUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocationUpdateStatus
         */
        Builder updateStatus(LocationUpdateStatus updateStatus);
    }

    static final class BuilderImpl implements Builder {
        private LocationState locationState;

        private List<String> stoppedActions = DefaultSdkAutoConstructList.getInstance();

        private String updateStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(LocationAttributes model) {
            locationState(model.locationState);
            stoppedActionsWithStrings(model.stoppedActions);
            updateStatus(model.updateStatus);
        }

        public final LocationState.Builder getLocationState() {
            return locationState != null ? locationState.toBuilder() : null;
        }

        @Override
        public final Builder locationState(LocationState locationState) {
            this.locationState = locationState;
            return this;
        }

        public final void setLocationState(LocationState.BuilderImpl locationState) {
            this.locationState = locationState != null ? locationState.build() : null;
        }

        public final Collection<String> getStoppedActions() {
            if (stoppedActions instanceof SdkAutoConstructList) {
                return null;
            }
            return stoppedActions;
        }

        @Override
        public final Builder stoppedActionsWithStrings(Collection<String> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copy(stoppedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stoppedActionsWithStrings(String... stoppedActions) {
            stoppedActionsWithStrings(Arrays.asList(stoppedActions));
            return this;
        }

        @Override
        public final Builder stoppedActions(Collection<FleetAction> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copyEnumToString(stoppedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stoppedActions(FleetAction... stoppedActions) {
            stoppedActions(Arrays.asList(stoppedActions));
            return this;
        }

        public final void setStoppedActions(Collection<String> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copy(stoppedActions);
        }

        public final String getUpdateStatus() {
            return updateStatus;
        }

        @Override
        public final Builder updateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
            return this;
        }

        @Override
        public final Builder updateStatus(LocationUpdateStatus updateStatus) {
            this.updateStatus(updateStatus == null ? null : updateStatus.toString());
            return this;
        }

        public final void setUpdateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
        }

        @Override
        public LocationAttributes build() {
            return new LocationAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
